// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef NET_ANDROID_KEYSTORE_H
#define NET_ANDROID_KEYSTORE_H

#include <jni.h>

#include <string>
#include <vector>

#include "base/basictypes.h"
#include "base/strings/string_piece.h"
#include "net/base/net_export.h"
#include "net/ssl/ssl_client_cert_type.h"

// Avoid including <openssl/evp.h> here.
typedef struct evp_pkey_st EVP_PKEY;

// Misc functions to access the Android platform KeyStore.

namespace net {
namespace android {

// Define a list of constants describing private key types. The
// values are shared with Java through org.chromium.net.PrivateKeyType.
// Example: PRIVATE_KEY_TYPE_RSA.
enum PrivateKeyType {
#define DEFINE_PRIVATE_KEY_TYPE(name,value)  PRIVATE_KEY_TYPE_ ## name = value,
#include "net/android/private_key_type_list.h"
#undef DEFINE_PRIVATE_KEY_TYPE
};

// Returns the modulus of a given RSAPrivateKey platform object,
// as a series of bytes, in big-endian representation. This can be
// used with BN_bin2bn() to convert to an OpenSSL BIGNUM.
//
// |private_key| is a JNI reference for the private key.
// |modulus| will receive the modulus bytes on success.
// Returns true on success, or false on failure (e.g. if the key
// is not RSA).
NET_EXPORT bool GetRSAKeyModulus(jobject private_key,
                                 std::vector<uint8>* modulus);

// Returns the Q parameter of a given DSAPrivateKey platform object,
// as a series of bytes, in big-endian representation. This can be used
// with BN_bin2bn() to convert to an OpenSSL BIGNUM.
// |private_key| is a JNI reference for the private key.
// |q| will receive the result bytes on success.
// Returns true on success, or false on failure (e.g. if the key is
// not DSA).
NET_EXPORT bool GetDSAKeyParamQ(jobject private_key,
                                std::vector<uint8>* q);

// Returns the order parameter of a given ECPrivateKey platform object,
// as a series of bytes, in big-endian representation. This can be used
// with BN_bin2bn() to convert to an OpenSSL BIGNUM.
// |private_key| is a JNI reference for the private key.
// |order| will receive the result bytes on success.
// Returns true on success, or false on failure (e.g. if the key is
// not EC).
bool GetECKeyOrder(jobject private_key,
                   std::vector<uint8>* order);

// Returns the encoded PKCS#8 representation of a private key.
// This only works on Android 4.0.3 and older releases for platform keys
// (i.e. all keys except those explicitely generated by the application).
// |private_key| is a JNI reference for the private key.
// |encoded| will receive the encoded data on success.
// Returns true on success, or false on failure (e.g. on 4.0.4 or higher).
bool GetPrivateKeyEncodedBytes(jobject private_key,
                               std::vector<uint8>* encoded);

// Compute the signature of a given message, which is actually a hash,
// using a private key. For more details, please read the comments for the
// rawSignDigestWithPrivateKey method in AndroidKeyStore.java.
//
// |private_key| is a JNI reference for the private key.
// |digest| is the input digest.
// |signature| will receive the signature on success.
// Returns true on success, false on failure.
//
NET_EXPORT bool RawSignDigestWithPrivateKey(
    jobject private_key,
    const base::StringPiece& digest,
    std::vector<uint8>* signature);


// Return the PrivateKeyType of a given private key.
// |private_key| is a JNI reference for the private key.
// Returns a PrivateKeyType, while will be CLIENT_CERT_INVALID_TYPE
// on error.
NET_EXPORT PrivateKeyType GetPrivateKeyType(jobject private_key);

// Returns a handle to the system EVP_PKEY object used to back a given
// private_key object. This must *only* be used for RSA private keys
// on Android < 4.2. Technically, this is only guaranteed to work if
// the system image contains a vanilla implementation of the Java
// API frameworks based on Harmony + OpenSSL.
//
// |private_key| is a JNI reference for the private key.
// Returns an EVP_PKEY* handle, or NULL in case of error.
//
// Note: Despite its name and return type, this function doesn't know
//       anything about OpenSSL, it just type-casts a system pointer that
//       is passed as an int through JNI. As such, it never increments
//       the returned key's reference count.
EVP_PKEY* GetOpenSSLSystemHandleForPrivateKey(jobject private_key);

// Register JNI methods
NET_EXPORT bool RegisterKeyStore(JNIEnv* env);

}  // namespace android
}  // namespace net

#endif  // NET_ANDROID_KEYSTORE_H
