// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef PPAPI_THUNK_RESOURCE_CREATION_API_H_
#define PPAPI_THUNK_RESOURCE_CREATION_API_H_

#include "ppapi/c/dev/pp_video_dev.h"
#include "ppapi/c/dev/ppb_file_chooser_dev.h"
#include "ppapi/c/dev/ppb_truetype_font_dev.h"
#include "ppapi/c/pp_bool.h"
#include "ppapi/c/pp_instance.h"
#include "ppapi/c/pp_resource.h"
#include "ppapi/c/ppb_audio.h"
#include "ppapi/c/ppb_audio_config.h"
#include "ppapi/c/ppb_file_system.h"
#include "ppapi/c/ppb_graphics_3d.h"
#include "ppapi/c/ppb_image_data.h"
#include "ppapi/c/ppb_input_event.h"
#include "ppapi/c/ppb_network_monitor.h"
#include "ppapi/c/ppb_websocket.h"
#include "ppapi/c/private/pp_private_font_charset.h"
#include "ppapi/shared_impl/api_id.h"
#include "ppapi/shared_impl/ppb_image_data_shared.h"

struct PP_Flash_Menu;
struct PP_FontDescription_Dev;
struct PP_BrowserFont_Trusted_Description;
struct PP_NetAddress_IPv4;
struct PP_NetAddress_IPv6;
struct PP_NetAddress_Private;
struct PP_Size;

namespace ppapi {

struct FileRefCreateInfo;
struct URLRequestInfoData;
struct URLResponseInfoData;

namespace thunk {

// A functional API for creating resource types. Separating out the creation
// functions here allows us to implement most resources as a pure "resource
// API", meaning all calls are routed on a per-resource-object basis. The
// creation functions are not per-object (since there's no object during
// creation) so need functional routing based on the instance ID.
class ResourceCreationAPI {
 public:
  virtual ~ResourceCreationAPI() {}

  virtual PP_Resource CreateFileIO(PP_Instance instance) = 0;
  virtual PP_Resource CreateFileRef(
      PP_Instance instance,
      const FileRefCreateInfo& serialized) = 0;
  virtual PP_Resource CreateFileSystem(PP_Instance instance,
                                       PP_FileSystemType type) = 0;
  virtual PP_Resource CreateIMEInputEvent(PP_Instance instance,
                                          PP_InputEvent_Type type,
                                          PP_TimeTicks time_stamp,
                                          struct PP_Var text,
                                          uint32_t segment_number,
                                          const uint32_t* segment_offsets,
                                          int32_t target_segment,
                                          uint32_t selection_start,
                                          uint32_t selection_end) = 0;
  virtual PP_Resource CreateKeyboardInputEvent(
      PP_Instance instance,
      PP_InputEvent_Type type,
      PP_TimeTicks time_stamp,
      uint32_t modifiers,
      uint32_t key_code,
      struct PP_Var character_text) = 0;
  virtual PP_Resource CreateMouseInputEvent(
      PP_Instance instance,
      PP_InputEvent_Type type,
      PP_TimeTicks time_stamp,
      uint32_t modifiers,
      PP_InputEvent_MouseButton mouse_button,
      const PP_Point* mouse_position,
      int32_t click_count,
      const PP_Point* mouse_movement) = 0;
  virtual PP_Resource CreateTouchInputEvent(
      PP_Instance instance,
      PP_InputEvent_Type type,
      PP_TimeTicks time_stamp,
      uint32_t modifiers) = 0;
  virtual PP_Resource CreateResourceArray(PP_Instance instance,
                                          const PP_Resource elements[],
                                          uint32_t size) = 0;
  virtual PP_Resource CreateTrueTypeFont(
      PP_Instance instance,
      const PP_TrueTypeFontDesc_Dev* desc) = 0;
  virtual PP_Resource CreateURLLoader(PP_Instance instance) = 0;
  virtual PP_Resource CreateURLRequestInfo(
      PP_Instance instance) = 0;

  virtual PP_Resource CreateWheelInputEvent(
      PP_Instance instance,
      PP_TimeTicks time_stamp,
      uint32_t modifiers,
      const PP_FloatPoint* wheel_delta,
      const PP_FloatPoint* wheel_ticks,
      PP_Bool scroll_by_page) = 0;

  virtual PP_Resource CreateAudio1_0(PP_Instance instance,
                                     PP_Resource config_id,
                                     PPB_Audio_Callback_1_0 audio_callback,
                                     void* user_data) = 0;
  virtual PP_Resource CreateAudio(PP_Instance instance,
                                  PP_Resource config_id,
                                  PPB_Audio_Callback audio_callback,
                                  void* user_data) = 0;
  virtual PP_Resource CreateAudioTrusted(PP_Instance instance) = 0;
  virtual PP_Resource CreateAudioConfig(PP_Instance instance,
                                        PP_AudioSampleRate sample_rate,
                                        uint32_t sample_frame_count) = 0;
  virtual PP_Resource CreateFileChooser(PP_Instance instance,
                                        PP_FileChooserMode_Dev mode,
                                        const PP_Var& accept_types) = 0;
  virtual PP_Resource CreateGraphics2D(PP_Instance instance,
                                       const PP_Size* size,
                                       PP_Bool is_always_opaque) = 0;
  virtual PP_Resource CreateGraphics3D(PP_Instance instance,
                                       PP_Resource share_context,
                                       const int32_t* attrib_list) = 0;
  virtual PP_Resource CreateGraphics3DRaw(PP_Instance instance,
                                          PP_Resource share_context,
                                          const int32_t* attrib_list) = 0;
  virtual PP_Resource CreateHostResolver(PP_Instance instance) = 0;
  virtual PP_Resource CreateHostResolverPrivate(PP_Instance instance) = 0;
  virtual PP_Resource CreateImageData(PP_Instance instance,
                                      PP_ImageDataFormat format,
                                      const PP_Size* size,
                                      PP_Bool init_to_zero) = 0;
  virtual PP_Resource CreateImageDataSimple(PP_Instance instance,
                                            PP_ImageDataFormat format,
                                            const PP_Size* size,
                                            PP_Bool init_to_zero) = 0;
  virtual PP_Resource CreateNetAddressFromIPv4Address(
      PP_Instance instance,
      const PP_NetAddress_IPv4* ipv4_addr) = 0;
  virtual PP_Resource CreateNetAddressFromIPv6Address(
      PP_Instance instance,
      const PP_NetAddress_IPv6* ipv6_addr) = 0;
  virtual PP_Resource CreateNetAddressFromNetAddressPrivate(
      PP_Instance instance,
      const PP_NetAddress_Private& private_addr) = 0;
  virtual PP_Resource CreateNetworkMonitor(PP_Instance instance) = 0;
  virtual PP_Resource CreatePrinting(PP_Instance instance) = 0;
  virtual PP_Resource CreateTCPServerSocketPrivate(PP_Instance instance) = 0;
  virtual PP_Resource CreateTCPSocket1_0(PP_Instance instace) = 0;
  virtual PP_Resource CreateTCPSocket(PP_Instance instance) = 0;
  virtual PP_Resource CreateTCPSocketPrivate(PP_Instance instace) = 0;
  virtual PP_Resource CreateUDPSocket(PP_Instance instace) = 0;
  virtual PP_Resource CreateUDPSocketPrivate(PP_Instance instace) = 0;
  virtual PP_Resource CreateVideoDestination(PP_Instance instance) = 0;
  virtual PP_Resource CreateVideoSource(PP_Instance instance) = 0;
  virtual PP_Resource CreateWebSocket(PP_Instance instance) = 0;
  virtual PP_Resource CreateX509CertificatePrivate(PP_Instance instance) = 0;
#if !defined(OS_NACL)
  virtual PP_Resource CreateAudioInput(PP_Instance instance) = 0;
  virtual PP_Resource CreateBroker(PP_Instance instance) = 0;
  virtual PP_Resource CreateBrowserFont(
      PP_Instance instance,
      const PP_BrowserFont_Trusted_Description* description) = 0;
  virtual PP_Resource CreateBuffer(PP_Instance instance, uint32_t size) = 0;
  virtual PP_Resource CreateFlashDRM(PP_Instance instance) = 0;
  virtual PP_Resource CreateFlashFontFile(
      PP_Instance instance,
      const PP_BrowserFont_Trusted_Description* description,
      PP_PrivateFontCharset charset) = 0;
  virtual PP_Resource CreateFlashMenu(PP_Instance instance,
                                      const PP_Flash_Menu* menu_data) = 0;
  virtual PP_Resource CreateFlashMessageLoop(PP_Instance instance) = 0;
  virtual PP_Resource CreatePlatformVerificationPrivate(
      PP_Instance instance) = 0;
  virtual PP_Resource CreateScrollbar(PP_Instance instance,
                                      PP_Bool vertical) = 0;
  virtual PP_Resource CreateTalk(PP_Instance instance) = 0;
  virtual PP_Resource CreateVideoCapture(PP_Instance instance) = 0;
  virtual PP_Resource CreateVideoDecoder(
      PP_Instance instance,
      PP_Resource context3d_id,
      PP_VideoDecoder_Profile profile) = 0;
#endif  // !defined(OS_NACL)

  static const ApiID kApiID = API_ID_RESOURCE_CREATION;
};

}  // namespace thunk
}  // namespace ppapi

#endif  // PPAPI_THUNK_RESOURCE_CREATION_API_H_
