// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/base/ime/mock_input_method.h"

namespace ui {

MockInputMethod::MockInputMethod(internal::InputMethodDelegate* delegate)
    : text_input_client_(NULL),
      is_sticky_text_input_client_(false) {
}

MockInputMethod::~MockInputMethod() {
}

void MockInputMethod::SetDelegate(internal::InputMethodDelegate* delegate) {
}

void MockInputMethod::SetFocusedTextInputClient(TextInputClient* client) {
  if (is_sticky_text_input_client_)
    return;
  if (text_input_client_ == client)
    return;
  text_input_client_ = client;
  if (client)
    OnTextInputTypeChanged(client);
}

void MockInputMethod::SetStickyFocusedTextInputClient(TextInputClient* client) {
  is_sticky_text_input_client_ = (client != NULL);
  if (text_input_client_ == client)
    return;
  text_input_client_ = client;
  if (client)
    OnTextInputTypeChanged(client);
}

void MockInputMethod::DetachTextInputClient(TextInputClient* client) {
  if (text_input_client_ == client) {
    text_input_client_ = NULL;
    is_sticky_text_input_client_ = false;
  }
}

TextInputClient* MockInputMethod::GetTextInputClient() const {
  return text_input_client_;
}

bool MockInputMethod::DispatchKeyEvent(const base::NativeEvent& native_event) {
  return false;
}

bool MockInputMethod::DispatchFabricatedKeyEvent(const ui::KeyEvent& event) {
  return false;
}

void MockInputMethod::Init(bool focused) {
}

void MockInputMethod::OnFocus() {
  FOR_EACH_OBSERVER(Observer, observer_list_, OnFocus());
}

void MockInputMethod::OnBlur() {
  FOR_EACH_OBSERVER(Observer, observer_list_, OnBlur());
}

bool MockInputMethod::OnUntranslatedIMEMessage(const base::NativeEvent& event,
                                               NativeEventResult* result) {
  FOR_EACH_OBSERVER(Observer, observer_list_, OnUntranslatedIMEMessage(event));
  if (result)
    *result = NativeEventResult();
  return false;
}

void MockInputMethod::OnTextInputTypeChanged(const TextInputClient* client) {
  FOR_EACH_OBSERVER(Observer, observer_list_, OnTextInputTypeChanged(client));
  FOR_EACH_OBSERVER(Observer, observer_list_, OnTextInputStateChanged(client));
}

void MockInputMethod::OnCaretBoundsChanged(const TextInputClient* client) {
  FOR_EACH_OBSERVER(Observer, observer_list_, OnCaretBoundsChanged(client));
}

void MockInputMethod::CancelComposition(const TextInputClient* client) {
}

void MockInputMethod::OnInputLocaleChanged() {
  FOR_EACH_OBSERVER(Observer, observer_list_, OnInputLocaleChanged());
}

std::string MockInputMethod::GetInputLocale() {
  return "";
}

base::i18n::TextDirection MockInputMethod::GetInputTextDirection() {
  return base::i18n::UNKNOWN_DIRECTION;
}

bool MockInputMethod::IsActive() {
  return true;
}

TextInputType MockInputMethod::GetTextInputType() const {
  return TEXT_INPUT_TYPE_NONE;
}

TextInputMode MockInputMethod::GetTextInputMode() const {
  return TEXT_INPUT_MODE_DEFAULT;
}

bool MockInputMethod::CanComposeInline() const {
  return true;
}

bool MockInputMethod::IsCandidatePopupOpen() const {
  return false;
}

void MockInputMethod::AddObserver(InputMethodObserver* observer) {
  observer_list_.AddObserver(static_cast<Observer*>(observer));
}

void MockInputMethod::RemoveObserver(InputMethodObserver* observer) {
  observer_list_.RemoveObserver(static_cast<Observer*>(observer));
}

}  // namespace ui
