/****************************************************************************
**
** Copyright (C) 2024 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the release tools of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

var componentSelectionPage;

function Component()
{
    var ifwVersion = installer.value("FrameworkVersion");
    var isOpenSource = "true";
    if (['false', 'no', '0'].indexOf(isOpenSource) >= 0) {
        // set license type for Commercial installations
        installer.setValue("QT_LICENSE_TYPE", "opensource");
    }
    else {
        // set lgpl and fdl/3rd party license agreement dependencies by default for opensource installations
        var lgplLicenseDependency = component.value("Dependencies");
        var lgplLicenseDependencyStr = "qt.license.thirdparty, qt.license.lgpl";
        if (lgplLicenseDependency)
            component.setValue("Dependencies", lgplLicenseDependency + "," + lgplLicenseDependencyStr)
        else
            component.setValue("Dependencies", lgplLicenseDependencyStr)
    }

    if (installer.value("os") == "mac") {
        var installNameToolCheck = installer.execute("/usr/bin/xcrun", ["-f", "install_name_tool"])[1];
        if (installNameToolCheck !== 0) {
            QMessageBox["warning"]("installNameToolCheckError",
                    qsTr("No install_name_tool found!"),
                    qsTr("You need to install Xcode (https://developer.apple.com/xcode/). If you already have Xcode installed, you need to set the active developer directory with 'xcode-select'.\n"));
        }
        var otoolCheck = installer.execute("/usr/bin/xcrun", ["-f", "otool"])[1];
        if (otoolCheck !== 0) {
            QMessageBox["warning"]("otoolCheckError",
                    qsTr("No otool found!"),
                    qsTr("You need to install Xcode (https://developer.apple.com/xcode/). If you already have Xcode installed, you need to set the active developer directory with 'xcode-select'.\n"));
        }
    }

    if (installer.versionMatches(ifwVersion, ">=4.0.0") && !installer.isCommandLineInstance()) {
        componentSelectionPage = gui.pageWidgetByObjectName("ComponentSelectionPage")
        if (componentSelectionPage) {
            componentSelectionPage.entered.connect(Component.prototype.componentPageEntered);
            componentSelectionPage.left.connect(Component.prototype.componentPageLeft);

            // After we have fetched categories in component selection page, above entered() -signal is not fired
            // so we need to call directly this function
            Component.prototype.componentPageEntered();
        }
    }
}

Component.prototype.componentPageEntered = function()
{
    // We need this temporary value, because we don't leave the component selection page so the subtitle is not reset when
    // we fetch categories, but all the scripts are re-evaluated, so we don't have other means to set the original title value
    if (installer.value("temporaryQtComponentPageSubTitleValue").trim().length <= 0 &&  componentSelectionPage.subTitle.trim().length > 0)
        installer.setValue("temporaryQtComponentPageSubTitleValue", componentSelectionPage.subTitle)

    var checkBox = gui.findChild(componentSelectionPage, "Archive");
    if (checkBox)
        checkBox.clicked.connect(Component.prototype.setPageSubTitle);
}

Component.prototype.componentPageLeft = function()
{
    installer.setValue("temporaryQtComponentPageSubTitleValue", "");
}

Component.prototype.setPageSubTitle = function()
{
    var checkBox = gui.findChild(componentSelectionPage, "Archive");
    if (!checkBox)
        return;

    if (checkBox.checked)
        componentSelectionPage.subTitle = qsTr("<center><font color=\"red\"><b>This is an archive of historical versions, which may contain known errors and security vulnerabilities.</b></font></center>")
    else
        componentSelectionPage.subTitle = installer.value("temporaryQtComponentPageSubTitleValue")
}

qmakeOutputInstallerKey = function(aComponent)
{
    var qmakeOutputInstallerKey = aComponent.name;
    // if installing packages 5.0.2 - 5.2.x, in later releases the addons/essentials split is removed from packages
    if (qmakeOutputInstallerKey.match("essentials$") || qmakeOutputInstallerKey.match("addons$")) {
        // try to find the parent
        if (qmakeOutputInstallerKey.lastIndexOf(".") !== -1) {
            qmakeOutputInstallerKey = qmakeOutputInstallerKey.slice(0, qmakeOutputInstallerKey.lastIndexOf("."));
        }
    }
    qmakeOutputInstallerKey += "_qmakeoutput";
    return qmakeOutputInstallerKey;
}

addInitQtPatchOperation = function(aComponent, aOS, aQtPath, aQmakePath, version)
{
    var installerKey = qmakeOutputInstallerKey(aComponent)
    aComponent.addOperation("ConsumeOutput", installerKey, aQmakePath, "-query");
    aComponent.addOperation("QtPatch", aOS, aQtPath, version, "QmakeOutputInstallerKey=" + installerKey);

    var isOpenSource = "true";
    if (['false', 'no', '0'].indexOf(isOpenSource) >= 0) {
        // in enterprise version we need to patch license information into dedicated binary files
        aComponent.addOperation( "ApplyProductKey", aOS, aQtPath, version);
    }
}

registerQtCreatorDocumentation = function(aComponent, aPath)
{
    var ifwVersion = installer.value("FrameworkVersion");
    var settingsFile = "@QtCreatorInstallerSettingsFile@";
    if (installer.versionMatches(ifwVersion, "<4.5.1")) {
        // IFW version older than 4.5.1 doesn't support @...@ settings
        settingsFile = installer.value("QtCreatorInstallerSettingsFile");
        if (settingsFile == "")
            return;
    }
    aComponent.addOperation("Settings",
        "path=" + settingsFile,
        "method=add_array_value",
        "key=Help/InstalledDocumentation",
        "value=" + "@TargetDir@" + aPath);
}

registerQtCreatorExampleSet = function(aComponent, aDisplayName, aDocumentationPath, aExamplesPath)
{
    var ifwVersion = installer.value("FrameworkVersion");
    var settingsFile = "@QtCreatorInstallerSettingsFile@";
    if (installer.versionMatches(ifwVersion, "<4.5.1")) {
        // IFW version older than 4.5.1 doesn't support @...@ settings
        settingsFile = installer.value("QtCreatorInstallerSettingsFile");
        if (settingsFile == "")
            return;
    }
    aComponent.addOperation("Settings",
        "path=" + settingsFile,
        "method=add_array_value",
        "key=Help/InstalledExamples",
        "value=" + aDisplayName + "|" + "@TargetDir@" + aDocumentationPath + "|" + "@TargetDir@" + aExamplesPath);
}

/*****
* Global function to be used by Qt6 binary packages.
* This function will write qt.conf to the /bin and /libexec directories.
* @param aComponentRootPath     Full path to the Qt package (under this path the /bin should exist)
* @param aQtInstallationName    "Qt-x.x.x"
*/
createQtConfFiles = function(aComponent, aComponentRootPath, aQtInstallationName)
{
    print("Creating qt.conf files for: " + aComponentRootPath);
    var qtConfBinPath = aComponentRootPath + "/" + "bin" + "/" + "qt.conf";
    print("Writing qt.conf file to: " + qtConfBinPath);
    examplesPath = "../../Examples/" + aQtInstallationName;
    documentationPath = "../../Docs/" + aQtInstallationName;
    aComponent.addOperation("Settings",
        "path=" + qtConfBinPath,
        "method=set",
        "key=Paths/Prefix",
        "value=..");
    aComponent.addOperation("Settings",
        "path=" + qtConfBinPath,
        "method=set",
        "key=Paths/Examples",
        "value=" + examplesPath);
    aComponent.addOperation("Settings",
        "path=" + qtConfBinPath,
        "method=set",
        "key=Paths/Documentation",
        "value=" + documentationPath);
    // Windows doesn't have libexec directory
    if (installer.value("os") == "x11" || installer.value("os") == "mac") {
        var qtConfLibexecPath = aComponentRootPath + "/" + "libexec" + "/" + "qt.conf";
        print("Writing qt.conf file to: " + qtConfLibexecPath);
        aComponent.addOperation("Settings",
            "path=" + qtConfLibexecPath,
            "method=set",
            "key=Paths/Prefix",
            "value=..");
        aComponent.addOperation("Settings",
            "path=" + qtConfLibexecPath,
            "method=set",
            "key=Paths/Examples",
            "value=" + examplesPath);
        aComponent.addOperation("Settings",
            "path=" + qtConfLibexecPath,
            "method=set",
            "key=Paths/Documentation",
            "value=" + documentationPath);
    }
}

/*****
* Global function to be used by Qt binary packages.
* This function will write qt.conf to the /bin directory. Examples and documentation paths
* will be over ridden to point into centralized doc&examples location
* @param aComponentRootPath     Full path to the Qt package (under this path the /bin should exist)
* @param aQtInstallationName    "Qt-x.x.x"
*/
patchQtExamplesAndDoc = function(aComponent, aComponentRootPath, aQtInstallationName)
{
    print("Patching doc & examples for: " + aComponentRootPath);
    var fileName = aComponentRootPath + "/" + "bin" + "/" + "qt.conf";
    print("qt.conf file: " + fileName);

    aComponent.addOperation("Settings",
        "path=" + fileName,
        "method=set",
        "key=Paths/Prefix",
        "value=..");

    aComponent.addOperation("Settings",
        "path=" + fileName,
        "method=set",
        "key=Paths/Examples",
        "value=../../Examples/" + aQtInstallationName);

    aComponent.addOperation("Settings",
        "path=" + fileName,
        "method=set",
        "key=Paths/Documentation",
        "value=../../Docs/" + aQtInstallationName);
}

patchQt4ExamplesAndDoc = function(aComponent, aComponentRootPath, aQtInstallationName)
{
    patchQtExamplesAndDoc(aComponent, aComponentRootPath, aQtInstallationName);

    print("Patching demos for: " + aComponentRootPath);
    var fileName = aComponentRootPath + "/" + "bin" + "/" + "qt.conf";
    print("qt.conf file: " + fileName);

    aComponent.addOperation("Settings",
        "path=" + fileName,
        "method=add_array_value",
        "key=Paths/Demos",
        "value=../../Examples/" + aQtInstallationName);
}

Component.prototype.createOperations = function()
{
    component.createOperations();
}
