// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/ozone/platform/wayland/test/test_wayland_server_thread.h"

#include <sys/socket.h>
#include <wayland-server.h>

#include <cstdlib>
#include <memory>
#include <utility>

#include "base/bind.h"
#include "base/files/file_util.h"
#include "base/files/scoped_file.h"
#include "base/run_loop.h"
#include "base/strings/string_number_conversions.h"
#include "base/synchronization/lock.h"
#include "ui/ozone/platform/wayland/test/test_gtk_primary_selection.h"
#include "ui/ozone/platform/wayland/test/test_zwp_primary_selection.h"

namespace wl {

namespace {
// TODO(1365887): This is a lock that workarounds a problem when wl_client_flush
// is called from multiple threads.
static base::Lock g_global_lock_;
}  // namespace

void DisplayDeleter::operator()(wl_display* display) {
  wl_display_destroy(display);
}

TestWaylandServerThread::TestWaylandServerThread()
    : Thread("test_wayland_server"),
      pause_event_(base::WaitableEvent::ResetPolicy::AUTOMATIC,
                   base::WaitableEvent::InitialState::NOT_SIGNALED),
      resume_event_(base::WaitableEvent::ResetPolicy::AUTOMATIC,
                    base::WaitableEvent::InitialState::NOT_SIGNALED),
      compositor_v4_(4),
      compositor_v3_(3),
      controller_(FROM_HERE) {}

TestWaylandServerThread::~TestWaylandServerThread() {
  if (client_)
    wl_client_destroy(client_);

  // Stop watching the descriptor here to guarantee that no new events will come
  // during or after the destruction of the display.
  controller_.StopWatchingFileDescriptor();

  Resume();
  Stop();
}

// static
void TestWaylandServerThread::FlushClientForResource(wl_resource* resource) {
  DCHECK(resource);
  base::AutoLock scoped_lock(g_global_lock_);
  wl_client_flush(wl_resource_get_client(resource));
}

bool TestWaylandServerThread::Start(const ServerConfig& config) {
  display_.reset(wl_display_create());
  if (!display_)
    return false;
  event_loop_ = wl_display_get_event_loop(display_.get());

  int fd[2];
  if (socketpair(AF_UNIX, SOCK_STREAM | SOCK_CLOEXEC, 0, fd) < 0)
    return false;
  base::ScopedFD server_fd(fd[0]);
  base::ScopedFD client_fd(fd[1]);

  if (wl_display_init_shm(display_.get()) < 0)
    return false;
  if (config.compositor_version == CompositorVersion::kV3) {
    if (!compositor_v3_.Initialize(display_.get()))
      return false;
  } else {
    if (!compositor_v4_.Initialize(display_.get()))
      return false;
  }
  if (!sub_compositor_.Initialize(display_.get()))
    return false;
  if (!viewporter_.Initialize(display_.get()))
    return false;
  if (!alpha_compositing_.Initialize(display_.get()))
    return false;
  if (!output_.Initialize(display_.get()))
    return false;
  SetupOutputs();

  if (!data_device_manager_.Initialize(display_.get()))
    return false;
  if (!SetupPrimarySelectionManager(config.primary_selection_protocol))
    return false;

  if (!seat_.Initialize(display_.get()))
    return false;
  if (config.shell_version == ShellVersion::kV6) {
    if (!zxdg_shell_v6_.Initialize(display_.get()))
      return false;
  } else {
    if (!xdg_shell_.Initialize(display_.get()))
      return false;
    if (!zaura_shell_.Initialize(display_.get()))
      return false;
  }
  if (!zcr_stylus_.Initialize(display_.get()))
    return false;
  if (!zcr_text_input_extension_v1_.Initialize(display_.get()))
    return false;
  if (!zwp_text_input_manager_v1_.Initialize(display_.get()))
    return false;
  if (!SetupExplicitSynchronizationProtocol(
          config.use_explicit_synchronization))
    return false;
  if (!zwp_linux_dmabuf_v1_.Initialize(display_.get()))
    return false;
  if (!overlay_prioritizer_.Initialize(display_.get()))
    return false;
  if (!wp_pointer_gestures_.Initialize(display_.get()))
    return false;

  client_ = wl_client_create(display_.get(), server_fd.release());
  if (!client_)
    return false;

  base::Thread::Options options;
  options.message_pump_factory = base::BindRepeating(
      &TestWaylandServerThread::CreateMessagePump, base::Unretained(this));
  if (!base::Thread::StartWithOptions(std::move(options)))
    return false;

  setenv("WAYLAND_SOCKET", base::NumberToString(client_fd.release()).c_str(),
         1);

  return true;
}

void TestWaylandServerThread::Pause() {
  task_runner()->PostTask(FROM_HERE,
                          base::BindOnce(&TestWaylandServerThread::DoPause,
                                         base::Unretained(this)));
  pause_event_.Wait();
}

void TestWaylandServerThread::Resume() {
  if (display_) {
    base::AutoLock scoped_lock(g_global_lock_);
    wl_display_flush_clients(display_.get());
  }
  resume_event_.Signal();
}

MockWpPresentation* TestWaylandServerThread::EnsureWpPresentation() {
  if (wp_presentation_.Initialize(display_.get()))
    return &wp_presentation_;
  return nullptr;
}

TestSurfaceAugmenter* TestWaylandServerThread::EnsureSurfaceAugmenter() {
  if (surface_augmenter_.Initialize(display_.get()))
    return &surface_augmenter_;
  return nullptr;
}

// By default, just make sure primary screen has bounds set. Otherwise delegates
// it, making it possible to emulate different scenarios, such as, multi-screen,
// lazy configuration, arbitrary ordering of the outputs metadata sending, etc.
void TestWaylandServerThread::SetupOutputs() {
  if (output_delegate_) {
    output_delegate_->SetupOutputs(&output_);
    return;
  }
  if (output_.GetRect().IsEmpty())
    output_.SetRect(gfx::Rect{0, 0, 800, 600});
}

bool TestWaylandServerThread::SetupPrimarySelectionManager(
    PrimarySelectionProtocol protocol) {
  switch (protocol) {
    case PrimarySelectionProtocol::kNone:
      return true;
    case PrimarySelectionProtocol::kZwp:
      primary_selection_device_manager_.reset(CreateTestSelectionManagerZwp());
      break;
    case PrimarySelectionProtocol::kGtk:
      primary_selection_device_manager_.reset(CreateTestSelectionManagerGtk());
      break;
  }
  return primary_selection_device_manager_->Initialize(display_.get());
}

bool TestWaylandServerThread::SetupExplicitSynchronizationProtocol(
    ShouldUseExplicitSynchronizationProtocol usage) {
  switch (usage) {
    case ShouldUseExplicitSynchronizationProtocol::kNone:
      return true;
    case ShouldUseExplicitSynchronizationProtocol::kUse:
      return zwp_linux_explicit_synchronization_v1_.Initialize(display_.get());
  }
  NOTREACHED();
  return false;
}

void TestWaylandServerThread::DoPause() {
  base::RunLoop().RunUntilIdle();
  pause_event_.Signal();
  resume_event_.Wait();
}

std::unique_ptr<base::MessagePump>
TestWaylandServerThread::CreateMessagePump() {
  auto pump = std::make_unique<base::MessagePumpLibevent>();
  pump->WatchFileDescriptor(wl_event_loop_get_fd(event_loop_), true,
                            base::MessagePumpLibevent::WATCH_READ, &controller_,
                            this);
  return std::move(pump);
}

void TestWaylandServerThread::OnFileCanReadWithoutBlocking(int fd) {
  wl_event_loop_dispatch(event_loop_, 0);
  if (display_) {
    base::AutoLock scoped_lock(g_global_lock_);
    wl_display_flush_clients(display_.get());
  }
}

void TestWaylandServerThread::OnFileCanWriteWithoutBlocking(int fd) {}

}  // namespace wl
