// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_CACHED_METADATA_HANDLER_H_
#define THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_CACHED_METADATA_HANDLER_H_

#include <stdint.h>
#include "third_party/blink/renderer/platform/heap/handle.h"
#include "third_party/blink/renderer/platform/wtf/forward.h"

namespace blink {

class CachedMetadata;

// Handler class for caching operations.
class CachedMetadataHandler
    : public GarbageCollectedFinalized<CachedMetadataHandler> {
 public:
  enum CacheType {
    kSendToPlatform,  // send cache data to blink::Platform::cacheMetadata
    kCacheLocally     // cache only in Resource's member variables
  };

  // Enum for marking serialized cached metadatas so that the deserializers
  // do not conflict.
  enum CachedMetadataType : uint32_t {
    kSingleEntry,    // the metadata is a single CachedMetadata entry
    kSourceKeyedMap  // the metadata is multiple CachedMetadata entries keyed by
                     // a source string.
  };

  virtual ~CachedMetadataHandler() = default;
  virtual void Trace(blink::Visitor* visitor) {}

  // Reset existing metadata, to allow setting new data.
  virtual void ClearCachedMetadata(CacheType = kCacheLocally) = 0;

  // Returns the encoding to which the cache is specific.
  virtual String Encoding() const = 0;

  virtual bool IsServedFromCacheStorage() const = 0;

 protected:
  CachedMetadataHandler() = default;
};

// A CachedMetadataHandler which stores one piece of metadata.
class SingleCachedMetadataHandler : public CachedMetadataHandler {
 public:
  // Caches the given metadata in association with this resource and suggests
  // that the platform persist it. The dataTypeID is a pseudo-randomly chosen
  // identifier that is used to distinguish data generated by the caller.
  virtual void SetCachedMetadata(uint32_t data_type_id,
                                 const char*,
                                 size_t,
                                 CacheType = kSendToPlatform) = 0;

  // Returns cached metadata of the given type associated with this resource.
  // This cached metadata can be pruned at any time.
  virtual scoped_refptr<CachedMetadata> GetCachedMetadata(
      uint32_t data_type_id) const = 0;

 protected:
  SingleCachedMetadataHandler() = default;
};

}  // namespace blink

#endif  // THIRD_PARTY_BLINK_RENDERER_PLATFORM_LOADER_FETCH_CACHED_METADATA_HANDLER_H_
