/****************************************************************************
**
** Copyright (C) 2016 by Southwest Research Institute (R)
** Contact: http://www.qt-project.org/legal
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <qfile.h>
#include <qdebug.h>

quint32 convertmantissa(qint32 i)
{
    quint32 m = i << 13; // Zero pad mantissa bits
    quint32 e = 0; // Zero exponent

    while (!(m & 0x00800000)) {   // While not normalized
        e -= 0x00800000;     // Decrement exponent (1<<23)
        m <<= 1;             // Shift mantissa
    }
    m &= ~0x00800000;          // Clear leading 1 bit
    e += 0x38800000;           // Adjust bias ((127-14)<<23)
    return m | e;            // Return combined number
}

// we first build these tables up and then print them out as a separate step in order
// to more closely map the implementation given in the paper.
quint32 basetable[512];
quint32 shifttable[512];

#define PRINTHEX(a) "0x" + QByteArray::number(a,16).toUpper() + "U,\n"

qint32 main(qint32 argc, char **argv)
{
    if (argc < 2) {
        qWarning() << "Must provide output filename as argument.";
        return -1;
    }

    QFile fid(QFile::decodeName(argv[1]));
    if (!fid.open(QIODevice::WriteOnly | QIODevice::Text)) {
        qWarning() << "Abort: Failed to open/create file" << fid.fileName();
        return -1;
    }
    quint32 i;

    fid.write("/* This file was generated by gen_qfloat16_tables.cpp */\n\n");
    fid.write("#include <QtCore/qfloat16.h>\n\n");

    fid.write("QT_BEGIN_NAMESPACE\n\n");
    fid.write("#if !defined(__F16C__) && !defined(__ARM_FP16_FORMAT_IEEE)\n\n");

    fid.write("const quint32 qfloat16::mantissatable[2048] = {\n");
    fid.write("0,\n");
    for (i = 1; i < 1024; i++)
        fid.write(PRINTHEX(convertmantissa(i)));
    for (i = 1024; i < 2048; i++)
        fid.write(PRINTHEX(0x38000000U + ((i - 1024) << 13)));
    fid.write("};\n\n");

    fid.write("const quint32 qfloat16::exponenttable[64] = {\n");
    fid.write("0,\n");
    for (i = 1; i < 31; i++)
        fid.write(PRINTHEX(i << 23));
    fid.write("0x47800000U,\n");  // 31
    fid.write("0x80000000U,\n");  // 32
    for (i = 33; i < 63; i++)
        fid.write(PRINTHEX(0x80000000U + ((i - 32) << 23)));
    fid.write("0xC7800000U,\n");  // 63
    fid.write("};\n\n");

    fid.write("const quint32 qfloat16::offsettable[64] = {\n");
    fid.write("0,\n");
    for (i = 1; i < 32; i++)
        fid.write("1024U,\n");
    fid.write("0,\n");
    for (i = 33; i < 64; i++)
        fid.write("1024U,\n");
    fid.write("};\n\n");

    qint32 e;
    for (i = 0; i < 256; ++i) {
        e = i - 127;
        if (e < -24) {   // Very small numbers map to zero
            basetable[i | 0x000] = 0x0000;
            basetable[i | 0x100] = 0x8000;
            shifttable[i | 0x000] = 24;
            shifttable[i | 0x100] = 24;

        } else if (e < -14) {             // Small numbers map to denorms
            basetable[i | 0x000] = (0x0400 >> (-e - 14));
            basetable[i | 0x100] = (0x0400 >> (-e - 14)) | 0x8000;
            shifttable[i | 0x000] = -e - 1;
            shifttable[i | 0x100] = -e - 1;

        } else if (e <= 15) {            // Normal numbers just lose precision
            basetable[i | 0x000] = ((e + 15) << 10);
            basetable[i | 0x100] = ((e + 15) << 10) | 0x8000;
            shifttable[i | 0x000] = 13;
            shifttable[i | 0x100] = 13;

        } else if (e < 128) {            // Large numbers map to Infinity
            basetable[i | 0x000] = 0x7C00;
            basetable[i | 0x100] = 0xFC00;
            shifttable[i | 0x000] = 24;
            shifttable[i | 0x100] = 24;

        } else {                     // Infinity and NaN's stay Infinity and NaN's
            basetable[i | 0x000] = 0x7C00;
            basetable[i | 0x100] = 0xFC00;
            shifttable[i | 0x000] = 13;
            shifttable[i | 0x100] = 13;
        }
    }

    fid.write("const quint32 qfloat16::basetable[512] = {\n");
    for (i = 0; i < 512; i++)
        fid.write(PRINTHEX(basetable[i]));

    fid.write("};\n\n");

    fid.write("const quint32 qfloat16::shifttable[512] = {\n");
    for (i = 0; i < 512; i++)
        fid.write(PRINTHEX(shifttable[i]));

    fid.write("};\n\n");

    fid.write("#endif // !__F16C__ && !__ARM_FP16_FORMAT_IEEE\n\n");
    fid.write("QT_END_NAMESPACE\n");
    fid.close();
    return 0;
}
