

  /** @polymerBehavior */
  Polymer.IronSelectableBehavior = {

    properties: {

      /**
       * If you want to use the attribute value of an element for `selected` instead of the index,
       * set this to the name of the attribute.
       *
       * @attribute attrForSelected
       * @type {string}
       */
      attrForSelected: {
        type: String,
        value: null
      },

      /**
       * Gets or sets the selected element. The default is to use the index of the item.
       *
       * @attribute selected
       * @type {string}
       */
      selected: {
        type: String,
        notify: true
      },

      /**
       * Returns the currently selected item.
       *
       * @attribute selectedItem
       * @type {Object}
       */
      selectedItem: {
        type: Object,
        readOnly: true,
        notify: true
      },

      /**
       * The event that fires from items when they are selected. Selectable
       * will listen for this event from items and update the selection state.
       * Set to empty string to listen to no events.
       *
       * @attribute activateEvent
       * @type {string}
       * @default 'tap'
       */
      activateEvent: {
        type: String,
        value: 'tap',
        observer: '_activateEventChanged'
      },

      /**
       * This is a CSS selector sting.  If this is set, only items that matches the CSS selector
       * are selectable.
       *
       * @attribute selectable
       * @type {string}
       */
      selectable: String,

      /**
       * The class to set on elements when selected.
       *
       * @attribute selectedClass
       * @type {string}
       */
      selectedClass: {
        type: String,
        value: 'iron-selected'
      },

      /**
       * The attribute to set on elements when selected.
       *
       * @attribute selectedAttribute
       * @type {string}
       */
      selectedAttribute: {
        type: String,
        value: null
      }

    },

    observers: [
      '_updateSelected(attrForSelected, selected)'
    ],

    excludedLocalNames: {
      'template': 1
    },

    created: function() {
      this._bindFilterItem = this._filterItem.bind(this);
      this._selection = new Polymer.IronSelection(this._applySelection.bind(this));
    },

    attached: function() {
      this._observer = this._observeItems(this);
      this._contentObserver = this._observeContent(this);
    },

    detached: function() {
      if (this._observer) {
        this._observer.disconnect();
      }
      if (this._contentObserver) {
        this._contentObserver.disconnect();
      }
      this._removeListener(this.activateEvent);
    },

    /**
     * Returns an array of selectable items.
     *
     * @property items
     * @type Array
     */
    get items() {
      var nodes = Polymer.dom(this).queryDistributedElements(this.selectable || '*');
      return Array.prototype.filter.call(nodes, this._bindFilterItem);
    },

    /**
     * Returns the index of the given item.
     *
     * @method indexOf
     * @param {Object} item
     * @returns Returns the index of the item
     */
    indexOf: function(item) {
      return this.items.indexOf(item);
    },

    /**
     * Selects the given value.
     *
     * @method select
     * @param {string} value the value to select.
     */
    select: function(value) {
      this.selected = value;
    },

    /**
     * Selects the previous item.
     *
     * @method selectPrevious
     */
    selectPrevious: function() {
      var length = this.items.length;
      var index = (Number(this._valueToIndex(this.selected)) - 1 + length) % length;
      this.selected = this._indexToValue(index);
    },

    /**
     * Selects the next item.
     *
     * @method selectNext
     */
    selectNext: function() {
      var index = (Number(this._valueToIndex(this.selected)) + 1) % this.items.length;
      this.selected = this._indexToValue(index);
    },

    _addListener: function(eventName) {
      this.listen(this, eventName, '_activateHandler');
    },

    _removeListener: function(eventName) {
      // There is no unlisten yet...
      // https://github.com/Polymer/polymer/issues/1639
      //this.removeEventListener(eventName, this._bindActivateHandler);
    },

    _activateEventChanged: function(eventName, old) {
      this._removeListener(old);
      this._addListener(eventName);
    },

    _updateSelected: function() {
      this._selectSelected(this.selected);
    },

    _selectSelected: function(selected) {
      this._selection.select(this._valueToItem(this.selected));
    },

    _filterItem: function(node) {
      return !this.excludedLocalNames[node.localName];
    },

    _valueToItem: function(value) {
      return (value == null) ? null : this.items[this._valueToIndex(value)];
    },

    _valueToIndex: function(value) {
      if (this.attrForSelected) {
        for (var i = 0, item; item = this.items[i]; i++) {
          if (this._valueForItem(item) == value) {
            return i;
          }
        }
      } else {
        return Number(value);
      }
    },

    _indexToValue: function(index) {
      if (this.attrForSelected) {
        var item = this.items[index];
        if (item) {
          return this._valueForItem(item);
        }
      } else {
        return index;
      }
    },

    _valueForItem: function(item) {
      return item[this.attrForSelected] || item.getAttribute(this.attrForSelected);
    },

    _applySelection: function(item, isSelected) {
      if (this.selectedClass) {
        this.toggleClass(this.selectedClass, isSelected, item);
      }
      if (this.selectedAttribute) {
        this.toggleAttribute(this.selectedAttribute, isSelected, item);
      }
      this._selectionChange();
      this.fire('iron-' + (isSelected ? 'select' : 'deselect'), {item: item});
    },

    _selectionChange: function() {
      this._setSelectedItem(this._selection.get());
    },

    // observe content changes under the given node.
    _observeContent: function(node) {
      var content = node.querySelector('content');
      if (content && content.parentElement === node) {
        return this._observeItems(node.domHost);
      }
    },

    // observe items change under the given node.
    _observeItems: function(node) {
      var observer = new MutationObserver(function() {
        if (this.selected != null) {
          this._updateSelected();
        }
      }.bind(this));
      observer.observe(node, {
        childList: true,
        subtree: true
      });
      return observer;
    },

    _activateHandler: function(e) {
      // TODO: remove this when https://github.com/Polymer/polymer/issues/1639 is fixed so we
      // can just remove the old event listener.
      if (e.type !== this.activateEvent) {
        return;
      }
      var t = e.target;
      var items = this.items;
      while (t && t != this) {
        var i = items.indexOf(t);
        if (i >= 0) {
          var value = this._indexToValue(i);
          this._itemActivate(value, t);
          return;
        }
        t = t.parentNode;
      }
    },

    _itemActivate: function(value, item) {
      if (!this.fire('iron-activate',
          {selected: value, item: item}, {cancelable: true}).defaultPrevented) {
        this.select(value);
      }
    }

  };

