// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef SERVICES_UI_WS_GPU_COMPOSITOR_FRAME_SINK_H_
#define SERVICES_UI_WS_GPU_COMPOSITOR_FRAME_SINK_H_

#include <memory>
#include <set>

#include "base/macros.h"
#include "base/memory/ref_counted.h"
#include "cc/ipc/compositor_frame.mojom.h"
#include "cc/ipc/mojo_compositor_frame_sink.mojom.h"
#include "cc/output/context_provider.h"
#include "cc/scheduler/begin_frame_source.h"
#include "cc/surfaces/display.h"
#include "cc/surfaces/display_client.h"
#include "cc/surfaces/frame_sink_id.h"
#include "cc/surfaces/surface_factory.h"
#include "cc/surfaces/surface_factory_client.h"
#include "cc/surfaces/surface_id.h"
#include "cc/surfaces/surface_id_allocator.h"
#include "mojo/public/cpp/bindings/binding.h"
#include "services/ui/surfaces/surfaces_context_provider.h"

namespace base {
class SingleThreadTaskRunner;
}

namespace gpu {
class GpuMemoryBufferManager;
}

namespace ui {

class DisplayCompositor;

namespace ws {

// Server side representation of a WindowSurface.
class GpuCompositorFrameSink : public cc::mojom::MojoCompositorFrameSink,
                               public cc::DisplayClient,
                               public cc::mojom::MojoCompositorFrameSinkPrivate,
                               public cc::SurfaceFactoryClient,
                               public cc::BeginFrameObserver {
 public:
  // TODO(fsamuel): DisplayCompositor should own
  // GpuCompositorFrameSink. GpuCompositorFrameSink should not
  // refer to GpuCompositorFrameSinkManager.
  GpuCompositorFrameSink(
      scoped_refptr<DisplayCompositor> display_compositor,
      const cc::FrameSinkId& frame_sink_id,
      gfx::AcceleratedWidget widget,
      gpu::GpuMemoryBufferManager* gpu_memory_buffer_manager,
      scoped_refptr<SurfacesContextProvider> context_provider,
      cc::mojom::MojoCompositorFrameSinkRequest request,
      cc::mojom::MojoCompositorFrameSinkClientPtr client);

  ~GpuCompositorFrameSink() override;

  // cc::mojom::MojoCompositorFrameSink:
  void SetNeedsBeginFrame(bool needs_begin_frame) override;
  void SubmitCompositorFrame(cc::CompositorFrame frame) override;

  // cc::mojom::MojoCompositorFrameSinkPrivate:
  void AddChildFrameSink(const cc::FrameSinkId& child_frame_sink_id) override;
  void RemoveChildFrameSink(
      const cc::FrameSinkId& child_frame_sink_id) override;

 private:
  void InitDisplay(gfx::AcceleratedWidget widget,
                   gpu::GpuMemoryBufferManager* gpu_memory_buffer_manager,
                   scoped_refptr<SurfacesContextProvider> context_provider);

  void DidReceiveCompositorFrameAck();

  // cc::DisplayClient implementation.
  void DisplayOutputSurfaceLost() override;
  void DisplayWillDrawAndSwap(bool will_draw_and_swap,
                              const cc::RenderPassList& render_passes) override;
  void DisplayDidDrawAndSwap() override;

  // cc::SurfaceFactoryClient implementation.
  void ReturnResources(const cc::ReturnedResourceArray& resources) override;
  void SetBeginFrameSource(cc::BeginFrameSource* begin_frame_source) override;

  // cc::BeginFrameObserver implementation.
  void OnBeginFrame(const cc::BeginFrameArgs& args) override;
  const cc::BeginFrameArgs& LastUsedBeginFrameArgs() const override;
  void OnBeginFrameSourcePausedChanged(bool paused) override;

  void UpdateNeedsBeginFramesInternal();

  const cc::FrameSinkId frame_sink_id_;
  scoped_refptr<base::SingleThreadTaskRunner> task_runner_;

  // TODO(fsamuel): We hold a reference to DisplayCompositor so we can talk to
  // SurfaceManager in the destructor. In the future, DisplayCompositor will own
  // GpuCompositorFrameSink.
  scoped_refptr<DisplayCompositor> display_compositor_;

  gfx::Size last_submitted_frame_size_;
  // GpuCompositorFrameSink holds a cc::Display if it created with
  // non-null gfx::AcceleratedWidget. In the window server, the display root
  // window's CompositorFrameSink will have a valid gfx::AcceleratedWidget.
  std::unique_ptr<cc::Display> display_;

  cc::LocalFrameId local_frame_id_;
  cc::SurfaceIdAllocator surface_id_allocator_;
  cc::SurfaceFactory surface_factory_;
  // Counts the number of CompositorFrames that have been submitted and have not
  // yet received an ACK.
  int ack_pending_count_ = 0;
  cc::ReturnedResourceArray surface_returned_resources_;

  // The begin frame source being observered. Null if none.
  cc::BeginFrameSource* begin_frame_source_ = nullptr;

  // The last begin frame args generated by the begin frame source.
  cc::BeginFrameArgs last_begin_frame_args_;

  // Whether a request for begin frames has been issued.
  bool needs_begin_frame_ = false;

  // Whether or not a frame observer has been added.
  bool added_frame_observer_ = false;

  cc::mojom::MojoCompositorFrameSinkClientPtr client_;
  mojo::Binding<cc::mojom::MojoCompositorFrameSink> binding_;

  DISALLOW_COPY_AND_ASSIGN(GpuCompositorFrameSink);
};

}  // namespace ws

}  // namespace ui

#endif  // SERVICES_UI_WS_GPU_COMPOSITOR_FRAME_SINK_H_
