/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-07-16
 * Description : digiKam image editor to adjust Hue, Saturation,
 *               and Lightness of picture.
 *
 * Copyright (C) 2004-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.
 
#include <tqcolor.h>
#include <tqgroupbox.h>
#include <tqhgroupbox.h>
#include <tqvgroupbox.h>
#include <tqvbox.h>
#include <tqhbuttongroup.h> 
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqframe.h>
#include <tqlabel.h>
#include <tqpushbutton.h>
#include <tqcombobox.h>
#include <tqwhatsthis.h>
#include <tqtooltip.h>
#include <tqtimer.h>

// KDE includes.

#include <knuminput.h>
#include <tdelocale.h>
#include <tdeapplication.h>
#include <tdeconfig.h>
#include <kcursor.h>
#include <kstandarddirs.h>
#include <kcolordialog.h>

// Digikam includes.

#include "imageiface.h"
#include "imagewidget.h"
#include "histogramwidget.h"
#include "colorgradientwidget.h"
#include "hslmodifier.h"
#include "dimg.h"

// Local includes.

#include "hspreviewwidget.h"
#include "imageeffect_hsl.h"
#include "imageeffect_hsl.moc"

namespace DigikamImagesPluginCore
{

ImageEffect_HSL::ImageEffect_HSL(TQWidget* parent)
               : Digikam::ImageDlgBase(parent, i18n("Hue/Saturation/Lightness"), "hsladjust", false)
{
    m_destinationPreviewData = 0L;
    setHelp("hsladjusttool.anchor", "digikam");

    m_previewWidget = new Digikam::ImageWidget("hsladjust Tool Dialog", plainPage(),
                                               i18n("<p>Here you can see the image "
                                                    "Hue/Saturation/Lightness adjustments preview. "
                                                    "You can pick color on image "
                                                    "to see the color level corresponding on histogram."));
    setPreviewAreaWidget(m_previewWidget); 

    // -------------------------------------------------------------
    
    TQWidget *gboxSettings     = new TQWidget(plainPage());
    TQGridLayout* gridSettings = new TQGridLayout(gboxSettings, 11, 4, spacingHint());

    TQLabel *label1 = new TQLabel(i18n("Channel:"), gboxSettings);
    label1->setAlignment ( TQt::AlignRight | TQt::AlignVCenter );
    m_channelCB = new TQComboBox( false, gboxSettings );
    m_channelCB->insertItem( i18n("Luminosity") );
    m_channelCB->insertItem( i18n("Red") );
    m_channelCB->insertItem( i18n("Green") );
    m_channelCB->insertItem( i18n("Blue") );
    TQWhatsThis::add( m_channelCB, i18n("<p>Select the histogram channel to display here:<p>"
                                       "<b>Luminosity</b>: display the image's luminosity values.<p>"
                                       "<b>Red</b>: display the red image-channel values.<p>"
                                       "<b>Green</b>: display the green image-channel values.<p>"
                                       "<b>Blue</b>: display the blue image-channel values.<p>"));

    m_scaleBG = new TQHButtonGroup(gboxSettings);
    m_scaleBG->setExclusive(true);
    m_scaleBG->setFrameShape(TQFrame::NoFrame);
    m_scaleBG->setInsideMargin( 0 );
    TQWhatsThis::add( m_scaleBG, i18n("<p>Select the histogram scale here.<p>"
                                     "If the image's maximal counts are small, you can use the linear scale.<p>"
                                     "Logarithmic scale can be used when the maximal counts are big; "
                                     "if it is used, all values (small and large) will be visible on the graph."));
    
    TQPushButton *linHistoButton = new TQPushButton( m_scaleBG );
    TQToolTip::add( linHistoButton, i18n( "<p>Linear" ) );
    m_scaleBG->insert(linHistoButton, Digikam::HistogramWidget::LinScaleHistogram);
    TDEGlobal::dirs()->addResourceType("histogram-lin", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
    TQString directory = TDEGlobal::dirs()->findResourceDir("histogram-lin", "histogram-lin.png");
    linHistoButton->setPixmap( TQPixmap( directory + "histogram-lin.png" ) );
    linHistoButton->setToggleButton(true);
    
    TQPushButton *logHistoButton = new TQPushButton( m_scaleBG );
    TQToolTip::add( logHistoButton, i18n( "<p>Logarithmic" ) );
    m_scaleBG->insert(logHistoButton, Digikam::HistogramWidget::LogScaleHistogram);
    TDEGlobal::dirs()->addResourceType("histogram-log", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
    directory = TDEGlobal::dirs()->findResourceDir("histogram-log", "histogram-log.png");
    logHistoButton->setPixmap( TQPixmap( directory + "histogram-log.png" ) );
    logHistoButton->setToggleButton(true);       

    TQHBoxLayout* l1 = new TQHBoxLayout();
    l1->addWidget(label1);
    l1->addWidget(m_channelCB);
    l1->addStretch(10);
    l1->addWidget(m_scaleBG);
    
    gridSettings->addMultiCellLayout(l1, 0, 0, 0, 4);

    // -------------------------------------------------------------

    TQVBox *histoBox   = new TQVBox(gboxSettings);
    m_histogramWidget = new Digikam::HistogramWidget(256, 140, histoBox, false, true, true);
    TQWhatsThis::add( m_histogramWidget, i18n("<p>Here you can see the target preview image histogram drawing "
                                             "of the selected image channel. This one is re-computed at any "
                                             "settings changes."));
    TQLabel *space = new TQLabel(histoBox);
    space->setFixedHeight(1);    
    m_hGradient = new Digikam::ColorGradientWidget( Digikam::ColorGradientWidget::Horizontal, 10, histoBox );
    m_hGradient->setColors( TQColor( "black" ), TQColor( "white" ) );
    
    gridSettings->addMultiCellWidget(histoBox, 1, 2, 0, 4);

    // -------------------------------------------------------------

    m_HSSelector = new KHSSelector(gboxSettings);
    TQWhatsThis::add( m_HSSelector, i18n("<p>Select the hue and saturation adjustments of the image here."));
    m_HSSelector->setMinimumSize(256, 142);
    gridSettings->addMultiCellWidget(m_HSSelector, 3, 3, 0, 4);

    m_HSPreview = new HSPreviewWidget(gboxSettings, spacingHint());
    TQWhatsThis::add( m_HSPreview, i18n("<p>You can see here a color preview of the hue and "
                                       "saturation adjustments."));
    m_HSPreview->setMinimumSize(256, 15);
    gridSettings->addMultiCellWidget(m_HSPreview, 4, 4, 0, 4);

    TQLabel *label2 = new TQLabel(i18n("Hue:"), gboxSettings);
    m_hInput       = new KDoubleNumInput(gboxSettings);
    m_hInput->setPrecision(0);
    m_hInput->setRange(-180.0, 180.0, 1.0, true);
    m_hInput->setValue(0.0);
    TQWhatsThis::add( m_hInput, i18n("<p>Set here the hue adjustment of the image."));
    gridSettings->addMultiCellWidget(label2, 5, 5, 0, 4);
    gridSettings->addMultiCellWidget(m_hInput, 6, 6, 0, 4);

    TQLabel *label3 = new TQLabel(i18n("Saturation:"), gboxSettings);
    m_sInput       = new KDoubleNumInput(gboxSettings);
    m_sInput->setPrecision(2);
    m_sInput->setRange(-100.0, 100.0, 0.01, true);
    m_sInput->setValue(0.0);
    TQWhatsThis::add( m_sInput, i18n("<p>Set here the saturation adjustment of the image."));
    gridSettings->addMultiCellWidget(label3, 7, 7, 0, 4);
    gridSettings->addMultiCellWidget(m_sInput, 8, 8, 0, 4);

    TQLabel *label4 = new TQLabel(i18n("Lightness:"), gboxSettings);
    m_lInput       = new KDoubleNumInput(gboxSettings);
    m_lInput->setPrecision(2);
    m_lInput->setRange(-100.0, 100.0, 0.01, true);
    m_lInput->setValue(0.0);
    TQWhatsThis::add( m_lInput, i18n("<p>Set here the lightness adjustment of the image."));    
    gridSettings->addMultiCellWidget(label4, 9, 9, 0, 4);
    gridSettings->addMultiCellWidget(m_lInput, 10, 10, 0, 4);

    gridSettings->setRowStretch(11, 10);
    setUserAreaWidget(gboxSettings);

    // -------------------------------------------------------------

    connect(m_HSSelector, TQ_SIGNAL(valueChanged(int, int)),
            this, TQ_SLOT(slotHSChanged(int, int)));

    connect(m_channelCB, TQ_SIGNAL(activated(int)),
            this, TQ_SLOT(slotChannelChanged(int)));

    connect(m_scaleBG, TQ_SIGNAL(released(int)),
            this, TQ_SLOT(slotScaleChanged(int)));

    connect(m_previewWidget, TQ_SIGNAL(spotPositionChangedFromTarget( const Digikam::DColor &, const TQPoint & )),
            this, TQ_SLOT(slotColorSelectedFromTarget( const Digikam::DColor & )));

    connect(m_hInput, TQ_SIGNAL(valueChanged (double)),
            this, TQ_SLOT(slotTimer()));

    connect(m_hInput, TQ_SIGNAL(valueChanged (double)),
            this, TQ_SLOT(slotHChanged(double)));
            
    connect(m_sInput, TQ_SIGNAL(valueChanged (double)),
            this, TQ_SLOT(slotTimer()));

    connect(m_sInput, TQ_SIGNAL(valueChanged (double)),
            this, TQ_SLOT(slotSChanged(double)));
            
    connect(m_lInput, TQ_SIGNAL(valueChanged (double)),
            this, TQ_SLOT(slotTimer()));
    
    connect(m_previewWidget, TQ_SIGNAL(signalResized()),
            this, TQ_SLOT(slotEffect()));              
            
    // -------------------------------------------------------------            

    enableButtonOK( false );
}

ImageEffect_HSL::~ImageEffect_HSL()
{
    m_histogramWidget->stopHistogramComputation();

    if (m_destinationPreviewData) 
       delete [] m_destinationPreviewData;
       
    delete m_histogramWidget;
    delete m_previewWidget;
}

void ImageEffect_HSL::slotChannelChanged(int channel)
{
    switch(channel)
    {
        case LuminosityChannel:
            m_histogramWidget->m_channelType = Digikam::HistogramWidget::ValueHistogram;
            m_hGradient->setColors( TQColor( "black" ), TQColor( "white" ) );
            break;
    
        case RedChannel:
            m_histogramWidget->m_channelType = Digikam::HistogramWidget::RedChannelHistogram;
            m_hGradient->setColors( TQColor( "black" ), TQColor( "red" ) );
            break;
    
        case GreenChannel:         
            m_histogramWidget->m_channelType = Digikam::HistogramWidget::GreenChannelHistogram;
            m_hGradient->setColors( TQColor( "black" ), TQColor( "green" ) );
            break;
    
        case BlueChannel:         
            m_histogramWidget->m_channelType = Digikam::HistogramWidget::BlueChannelHistogram;
            m_hGradient->setColors( TQColor( "black" ), TQColor( "blue" ) );
            break;
    }

    m_histogramWidget->repaint(false);
}

void ImageEffect_HSL::slotScaleChanged(int scale)
{
    m_histogramWidget->m_scaleType = scale;
    m_histogramWidget->repaint(false);
}

void ImageEffect_HSL::slotColorSelectedFromTarget( const Digikam::DColor &color )
{
    m_histogramWidget->setHistogramGuideByColor(color);
}

void ImageEffect_HSL::slotHSChanged(int h, int s)
{
    double hue = double(h);
    if (h >= 180 && h <= 359)
        hue = double(h) - 359.0;

    double sat = ((double)s * (200.0/255.0)) - 100.0;

    m_hInput->blockSignals(true);       
    m_sInput->blockSignals(true);       
    m_hInput->setValue(hue);
    m_sInput->setValue(sat);
    m_hInput->blockSignals(false);       
    m_sInput->blockSignals(false);   
    slotTimer();    
}

void ImageEffect_HSL::slotHChanged(double h)
{
    int hue = int(h);
    if (h >= -180 && h < 0)
        hue = int(h) + 359;

    m_HSSelector->blockSignals(true);       
    m_HSSelector->setXValue(hue);
    m_HSSelector->blockSignals(false);       
}

void ImageEffect_HSL::slotSChanged(double s)
{
    int sat = (int)((s + 100.0) * (255.0/200.0));

    m_HSSelector->blockSignals(true);       
    m_HSSelector->setYValue(sat);
    m_HSSelector->blockSignals(false);       
}

void ImageEffect_HSL::readUserSettings()
{
    TDEConfig* config = kapp->config();
    config->setGroup("hsladjust Tool Dialog");
    m_channelCB->setCurrentItem(config->readNumEntry("Histogram Channel", 0));    // Luminosity.
    m_scaleBG->setButton(config->readNumEntry("Histogram Scale", Digikam::HistogramWidget::LogScaleHistogram));
    m_hInput->setValue(config->readDoubleNumEntry("HueAjustment", 0.0));
    m_sInput->setValue(config->readDoubleNumEntry("SaturationAjustment", 0.0));
    m_lInput->setValue(config->readDoubleNumEntry("LighnessAjustment", 0.0));
    slotHChanged(m_hInput->value());
    slotSChanged(m_sInput->value());
    slotChannelChanged(m_channelCB->currentItem());
    slotScaleChanged(m_scaleBG->selectedId());
}

void ImageEffect_HSL::writeUserSettings()
{
    TDEConfig* config = kapp->config();
    config->setGroup("hsladjust Tool Dialog");
    config->writeEntry("Histogram Channel", m_channelCB->currentItem());
    config->writeEntry("Histogram Scale", m_scaleBG->selectedId());
    config->writeEntry("HueAjustment", m_hInput->value());
    config->writeEntry("SaturationAjustment", m_sInput->value());
    config->writeEntry("LighnessAjustment", m_lInput->value());
    config->sync();
}

void ImageEffect_HSL::resetValues()
{
    m_hInput->blockSignals(true);	
    m_sInput->blockSignals(true);	
    m_lInput->blockSignals(true);	
    m_hInput->setValue(0.0);
    m_sInput->setValue(0.0);
    m_lInput->setValue(0.0);
    slotHChanged(0.0);
    slotSChanged(0.0);
    m_hInput->blockSignals(false);	
    m_sInput->blockSignals(false);	
    m_lInput->blockSignals(false);	
} 

void ImageEffect_HSL::slotEffect()
{
    kapp->setOverrideCursor( KCursor::waitCursor() );

    double hu  = m_hInput->value();
    double sa  = m_sInput->value();    
    double lu  = m_lInput->value();
    
    enableButtonOK( hu != 0.0 || sa != 0.0 || lu != 0.0);
    
    m_HSPreview->setHS(hu, sa);
    m_histogramWidget->stopHistogramComputation();

    if (m_destinationPreviewData) 
       delete [] m_destinationPreviewData;

    Digikam::ImageIface* iface = m_previewWidget->imageIface();
    m_destinationPreviewData   = iface->getPreviewImage();
    int w                      = iface->previewWidth();
    int h                      = iface->previewHeight();
    bool a                     = iface->previewHasAlpha();
    bool sb                    = iface->previewSixteenBit();

    Digikam::DImg preview(w, h, sb, a, m_destinationPreviewData);
    Digikam::HSLModifier cmod;
    cmod.setHue(hu);
    cmod.setSaturation(sa);
    cmod.setLightness(lu);
    cmod.applyHSL(preview);
    iface->putPreviewImage(preview.bits());

    m_previewWidget->updatePreview();

    // Update histogram.
   
    memcpy(m_destinationPreviewData, preview.bits(), preview.numBytes());
    m_histogramWidget->updateData(m_destinationPreviewData, w, h, sb, 0, 0, 0, false);

    kapp->restoreOverrideCursor();
}

void ImageEffect_HSL::finalRendering()
{
    kapp->setOverrideCursor( KCursor::waitCursor() );

    double hu  = m_hInput->value();
    double sa  = m_sInput->value();    
    double lu  = m_lInput->value();

    Digikam::ImageIface* iface = m_previewWidget->imageIface();
    uchar *data                = iface->getOriginalImage();
    int w                      = iface->originalWidth();
    int h                      = iface->originalHeight();
    bool a                     = iface->originalHasAlpha();
    bool sb                    = iface->originalSixteenBit();
    Digikam::DImg original(w, h, sb, a, data);
    delete [] data;

    Digikam::HSLModifier cmod;
    cmod.setHue(hu);
    cmod.setSaturation(sa);
    cmod.setLightness(lu);
    cmod.applyHSL(original);

    iface->putOriginalImage(i18n("HSL Adjustments"), original.bits());
    kapp->restoreOverrideCursor();
    accept();
}

}  // NameSpace DigikamImagesPluginCore

