/***************************************************************************
*   Copyright (C) 2004-2009 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <tqprocess.h>
#include <tqwaitcondition.h>
#include <tqapplication.h>
#include <tqbuffer.h>

#include "fileimporterbibtex.h"
#include "fileimporterexternal.h"

namespace BibTeX
{

    FileImporterExternal::FileImporterExternal( Importer importer, File::FileFormat fileformat )
            : FileImporter(), m_importer( importer ), m_fileformat( fileformat )
    {
        wc = new TQWaitCondition();
    }


    FileImporterExternal::~FileImporterExternal()
    {
        delete wc;
    }

    File* FileImporterExternal::load( TQIODevice *iodevice )
    {
        m_mutex.lock();
        File * result = NULL;
        TQBuffer buffer;

        if ( fetchInput( iodevice, buffer ) )
        {
            buffer.open( IO_ReadOnly );
            FileImporter * bibtexImporter = new FileImporterBibTeX();
            result = bibtexImporter->load( &buffer );
            buffer.close();
            delete bibtexImporter;
        }

        m_mutex.unlock();
        return result;
    }

    void FileImporterExternal::cancel()
    {
        if ( process != NULL )
            process->kill();
    }

    bool FileImporterExternal::fetchInput( TQIODevice *input, TQBuffer &output )
    {
        bool result = false;

        TQString commandLine;
        switch ( m_fileformat )
        {
        default:
            return false;
        }
        TQStringList args = TQStringList::split( ' ', commandLine );

        output.open( IO_WriteOnly );
        writeTo = new TQTextStream( &output );
        writeTo->setEncoding( TQTextStream::UnicodeUTF8 );

        TQApplication::setOverrideCursor( TQt::waitCursor );

        process = new TQProcess( args );
        connect( process, TQ_SIGNAL( processExited() ), this, TQ_SLOT( slotProcessExited() ) );
        connect( process, TQ_SIGNAL( readyReadStdout() ), this, TQ_SLOT( slotReadProcessOutput() ) );
        connect( process, TQ_SIGNAL( readyReadStderr() ), this, TQ_SLOT( slotReadProcessOutput() ) );
        connect( process, TQ_SIGNAL( wroteToStdin() ), this, TQ_SLOT( slotWroteToStdin() ) );

        if ( process->start() )
        {
            while ( !process->isRunning() )
            {
                wc->wait( 250 );
                tqApp->processEvents();
            }
            tqApp->processEvents();
            TQByteArray buf = input->readAll();
            process->writeToStdin( buf );
            tqApp->processEvents();
            while ( process->isRunning() )
            {
                wc->wait( 250 );
                tqApp->processEvents();
            }

            result = process->normalExit();
        }

        disconnect( process, TQ_SIGNAL( wroteToStdin() ), this, TQ_SLOT( slotWroteToStdin() ) );
        disconnect( process, TQ_SIGNAL( readyReadStdout() ), this, TQ_SLOT( slotReadProcessOutput() ) );
        disconnect( process, TQ_SIGNAL( readyReadStderr() ), this, TQ_SLOT( slotReadProcessOutput() ) );
        disconnect( process, TQ_SIGNAL( processExited() ), this, TQ_SLOT( slotProcessExited() ) );
        delete( process );
        process = NULL;
        delete writeTo;
        writeTo = NULL;
        output.close();

        TQApplication::restoreOverrideCursor();
        return result;
    }

    void FileImporterExternal::slotProcessExited()
    {
        wc->wakeAll();
    }

    void FileImporterExternal::slotReadProcessOutput()
    {
        if ( writeTo != NULL )
            while ( process->canReadLineStdout() )
            {
                TQString line = process->readLineStdout();
                //                 tqDebug( "line=%s", line.latin1() );
                ( *writeTo ) << line.latin1();
                ( *writeTo ) << endl;
            }
    }

    void FileImporterExternal::slotWroteToStdin()
    {
        process->closeStdin();
    }


}
#include "fileimporterexternal.moc"
