/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 * Copyright (C) 2005 Markus Brueffer <markus@brueffer.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPQUEUEVIEW_H
#define KFTPQUEUEVIEW_H

#include <tqguardedptr.h>
#include <tqintdict.h>
#include <tqlayout.h>
#include <tqlabel.h>

#include <tdeaction.h>
#include <tdelistview.h>

#include "kftpqueue.h"

class TDEToolBar;
class TDEListViewSearchLine;

namespace KFTPWidgets {

class ListView;
class QueueView;

/**
 * A visual representation of a queued object.
 */
class QueueViewItem : public TQListViewItem
{
public:
    /**
     * Class constructor.
     *
     * @param view Queue view widget
     * @param object Queue object
     * @param parent Parent list view
     */
    QueueViewItem(QueueView *view, KFTPQueue::QueueObject *object, TQListView *parent);
    
    /**
     * Class constructor.
     *
     * @param view Queue view widget
     * @param object Queue object
     * @param parent Parent item
     */
    QueueViewItem(QueueView *view, KFTPQueue::QueueObject *object, TQListViewItem *parent);
    
    /**
     * Class destructor.
     */
    ~QueueViewItem();
    
    /**
     * Refresh the visual representation with data from the actual queue object.
     */
    void refresh();
    
    /**
     * Associate a queue object with this item.
     *
     * @param object A valid object pointer
     */
    void setObject(KFTPQueue::QueueObject *object) { m_queueObject = object; }
    
    /**
     * Returns the currently associated queue object.
     */
    KFTPQueue::QueueObject *getObject() const { return m_queueObject; }
    
    /**
     * Returns the last child item.
     */
    TQListViewItem *lastChild() const { return m_lastChild; }
    
    /**
     * Moves this item one position up.
     */
    void moveUp();
    
    /**
     * Moves this item one position down.
     */
    void moveDown();
    
    /**
     * Moves this item to parent's top.
     */
    void moveToTop();
    
    /**
     * Moves this item to parent's bottom.
     */
    void moveToBottom();
    
    /**
     * @overload
     * Reimplemented from TQListViewItem.
     */
    void insertItem(TQListViewItem *newChild);
    
    /**
     * @overload
     * Reimplemented from TQListViewItem.
     */
    void takeItem(TQListViewItem *item);
    
    /**
     * @overload
     * Reimplemented from TQListViewItem for text colors.
     */
    void paintCell(TQPainter *p, const TQColorGroup &cg, int column, int width, int alignment);
    
    /**
     * @overload
     * Reimplemented from TQListViewItem.
     */
    void sortChildItems(int, bool) {}
protected:
    /**
     * Init the item.
     */
    void init();
private:
    TQListViewItem *m_lastChild;
    
    KFTPQueue::QueueObject *m_queueObject;
    QueueView *m_queueView;
    long m_queueId;
};

/**
 * The list view widget.
 *
 * @author Jernej Kos
 */
class QueueListView : public ListView {
public:
    /**
     * Class constructor.
     *
     * @param parent Parent widget
     */
    QueueListView(TQWidget *parent);
    
    /**
     * @overload
     * Reimplemented from TQListView.
     */
    void insertItem(TQListViewItem *item);
    
    /**
     * @overload
     * Reimplemented from TQListView.
     */
    void setSorting(int, bool = true) { TQListView::setSorting(-1); }
};

/**
 * A widget for displaying and manipulating the current queue.
 *
 * @author Jernej Kos
 */
class QueueView : public TQWidget
{
friend class QueueViewItem;
TQ_OBJECT
public:
    /**
     * Class constructor.
     */
    QueueView(TQWidget *parent, const char *name);
    
    /**
     * Load queue list layout from the configuration file.
     */
    void loadLayout();
    
    /**
     * Save queue list layout to the configuration file.
     */
    void saveLayout();
public slots:    
    void updateActions();
protected:
    /**
     * Initialize actions.
     */
    void initActions();
    
    /**
     * Initialize toolbar widgets.
     */
    void initToolBar();
private:
    TDEActionCollection *m_actionCollection;
    
    // Actions
    TDEAction *m_launchAction;
    TDEAction *m_abortAction;
    TDEAction *m_removeAction;
    TDEAction *m_removeAllAction;
    TDEAction *m_moveUpAction;
    TDEAction *m_moveDownAction;
    TDEAction *m_moveTopAction;
    TDEAction *m_moveBottomAction;
    TDEAction *m_editAction;
    
    // Toolbar Actions
    TDEAction *m_loadAction;
    TDEAction *m_saveAction;
    TDEAction *m_startAction;
    TDEAction *m_pauseAction;
    TDEAction *m_stopAction;
    TDEAction *m_addAction;
    TDEAction *m_searchAction;
    TDEToggleAction *m_filterAction;
    
    TDEListViewSearchLine *m_searchField;

    TDEToolBar *m_toolBar;
    TDEToolBar *m_searchToolBar;
    QueueListView *m_queue;
    
    TQIntDict<QueueViewItem> m_queuedItems;
private slots:
    void slotObjectRemoved(long);
    void slotObjectUpdated();
    void slotTransferAdded(KFTPQueue::Transfer*);
    void slotSiteAdded(KFTPQueue::Site*);
    
    void contextMenuRequested(TDEListView*, TQListViewItem*, const TQPoint&);
    
    void slotSearchEraseClicked();
    
    // Slots for actions
    void slotLaunch();
    void slotAbort();
    void slotRemove();
    void slotRemoveAll();
    void slotMoveUp();
    void slotMoveDown();
    void slotMoveTop();
    void slotMoveBottom();
    void slotEdit();
    
    void slotLoad();
    void slotSave();
    void slotStart();
    void slotPause();
    void slotStop();
    void slotAdd();
    void slotSearch();
    void slotFilter();
    
    void slotDownloadLimitChanged(int value);
    void slotUploadLimitChanged(int value);
    void slotThreadCountChanged(int value);
};

}

#endif
