/*
 * This file is part of the KDE project
 *
 * Copyright (c) 2005 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "noisefilter.h"

#include <stdlib.h>
#include <vector>

#include <tqpoint.h>

#include <kdebug.h>
#include <kiconloader.h>
#include <tdeinstance.h>
#include <kgenericfactory.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <knuminput.h>
#include <kstandarddirs.h>
#include <tdetempfile.h>

#include <kis_image.h>
#include <kis_iterators_pixel.h>
#include <kis_layer.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_types.h>

#include "kis_wdg_noise.h"
#include "wdgnoiseoptions.h"

typedef KGenericFactory<ChalkNoiseFilter> ChalkNoiseFilterFactory;
K_EXPORT_COMPONENT_FACTORY( chalknoisefilter, ChalkNoiseFilterFactory( "chalk" ) )

ChalkNoiseFilter::ChalkNoiseFilter(TQObject *parent, const char *name, const TQStringList &)
        : KParts::Plugin(parent, name)
{
    setInstance(ChalkNoiseFilterFactory::instance());


    if (parent->inherits("KisFilterRegistry")) {
        KisFilterRegistry * manager = dynamic_cast<KisFilterRegistry *>(parent);
        manager->add(new KisFilterNoise());
    }
}

ChalkNoiseFilter::~ChalkNoiseFilter()
{
}

KisFilterNoise::KisFilterNoise() : KisFilter(id(), "other", i18n("&Random Noise..."))
{
}

KisFilterConfiguration* KisFilterNoise::configuration(TQWidget* w)
{
    KisWdgNoise* wN = dynamic_cast<KisWdgNoise*>(w);
    KisFilterConfiguration* config = new KisFilterConfiguration(id().id(), 1);
    if(wN)
    {
        config->setProperty("level", wN->widget()->intLevel->value() );
        config->setProperty("opacity", wN->widget()->intOpacity->value() );
    }
    return config;
}

KisFilterConfigWidget * KisFilterNoise::createConfigurationWidget(TQWidget* parent, KisPaintDeviceSP dev)
{
    return new KisWdgNoise((KisFilter*)this, (TQWidget*)parent, i18n("Configuration of noise filter").ascii());
}

void KisFilterNoise::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, KisFilterConfiguration* config, const TQRect& rect)
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);
    
    setProgressTotalSteps(rect.width() * rect.height());

    KisColorSpace * cs = src->colorSpace();
    TQ_INT32 psize = cs->pixelSize();
    
    TQVariant value;
    int level = (config && config->getProperty("level", value)) ? value.toInt() : 50;
    int opacity = (config && config->getProperty("opacity", value)) ? value.toInt() : 100;
    
    KisRectIteratorPixel dstIt = dst->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), true );
    KisRectIteratorPixel srcIt = src->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), false);
    
    TQ_UINT8* interm = new TQ_UINT8[ cs->pixelSize() ];
    TQ_UINT32 threshold = (RAND_MAX / 100) * (100 - level);
    
    TQ_UINT8 weights[2];
    weights[0] = (255 * opacity) / 100; weights[1] = 255 - weights[0];
    const TQ_UINT8* pixels[2];
    pixels[0] = interm;
    while(!srcIt.isDone())
    {
        if(rand() > threshold)
        {
            TQColor c = tqRgb((double)rand()/RAND_MAX * 255,(double)rand()/RAND_MAX * 255,(double)rand()/RAND_MAX * 255);
            cs->fromTQColor( c, interm, 0 );
            pixels[1] = srcIt.oldRawData();
            cs->mixColors( pixels, weights, 2, dstIt.rawData() );
        }
        ++srcIt;
        ++dstIt;
        incProgress();
    }
    
    delete interm;
    setProgressDone(); // Must be called even if you don't really support progression
}
