/***************************************************************************
                          showrecord.h  -  description
                             -------------------
    begin                : Thu Dec 28 2000
    copyright            : (C) 2000-2001 by Eggert Ehmke
    email                : eggert.ehmke@berlin.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SHOWRECORDELEM_H
#define SHOWRECORDELEM_H

//TQt headers
#include <tqdom.h>
#include <tqstring.h>
#include <tqlistview.h>
#include <tqdatetime.h>

//KDE headers
#include <tdelocale.h>
#include <tdeglobal.h>
#include <kdebug.h>
#include <mimelib/datetime.h>
#include <kmdcodec.h>

//KShowmail headers
#include "showlistviewitem.h"
#include "decodeRFC2047.h"
#include "constants.h"
#include "kshowmailview.h"
#include "showheaderdialog.h"
#include "headerfilter.h"
#include "filterlog.h"

using namespace Constants;

//forward class declarations
class KshowmailView;
class FilterElem;

/**
 * @short Represents a mail.
 *
 * It contains UID, Subject, From, To and date of the mail. Subject, From and To
 * will be read out from the given header. It also contains pointer to the
 * corresponding list view entry and maybe to a matching filter.
 * All mail objects are held by ShowRecord.
 * @see ShowRecord
 * @see ConfigElem
 * @author Eggert Ehmke
 */

class ShowRecordElem
{

  public:

    /**
     * Returned by showHeader(), if the user has
     * clicked OK.
     */
    static const int continueShowHeaders;

    /**
     * Returned by showHeader(), if the user has
     * clicked Cancel.
     */
    static const int cancelShowHeaders;

    /**
     * Constructor
     */
	ShowRecordElem ();

    /**
     * Constructor
     * @param number number of the mail on the server
     * @param uid Unique ID of the mail
     * @param isNew true - mail is new; false - mail is not new
     */
    ShowRecordElem( int number, TQString& uid, bool isNew );

    /**
     * Sets the mail header and extracts From, To, Subject, Date and Content Type.
     * All parts of the header are coded according RFC 2047.
     * @param header the mail header
     */
    void setHeader( const TQString& header );

    /**
     * Returns the whole header.
     * @return the mail header
     */
    TQString header() const;

    /**
     * Returns the sender address.
     * @return sender address
     */
    TQString from() const;

    /**
     * Returns the addressee.
     * @return the addressee
     */
    TQString to() const;

    /**
     * Returns the subject.
     * @return mail subject
     */
    TQString subject() const;

    /**
     * Returns the stored content type.
     * @return content type
     */
    TQString content() const;


    /**
     * Returns the date on which the mail was sent, formated to the current
     * locale's conventions.
     * @return the date
     */
    TQString date() const;

    /**
     * Returns the date on which the mail was sent.
     * The format corresponds to the ISO 8601 extended specification for
     * representations of dates and times, which is YYYY-MM-DDTHH:MM:SS
     * @return the date
     */
    TQString strUnixTime () const;

    /**
     * Returns the date and time on which the mail was sent.
     * @return the date
     */
    TQDateTime sentDateTime() const;

    /**
     * Sets the unique mail ID.
     * @param uid the ID
     */
    void setUIDL( const TQString& uid );

    /**
     * Returns the unique ID of this mail.
     * @return UID
     */
    TQString uidl() const;

    /**
     * Sets the size of the mail.
     * @param size Size
     */
    void setSize( int size );

    /**
     * Returns the mail size as numeric value.
     * @return size of the mail
     */
    int size() const;

    /**
     * Returns the mail size as string.
     * The String has a length of 8 and the number is right-aligned to allow sorting.
     */
    TQString strSize () const;

    /**
     * Returns the mails size as string with prefix
     */
    TQString strSizePrefix() const;

    /**
     * Sets whether the mail is new or not.
     * @param isnew true - mail is new; false - mail is not new
     */
    void setNew( bool isnew );

    /**
     * Returns whether the mail is new or not.
     * @return true - mail is new
     * @return false - mail is not new
     */
    bool isNew() const;

    /**
     * Returns whether the mail is new or not.
     * @return localized String "new", if the mail is new
     * @return localized String "old", if the mail is not new
     */
    TQString state() const;

    /**
     * Sets the number of this mail.
     * @param n mail number
     */
    void setNumber( int n );

    /**
     * Returns the number of this mail.
     * @return mail number
     */
    int number() const;

    /**
     * Sets the pointer to the corresponding list view item.
     * @see m_pItem
     * @see ShowListViewItem
     * @param item pointer to the corresponding item
     */
    void setViewItem( ShowListViewItem* item );

    /**
     * Returns the pointer to the corresponding list view item.
     * @see m_pItem
     * @see ShowListViewItem
     * @return pointer to the item
     */
    ShowListViewItem* viewItem() const;

    /**
     * Looks for a filter matching the mail header and returns the action recommend by this filter.
     * If the action is MARK, it set markAtViewRefresh to True so the related view entry will be marked
     * at the next view refresh.
     * @param filter pointer to the header filter
     * @param account name of the account
     * @param mailbox reference to a TQString object in which the mailbox name shall be written if the action is MOVE
     * @param log pointer to the filter log
     * @return recommend action
     * @see FilterAction_Type
     * @see HeaderFilter
     * @see markAtViewRefresh
     */
    FilterAction_Type applyHeaderFilter( HeaderFilter* filter, TQString account, TQString& mailbox, FilterLog* log = NULL );

    /**
     * Read a mail (header, size, uid and number) from the given DOM element
     * and store it in this mail object.
     * The element has to contains the attributes "uidl", "size", "number" and
     * a child node called "header".
     * @param elem a DOM element which contains a mail
     */
    void readOptions( TQDomElement& elem );

    /**
     * Creates a new DOM element in the given DOM document and add it to given
     * DOM parent element
     * @param doc the DOM document in that all options are stored
     * @param parent the parent element (account) of the mail
     */
    void saveOptions( TQDomDocument& doc, TQDomElement& parent );

    /**
     * Returns whether the mail is selected in the list view.
     * @return true - mail is selected
     * @return false - mail is not selected
     */
    bool isSelected() const;

    /**
     * Decodes the given mail body.
     * The mail object has to do this job, because
     * it has stored the mail header and therefore it knows the content
     * type and encoding of the mail.
     * It decodes just the text or html part of the body. The rest of it
     * will be rejected.
     * @param body the encoded mail (including header)
     * @param preferHTML decode HTML part if present
     * @return decoded mail body
     */
    TQString decodeMailBody( TQByteArray body, bool preferHTML ) const;

    /**
     * Shows the header in a dialog box.
     * @param account account name
     * @return continueShowHeaders - the user has OK clicked. The invoking function will show the next header
     * @return cancelShowHeaders - the user has Cancel clicked. The invoking function will not show the next header.
     */
    int showHeader( TQString& account );

    /**
     * Writes this mail into the Move Log.
     * @param log pointer to the log
     * @param account name of the account
     * @param mailbox name of the mailbox
     */
    void writeToMoveLog( FilterLog* log, TQString account, TQString mailbox );

    /**
     * Writes this mail into the Delete Log.
     * @param log pointer to the log
     * @param account name of the account
     */
    void writeToDeleteLog( FilterLog* log, TQString account );

    /**
     * The mail will be marked at the next view refresh.
     */
    void setMarkAtNextViewRefresh();

  private:

    /**
     * Mail header; coded according RFC 2047
     */
    TQCString m_header;

    /**
     * The sender address; coded according RFC 2047
     */
    TQCString m_from;

    /**
     * The addressee; coded according RFC 2047
     */
    TQCString m_to;

    /**
     * The subject; coded according RFC 2047
     */
    TQCString m_subject;

    /**
     * The date on which the mail was sent
     */
    TQDateTime m_unixDate;

    /**
     * The content type
     */
    TQCString m_content;

    /**
     * The unique ID of the mail.
     */
    TQString m_uid;

    /**
     * Size of the mail
     */
    int m_size;

    /**
     * It is set to true when the mail is new.
     */
    bool m_new;

    /**
     * The mail number of the account
     */
    int m_nNumber;

    /**
     * true - the mail shall be marked at the next mail view refresh.
     * It will be set to True by applyHeaderFilter() if the recommend filter action is MARK.
     * It is used by setViewItem(). This methode marks the related list entry and after then it set
     * this variable to False.
     */
    bool markAtViewRefresh;

    /**
     * Pointer to the corresponding list view entry. Used to check
     * whether the entry is highlighted and we have to process this mail.
     * @see ShowListViewItem
     */
    ShowListViewItem* m_pItem;

    /**
     * Searches in the header for a line which starts with the
     * given item.
     * For example:
     * scanHeader( "From") will find the line
     * "From: Ulrich Weigelt <ulrich.weigelt@gmx.de>"
     * and returns "Ulrich Weigelt <ulrich.weigelt@gmx.de>"
     * Returns an empty string (""), if nothing was found.
     * @param item the search item
     * @return the content of the found line
     */
    TQCString scanHeader( const TQString& item ) const;

    /**
     * Sets the given string as sender address
     * @param from sender address
     */
    void setFrom( const TQCString& from );

    /**
     * Sets the given string as recipient address
     * @param to recipient address
     */
    void setTo( const TQCString& to );

    /**
     * Sets the given string as subject
     * @param subject the mail subject
     */
    void setSubject( const TQCString& subject );

    /**
     * Converts the given string to a date-time value and stores it.
     * @param date the date
     */
    void setDate( const TQCString& date );

    /**
     * Sets the given string as content type.
     * @param content the content type
     */
    void setContent( const TQCString& content );

    /**
     * Returns the boundary, if the mail has a multi part body.
     * Otherwise it returns an empty string.
     * @return boundary
     */
    TQString getBoundary() const;

    /**
     * Returns the char set of the content (e.g. iso-8859-1).
     * If no char set is denoted, it will returns an empty string.
     * @return charset
     */
    TQString getCharset() const;

    /**
     * Returns the content transfer encoding of the content
     * (e.g. quoted-printable).
     * If no encoding is denoted, it will returns an empty string.
     * @return content transfer encoding
     */
    TQString getTransferEncoding() const;

};

#endif
