/***************************************************************************
 *
 * tdenetman-devicestore_dbus.cpp - A NetworkManager frontend for TDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa        <hschaa@suse.de>, <helmut.schaa@gmx.de>
 * Author: Timothy Pearson <kb9vqf@pearsoncomputing.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

// tqt headers
#include <tqwidget.h>
#include <tqcombobox.h>
#include <tqtabwidget.h>
#include <tqpushbutton.h>
#include <tqwidgetstack.h>
#include <tqapplication.h>
#include <tqlabel.h>

// tde headers
#include <kiconloader.h>
#include <kdebug.h>
#include <kpushbutton.h>
#include <tdelocale.h>
#include <tdemessagebox.h>

// tdenm headers
#include "tdenetman-connection_setting_widget_interface.h"
#include "tdenetman-connection_settings_dialog.h"
#include "tdenetman-connection_setting_cdma_widget.h"
#include "tdenetman-connection_setting_gsm_widget.h"
#include "tdenetman-connection_setting_ppp_widget.h"
#include "tdenetman-connection_setting_serial_widget.h"
#include "tdenetman-connection_setting_wireless_widget.h"
#include "tdenetman-connection_setting_wireless_security_widget.h"
#include "tdenetman-connection_setting_ipv4_widget.h"
#include "tdenetman-connection_setting_info_widget.h"
#include "tdenetman-connection_setting_vpn_widget.h"
#include "tdenetman-wireless_network.h"
#include "tdenetman-wireless_manager.h"

using namespace ConnectionSettings;

ConnectionSettingsDialogImpl::ConnectionSettingsDialogImpl(TDENetworkConnection* conn, bool new_conn, TQByteArray networkextid, TQWidget* parent, const char* name, bool modal, WFlags fl)
	: ConnectionSettingsDialog(parent, name, modal, fl)
	, _disable_next_button(false)
	, _conn(conn)
	, _new_conn(new_conn)
	, _networkextid(networkextid)
{
	updateDialogForDeviceType();
	
	// get notified if device combo changes
	connect(btnConnect, TQ_SIGNAL(clicked()), this, TQ_SLOT( slotConnect()) );
	connect(pbNext, TQ_SIGNAL(clicked()), this, TQ_SLOT( slotNext()) );
	connect(pbBack, TQ_SIGNAL(clicked()), this, TQ_SLOT( slotBack()) );
	connect(pbCancel, TQ_SIGNAL(clicked()), this, TQ_SLOT( slotCancel()) );
	connect(pbSave, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotSave()) );

	// nice images for the buttons
	btnConnect->setIconSet(SmallIcon ("connect_creating", TQIconSet::Automatic));
	pbNext->setIconSet(SmallIcon ("1rightarrow", TQIconSet::Automatic));
	pbBack->setIconSet(SmallIcon ("1leftarrow", TQIconSet::Automatic));
	pbCancel->setIconSet(SmallIcon("cancel", TQIconSet::Automatic));
	pbSave->setIconSet(SmallIcon("ok", TQIconSet::Automatic));

	// enable or disable buttons according to the current state
	slotEnableButtons();
}

ConnectionSettingsDialogImpl::~ConnectionSettingsDialogImpl()
{
	kdDebug() << "ConnectionSettingsDialogImpl::~ConnectionSettingsDialogImpl" << endl;
}

TQValueList<WidgetInterface*>
ConnectionSettingsDialogImpl::createWidgetsForWireless(TDENetworkConnection* conn, bool new_conn)
{
	TQValueList<WidgetInterface*> ret;

	// widgetlist for wireless connection
	ret.append(new ConnectionSettings::WirelessWidgetImpl(conn, new_conn, TQByteArray(), this));
	ret.append(new ConnectionSettings::WirelessSecurityWidgetImpl(conn, new_conn, this));
	ret.append(new ConnectionSettings::IPv4WidgetImpl(conn, this));
	ret.append(new ConnectionSettings::InfoWidgetImpl(conn, this));

	if (ret.isEmpty())
	{
		// we have a problem here, NetworkManager asked for a setting we do not have
		kdError() << k_funcinfo << "Unexpected setting requested" << endl;
	}

	return ret;
}

TQValueList<WidgetInterface*>
ConnectionSettingsDialogImpl::createWidgetsForWirelessKnownESSID(TDENetworkConnection* conn, bool new_conn, const TQByteArray& essid)
{
	TQValueList<WidgetInterface*> ret;

	// widgetlist for wireless connection
	ret.append(new ConnectionSettings::WirelessWidgetImpl(conn, new_conn, essid, this));
	ret.append(new ConnectionSettings::WirelessSecurityWidgetImpl(conn, new_conn, this));
	ret.append(new ConnectionSettings::IPv4WidgetImpl(conn, this));
	ret.append(new ConnectionSettings::InfoWidgetImpl(conn, this));

	if (ret.isEmpty())
	{
		// we have a problem here, NetworkManager asked for a setting we do not have
		kdError() << k_funcinfo << "Unexpected setting requested" << endl;
	}

	return ret;
}

TQValueList<WidgetInterface*>
ConnectionSettingsDialogImpl::createWidgetsForWired(TDENetworkConnection* conn, bool new_conn)
{
	TQValueList<WidgetInterface*> ret;

	// widgetlist for wired connection
	ret.append(new ConnectionSettings::IPv4WidgetImpl(conn, this));
	ret.append(new ConnectionSettings::InfoWidgetImpl(conn, this));

	return ret;
}

TQValueList<WidgetInterface*>
ConnectionSettingsDialogImpl::createWidgetsForVPN(TDENetworkConnection* conn, bool new_conn)
{
	TQValueList<WidgetInterface*> ret;

	// widgetlist for virtual private connection
	ret.append(new ConnectionSettings::VPNWidgetImpl(conn, new_conn, this));
	ret.append(new ConnectionSettings::InfoWidgetImpl(conn, this));

	return ret;
}

TQValueList<WidgetInterface*>
ConnectionSettingsDialogImpl::createWidgetsForCDMA(TDENetworkConnection* conn, bool new_conn)
{
	TQValueList<WidgetInterface*> ret;

	// widgetlist for wired connection
	ret.append(new ConnectionSettings::CDMAWidgetImpl(conn, this));
	ret.append(new ConnectionSettings::SerialWidgetImpl(conn, this));
	ret.append(new ConnectionSettings::PPPWidgetImpl(conn, this));
	ret.append(new ConnectionSettings::IPv4WidgetImpl(conn, this));
	ret.append(new ConnectionSettings::InfoWidgetImpl(conn, this));

	return ret;
}

TQValueList<WidgetInterface*>
ConnectionSettingsDialogImpl::createWidgetsForGSM(TDENetworkConnection* conn, bool new_conn)
{
	TQValueList<WidgetInterface*> ret;

	// widgetlist for wired connection
	ret.append(new ConnectionSettings::GSMWidgetImpl(conn, this));
	ret.append(new ConnectionSettings::PPPWidgetImpl(conn, this));
	ret.append(new ConnectionSettings::SerialWidgetImpl(conn, this));
	ret.append(new ConnectionSettings::IPv4WidgetImpl(conn, this));
	ret.append(new ConnectionSettings::InfoWidgetImpl(conn, this));

	return ret;
}

void
ConnectionSettingsDialogImpl::createWidgetsForConnection(TDENetworkConnection* conn, bool new_conn)
{
	/*
	  Currently two modes:
	    * dev == NULL -> Connection should be edited without the wish to start a connection
	    * dev != NULL -> A connection should be edited for connection on this device
	*/

	if (!conn) {
		// TODO: create an empty widget and show an error
		kdWarning() << k_funcinfo << "Not handled yet" << endl;
	}
	else {
		TQValueList<WidgetInterface*> widgets;
		// TODO: move to a factory class
		if (conn->type() == TDENetworkConnectionType::WiFi) {
			if (_networkextid.isNull()) {
				widgets = createWidgetsForWireless(conn, new_conn);
			}
			else {
				widgets = createWidgetsForWirelessKnownESSID(conn, new_conn, _networkextid);
			}
		}
		else if (conn->type() == TDENetworkConnectionType::WiredEthernet) {
			widgets = createWidgetsForWired(conn, new_conn);
		}
		else if (conn->type() == TDENetworkConnectionType::Modem) {
			TDEModemConnection* modemconn = dynamic_cast<TDEModemConnection*>(conn);
			if (modemconn) {
				if (modemconn->type == TDEModemConnectionType::CDMA) {
					widgets = createWidgetsForCDMA(conn, new_conn);
				}
				else if (modemconn->type == TDEModemConnectionType::GSM) {
					widgets = createWidgetsForGSM(conn, new_conn);
				}
				else {
					kdWarning() << k_funcinfo << "Specific modem type not handled yet" << endl;
				}
			}
		}
		else if (conn->type() == TDENetworkConnectionType::VPN) {
			widgets = createWidgetsForVPN(conn, new_conn);
		}
		else {
			kdWarning() << k_funcinfo << "Not handled yet" << endl;
		}

		int id;
		for (TQValueList<WidgetInterface*>::Iterator it = widgets.begin(); it != widgets.end(); ++it) {
			id = wstackSettings->addWidget(*it);
			_widgetIds.append(id);
		}

		if (widgets.begin() != widgets.end()) {
			activateWidget(*widgets.begin());
		}
	}
}

void
ConnectionSettingsDialogImpl::updateDialogForDeviceType()
{
	// clear our tabview first
	TQWidget *page = NULL;
	while ( (page = wstackSettings->visibleWidget()) != NULL)
	{
		wstackSettings->removeWidget(page);
		delete page;
	}

	if (_conn) {
		createWidgetsForConnection(_conn, _new_conn);
	}
	else {
		// this should never happen
		TQLabel* lbl = new TQLabel(wstackSettings, "Unknown Device Type");
		wstackSettings->addWidget(lbl);

		wstackSettings->raiseWidget(lbl);
	}
}

int
ConnectionSettingsDialogImpl::getDeviceTypeFromConnection(TDENetworkConnection* conn)
{
	if (conn->type() == TDENetworkConnectionType::WiFi) {
		return TDENetworkDeviceType::WiFi;
	}
	else if (conn->type() == TDENetworkConnectionType::WiredEthernet) {
		return TDENetworkDeviceType::WiredEthernet;
	}
	else if (conn->type() == TDENetworkConnectionType::Modem) {
		return TDENetworkDeviceType::Modem;
	}

	return TDENetworkDeviceType::Other;
}

void
ConnectionSettingsDialogImpl::slotConnect()
{
	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	// add/save the connection
	if (nm && nm->saveConnection(_conn))
	{
		// activate device
		if (_conn)
		{
	// 		// VPN connection needs a special specific object
	// 		if (_conn->getType() == NM_SETTING_VPN_SETTING_NAME) {
	// 			TQT_DBusObjectPath act_conn = nm->getDefaultActiveConnection();
	// 			TQT_DBusObjectPath device   = nm->getDeviceForActiveConnection(act_conn);
	// 			nm->ActivateConnectionAsync(id, "org.freedesktop.NetworkManagerUserSettings", _conn->getObjectPath(), device, act_conn, err);
	// 		}
	// 		else {
				// we need to call ActivateDevice async
				nm->initiateConnection(_conn->UUID);
	// 		}
		}
	}
	else {
		KMessageBox::error(this, i18n("<qt><b>Unable to save network connection!</b><p>Potential causes:<br> * Insufficient permissions<br> * NetworkManager not running<br> * DBUS failure</qt>"), i18n("Unable to perform requested operation"));
	}

	emit connectionSaved();
	this->close(true);
}

void
ConnectionSettingsDialogImpl::slotSave()
{
	// Make sure settings are committed
	TQValueList<int>::Iterator current = _widgetIds.find(wstackSettings->id(wstackSettings->visibleWidget()));
	if (current != _widgetIds.fromLast())
	{
		WidgetInterface* widget = NULL;

		// let the widget know about it being the active one
		widget = dynamic_cast<WidgetInterface*>(wstackSettings->widget(*current));
		if (widget) {
			deactivateWidget(widget);
		}
	}

	TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();

	// save the connection
	if (!nm || !nm->saveConnection(_conn))
	{
		KMessageBox::error(this, i18n("<qt><b>Unable to save network connection!</b><p>Potential causes:<br> * Insufficient permissions<br> * NetworkManager not running<br> * DBUS failure</qt>"), i18n("Unable to perform requested operation"));
	}

	emit connectionSaved();
	this->close(true);
}

void
ConnectionSettingsDialogImpl::slotCancel()
{
	close();
}

void
ConnectionSettingsDialogImpl::slotNext()
{
	TQValueList<int>::Iterator current = _widgetIds.find(wstackSettings->id(wstackSettings->visibleWidget()));
	if (current != _widgetIds.fromLast())
	{
		WidgetInterface* widget = NULL;

		// let the widget know about it being the active one
		widget = dynamic_cast<WidgetInterface*>(wstackSettings->widget(*current));
		if (widget) {
			deactivateWidget(widget);
		}

		// next widget
		current++;
	
		// let the widget know about it being the active one
		widget = dynamic_cast<WidgetInterface*>(wstackSettings->widget(*current));
		if (widget) {
			activateWidget(widget);
		}
	}
	slotEnableButtons();
}

void
ConnectionSettingsDialogImpl::slotBack()
{
	TQValueList<int>::Iterator current = _widgetIds.find(wstackSettings->id(wstackSettings->visibleWidget()));
	if (current != _widgetIds.begin())
	{
		WidgetInterface* widget = NULL;

		// let the widget know about it being the active one
		widget = dynamic_cast<WidgetInterface*>(wstackSettings->widget(*current));
		if (widget) {
			deactivateWidget(widget);
		}

		// one back
		current--;

		// let the widget know about it being the active one
		widget = dynamic_cast<WidgetInterface*>(wstackSettings->widget(*current));
		if (widget) {
			activateWidget(widget);
		}
	}
	slotEnableButtons();
}

void
ConnectionSettingsDialogImpl::activateWidget(WidgetInterface* widget)
{
	// allow the widget to press next
	connect(widget, TQ_SIGNAL(next()), this, TQ_SLOT(slotNext()));

	// allow the widget to do some initialization
	widget->Activate();

	// show the widget
	wstackSettings->raiseWidget(widget);

}

void
ConnectionSettingsDialogImpl::deactivateWidget(WidgetInterface* widget)
{
	// allow the widget to press next
	disconnect(widget, TQ_SIGNAL(next()), this, TQ_SLOT(slotNext()));

	// allow the widget to do some initialization
	widget->Deactivate();
}

void
ConnectionSettingsDialogImpl::slotEnableButtons()
{
	// enable the buttons according to the current state
	
	TQValueList<int>::Iterator current = _widgetIds.find(wstackSettings->id(wstackSettings->visibleWidget()));
	bool enabled;

	/*
	  Next: enabled if another widget is available
	*/
	enabled = true;
	if (current == _widgetIds.fromLast() || _disable_next_button) {
		enabled = false;
	}
	pbNext->setEnabled(enabled);

	/*
	 Back: enabled if the current widget has a predecessor
	*/
	enabled = true;
	if (current == _widgetIds.begin()) {
		enabled = false;
	}
	pbBack->setEnabled(enabled);

	/*
	 Connect: only show connect if the connection is valid
	*/
	if (_conn) {
		TDEGlobalNetworkManager* nm = TDEGlobal::networkManager();
		TDENetworkErrorStringMap errorStringMap;
		TDENetworkConnectionErrorFlags::TDENetworkConnectionErrorFlags errorFlags;
		bool ret;
		ret = (nm ? nm->verifyConnectionSettings(_conn, &errorFlags, &errorStringMap) : 0);
		btnConnect->setEnabled(ret);
		if (!ret) {
			// FIXME
			// Check the error flags and the current dialog type to determine if an error message is warranted
			// KMessageBox::error(this, errorString, i18n("Invalid Settings Detected"));
		}
		pbSave->setEnabled(nm ? nm->verifyConnectionSettings(_conn) : 0);
	}
}

#include "tdenetman-connection_settings_dialog.moc"
