/*  This file is part of the KDE KMobile library
    Copyright (C) 2003 Helge Deller <deller@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#include <tqdragobject.h>
#include <kprinter.h>
#include <tqpainter.h>
#include <tqpaintdevicemetrics.h>

#include <tdeglobal.h>
#include <tdelocale.h>
#include <kiconloader.h>
#include <tdemenubar.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <tdeaccel.h>
#include <tdeio/netaccess.h>
#include <tdefiledialog.h>
#include <tdemessagebox.h>
#include <tdeconfig.h>

#include <kedittoolbar.h>

#include <tdestdaccel.h>
#include <tdeaction.h>
#include <kstdaction.h>

#include <kpushbutton.h>
#include <kdebug.h>
#include <kstandarddirs.h>

#include "kmobile.h"
#include "pref.h"

#include "systemtray.h"

#include "kmobileitem.h"
#include "kmobile_selectiondialog.h"

KMobile::KMobile()
    : TDEMainWindow( 0, "kmobile" )
{
    m_config = new TDEConfig("kmobilerc");

    m_view = new KMobileView(this, m_config);

    // tell the TDEMainWindow that this is indeed the main widget
    setCentralWidget(m_view);

    // then, setup our actions
    setupActions();

    toolBar()->show();
    statusBar()->show();

    // apply the saved mainwindow settings, if any, and ask the mainwindow
    // to automatically save settings if changed: window size, toolbar
    // position, icon size, etc.
    setAutoSaveSettings();

    // allow the view to change the statusbar and caption
    connect(m_view, TQ_SIGNAL(signalChangeStatusbar(const TQString&)),
            this,   TQ_SLOT(changeStatusbar(const TQString&)));

    // restore all configured devices
    restoreAll();

    // setup the system tray
    m_systemTray = new SystemTray(this, "systemTray");
    m_systemTray->show();
    connect(m_systemTray, TQ_SIGNAL(quitSelected()), this, TQ_SLOT(slotQuit()));
}

KMobile::~KMobile()
{
   delete m_config;
}

void KMobile::setupActions()
{
    KStdAction::close(this, TQ_SLOT(dockApplication()), actionCollection());
    KStdAction::quit(kapp, TQ_SLOT(quit()), actionCollection());

    m_toolbarAction = KStdAction::showToolbar(this, TQ_SLOT(optionsShowToolbar()), actionCollection());
    optionsShowToolbar();
    m_statusbarAction = KStdAction::showStatusbar(this, TQ_SLOT(optionsShowStatusbar()), actionCollection());

    KStdAction::keyBindings(this, TQ_SLOT(optionsConfigureKeys()), actionCollection());
    KStdAction::configureToolbars(this, TQ_SLOT(optionsConfigureToolbars()), actionCollection());
    KStdAction::preferences(this, TQ_SLOT(optionsPreferences()), actionCollection());

    new TDEAction(i18n("&Add Device..."), "folder-new", 0,
		this, TQ_SLOT(addDevice()), actionCollection(), "device_add");
    new TDEAction( KGuiItem( i18n("&Remove Device"), "edittrash", i18n("Remove this device") ),
		"Delete", this,  TQ_SLOT(removeDevice()), actionCollection(), "device_remove");
    new TDEAction(i18n("Re&name Device..."), 0, Key_F2,
		this, TQ_SLOT(renameDevice()), actionCollection(), "device_rename");
    new TDEAction(i18n("&Configure Device..."), "configure", 0,
		this, TQ_SLOT(configDevice()), actionCollection(), "device_configure");

    createGUI();

    connect( kapp, TQ_SIGNAL(aboutToQuit()), this, TQ_SLOT(saveAll()) );
}


void KMobile::dockApplication()
{
    // dock to system tray
    hide();
}

bool KMobile::queryClose()
{
    dockApplication();
    return false;
}

bool KMobile::queryExit()
{
    dockApplication();
    return false;
}

void KMobile::slotQuit()
{
    kapp->quit();
}

void KMobile::showMinimized()
{
    dockApplication();
}


void KMobile::saveAll()
{
    m_view->saveAll(); 
}

void KMobile::restoreAll()
{
    m_view->restoreAll(); 
}

void KMobile::fileSave()
{
    // this slot is called whenever the File->Save menu is selected,
    // the Save shortcut is pressed (usually CTRL+S) or the Save toolbar
    // button is clicked

    // save the current file
}


void KMobile::saveProperties(TDEConfig *)
{
    // the 'config' object points to the session managed
    // config file.  anything you write here will be available
    // later when this app is restored
}

void KMobile::readProperties(TDEConfig *config)
{
    // the 'config' object points to the session managed
    // config file.  this function is automatically called whenever
    // the app is being restored.  read in here whatever you wrote
    // in 'saveProperties'

    TQString url = config->readPathEntry("lastURL");
}

void KMobile::optionsShowToolbar()
{
    // this is all very cut and paste code for showing/hiding the
    // toolbar
    if (m_toolbarAction->isChecked())
        toolBar()->show();
    else
        toolBar()->hide();
}

void KMobile::optionsShowStatusbar()
{
    // this is all very cut and paste code for showing/hiding the
    // statusbar
    if (m_statusbarAction->isChecked())
        statusBar()->show();
    else
        statusBar()->hide();
}

void KMobile::optionsConfigureKeys()
{
    KKeyDialog::configure(actionCollection());
}

void KMobile::optionsConfigureToolbars()
{
    // use the standard toolbar editor
    saveMainWindowSettings(TDEGlobal::config(), autoSaveGroup());
}

void KMobile::newToolbarConfig()
{
    // this slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
    // recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
    createGUI();

    applyMainWindowSettings(TDEGlobal::config(), autoSaveGroup());
}

void KMobile::optionsPreferences()
{
    // popup some sort of preference dialog, here
#if 0
    KMobilePreferences dlg;
    if (dlg.exec())
    {
        // redo your settings
    }
#endif
}


void KMobile::renameDevice()
{
    // rename the current selected device
    TQIconViewItem *item = m_view->currentItem();
    if (item)
       item->rename();
}

/*
 * Add a new Device (Dialog)
 */

void KMobile::addDevice()
{
  KMobile_SelectionDialog *dialog = new KMobile_SelectionDialog(m_view);
  if (!dialog)
    return;

  dialog->setCaption( i18n("Add New Mobile or Portable Device") );

  dialog->helpText->setText( i18n("Please select the category to which your new device belongs:") );
  dialog->addButton->setText( i18n("&Scan for New Devices...") );
  dialog->addButton->setDisabled(true);
  dialog->iconView->connect( dialog->iconView, TQ_SIGNAL(doubleClicked(TQIconViewItem*)),
			dialog, TQ_SLOT(accept()) );
  dialog->selectButton->setText( i18n("&Add") );
  dialog->selectButton->connect( dialog->selectButton, TQ_SIGNAL(clicked()), dialog, TQ_SLOT(accept()) );
  dialog->cancelButton->connect( dialog->cancelButton, TQ_SIGNAL(clicked()), dialog, TQ_SLOT(reject()) );

  TDETrader::OfferList list = KMobileItem::getMobileDevicesList();
  TDETrader::OfferListIterator it;
  KService::Ptr ptr;
  for ( it = list.begin(); it != list.end(); ++it ) {
    ptr = *it;
    kdDebug() << TQString("LIBRARY: '%1', NAME: '%2', ICON: '%3', COMMENT: '%4'\n")
		.arg(ptr->library()).arg(ptr->name()).arg(ptr->icon()).arg(ptr->comment());

    TQString iconName = ptr->icon();
    if (iconName.isEmpty())
	iconName = KMOBILE_ICON_UNKNOWN;
    TQPixmap pm = TDEGlobal::instance()->iconLoader()->loadIcon(iconName, TDEIcon::Desktop );
    
    TQIconViewItem *item;
    item = new TQIconViewItem( dialog->iconView, ptr->name(), pm );

    //if (!ptr->comment().isNull())
    //	TQToolTip::add(item->pixmap(), ptr->comment() );
  }

  int index = -1;
  if (dialog->exec() == TQDialog::Accepted)
     index = dialog->iconView->currentItem()->index(); // get index of selected item
  delete dialog;

  if (index<0 || index>=(int)list.count())
    return;

  ptr = list[index];

  // add the new device to the list
  if (!m_view->addNewDevice(m_config, ptr)) {
	KMessageBox::error(this, 
		TQString("<qt>KMobile could not load the <b>%1</b> Device Driver.<p>"
		     "Please use the Skeleton- or Gnokii Device Driver during development.<p>"
		     "This driver will still be visible, but you won't be able to access it "
		     "from Konqueror or any other application.</qt>").arg(ptr->name()),
		kapp->name());
  }

  saveAll();
}

#if 0
/*
 * show dialog to user, in which he may choose and select one of the already
 * configured mobile devices.
 */
KMobileDevice * KMobileFactory::chooseDeviceDialog( TQWidget *parent, 
		enum KMobileDevice::ClassType /*type*/, enum KMobileDevice::Capabilities /*caps*/ )
{
  int num;

  m_parent = parent;

  // do we already have some devices configured ?
  num = readDevicesList();
  if (!num) {
     int answ;
     answ = KMessageBox::questionYesNo(parent,
		i18n( "<qt>You have no mobile devices configured yet.<p>"
			"Do you want to add a device now ?</qt>" ),
		i18n( "TDE Mobile Device Access" ), KStdGuiItem::add(), i18n("Do Not Add") );
     if (answ != KMessageBox::Yes)
	return 0L;
     // add a new device
     addDeviceDialog(parent);
  }
  num = readDevicesList();
  if (!num) 
    return 0L;

  // let the user select one of the configured devices
  KMobile_selectiondialog *dialog = new KMobile_selectiondialog(parent);
  if (!dialog)
    return 0L;

  dialog->addButton->connect( dialog->addButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(slotAddDevice()) );
  dialog->iconView->connect( dialog->iconView, TQ_SIGNAL(doubleClicked(TQIconViewItem*)),
			dialog, TQ_SLOT(accept()) );
  dialog->selectButton->connect( dialog->selectButton, TQ_SIGNAL(clicked()), dialog, TQ_SLOT(accept()) );
  dialog->cancelButton->connect( dialog->cancelButton, TQ_SIGNAL(clicked()), dialog, TQ_SLOT(reject()) );

  for (int i=0; i<countDevices(); i++) {
    KService::Ptr ptr;
    ptr = ServiceForEntry(i);
    if (!ptr)
	continue;

    // kdDebug() << TQString("LIBRARY: '%1', NAME: '%2', ICON: '%3', COMMENT: '%4'   #%5\n")
    //		.arg(ptr->library()).arg(ptr->name()).arg(ptr->icon()).arg(ptr->comment()).arg(i);

    TQString iconName = ptr->icon();
    if (iconName.isEmpty())
	iconName = KMOBILE_ICON_UNKNOWN;
    TQPixmap pm( ::locate("icon", iconName+".png") );
    
    TQIconViewItem *item;
    item = new TQIconViewItem( dialog->iconView, ptr->name(), pm );

  }

  int index = -1;
  if (dialog->exec() == TQDialog::Accepted)
     index = dialog->iconView->currentItem()->index(); // get index of selected item
  delete dialog;

  if (index<0 || index>=countDevices())
    return 0L;

  return getDevice(index);
}
#endif


void KMobile::removeDevice()
{
    // remove the current selected device
    TQIconViewItem *item = m_view->currentItem();
    if (item)
       m_view->removeDevice( item->text() );
}

void KMobile::configDevice()
{
    // configure the current selected device
    TQIconViewItem *item = m_view->currentItem();
    if (item)
       m_view->configDevice( item->text() );
}

void KMobile::changeStatusbar(const TQString& text)
{
    // display the text on the statusbar
    statusBar()->message(text);
}

#include "kmobile.moc"
