/*

 ark -- archiver for the KDE project

 Copyright (C)

 1997-1999: Rob Palmbos palm9744@kettering.edu
 2000: Corel Corporation (author: Emily Ezust, emilye@corel.com)
 2001: Corel Corporation (author: Michael Jarrett, michaelj@corel.com)

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/

// C includes
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

// QT includes
#include <tqdir.h>

// KDE includes
#include <kdebug.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <kprocess.h>

// ark includes
#include "arkwidget.h"
#include "settings.h"
#include "ar.h"

ArArch::ArArch( ArkWidget *_gui, const TQString & _fileName )
  : Arch(_gui, _fileName )
{
  m_archiver_program = m_unarchiver_program = "ar";
  verifyCompressUtilityIsAvailable( m_archiver_program );
  verifyUncompressUtilityIsAvailable( m_unarchiver_program );

  // Do not set headerString - there is none for Ar
  m_numCols = 5;
  m_dateCol = 4; m_fixYear = 8; m_repairMonth = 5; m_fixDay = 6; m_fixTime = 7;

  m_archCols.append(new ArchColumns(1, TQRegExp("[a-zA-Z-]+"), 12)); // Perms
  m_archCols.append(new ArchColumns(2, TQRegExp("[^\\s]+"), 128)); //User/grp
  m_archCols.append(new ArchColumns(3, TQRegExp("[0-9]+"))); // Size
  m_archCols.append(new ArchColumns(5, TQRegExp("[a-zA-Z]+"), 4)); // Month
  m_archCols.append(new ArchColumns(6, TQRegExp("[0-9]+"), 2)); // Day
  m_archCols.append(new ArchColumns(7, TQRegExp("[0-9:]+"), 6)); // Time
  m_archCols.append(new ArchColumns(8, TQRegExp("[0-9]+"), 5)); // Year
  m_archCols.append(new ArchColumns(0, TQRegExp("[^\\s][^\\n]+"), 4096));// File

  kdDebug(1601) << "ArArch constructor" << endl;
}

void ArArch::setHeaders()
{
  ColumnList list;
  list.append( FILENAME_COLUMN );
  list.append( PERMISSION_COLUMN );
  list.append( OWNER_GROUP_COLUMN );
  list.append( SIZE_COLUMN );
  list.append( TIMESTAMP_COLUMN );

  emit headers( list );
}

void ArArch::open()
{
  kdDebug(1601) << "+ArArch::open" << endl;
  setHeaders();

  m_buffer = "";

  TDEProcess *kp = m_currentProcess = new TDEProcess;
  *kp << m_archiver_program << "vt" << m_filename;
  connect( kp, TQ_SIGNAL(receivedStdout(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedTOC(TDEProcess*, char*, int)));
  connect( kp, TQ_SIGNAL(receivedStderr(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));

  connect( kp, TQ_SIGNAL(processExited(TDEProcess*)), this,
	   TQ_SLOT(slotOpenExited(TDEProcess*)));

  if (kp->start(TDEProcess::NotifyOnExit, TDEProcess::AllOutput) == false)
    {
      KMessageBox::error( 0, i18n("Could not start a subprocess.") );
      emit sigOpen(this, false, TQString(), 0 );
    }
  kdDebug(1601) << "-ArArch::open" << endl;
}

void ArArch::create()
{
  TDEProcess *kp = m_currentProcess = new TDEProcess;
  kp->clearArguments();
  *kp << m_archiver_program << "c" << m_filename;

  connect( kp, TQ_SIGNAL(receivedStdout(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));
  connect( kp, TQ_SIGNAL(receivedStderr(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));

  if (kp->start(TDEProcess::Block) == false)
    {
      KMessageBox::error( 0, i18n("Could not start a subprocess.") );
      emit sigCreate(this, false, m_filename,
		     Arch::Extract | Arch::Delete | Arch::Add
		     | Arch::View);
    }
  else
    emit sigCreate(this, true, m_filename,
		   Arch::Extract | Arch::Delete | Arch::Add
		   | Arch::View);
}

void ArArch::addFile( const TQStringList &urls )
{
  kdDebug(1601) << "+ArArch::addFile" << endl;
  TDEProcess *kp = m_currentProcess = new TDEProcess;
  kp->clearArguments();
  *kp << m_archiver_program;

  if (ArkSettings::replaceOnlyWithNewer())
	  *kp << "ru";
  else
	  *kp << "r";

  *kp << m_filename;

  TQStringList::ConstIterator iter;
  KURL url( urls.first() );
  TQDir::setCurrent( url.directory() );
  for (iter = urls.begin(); iter != urls.end(); ++iter )
  {
    KURL fileURL( *iter );
    *kp << fileURL.fileName();
  }

  connect( kp, TQ_SIGNAL(receivedStdout(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));
  connect( kp, TQ_SIGNAL(receivedStderr(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));

  connect( kp, TQ_SIGNAL(processExited(TDEProcess*)), this,
	   TQ_SLOT(slotAddExited(TDEProcess*)));

  if (kp->start(TDEProcess::NotifyOnExit, TDEProcess::AllOutput) == false)
    {
      KMessageBox::error( 0, i18n("Could not start a subprocess.") );
      emit sigAdd(false);
    }

  kdDebug(1601) << "-ArArch::addFile" << endl;
}

void ArArch::unarchFileInternal()
{
  // if m_fileList is empty, we extract all.
  // if m_destDir is empty, abort with error.

  kdDebug(1601) << "+ArArch::unarchFile" << endl;
  TQString dest;

  if (m_destDir.isEmpty() || m_destDir.isNull())
    {
      kdError(1601) << "There was no extract directory given." << endl;
      return;
    }
  else dest = m_destDir;

  // ar has no option to specify the destination directory
  // so I have to change to it.

  bool ret = TQDir::setCurrent(dest);
 // I already checked the validity of the dir before coming here
  Q_ASSERT(ret);

  TDEProcess *kp = m_currentProcess = new TDEProcess;
  kp->clearArguments();

  *kp << m_archiver_program;
  *kp << "vx";
  *kp << m_filename;

  // if the list is empty, no filenames go on the command line,
  // and we then extract everything in the archive.
  if (m_fileList)
    {
      for ( TQStringList::Iterator it = m_fileList->begin();
	    it != m_fileList->end(); ++it )
	{
	  *kp << (*it);
	}
    }

  connect( kp, TQ_SIGNAL(receivedStdout(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));
  connect( kp, TQ_SIGNAL(receivedStderr(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));

  connect( kp, TQ_SIGNAL(processExited(TDEProcess*)), this,
	   TQ_SLOT(slotExtractExited(TDEProcess*)));

  if (kp->start(TDEProcess::NotifyOnExit, TDEProcess::AllOutput) == false)
    {
      KMessageBox::error( 0, i18n("Could not start a subprocess.") );
      emit sigExtract(false);
    }
}

void ArArch::remove(TQStringList *list)
{
  kdDebug(1601) << "+ArArch::remove" << endl;

  if (!list)
    return;

  TDEProcess *kp = m_currentProcess = new TDEProcess;
  kp->clearArguments();

  *kp << m_archiver_program << "d" << m_filename;
  for ( TQStringList::Iterator it = list->begin();
	it != list->end(); ++it )
    {
      TQString str = *it;
      *kp << str;
    }

  connect( kp, TQ_SIGNAL(receivedStdout(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));
  connect( kp, TQ_SIGNAL(receivedStderr(TDEProcess*, char*, int)),
	   this, TQ_SLOT(slotReceivedOutput(TDEProcess*, char*, int)));

  connect( kp, TQ_SIGNAL(processExited(TDEProcess*)), this,
	   TQ_SLOT(slotDeleteExited(TDEProcess*)));

  if (kp->start(TDEProcess::NotifyOnExit, TDEProcess::AllOutput) == false)
    {
      KMessageBox::error( 0, i18n("Could not start a subprocess.") );
      emit sigDelete(false);
    }

  kdDebug(1601) << "-ArArch::remove" << endl;
}


#include "ar.moc"
