/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
**
** This file is part of TQt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/
#include <scriptobject.h>
#include "command.h"
#include "formwindow.h"
#include "widgetfactory.h"
#include "propertyeditor.h"
#include "metadatabase.h"
#include "widgetdatabase.h"
#include "mainwindow.h"
#include "hierarchyview.h"
#include "workspace.h"
#include "actioneditorimpl.h"
#include "actiondnd.h"
#include "formfile.h"

#include <tqmap.h>
#include <tqfeatures.h>
#include <tqwidget.h>
#include <tqmetaobject.h>
#include <tqapplication.h>
#include <tqlayout.h>
#include <tqmessagebox.h>
#include <tqlistbox.h>
#include <tqiconview.h>
#include <tqmultilineedit.h>
#include <tqptrstack.h>
#include <tqheader.h>
#ifndef TQT_NO_TABLE
#include <tqtable.h>
#endif
#include <tqaction.h>

#include <tdelocale.h>


CommandHistory::CommandHistory( int s )
    : current( -1 ), steps( s ), savedAt( -1 )
{
    history.setAutoDelete( true );
    modified = false;
    compressedCommand = 0;
}

void CommandHistory::addCommand( Command *cmd, bool tryCompress )
{
    if ( tryCompress ) {
	if ( !compressedCommand ||
	     compressedCommand->type() != cmd->type() ||
	     !compressedCommand->canMerge( cmd ) ) {
	    checkCompressedCommand();
	    compressedCommand = 0;
	}

	if ( compressedCommand ) {
	    compressedCommand->merge( cmd );
	    modified = true;
	    modificationChanged( modified );
	    return;
	}
	compressedCommand = cmd;
    } else {
	checkCompressedCommand();
    }

    if ( current < (int)history.count() - 1 ) {
	if ( current < savedAt )
	    savedAt = -2;

	TQPtrList<Command> commands;
	commands.setAutoDelete( false );

	for( int i = 0; i <= current; ++i ) {
	    commands.insert( i, history.at( 0 ) );
	    history.take( 0 );
	}

	commands.append( cmd );
	history.clear();
	history = commands;
	history.setAutoDelete( true );
    } else {
	history.append( cmd );
    }

    if ( (int)history.count() > steps ) {
	savedAt--;
	history.removeFirst();
    } else {
	++current;
    }

    emitUndoRedo();
    modified = true;
    modificationChanged( modified );
}

void CommandHistory::undo()
{
    checkCompressedCommand();
    compressedCommand = 0;
    if ( current > -1 ) {
	history.at( current )->unexecute();
	--current;
    }
    emitUndoRedo();
    modified = savedAt != current;
    modificationChanged( modified );
}

void CommandHistory::redo()
{
    checkCompressedCommand();
    compressedCommand = 0;
    if ( current > -1 ) {
	if ( current < (int)history.count() - 1 ) {
	    ++current;
	    history.at( current )->execute();
	}
    } else {
	if ( history.count() > 0 ) {
	    ++current;
	    history.at( current )->execute();
	}
    }
    emitUndoRedo();
    modified = savedAt != current;
    modificationChanged( modified );
}

void CommandHistory::emitUndoRedo()
{
    Command *undoCmd = 0;
    Command *redoCmd = 0;

    if ( current >= 0 && current < (int)history.count() )
	undoCmd = history.at( current );
    if ( current + 1 >= 0 && current + 1 < (int)history.count() )
	redoCmd = history.at( current + 1 );

    bool ua = (undoCmd != 0);
    TQString uc;
    if ( ua )
	uc = undoCmd->name();
    bool ra = (redoCmd != 0);
    TQString rc;
    if ( ra )
	rc = redoCmd->name();
    emit undoRedoChanged( ua, ra, uc, rc );
}

void CommandHistory::setModified( bool m )
{
    modified = m;
    if ( !modified )
	savedAt = current;
    modificationChanged( modified );
}

bool CommandHistory::isModified() const
{
    return modified;
}

void CommandHistory::checkCompressedCommand()
{
    if ( compressedCommand && compressedCommand->type() == Command::SetProperty ) {
	Command *c = compressedCommand;
	compressedCommand = 0;
	if ( !( (SetPropertyCommand*)c )->checkProperty() ) {
	    history.remove( current );
	    --current;
	    emitUndoRedo();
	}
    }
}

// ------------------------------------------------------------

Command::Command( const TQString &n, FormWindow *fw )
    : cmdName( n ), formWin( fw )
{
}

Command::~Command()
{
}

TQString Command::name() const
{
    return cmdName;
}

FormWindow *Command::formWindow() const
{
    return formWin;
}

void Command::merge( Command * )
{
}

bool Command::canMerge( Command * )
{
    return false;
}


// ------------------------------------------------------------

ResizeCommand::ResizeCommand( const TQString &n, FormWindow *fw,
			      TQWidget *w, const TQRect &oldr, const TQRect &nr )
    : Command( n, fw ), widget( w ), oldRect( oldr ), newRect( nr )
{
}

void ResizeCommand::execute()
{
    widget->setGeometry( newRect );
    formWindow()->updateSelection( widget );
    formWindow()->emitUpdateProperties( widget );
    if ( WidgetFactory::layoutType( widget ) != WidgetFactory::NoLayout )
	formWindow()->updateChildSelections( widget );
}

void ResizeCommand::unexecute()
{
    widget->setGeometry( oldRect );
    formWindow()->updateSelection( widget );
    formWindow()->emitUpdateProperties( widget );
    if ( WidgetFactory::layoutType( widget ) != WidgetFactory::NoLayout )
	formWindow()->updateChildSelections( widget );
}

// ------------------------------------------------------------

InsertCommand::InsertCommand( const TQString &n, FormWindow *fw,
			      TQWidget *w, const TQRect &g )
    : Command( n, fw ), widget( w ), geometry( g )
{
}

void InsertCommand::execute()
{
    if ( geometry.size() == TQSize( 0, 0 ) ) {
	widget->move( geometry.topLeft() );
	widget->adjustSize();
    } else {
	TQSize s = geometry.size().expandedTo( widget->minimumSize() );
	s = s.expandedTo( widget->minimumSizeHint() );
	TQRect r( geometry.topLeft(), s );
	widget->setGeometry( r );
    }
    widget->show();
    formWindow()->widgets()->insert( widget, widget );
    formWindow()->clearSelection( false );
    formWindow()->selectWidget( widget );
    formWindow()->mainWindow()->objectHierarchy()->widgetInserted( widget );
}

void InsertCommand::unexecute()
{
    widget->hide();
    formWindow()->selectWidget( widget, false );
    formWindow()->widgets()->remove( widget );
    formWindow()->mainWindow()->objectHierarchy()->widgetRemoved( widget );
 }

// ------------------------------------------------------------

MoveCommand::MoveCommand( const TQString &n, FormWindow *fw,
			  const TQWidgetList &w,
			  const TQValueList<TQPoint> op,
			  const TQValueList<TQPoint> np,
			  TQWidget *opr, TQWidget *npr )
    : Command( n, fw ), widgets( w ), oldPos( op ), newPos( np ),
      oldParent( opr ), newParent( npr )
{
    widgets.setAutoDelete( false );
}

void MoveCommand::merge( Command *c )
{
    MoveCommand *cmd = (MoveCommand*)c;
    newPos = cmd->newPos;
}

bool MoveCommand::canMerge( Command *c )
{
    MoveCommand *cmd = (MoveCommand*)c;
    return widgets == cmd->widgets;
}


void MoveCommand::execute()
{
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	if ( !w->parentWidget() || WidgetFactory::layoutType( w->parentWidget() ) == WidgetFactory::NoLayout ) {
	    if ( newParent && oldParent && newParent != oldParent ) {
		TQPoint pos = newParent->mapFromGlobal( w->mapToGlobal( TQPoint( 0,0 ) ) );
		w->reparent( newParent, pos, true );
		formWindow()->raiseSelection( w );
		formWindow()->raiseChildSelections( w );
		formWindow()->widgetChanged( w );
		formWindow()->mainWindow()->objectHierarchy()->widgetRemoved( w );
		formWindow()->mainWindow()->objectHierarchy()->widgetInserted( w );
	    }
	    w->move( newPos[ widgets.at() ] );
	}
	formWindow()->updateSelection( w );
	formWindow()->updateChildSelections( w );
	formWindow()->emitUpdateProperties( w );
    }
}

void MoveCommand::unexecute()
{
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	if ( !w->parentWidget() || WidgetFactory::layoutType( w->parentWidget() ) == WidgetFactory::NoLayout ) {
	    if ( newParent && oldParent && newParent != oldParent ) {
		TQPoint pos = oldParent->mapFromGlobal( w->mapToGlobal( TQPoint( 0,0 ) ) );
		w->reparent( oldParent, pos, true );
		formWindow()->raiseSelection( w );
		formWindow()->raiseChildSelections( w );
		formWindow()->widgetChanged( w );
		formWindow()->mainWindow()->objectHierarchy()->widgetRemoved( w );
		formWindow()->mainWindow()->objectHierarchy()->widgetInserted( w );
	    }
	    w->move( oldPos[ widgets.at() ] );
	}
	formWindow()->updateSelection( w );
	formWindow()->updateChildSelections( w );
	formWindow()->emitUpdateProperties( w );
    }
}

// ------------------------------------------------------------

DeleteCommand::DeleteCommand( const TQString &n, FormWindow *fw,
			      const TQWidgetList &w )
    : Command( n, fw ), widgets( w )
{
    widgets.setAutoDelete( false );
}

void DeleteCommand::execute()
{
    formWindow()->setPropertyShowingBlocked( true );
    connections.clear();
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	w->hide();
	TQString s = w->name();
	s.prepend( "qt_dead_widget_" );
	w->setName( s.utf8() );
	formWindow()->selectWidget( w, false );
	formWindow()->widgets()->remove( w );
	TQValueList<MetaDataBase::Connection> conns = MetaDataBase::connections( formWindow(), w );
	connections.insert( w, conns );
	TQValueList<MetaDataBase::Connection>::Iterator it = conns.begin();
	for ( ; it != conns.end(); ++it ) {
	    MetaDataBase::removeConnection( formWindow(), (*it).sender,
					    (*it).signal, (*it).receiver, (*it).slot );
	}
    }
    formWindow()->setPropertyShowingBlocked( false );
    formWindow()->emitShowProperties();
    formWindow()->mainWindow()->objectHierarchy()->widgetsRemoved( widgets );

}

void DeleteCommand::unexecute()
{
    formWindow()->setPropertyShowingBlocked( true );
    formWindow()->clearSelection( false );
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	w->show();
	TQString s = w->name();
	s.remove( 0, TQString( "qt_dead_widget_" ).length() );
	w->setName( s.utf8() );
	formWindow()->widgets()->insert( w, w );
	formWindow()->selectWidget( w );
	TQValueList<MetaDataBase::Connection> conns = *connections.find( w );
	TQValueList<MetaDataBase::Connection>::Iterator it = conns.begin();
	for ( ; it != conns.end(); ++it ) {
	    MetaDataBase::addConnection( formWindow(), (*it).sender,
					 (*it).signal, (*it).receiver, (*it).slot );
	}
    }
    formWindow()->setPropertyShowingBlocked( false );
    formWindow()->emitShowProperties();
    formWindow()->mainWindow()->objectHierarchy()->widgetsInserted( widgets );
}

// ------------------------------------------------------------

SetPropertyCommand::SetPropertyCommand( const TQString &n, FormWindow *fw,
					TQObject *w, PropertyEditor *e,
					const TQString &pn, const TQVariant &ov,
					const TQVariant &nv, const TQString &ncut,
					const TQString &ocut, bool reset )
    : Command( n, fw ), widget( w ), editor( e ), propName( pn ),
      oldValue( ov ), newValue( nv ), oldCurrentItemText( ocut ), newCurrentItemText( ncut ),
      wasChanged( true ), isResetCommand( reset )
{
    wasChanged = MetaDataBase::isPropertyChanged( w, propName );
}


void SetPropertyCommand::execute()
{
    if ( !wasChanged )
	MetaDataBase::setPropertyChanged( widget, propName, true );
    if ( isResetCommand ) {
	MetaDataBase::setPropertyChanged( widget, propName, false );
	if ( WidgetFactory::resetProperty( widget, propName ) ) {
	    if ( !formWindow()->isWidgetSelected( widget ) && widget != formWindow() )
		formWindow()->selectWidget( widget );
	    if ( editor->widget() != widget )
		editor->setWidget( widget, formWindow() );
	    editor->propertyList()->setCurrentProperty( propName );
	    PropertyItem *i = (PropertyItem*)editor->propertyList()->currentItem();
	    if ( !i )
		return;
	    i->setValue( widget->property( propName.latin1() ) );
	    i->setChanged( false );
	    editor->refetchData();
	    editor->emitWidgetChanged();
	    return;
	}
    }
    setProperty( newValue, newCurrentItemText );
}

void SetPropertyCommand::unexecute()
{
    if ( !wasChanged )
	MetaDataBase::setPropertyChanged( widget, propName, false );
    if ( isResetCommand )
	MetaDataBase::setPropertyChanged( widget, propName, true );
    setProperty( oldValue, oldCurrentItemText );
}

bool SetPropertyCommand::canMerge( Command *c )
{
    SetPropertyCommand *cmd = (SetPropertyCommand*)c;
    const TQMetaProperty *p =
	widget->metaObject()->property( widget->metaObject()->findProperty( propName.latin1(), true ), true );
    if ( !p ) {
	if ( propName == "toolTip" || propName == "whatsThis" )
	    return true;
	if ( widget->inherits( "CustomWidget" ) ) {
	    MetaDataBase::CustomWidget *cw = ( (CustomWidget*)widget )->customWidget();
	    if ( !cw )
		return false;
	    for ( TQValueList<MetaDataBase::Property>::Iterator it = cw->lstProperties.begin(); it != cw->lstProperties.end(); ++it ) {
		if ( TQString( (*it ).property ) == propName ) {
		    if ( (*it).type == "String" || (*it).type == "CString" || (*it).type == "Int" || (*it).type == "UInt" )
			return true;
		}
	    }
	}
	return false;
    }
    TQVariant::Type t = TQVariant::nameToType( p->type() );
    return ( cmd->propName == propName &&
	     t == TQVariant::String || t == TQVariant::CString || t == TQVariant::Int || t == TQVariant::UInt );
}

void SetPropertyCommand::merge( Command *c )
{
    SetPropertyCommand *cmd = (SetPropertyCommand*)c;
    newValue = cmd->newValue;
    newCurrentItemText = cmd->newCurrentItemText;
}

bool SetPropertyCommand::checkProperty()
{
    if ( propName == "name" /*|| propName == "itemName"*/ ) { // ### fix that
	TQString s = newValue.toString();
	if ( !formWindow()->unify( widget, s, false ) ) {
	    TQMessageBox::information( formWindow()->mainWindow(),
				      i18n("Set 'name' Property" ),
				      i18n("The name of a widget must be unique.\n"
                            "'%1' is already used in form '%2',\n"
                            "so the name has been reverted to '%3'." ).
				      arg( newValue.toString() ).
				      arg( formWindow()->name() ).
				      arg( oldValue.toString() ));
	    setProperty( oldValue, oldCurrentItemText, false );
	    return false;
	}
	if ( s.isEmpty() ) {
	    TQMessageBox::information( formWindow()->mainWindow(),
				      i18n("Set 'name' Property" ),
				      i18n("The name of a widget must not be null.\n"
                            "The name has been reverted to '%1'." ).
				      arg( oldValue.toString() ));
	    setProperty( oldValue, oldCurrentItemText, false );
	    return false;
	}

	if ( widget->parent() && widget->parent()->inherits( "FormWindow" ) )
	    formWindow()->mainWindow()->formNameChanged( (FormWindow*)( (TQWidget*)widget )->parentWidget() );
    }
    return true;
}

void SetPropertyCommand::setProperty( const TQVariant &v, const TQString &currentItemText, bool select )
{
    if ( !formWindow()->isWidgetSelected( widget ) && !formWindow()->isMainContainer( widget ) && select )
	formWindow()->selectWidget( widget );
    if ( editor->widget() != widget && select )
	editor->setWidget( widget, formWindow() );
    if ( select )
	editor->propertyList()->setCurrentProperty( propName );

    const TQMetaProperty *p =
	widget->metaObject()->property( widget->metaObject()->findProperty( propName.latin1(), true ), true );
    if ( !p ) {
	if ( propName == "hAlign" ) {
	    p = widget->metaObject()->property( widget->metaObject()->findProperty( "alignment", true ), true );
	    int align = widget->property( "alignment" ).toInt();
	    align &= ~( AlignHorizontal_Mask );
	    align |= p->keyToValue( currentItemText.utf8() );
	    widget->setProperty( "alignment", TQVariant( align ) );
	} else if ( propName == "vAlign" ) {
	    p = widget->metaObject()->property( widget->metaObject()->findProperty( "alignment", true ), true );
	    int align = widget->property( "alignment" ).toInt();
	    align &= ~( AlignVertical_Mask );
	    align |= p->keyToValue( currentItemText.utf8() );
	    widget->setProperty( "alignment", TQVariant( align ) );
	} else if ( propName == "wordwrap" ) {
	    int align = widget->property( "alignment" ).toInt();
	    align &= ~WordBreak;
	    if ( v.toBool() )
		align |= WordBreak;
	    widget->setProperty( "alignment", TQVariant( align ) );
	} else if ( propName == "layoutSpacing" ) {
	    MetaDataBase::setSpacing( WidgetFactory::containerOfWidget( static_cast<TQWidget*>( editor->widget() ) ), v.toInt() );
	} else if ( propName == "layoutMargin" ) {
	    MetaDataBase::setMargin( WidgetFactory::containerOfWidget( static_cast<TQWidget*>( editor->widget() ) ), v.toInt() );
	} else if ( propName == "toolTip" || propName == "whatsThis" || propName == "database" || propName == "frameworkCode" ) {
	    MetaDataBase::setFakeProperty( editor->widget(), propName, v );
	} else if ( editor->widget()->inherits( "CustomWidget" ) ) {
	    MetaDataBase::CustomWidget *cw = ( (CustomWidget*)widget )->customWidget();
	    if ( cw ) {
		MetaDataBase::setFakeProperty( editor->widget(), propName, v );
	    }
	}
	editor->refetchData();
	editor->emitWidgetChanged();
	( ( PropertyItem* )editor->propertyList()->currentItem() )->setChanged( MetaDataBase::isPropertyChanged( widget, propName ) );
#ifndef TQT_NO_SQL
	if ( propName == "database" )
	    formWindow()->mainWindow()->objectHierarchy()->databasePropertyChanged( (TQWidget*)widget, MetaDataBase::fakeProperty( widget, "database" ).toStringList() );
#endif
	return;
    }

    if ( p->isSetType() ) {
	;
    } else if ( p->isEnumType() ) {
	widget->setProperty( propName.latin1(), p->keyToValue( currentItemText.utf8() ) );
    } else {
	TQVariant ov;
	if ( propName == "name" || propName == "itemName" )
	    ov = widget->property( propName.latin1() );
	int oldSerNum = -1;
	if ( v.type() == TQVariant::Pixmap )
	    oldSerNum = v.toPixmap().serialNumber();
	widget->setProperty( propName.latin1(), v );
	if ( oldSerNum != -1 && oldSerNum != widget->property( propName.latin1() ).toPixmap().serialNumber() )
	    MetaDataBase::setPixmapKey( formWindow(),
					widget->property( propName.latin1() ).toPixmap().serialNumber(),
					MetaDataBase::pixmapKey( formWindow(), oldSerNum ) );
	if ( propName == "cursor" )
	    MetaDataBase::setCursor( (TQWidget*)widget, v.toCursor() );
	if ( propName == "name" && widget->isWidgetType() ) {
	    formWindow()->mainWindow()->objectHierarchy()->namePropertyChanged( (TQWidget*)widget, ov );
	    if ( formWindow()->isMainContainer( widget ) )
		formWindow()->setName( v.toCString() );
	}
	if ( propName == "name" && widget->inherits( "TQAction" ) &&
	     formWindow()->mainContainer() &&
	     formWindow()->mainContainer()->inherits( "TQMainWindow" ) ) {
	    formWindow()->mainWindow()->actioneditor()->updateActionName( (TQAction*)widget );
	}
	if ( propName == "iconSet" && widget->inherits( "TQAction" ) &&
	     formWindow()->mainContainer() &&
	     formWindow()->mainContainer()->inherits( "TQMainWindow" ) ) {
	    formWindow()->mainWindow()->actioneditor()->updateActionIcon( (TQAction*)widget );
	}
	if ( propName == "caption" ) {
	    if ( formWindow()->isMainContainer( widget ) )
		formWindow()->setCaption( v.toString() );
	}
	if ( propName == "icon" ) {
	    if ( formWindow()->isMainContainer( widget ) )
		formWindow()->setIcon( v.toPixmap() );
	}
    }
    editor->refetchData();
    if ( editor->propertyList()->currentItem() && select ) {
	( ( PropertyItem* )editor->propertyList()->currentItem() )->showEditor();
	( ( PropertyItem* )editor->propertyList()->currentItem() )->setChanged( MetaDataBase::isPropertyChanged( widget, propName ) );
    }
    editor->emitWidgetChanged();
    formWindow()->killAccels( widget );
}

// ------------------------------------------------------------

LayoutHorizontalCommand::LayoutHorizontalCommand( const TQString &n, FormWindow *fw,
						  TQWidget *parent, TQWidget *layoutBase,
						  const TQWidgetList &wl )
    : Command( n, fw ), layout( wl, parent, fw, layoutBase )
{
}

void LayoutHorizontalCommand::execute()
{
    formWindow()->clearSelection( false );
    layout.doLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

void LayoutHorizontalCommand::unexecute()
{
    formWindow()->clearSelection( false );
    layout.undoLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

// ------------------------------------------------------------

LayoutHorizontalSplitCommand::LayoutHorizontalSplitCommand( const TQString &n, FormWindow *fw,
							    TQWidget *parent, TQWidget *layoutBase,
							    const TQWidgetList &wl )
    : Command( n, fw ), layout( wl, parent, fw, layoutBase, true, true )
{
}

void LayoutHorizontalSplitCommand::execute()
{
    formWindow()->clearSelection( false );
    layout.doLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

void LayoutHorizontalSplitCommand::unexecute()
{
    formWindow()->clearSelection( false );
    layout.undoLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

// ------------------------------------------------------------

LayoutVerticalCommand::LayoutVerticalCommand( const TQString &n, FormWindow *fw,
					      TQWidget *parent, TQWidget *layoutBase,
					      const TQWidgetList &wl )
    : Command( n, fw ), layout( wl, parent, fw, layoutBase )
{
}

void LayoutVerticalCommand::execute()
{
    formWindow()->clearSelection( false );
    layout.doLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

void LayoutVerticalCommand::unexecute()
{
    formWindow()->clearSelection( false );
    layout.undoLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

// ------------------------------------------------------------

LayoutVerticalSplitCommand::LayoutVerticalSplitCommand( const TQString &n, FormWindow *fw,
							TQWidget *parent, TQWidget *layoutBase,
							const TQWidgetList &wl )
    : Command( n, fw ), layout( wl, parent, fw, layoutBase, true, true )
{
}

void LayoutVerticalSplitCommand::execute()
{
    formWindow()->clearSelection( false );
    layout.doLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

void LayoutVerticalSplitCommand::unexecute()
{
    formWindow()->clearSelection( false );
    layout.undoLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

// ------------------------------------------------------------

LayoutGridCommand::LayoutGridCommand( const TQString &n, FormWindow *fw,
				      TQWidget *parent, TQWidget *layoutBase,
				      const TQWidgetList &wl, int xres, int yres )
    : Command( n, fw ), layout( wl, parent, fw, layoutBase, TQSize( TQMAX(5,xres), TQMAX(5,yres) ) )
{
}

void LayoutGridCommand::execute()
{
    formWindow()->clearSelection( false );
    layout.doLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

void LayoutGridCommand::unexecute()
{
    formWindow()->clearSelection( false );
    layout.undoLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
}

// ------------------------------------------------------------

BreakLayoutCommand::BreakLayoutCommand( const TQString &n, FormWindow *fw,
					TQWidget *layoutBase, const TQWidgetList &wl )
    : Command( n, fw ), lb( layoutBase ), widgets( wl )
{
    WidgetFactory::LayoutType lay = WidgetFactory::layoutType( layoutBase );
    spacing = MetaDataBase::spacing( layoutBase );
    margin = MetaDataBase::margin( layoutBase );
    layout = 0;
    if ( lay == WidgetFactory::HBox )
	layout = new HorizontalLayout( wl, layoutBase, fw, layoutBase, false, layoutBase->inherits( "TQSplitter" ) );
    else if ( lay == WidgetFactory::VBox )
	layout = new VerticalLayout( wl, layoutBase, fw, layoutBase, false, layoutBase->inherits( "TQSplitter" ) );
    else if ( lay == WidgetFactory::Grid )
	layout = new GridLayout( wl, layoutBase, fw, layoutBase, TQSize( TQMAX( 5, fw->grid().x()), TQMAX( 5, fw->grid().y()) ), false );
}

void BreakLayoutCommand::execute()
{
    if ( !layout )
	return;
    formWindow()->clearSelection( false );
    layout->breakLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() )
	w->resize( TQMAX( 16, w->width() ), TQMAX( 16, w->height() ) );
}

void BreakLayoutCommand::unexecute()
{
    if ( !layout )
	return;
    formWindow()->clearSelection( false );
    layout->doLayout();
    formWindow()->mainWindow()->objectHierarchy()->rebuild();
    MetaDataBase::setSpacing( WidgetFactory::containerOfWidget( lb ), spacing );
    MetaDataBase::setMargin( WidgetFactory::containerOfWidget( lb ), margin );
}

// ------------------------------------------------------------

MacroCommand::MacroCommand( const TQString &n, FormWindow *fw,
			    const TQPtrList<Command> &cmds )
    : Command( n, fw ), commands( cmds )
{
}

void MacroCommand::execute()
{
    for ( Command *c = commands.first(); c; c = commands.next() )
	c->execute();
}

void MacroCommand::unexecute()
{
    for ( Command *c = commands.last(); c; c = commands.prev() )
	c->unexecute();
}

// ------------------------------------------------------------

AddTabPageCommand::AddTabPageCommand( const TQString &n, FormWindow *fw,
				      TQTabWidget *tw, const TQString &label )
    : Command( n, fw ), tabWidget( tw ), tabLabel( label )
{
    tabPage = new QDesignerWidget( formWindow(), tabWidget, "tab" );
    tabPage->hide();
    index = -1;
    MetaDataBase::addEntry( tabPage );
}

void AddTabPageCommand::execute()
{
    if ( index == -1 )
	index = ( (QDesignerTabWidget*)tabWidget )->count();
    tabWidget->insertTab( tabPage, tabLabel, index );
    tabWidget->showPage( tabPage );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( tabWidget );
}

void AddTabPageCommand::unexecute()
{
    tabWidget->removePage( tabPage );
    tabPage->hide();
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( tabWidget );
}

// ------------------------------------------------------------
// ------------------------------------------------------------

AddToolBoxPageCommand::AddToolBoxPageCommand( const TQString &n, FormWindow *fw,
                      TQToolBox *tw, const TQString &_label )
    : Command( n, fw ), toolBox( tw ), label( _label )
{
    page = new QDesignerWidget( formWindow(), toolBox, "tab" );
    page->hide();
    index = -1;
    MetaDataBase::addEntry( page );
}

void AddToolBoxPageCommand::execute()
{
    if ( index == -1 )
    index = ( (EditorToolBox*)toolBox)->count();
    toolBox->insertItem(index, page, label);
    toolBox->setCurrentItem( page );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( toolBox );
}

void AddToolBoxPageCommand::unexecute()
{
    toolBox->removeItem( page );
    page->hide();
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( toolBox );
}

// ------------------------------------------------------------

MoveTabPageCommand::MoveTabPageCommand( const TQString &n, FormWindow *fw,
				      TQTabWidget *tw, TQWidget* page, const TQString& label, int nIndex, int oIndex )
    : Command( n, fw ), tabWidget( tw ), tabPage( page ), tabLabel( label )
{
    newIndex = nIndex;
    oldIndex = oIndex;
}

void MoveTabPageCommand::execute()
{
    ((QDesignerTabWidget*)tabWidget )->removePage( tabPage );
    ((QDesignerTabWidget*)tabWidget )->insertTab( tabPage, tabLabel, newIndex );
    ((QDesignerTabWidget*)tabWidget )->showPage( tabPage );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( tabWidget );
}

void MoveTabPageCommand::unexecute()
{
    ((QDesignerTabWidget*)tabWidget )->removePage( tabPage );
    ((QDesignerTabWidget*)tabWidget )->insertTab( tabPage, tabLabel, oldIndex );
    ((QDesignerTabWidget*)tabWidget )->showPage( tabPage );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( tabWidget );
}

// ------------------------------------------------------------

DeleteTabPageCommand::DeleteTabPageCommand( const TQString &n, FormWindow *fw,
					    TQTabWidget *tw, TQWidget *page )
    : Command( n, fw ), tabWidget( tw ), tabPage( page )
{
    tabLabel = ( (QDesignerTabWidget*)tabWidget )->pageTitle();
    index = ( (QDesignerTabWidget*)tabWidget )->currentPage();
}

void DeleteTabPageCommand::execute()
{
    tabWidget->removePage( tabPage );
    tabPage->hide();
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( tabWidget );
}

void DeleteTabPageCommand::unexecute()
{
    tabWidget->insertTab( tabPage, tabLabel, index );
    tabWidget->showPage( tabPage );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( tabWidget );
}

// ------------------------------------------------------------

DeleteToolBoxPageCommand::DeleteToolBoxPageCommand( const TQString &n, FormWindow *fw,
                        TQToolBox *tw, TQWidget *_page )
    : Command( n, fw ), toolBox( tw ), page( _page )
{
    label = ( (EditorToolBox*)toolBox )->pageTitle();
    index = ( (EditorToolBox*)toolBox )->currentPage();
}

void DeleteToolBoxPageCommand::execute()
{
    toolBox->removeItem( page );
    page->hide();
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( toolBox );
}

void DeleteToolBoxPageCommand::unexecute()
{
    toolBox->insertItem(index, page, label);
    toolBox->setCurrentItem( page );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->tabsChanged( toolBox );
}

// ------------------------------------------------------------

AddWizardPageCommand::AddWizardPageCommand( const TQString &n, FormWindow *fw,
					    TQWizard *w, const TQString &label, int i, bool s )
    : Command( n, fw ), wizard( w ), pageLabel( label )
{
    page = new QDesignerWidget( formWindow(), wizard, "page" );
    page->hide();
    index = i;
    show = s;
    MetaDataBase::addEntry( page );
}

void AddWizardPageCommand::execute()
{
    if ( index == -1 )
	index = wizard->pageCount();
    wizard->insertPage( page, pageLabel, index );
    if ( show )
      ( (QDesignerWizard*)wizard )->setCurrentPage( ( (QDesignerWizard*)wizard )->pageNum( page ) );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->pagesChanged( wizard );
}

void AddWizardPageCommand::unexecute()
{
    wizard->removePage( page );
    page->hide();
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->pagesChanged( wizard );
}

// ------------------------------------------------------------

DeleteWizardPageCommand::DeleteWizardPageCommand( const TQString &n, FormWindow *fw,
						  TQWizard *w, int i, bool s )
    : Command( n, fw ), wizard( w ), index( i )
{
    show = s;
}

void DeleteWizardPageCommand::execute()
{
    page = wizard->page( index );
    pageLabel = wizard->title( page );
    wizard->removePage( page );
    page->hide();
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->pagesChanged( wizard );
}

void DeleteWizardPageCommand::unexecute()
{
    wizard->insertPage( page, pageLabel, index );
    if ( show )
      ( (QDesignerWizard*)wizard )->setCurrentPage( ( (QDesignerWizard*)wizard )->pageNum( page ) );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->pagesChanged( wizard );
}

// ------------------------------------------------------------

RenameWizardPageCommand::RenameWizardPageCommand( const TQString &n, FormWindow *fw,
						  TQWizard *w, int i, const TQString& name )
    : Command( n, fw ), wizard( w ), index( i ), label( name )
{

}

void RenameWizardPageCommand::execute()
{
    page = wizard->page( index );
    TQString oldLabel = wizard->title( page );

    wizard->setTitle( page, label );
    label = oldLabel;
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
}

void RenameWizardPageCommand::unexecute()
{
    execute();
}

// ------------------------------------------------------------

SwapWizardPagesCommand::SwapWizardPagesCommand( const TQString &n, FormWindow *fw, TQWizard *w, int i1, int i2 )
    : Command( n, fw ), wizard( w ), index1( i1 ), index2( i2 )
{
}

void SwapWizardPagesCommand::execute()
{
    TQWidget *page1 = wizard->page( index1 );
    TQWidget *page2 = wizard->page( index2 );
    TQString page1Label = wizard->title( page1 );
    TQString page2Label = wizard->title( page2 );
    wizard->removePage( page1 );
    wizard->removePage( page2 );
    wizard->insertPage( page1, page1Label, index2 );
    wizard->insertPage( page2, page2Label, index1 );
    formWindow()->emitUpdateProperties( formWindow()->currentWidget() );
    formWindow()->mainWindow()->objectHierarchy()->pagesChanged( wizard );
}

void SwapWizardPagesCommand::unexecute()
{
    execute();
}

// ------------------------------------------------------------

AddConnectionCommand::AddConnectionCommand( const TQString &name, FormWindow *fw,
					    MetaDataBase::Connection c )
    : Command( name, fw ), connection( c )
{
}

void AddConnectionCommand::execute()
{
    MetaDataBase::addConnection( formWindow(), connection.sender,
				 connection.signal, connection.receiver, connection.slot );
#ifndef KOMMANDER
    if ( connection.receiver == formWindow()->mainContainer() )
    {
	// tqDebug("AddConnectionCommand::execute(): Would have called EventList::setup()");
	formWindow()->mainWindow()->propertyeditor()->eventList()->setup();
    }
#endif
}

void AddConnectionCommand::unexecute()
{
    MetaDataBase::removeConnection( formWindow(), connection.sender,
				    connection.signal, connection.receiver, connection.slot );
#ifndef KOMMANDER
    if ( connection.receiver == formWindow()->mainContainer() )
    {
	// tqDebug("AddConnectionCommand::unexecute(): Would have called EventList::setup()");
	formWindow()->mainWindow()->propertyeditor()->eventList()->setup();
    }
#endif
}

// ------------------------------------------------------------

RemoveConnectionCommand::RemoveConnectionCommand( const TQString &name, FormWindow *fw,
						  MetaDataBase::Connection c )
    : Command( name, fw ), connection( c )
{
}

void RemoveConnectionCommand::execute()
{
    MetaDataBase::removeConnection( formWindow(), connection.sender,
				    connection.signal, connection.receiver, connection.slot );
#ifndef KOMMANDER
    if ( connection.receiver == formWindow()->mainContainer() )
    {
        // tqDebug("RemoveConnectionCommand::execute(): Would have called EventList::setup()");
	formWindow()->mainWindow()->propertyeditor()->eventList()->setup();
    }
#endif
}

void RemoveConnectionCommand::unexecute()
{
    MetaDataBase::addConnection( formWindow(), connection.sender,
				 connection.signal, connection.receiver, connection.slot );
#ifndef KOMMANDER
    if ( connection.receiver == formWindow()->mainContainer() )
    {
        // tqDebug("RemoveConnectionCommand::unexecute(): Would have called EventList::setup()");
	formWindow()->mainWindow()->propertyeditor()->eventList()->setup();
    }
#endif
}

// ------------------------------------------------------------

/* TODO : We don't need these commands. */
AddSlotCommand::AddSlotCommand( const TQString &name, FormWindow *fw, const TQCString &s,
				const TQString& spec, const TQString &a, const TQString &l, const TQString &rt )
    : Command( name, fw ), slot( s ), specifier( spec ), access( a ), language( l ), returnType( rt )
{
	// tqDebug("AddSlotCommand::AddSlotCommand()");
}

void AddSlotCommand::execute()
{
    MetaDataBase::addSlot( formWindow(), slot, specifier, access, language, returnType );
#ifndef KOMMANDER
    formWindow()->mainWindow()->slotsChanged();
#endif
    if ( formWindow()->formFile() )
	formWindow()->formFile()->setModified( true );
}

void AddSlotCommand::unexecute()
{
    MetaDataBase::removeSlot( formWindow(), slot, specifier, access, language, returnType );
#ifndef KOMMANDER
    formWindow()->mainWindow()->slotsChanged();
#endif
    if ( formWindow()->formFile() )
	formWindow()->formFile()->setModified( true );
}

// ------------------------------------------------------------

RemoveSlotCommand::RemoveSlotCommand( const TQString &name, FormWindow *fw, const TQCString &s,
				      const TQString& spec, const TQString &a, const TQString &l, const TQString &rt )
    : Command( name, fw ), slot( s ), specifier( spec ), access( a ), language( l ), returnType( rt )
{
	// tqDebug("RemoveSlotCommand::RemoveSlotCommand()");
}

void RemoveSlotCommand::execute()
{
    MetaDataBase::removeSlot( formWindow(), slot, specifier, access, language, returnType );
#ifndef KOMMANDER
    formWindow()->mainWindow()->slotsChanged();
#endif
    if ( formWindow()->formFile() )
	formWindow()->formFile()->setModified( true );
}

void RemoveSlotCommand::unexecute()
{
    MetaDataBase::addSlot( formWindow(), slot, specifier, access, language, returnType );
#ifndef KOMMANDER
    formWindow()->mainWindow()->slotsChanged();
#endif
    if ( formWindow()->formFile() )
	formWindow()->formFile()->setModified( true );
}

// ------------------------------------------------------------

LowerCommand::LowerCommand( const TQString &name, FormWindow *fw, const TQWidgetList &w )
    : Command( name, fw ), widgets( w )
{
}

void LowerCommand::execute()
{
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	w->lower();
	formWindow()->raiseSelection( w );
    }

}

void LowerCommand::unexecute()
{
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	w->raise();
	formWindow()->raiseSelection( w );
    }
}

// ------------------------------------------------------------

RaiseCommand::RaiseCommand( const TQString &name, FormWindow *fw, const TQWidgetList &w )
    : Command( name, fw ), widgets( w )
{
}

void RaiseCommand::execute()
{
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	w->raise();
	formWindow()->raiseSelection( w );
    }

}

void RaiseCommand::unexecute()
{
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	w->lower();
	formWindow()->raiseSelection( w );
    }
}

// ------------------------------------------------------------

PasteCommand::PasteCommand( const TQString &n, FormWindow *fw,
			      const TQWidgetList &w )
    : Command( n, fw ), widgets( w )
{
}

void PasteCommand::execute()
{
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	w->show();
	formWindow()->selectWidget( w );
	formWindow()->widgets()->insert( w, w );
	formWindow()->mainWindow()->objectHierarchy()->widgetInserted( w );
    }
}

void PasteCommand::unexecute()
{
    for ( TQWidget *w = widgets.first(); w; w = widgets.next() ) {
	w->hide();
	formWindow()->selectWidget( w, false );
	formWindow()->widgets()->remove( w );
	formWindow()->mainWindow()->objectHierarchy()->widgetRemoved( w );
    }
 }

// ------------------------------------------------------------

TabOrderCommand::TabOrderCommand( const TQString &n, FormWindow *fw,
				  const TQWidgetList &ol, const TQWidgetList &nl )
    : Command( n, fw ), oldOrder( ol ), newOrder( nl )
{
}

void TabOrderCommand::merge( Command *c )
{
    TabOrderCommand *cmd = (TabOrderCommand*)c;
    newOrder = cmd->newOrder;
}

bool TabOrderCommand::canMerge( Command * )
{
    return true;
}

void TabOrderCommand::execute()
{
    MetaDataBase::setTabOrder( formWindow(), newOrder );
}

void TabOrderCommand::unexecute()
{
    MetaDataBase::setTabOrder( formWindow(), oldOrder );
}

// ------------------------------------------------------------

PopulateListBoxCommand::PopulateListBoxCommand( const TQString &n, FormWindow *fw,
						TQListBox *lb, const TQValueList<Item> &items )
    : Command( n, fw ), newItems( items ), listbox( lb )
{
    TQListBoxItem *i = 0;
    for ( i = listbox->firstItem(); i; i = i->next() ) {
	Item item;
	if ( i->pixmap() )
	    item.pix = *i->pixmap();
	item.text = i->text();
	oldItems.append( item );
    }
}

void PopulateListBoxCommand::execute()
{
    listbox->clear();
    for ( TQValueList<Item>::Iterator it = newItems.begin(); it != newItems.end(); ++it ) {
	Item i = *it;
	if ( !i.pix.isNull() )
	    (void)new TQListBoxPixmap( listbox, i.pix, i.text );
	else
	    (void)new TQListBoxText( listbox, i.text );
    }
    formWindow()->mainWindow()->propertyeditor()->refetchData();
}

void PopulateListBoxCommand::unexecute()
{
    listbox->clear();
    for ( TQValueList<Item>::Iterator it = oldItems.begin(); it != oldItems.end(); ++it ) {
	Item i = *it;
	if ( !i.pix.isNull() )
	    (void)new TQListBoxPixmap( listbox, i.pix, i.text );
	else
	    (void)new TQListBoxText( listbox, i.text );
    }
    formWindow()->mainWindow()->propertyeditor()->refetchData();
}

// ------------------------------------------------------------

PopulateIconViewCommand::PopulateIconViewCommand( const TQString &n, FormWindow *fw,
						  TQIconView *iv, const TQValueList<Item> &items )
    : Command( n, fw ), newItems( items ), iconview( iv )
{
#ifndef TQT_NO_ICONVIEW
    TQIconViewItem *i = 0;
    for ( i = iconview->firstItem(); i; i = i->nextItem() ) {
	Item item;
	if ( i->pixmap() )
	    item.pix = *i->pixmap();
	item.text = i->text();
	oldItems.append( item );
    }
#endif
}

void PopulateIconViewCommand::execute()
{
#ifndef TQT_NO_ICONVIEW
    iconview->clear();
    for ( TQValueList<Item>::Iterator it = newItems.begin(); it != newItems.end(); ++it ) {
	Item i = *it;
	(void)new TQIconViewItem( iconview, i.text, i.pix );
    }
#endif
}

void PopulateIconViewCommand::unexecute()
{
#ifndef TQT_NO_ICONVIEW
    iconview->clear();
    for ( TQValueList<Item>::Iterator it = oldItems.begin(); it != oldItems.end(); ++it ) {
	Item i = *it;
	(void)new TQIconViewItem( iconview, i.text, i.pix );
    }
#endif
}

// ------------------------------------------------------------

PopulateListViewCommand::PopulateListViewCommand( const TQString &n, FormWindow *fw,
						  TQListView *lv, TQListView *from )
    : Command( n, fw ), listview( lv )
{
    newItems = new TQListView();
    newItems->hide();
    transferItems( from, newItems );
    oldItems = new TQListView();
    oldItems->hide();
    transferItems( listview, oldItems );
}

void PopulateListViewCommand::execute()
{
    listview->clear();
    transferItems( newItems, listview );
}

void PopulateListViewCommand::unexecute()
{
    listview->clear();
    transferItems( oldItems, listview );
}

void PopulateListViewCommand::transferItems( TQListView *from, TQListView *to )
{
    TQHeader *header = to->header();
    while ( header->count() )
	to->removeColumn( 0 );
    TQHeader *h2 = from->header();
    for ( int i = 0; i < h2->count(); ++i ) {
	to->addColumn( h2->label( i ) );
	if ( h2->iconSet( i ) && !h2->iconSet( i )->pixmap().isNull() )
	    header->setLabel( i, *h2->iconSet( i ), h2->label( i ) );
	header->setResizeEnabled( h2->isResizeEnabled( i ), i );
	header->setClickEnabled( h2->isClickEnabled( i ), i );
    }

    TQListViewItemIterator it( from );
    TQPtrStack<TQListViewItem> fromParents, toParents;
    fromParents.push( 0 );
    toParents.push( 0 );
    TQPtrStack<TQListViewItem> toLasts;
    TQListViewItem *fromLast = 0;
    toLasts.push( 0 );
    int cols = from->columns();
    to->setSorting( -1 );
    from->setSorting( -1 );
    for ( ; it.current(); ++it ) {
	TQListViewItem *i = it.current();
	if ( i->parent() == fromParents.top() ) {
	    TQListViewItem *pi = toParents.top();
	    TQListViewItem *ni = 0;
	    if ( pi )
		ni = new TQListViewItem( pi, toLasts.top() );
	    else
		ni = new TQListViewItem( to, toLasts.top() );
	    for ( int c = 0; c < cols; ++c ) {
		ni->setText( c, i->text( c ) );
		if ( i->pixmap( c ) )
		    ni->setPixmap( c, *i->pixmap( c ) );
	    }
	    toLasts.pop();
	    toLasts.push( ni );
	    if ( pi )
		pi->setOpen( true );
	} else {
	    if ( i->parent() == fromLast ) {
		fromParents.push( fromLast );
		toParents.push( toLasts.top() );
		toLasts.push( 0 );
		TQListViewItem *pi = toParents.top();
		TQListViewItem *ni = 0;
		if ( pi )
		    ni = new TQListViewItem( pi );
		else
		    ni = new TQListViewItem( to );
		for ( int c = 0; c < cols; ++c ) {
		    ni->setText( c, i->text( c ) );
		    if ( i->pixmap( c ) )
			ni->setPixmap( c, *i->pixmap( c ) );
		}
		toLasts.pop();
		toLasts.push( ni );
		if ( pi )
		    pi->setOpen( true );
	    } else {
		while ( fromParents.top() != i->parent() ) {
		    fromParents.pop();
		    toParents.pop();
		    toLasts.pop();
		}

		TQListViewItem *pi = toParents.top();
		TQListViewItem *ni = 0;
		if ( pi )
		    ni = new TQListViewItem( pi, toLasts.top() );
		else
		    ni = new TQListViewItem( to, toLasts.top() );
		for ( int c = 0; c < cols; ++c ) {
		    ni->setText( c, i->text( c ) );
		    if ( i->pixmap( c ) )
			ni->setPixmap( c, *i->pixmap( c ) );
		}
		if ( pi )
		    pi->setOpen( true );
		toLasts.pop();
		toLasts.push( ni );
	    }
	}
	fromLast = i;
    }
}



// ------------------------------------------------------------

PopulateMultiLineEditCommand::PopulateMultiLineEditCommand( const TQString &n, FormWindow *fw,
							    TQMultiLineEdit *mle, const TQString &txt )
    : Command( n, fw ), newText( txt ), mlined( mle )
{
    oldText = mlined->text();
    wasChanged = MetaDataBase::isPropertyChanged( mlined, "text" );
}

void PopulateMultiLineEditCommand::execute()
{
    mlined->setText( newText );
    MetaDataBase::setPropertyChanged( mlined, "text", true );
    formWindow()->emitUpdateProperties( mlined );
}

void PopulateMultiLineEditCommand::unexecute()
{
    mlined->setText( oldText );
    MetaDataBase::setPropertyChanged( mlined, "text", wasChanged );
    formWindow()->emitUpdateProperties( mlined );
}

// ------------------------------------------------------------

PopulateTableCommand::PopulateTableCommand( const TQString &n, FormWindow *fw, TQTable *t,
					    const TQValueList<Row> &rows,
					    const TQValueList<Column> &columns )
    : Command( n, fw ), newRows( rows ), newColumns( columns ), table( t )
{
#ifndef TQT_NO_TABLE
    int i = 0;
    TQMap<TQString, TQString> columnFields = MetaDataBase::columnFields( table);
    for ( i = 0; i < table->horizontalHeader()->count(); ++i ) {
	PopulateTableCommand::Column col;
	col.text = table->horizontalHeader()->label( i );
	if ( table->horizontalHeader()->iconSet( i ) )
	    col.pix = table->horizontalHeader()->iconSet( i )->pixmap();
	col.field = *columnFields.find( col.text );
	oldColumns.append( col );
    }
    for ( i = 0; i < table->verticalHeader()->count(); ++i ) {
	PopulateTableCommand::Row row;
	row.text = table->verticalHeader()->label( i );
	if ( table->verticalHeader()->iconSet( i ) )
	    row.pix = table->verticalHeader()->iconSet( i )->pixmap();
	oldRows.append( row );
    }
#endif
}

void PopulateTableCommand::execute()
{
#ifndef TQT_NO_TABLE
    TQMap<TQString, TQString> columnFields;
    table->setNumCols( newColumns.count() );
    int i = 0;
    for ( TQValueList<Column>::Iterator cit = newColumns.begin(); cit != newColumns.end(); ++cit, ++i ) {
	table->horizontalHeader()->setLabel( i, (*cit).pix, (*cit).text );
	if ( !(*cit).field.isEmpty() )
	    columnFields.insert( (*cit).text, (*cit).field );
    }
    MetaDataBase::setColumnFields( table, columnFields );
    table->setNumRows( newRows.count() );
    i = 0;
    for ( TQValueList<Row>::Iterator rit = newRows.begin(); rit != newRows.end(); ++rit, ++i )
	table->verticalHeader()->setLabel( i, (*rit).pix, (*rit).text );
#endif
}

void PopulateTableCommand::unexecute()
{
#ifndef TQT_NO_TABLE
    TQMap<TQString, TQString> columnFields;
    table->setNumCols( oldColumns.count() );
    int i = 0;
    for ( TQValueList<Column>::Iterator cit = oldColumns.begin(); cit != oldColumns.end(); ++cit, ++i ) {
	table->horizontalHeader()->setLabel( i, (*cit).pix, (*cit).text );
	if ( !(*cit).field.isEmpty() )
	    columnFields.insert( (*cit).text, (*cit).field );
    }
    MetaDataBase::setColumnFields( table, columnFields );
    table->setNumRows( oldRows.count() );
    i = 0;
    for ( TQValueList<Row>::Iterator rit = oldRows.begin(); rit != oldRows.end(); ++rit, ++i )
	table->verticalHeader()->setLabel( i, (*rit).pix, (*rit).text );
#endif
}

// ------------------------------------------------------------

AddActionToToolBarCommand::AddActionToToolBarCommand( const TQString &n, FormWindow *fw,
						      TQAction *a, QDesignerToolBar *tb, int idx )
    : Command( n, fw ), action( a ), toolBar( tb ), index( idx )
{
}

void AddActionToToolBarCommand::execute()
{
    action->addTo( toolBar );

    if ( action->inherits( "QDesignerAction" ) ) {
	TQString s = ( (QDesignerAction*)action )->widget()->name();
	if ( s.startsWith( "qt_dead_widget_" ) ) {
	    s.remove( 0, TQString( "qt_dead_widget_" ).length() );
	    ( (QDesignerAction*)action )->widget()->setName( s.utf8() );
	}
    }

    if ( action->inherits( "QDesignerAction" ) ) {
	toolBar->insertAction( ( (QDesignerAction*)action )->widget(), action );
	( (QDesignerAction*)action )->widget()->installEventFilter( toolBar );
    } else if ( action->inherits( "QDesignerActionGroup" ) ) {
	if ( ( (QDesignerActionGroup*)action )->usesDropDown() ) {
	    toolBar->insertAction( ( (QDesignerActionGroup*)action )->widget(), action );
	    ( (QDesignerActionGroup*)action )->widget()->installEventFilter( toolBar );
	}
    } else if ( action->inherits( "QSeparatorAction" ) ) {
	toolBar->insertAction( ( (QSeparatorAction*)action )->widget(), action );
	( (QSeparatorAction*)action )->widget()->installEventFilter( toolBar );
    }
    if ( !action->inherits( "TQActionGroup" ) || ( (TQActionGroup*)action )->usesDropDown()) {
	if ( index == -1 )
	    toolBar->appendAction( action );
	else
	    toolBar->insertAction( index, action );
	toolBar->reInsert();
	TQObject::connect( action, TQ_SIGNAL( destroyed() ), toolBar, TQ_SLOT( actionRemoved() ) );
    } else {
	TQObjectList clo = action->childrenListObject();
	if (!clo.isEmpty()) {
	    TQObjectListIt it( clo );
	    int i = 0;
	    while ( it.current() ) {
		TQObject *o = it.current();
		++it;
		if ( !o->inherits( "TQAction" ) )
		    continue;
		// ### fix it for nested actiongroups
		if ( o->inherits( "QDesignerAction" ) ) {
		    QDesignerAction *ac = (QDesignerAction*)o;
		    toolBar->insertAction( ac->widget(), ac );
		    ac->widget()->installEventFilter( toolBar );
		    if ( index == -1 )
			toolBar->appendAction( ac );
		    else
			toolBar->insertAction( index + (i++), ac );
		}
	    }
	}
	toolBar->reInsert();
	TQObject::connect( action, TQ_SIGNAL( destroyed() ), toolBar, TQ_SLOT( actionRemoved() ) );
    }
}

void AddActionToToolBarCommand::unexecute()
{
    if ( action->inherits( "QDesignerAction" ) ) {
	TQString s = ( (QDesignerAction*)action )->widget()->name();
	s.prepend( "qt_dead_widget_" );
	( (QDesignerAction*)action )->widget()->setName( s.utf8() );
    }

    toolBar->removeAction( action );
    action->removeFrom( toolBar );
    TQObject::disconnect( action, TQ_SIGNAL( destroyed() ), toolBar, TQ_SLOT( actionRemoved() ) );
    if ( !action->inherits( "TQActionGroup" ) || ( (TQActionGroup*)action )->usesDropDown()) {
	action->removeEventFilter( toolBar );
    } else {
	TQObjectList clo = action->childrenListObject();
	if ( !clo.isEmpty() ) {
	    TQObjectListIt it( clo );
	    while ( it.current() ) {
		TQObject *o = it.current();
		++it;
		if ( !o->inherits( "TQAction" ) )
		    continue;
		if ( o->inherits( "QDesignerAction" ) ) {
		    o->removeEventFilter( toolBar );
		    toolBar->removeAction( (TQAction*)o );
		}
	    }
	}
    }
}

// ------------------------------------------------------------

AddActionToPopupCommand::AddActionToPopupCommand( const TQString &n, FormWindow *fw,
						  TQAction *a, QDesignerPopupMenu *p, int idx )
    : Command( n, fw ), action( a ), popup( p ), index( idx )
{
}

void AddActionToPopupCommand::execute()
{
    if ( action->inherits( "TQActionGroup" ) ) {
	if ( ( (TQActionGroup*)action )->usesDropDown() ) {
	    action->addTo( popup );
	    popup->insertAction( index, action );
	} else {
	    action->addTo( popup );
	    TQObjectList clo = action->childrenListObject();
	    if ( !clo.isEmpty() ) {
		TQObjectListIt it( clo );
		int i = 0;
		while ( it.current() ) {
		    TQObject *o = it.current();
		    ++it;
		    if ( !o->inherits( "TQAction" ) )
			continue;
		    QDesignerAction *ac = (QDesignerAction*)o;
		    popup->insertAction( index + (i++), ac );
		}
	    }
	}
	popup->reInsert();
	TQObject::connect( action, TQ_SIGNAL( destroyed() ), popup, TQ_SLOT( actionRemoved() ) );
    } else {
	if ( !action->inherits( "QDesignerAction" ) || ( (QDesignerAction*)action )->supportsMenu() ) {
	    action->addTo( popup );
	    popup->insertAction( index, action );
	    popup->reInsert();
	    TQObject::connect( action, TQ_SIGNAL( destroyed() ), popup, TQ_SLOT( actionRemoved() ) );
	}
    }
}

void AddActionToPopupCommand::unexecute()
{
    action->removeFrom( popup );
    popup->removeAction( action );
    TQObject::disconnect( action, TQ_SIGNAL( destroyed() ), popup, TQ_SLOT( actionRemoved() ) );
    if ( !action->inherits( "TQActionGroup" ) || ( (TQActionGroup*)action )->usesDropDown()) {
	action->removeEventFilter( popup );
    } else {
	TQObjectList clo = action->childrenListObject();
	if ( !clo.isEmpty() ) {
	    TQObjectListIt it( clo );
	    while ( it.current() ) {
		TQObject *o = it.current();
		++it;
		if ( !o->inherits( "TQAction" ) )
		    continue;
		if ( o->inherits( "QDesignerAction" ) ) {
		    o->removeEventFilter( popup );
		    popup->removeAction( (TQAction*)o );
		}
	    }
	}
    }
}

// ------------------------------------------------------------

AddMenuCommand::AddMenuCommand( const TQString &n, FormWindow *fw, TQMainWindow *mw )
    : Command( n, fw ), menuBar( 0 ), popup( 0 ), mainWindow( mw ), id( -1 ), name( "Menu" )
{
}

void AddMenuCommand::execute()
{
    if ( !popup ) {
	TQString n = "PopupMenu";
	popup = new QDesignerPopupMenu( mainWindow );
	formWindow()->unify( popup, n, true );
	popup->setName( n.utf8() );
    }
    if ( !mainWindow->child( 0, "TQMenuBar" ) ) {
	menuBar = new QDesignerMenuBar( (TQWidget*)mainWindow );
	menuBar->setName( "menubar" );
    } else {
	menuBar = (QDesignerMenuBar*)mainWindow->menuBar();
    }
    if ( id == -1 )
	id = mainWindow->menuBar()->insertItem( name, popup );
    else
	id = mainWindow->menuBar()->insertItem( name, popup, id, index );
    formWindow()->killAccels( formWindow() );
}

void AddMenuCommand::unexecute()
{
    if ( !popup || !menuBar )
	return;
    menuBar->removeItem( id );
    formWindow()->killAccels( formWindow() );
}

// ------------------------------------------------------------

RenameMenuCommand::RenameMenuCommand( const TQString &n, FormWindow *fw, QDesignerMenuBar *mb,
				      int i, const TQString &on, const TQString &nn )
    : Command( n, fw ), menuBar( mb ), id( i ), oldName( on ), newName( nn )
{
}

void RenameMenuCommand::execute()
{
    menuBar->changeItem( id, newName );
    formWindow()->killAccels( formWindow() );
}

void RenameMenuCommand::unexecute()
{
    menuBar->changeItem( id, oldName );
    formWindow()->killAccels( formWindow() );
}

// ------------------------------------------------------------

MoveMenuCommand::MoveMenuCommand( const TQString &n, FormWindow *fw, QDesignerMenuBar *mb,
				  QDesignerPopupMenu *p, int fidx, int tidx, const TQString &txt )
    : Command( n, fw ), menuBar( mb ), popup( p ), fromIdx( fidx ), toIdx( tidx ), text( txt )
{
}

void MoveMenuCommand::execute()
{
    menuBar->removeItem( menuBar->idAt( fromIdx ) );
    menuBar->insertItem( text, popup, -1, toIdx );
    formWindow()->killAccels( formWindow() );
}

void MoveMenuCommand::unexecute()
{
    menuBar->removeItem( menuBar->idAt( toIdx ) );
    menuBar->insertItem( text, popup, -1, fromIdx );
    formWindow()->killAccels( formWindow() );
}

// ------------------------------------------------------------

AddToolBarCommand::AddToolBarCommand( const TQString &n, FormWindow *fw, TQMainWindow *mw )
    : Command( n, fw ), toolBar( 0 ), mainWindow( mw )
{
}

void AddToolBarCommand::execute()
{
    if ( !toolBar ) {
	toolBar = new QDesignerToolBar( mainWindow );
	TQString n = "Toolbar";
	formWindow()->unify( toolBar, n, true );
	toolBar->setName( n.utf8() );
	mainWindow->addToolBar( toolBar, n );
    } else {
	toolBar->show();
    }
}

void AddToolBarCommand::unexecute()
{
    toolBar->hide();
}
#include "command.moc"
