/***************************************************************************
 *   Copyright (C) 2006-2012 by Thomas Schweitzer                          *
 *   thomas-schweitzer(at)arcor.de                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2.0 as   *
 *   published by the Free Software Foundation.                            *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program in the file LICENSE.GPL; if not, write to the *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "config.h"
#include "SettingsPaths.h"

#include <tqapplication.h>
#include <tqdir.h>

#include <stdlib.h>

/*
    \class SettingsPaths
    \brief SettingsPaths is a pure static functions class from which info
    about the paths needed for settings can be retrieved.
*/

bool     SettingsPaths::m_alreadyInitialized = false;
bool     SettingsPaths::m_portableMode = false;
TQString SettingsPaths::m_applicationBinaryPath = "";
TQString SettingsPaths::m_globalFilesPath = "";
TQString SettingsPaths::m_indenterPath = "";
TQString SettingsPaths::m_settingsPath = "";
TQString SettingsPaths::m_tempPath = "";

/*
    \brief Initializes all available information about the paths.

    Mainly during this init it is detected whether to start in portable mode or not. This is
    done by testing whether the directory "config" is in the same directory as this
    applications executable file.
    In portable mode all data is ONLY written to subdirectories of the applications executable file.
    Means also that the directory "indenters" has to be there.
    In not portable mode (multiuser mode) only users home directory is used for writing config data.
 */
void SettingsPaths::init()
{
	m_alreadyInitialized = true;

	tqDebug("Initializing application paths.");

	// Get the applications binary path, with respect to MacOSXs use of the .app folder.
	m_applicationBinaryPath = tqApp->applicationDirPath();
	// Remove any trailing slashes
	while (m_applicationBinaryPath.right(1) == "/")
	{
		m_applicationBinaryPath.truncate(m_applicationBinaryPath.length() - 1);
	}

	// If the "config" directory is a subdir of the applications binary path, use this one
	// (portable mode)
	m_settingsPath = m_applicationBinaryPath + "/config";
	if (TQFile::exists(m_settingsPath))
	{
		m_portableMode = true;
		TQDir dirCreator;
		m_globalFilesPath = m_applicationBinaryPath;
		m_indenterPath    = m_applicationBinaryPath + "/indenters";
		dirCreator.mkdir(m_settingsPath);
		m_tempPath = m_applicationBinaryPath + "/temp";
		//TODO: If the portable drive has write protection, use local temp path and clean it up on exit.
		dirCreator.mkdir(m_tempPath);
	}
	// ... otherwise use the system specific global application data path.
	else
	{
		m_portableMode = false;
		TQDir dirCreator;
		// Remove any trailing slashes.
		m_settingsPath = TQDir::homeDirPath();
		while (m_settingsPath.right(1) == "/")
		{
			m_settingsPath.truncate(m_settingsPath.length() - 1);
		}
		m_settingsPath    = m_settingsPath + "/.universalindentgui";
		m_globalFilesPath = APP_SHARE_PATH;
		dirCreator.mkdir(m_settingsPath);
		// If a highlighter config file does not exist in the users home config dir
		// copy the default config file over there.
		if (!TQFile::exists(m_settingsPath + "/UiGuiSyntaxHighlightConfig.ini"))
		{
			TQString copyCmd("cp " + m_globalFilesPath + "/config/UiGuiSyntaxHighlightConfig.ini " +
			        m_settingsPath + "/UiGuiSyntaxHighlightConfig.ini");
			int res = system(copyCmd.local8Bit().data()); 
			if (res == -1)
			{
				tqWarning("Unable to copy default highlighter config to the user home config folder.");
			}
		}
		m_indenterPath = m_globalFilesPath + "/indenters";

		m_tempPath = getenv("TMPDIR");
		if (m_tempPath.isEmpty())
			m_tempPath = getenv("TEMP");
		if (m_tempPath.isEmpty())
			m_tempPath = getenv("TMP");
		if (m_tempPath.isEmpty()) 
			m_tempPath = "/tmp";
		while (m_tempPath.right(1) == "/")
		{
			m_tempPath.truncate(m_tempPath.length() - 1);
		}
		// On Unix based systems create a random temporary directory for security
		// reasons. Otherwise an evil human being could create a symbolic link
		// to an important existing file which gets overwritten when UiGUI writes
		// into this normally temporary but linked file.
		m_tempPath = m_tempPath + "/UniversalIndentGUI-XXXXXX";
		m_tempPath = mkdtemp(m_tempPath.local8Bit().data());
	}

	tqDebug("Paths are:\n"
	        "    m_applicationBinaryPath = %s\n"
					"    m_settingsPath = %s\n"
			    "    m_globalFilesPath = %s\n"
					"    m_indenterPath = %s\n"
					"    m_tempPath = %s\n"
					"Running in portable mode = %s",
					m_applicationBinaryPath.local8Bit().data(),
					m_settingsPath.local8Bit().data(),
					m_globalFilesPath.local8Bit().data(),
	        m_indenterPath.local8Bit().data(),
					m_tempPath.local8Bit().data(),
					(m_portableMode ? "Yes" : "No"));
}

/*
    \brief Returns the path of the applications executable.
 */
const TQString SettingsPaths::getApplicationBinaryPath()
{
	if (!m_alreadyInitialized)
	{
		SettingsPaths::init();
	}
	return m_applicationBinaryPath;
}

/*
    \brief Returns the path where all settings are being/should be written to.
 */
const TQString SettingsPaths::getSettingsPath()
{
	if (!m_alreadyInitialized)
	{
		SettingsPaths::init();
	}
	return m_settingsPath;
}

/*
    \brief Returns the path where the files concerning all users reside. For example translations.
 */
const TQString SettingsPaths::getGlobalFilesPath()
{
	if (!m_alreadyInitialized)
	{
		SettingsPaths::init();
	}
	return m_globalFilesPath;
}

/*
    \brief Returns the path where the indenter executables reside.
 */
const TQString SettingsPaths::getIndenterPath()
{
	if (!m_alreadyInitialized)
	{
		SettingsPaths::init();
	}
	return m_indenterPath;
}

/*
    \brief Returns the path where the where all temporary data should be written to.
 */
const TQString SettingsPaths::getTempPath()
{
	if (!m_alreadyInitialized)
	{
		SettingsPaths::init();
	}
	return m_tempPath;
}

/*
    \brief Returns true if portable mode shall be used.
 */
bool SettingsPaths::getPortableMode()
{
	if (!m_alreadyInitialized)
	{
		SettingsPaths::init();
	}
	return m_portableMode;
}

/*
    \brief Completely deletes the created temporary directory with all of its content.
 */
void SettingsPaths::cleanAndRemoveTempDir()
{
	TQString removeCmd("rm -rf " + m_tempPath);
	int res = system(removeCmd.local8Bit().data()); 
	if (res == -1)
	{
		tqWarning("Unable to remove temporary folder %s.", m_tempPath.local8Bit().data());
	}
}
