/**************************************************************************
*   Copyright (C) 2006 by Michel Ludwig (michel.ludwig@kdemail.net)       *
***************************************************************************/

/**************************************************************************
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#ifndef EDITORKEYSEQUENCEMANAGER_H
#define EDITORKEYSEQUENCEMANAGER_H

#include <tqevent.h>
#include <tqmap.h>
#include <tqobject.h>
#include <tqstring.h>
#include <tqstringlist.h>

// include <tdeaction.h>
#include <kate/view.h>

namespace KileJScript {
	class JScript;
	class Manager;
}

class KileInfo;

namespace KileEditorKeySequence {
	/**
	 * This class represents an action that can be assigned to an editor key sequence.
	 **/
 	class Action {
		public:
			Action();
			virtual ~Action();

			/**
			 * The main method, which implements the "action" itself.
			 **/
			virtual void execute() = 0;

			/**
			 * Returns a textual representation of the action.
			 **/
			virtual TQString getDescription() const;
 	};

	/**
	 * This class represents the execution of a JavaScript in Kile.
	 **/
	class ExecuteJScriptAction : public Action {
		public:
			ExecuteJScriptAction(KileJScript::JScript *jScript, KileJScript::Manager *jScriptManager);
			virtual ~ExecuteJScriptAction();

			virtual void execute();
			virtual TQString getDescription() const;

		protected:
			KileJScript::JScript *m_jScript;
			KileJScript::Manager *m_jScriptManager;
	};

	// forward declaration
	class Recorder;

	/**
	 * This manager class is responsible for handling the key sequences that get assigned 
	 * to actions. Currently, every key sequence can only trigger one single action.
	 *
	 * Whenever a watched key sequence is typed, the manager triggers the corresponding
	 * action. The only characters that are allowed in key sequences are those that make
	 * the cursor advance by one position, i.e. for example tabs are not allowed in key
	 * sequences.
	 **/
	class Manager : public TQObject {
		TQ_OBJECT
  
	
		friend class Recorder;

		public:
			/**
			 * Constructs a new manager object.
			 **/
			Manager(KileInfo* kileInfo, TQObject *parent = 0, const char *name = 0);
			virtual ~Manager();

			/**
			 * Adds a new consequence and the corresponding action.
			 * @param seq the key sequence
			 * @param action the action for the sequence
			 **/
			void addAction(const TQString& seq, Action *action);

			/** 
			 * Convenience method. Adds a key sequence-to-action map to this 
			 * manager, removing any existing mappings.
			 * @warning This method overrides any exising mappings !
			 **/
			void addActionMap(const TQMap<TQString, Action*>& map);

			/**
			 * Removes all the mappings.
			 **/
			void clear();

			/**
			 * Returns a list of all the key sequences that are currently being
			 * watched.
			 **/
			const TQStringList& getWatchedKeySequences();

			/**
			 * Returns the key sequence that corresponds to an action.
			 * @param a the action that is considered
			 **/
			TQString getKeySequence(const Action* a);

			/**
			 * Returns the action that corresponds to a key sequence.
			 **/
			Action* getAction(const TQString& seq);

			/**
			 * Remove a key sequence, i.e. the key sequence is no longer watched.
			 * @param seq the key sequence that should be removed
			 **/
			void removeKeySequence(const TQString& seq);

			/**
			 * Convenience method. Removes every key sequence contained in the list.
			 * @see removeKeySequence(const TQString& seq)
			 **/
			void removeKeySequence(const TQStringList& l);

			/**
			 * @warning not implemented yet !
			 **/
			void setEditorKeySequence(const TQString& seq, Action *action);

			/**
			 * Checks whether the sequence "seq" is already assigned to an action.
			 * This method also checks whether a longer sequence that starts with
			 * "seq" is assigned to an action.
			 * @param seq the sequence that should be checked
			 * @return "true" if and only the sequence "seq" or another sequence
			 *                that starts with "seq" is assigned to an action
			 **/
			bool isSequenceAssigned(const TQString& seq) const;

			/**
			 * Performs a few checks on a key sequence.
			 * @returns in the first component: 0 if the sequence is free; 1
			 *          if the sequence is assigned; 2 if there is a longer,
			 *          currently stored sequence that starts with "seq"; 3 
			 *          if "seq" starts with a shorter sequence that is currently
			 *          stored
			 *
			 *          in the second component: a string that corresponds to one
			 *          of the previous cases (in the case 0: TQString())
			 **/
			TQPair<int, TQString> checkSequence(const TQString& seq, const TQString& skip = TQString());
	
		signals:
			/**
			 * Emitted whenever the set of watched key sequences changes.
			 **/
			void watchedKeySequencesChanged();

		protected slots:
			/**
			 * Signalises to the manager that a (watched) sequence has been typed.
			 * @param seq the sequence that has been typed
			 **/
			void keySequenceTyped(const TQString& seq);

		protected:
			KileInfo *m_kileInfo;
			TQMap<TQString, Action*> m_actionMap;
			TQStringList m_watchedKeySequencesList;
	};

	/**
	 * This class keeps track of the characters that are typed. It is used in 
	 * conjunction with a Kate view and a KileEditorKeySequence::Manager.
	 **/
	class Recorder : public TQObject {
		TQ_OBJECT
  
		public:
			Recorder(Kate::View *view, Manager *manager);
			virtual ~Recorder();

		signals:
			/**
			 * Emitted whenever a key sequence that is currently watched has 
			 * been typed.
			 **/
			void detectedTypedKeySequence(const TQString& seq);
	

		public slots:
			/**
			 * Reloads the key sequences that this recorders watches.
			 **/
			void reloadWatchedKeySequences();

		protected:
			Manager *m_manager;
			TQString m_typedSequence;
			uint m_maxSequenceLength;
			uint m_oldCol, m_oldLine;
			Kate::View* m_view;
			TQStringList m_watchedKeySequencesList;

			virtual bool eventFilter(TQObject *o, TQEvent *e);

			/**
			 * Checks whether a key sequence is currently watched.
			 * @param s the key sequence that should be checked
			 **/
			bool seekForKeySequence(const TQString& s);
	};
}

#endif
