/* This file is part of the KDE project
   Copyright (C) 2003 Daniel Molkentin <molkentin@kde.org>
   Copyright (C) 2003 Joseph Wenninger <jowenn@kde.org>
   Copyright (C) 2003-2006 Jaroslaw Staniek <js@iidea.pl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef KEXIDB_DRIVER_MNGR_H
#define KEXIDB_DRIVER_MNGR_H

#include <tqobject.h>
#include <tqcstring.h>
#include <tqmap.h>
#include <tqdict.h>

#include <klibloader.h>
#include <kservice.h>

#include <kexidb/driver.h>

namespace KexiDB {

class DriverManagerInternal;
class Connection;
class ConnectionData;

//! Database driver management, e.g. finding and loading drivers.
class KEXI_DB_EXPORT DriverManager : public TQObject, public KexiDB::Object
{
	public:
		typedef TQMap<TQString, KService::Ptr> ServicesMap;

		DriverManager();
		virtual ~DriverManager();

		/*! Tries to load db driver with named name \a name.
			The name is case insensitive.
			\return db driver, or 0 if error (then error message is also set) */
		Driver* driver(const TQString& name);

		/*! returns list of available drivers names. 
			That drivers can be loaded by first use of driver() method. */
		const TQStringList driverNames();

		/*! returns information list of available drivers. 
			That drivers can be loaded by first use of driver() method. */
		const KexiDB::Driver::InfoMap driversInfo();

		/*! \return information about driver's named with \a name.
			The name is case insensitive. 
			You can check if driver information is not found calling 
			Info::name.isEmpty() (then error message is also set). */
		KexiDB::Driver::Info driverInfo(const TQString &name);

		/*! \return service information about driver's named with \a name.
			The name is case insensitive. 
			In most cases you can use driverInfo() instead. */
		KService::Ptr serviceInfo(const TQString &name);

		/*! \return a map structure of the services. Not necessary for everyday use. */
		const ServicesMap& services();

		/*! Looks up a drivers list by MIME type of database file.
		 Only file-based database drivers are checked.
		 The lookup is case insensitive.
		 \return driver name or null string if no driver found.
		*/
		TQString lookupByMime(const TQString &mimeType);

		//! server error is set if there is error at KService level (useful for debugging)
		virtual TQString serverErrorMsg();
		virtual int serverResult();
		virtual TQString serverResultName();

		/*! HTML information about possible problems encountered.
		 It's displayed in 'details' section, if an error encountered. 
		 Currently it contains a list of incompatible db drivers. 
		 Used in KexiStartupHandler::detectDriverForFile(). */
		TQString possibleProblemsInfoMsg() const;

	protected:
		virtual void drv_clearServerResult();

	private:
		DriverManagerInternal *d_int;
};

} //namespace KexiDB

#endif
