/* This file is part of the KDE project
   Copyright (C) 2004 Dag Andersen <danders@get2net.dk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation;
   version 2 of the License.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef KPTMAP_H
#define KPTMAP_H


#include <tqmap.h>
#include <tqdatetime.h>
#include <tqstring.h>
#include <tqpair.h>
#include <tqvaluelist.h>

#include <kdebug.h>

namespace KPlato
{

namespace Map {
enum State { None=0, NonWorking=1, Working=2 };
} // Map namespace

typedef TQMap<TQString, int> DateMapType;
class DateMap : public DateMapType
{
public:
    DateMap() {}
    virtual ~DateMap() {}

    virtual bool contains(TQDate date) const { return DateMapType::contains(date.toString(TQt::ISODate)); }

    void insert(TQString date, int state=Map::NonWorking) {
        //kdDebug()<<k_funcinfo<<date<<"="<<state<<endl;
        if (state == Map::None)
            DateMapType::remove(date);
        else
            DateMapType::insert(date, state);
    }
    void insert(TQDate date, int state=Map::NonWorking) { insert(date.toString(TQt::ISODate), state); }

    void remove(TQDate date) {
        //kdDebug()<<k_funcinfo<<date.toString(TQt::ISODate)<<endl;
        DateMapType::remove(date.toString(TQt::ISODate));
    }

    int state(TQString date) {
        DateMapType::iterator it = find(date);
        if (it == end()) return 0;
        else return it.data();
    }
    int state(TQDate date) { return state(date.toString(TQt::ISODate)); }

    bool operator==(const DateMap &m) const { 
        return keys() == m.keys() && values() == m.values(); 
    }
    bool operator!=(const DateMap &m) const { 
        return keys() != m.keys() || values() != m.values(); 
    }
    
    // boolean use
    void toggle(TQString date, int state=Map::NonWorking) {
        //kdDebug()<<k_funcinfo<<date<<"="<<state<<endl;
        if (DateMapType::contains(date))
            DateMapType::remove(date);
        else
            DateMapType::insert(date, state);
    }
    void toggle(TQDate date, int state=Map::NonWorking) { return toggle(date.toString(TQt::ISODate)); }
    void toggleClear(TQString date, int state=Map::NonWorking) {
        //kdDebug()<<k_funcinfo<<date<<"="<<state<<endl;
        bool s = DateMapType::contains(date);
        clear();
        if (!s) insert(date, state);
    }
    void toggleClear(TQDate date, int state=Map::NonWorking) { toggleClear(date.toString(TQt::ISODate)); }
};

typedef TQMap<int, int> IntMapType;
class IntMap : public IntMapType
{
public:
    IntMap() {}
    virtual ~IntMap() {}

    void insert(int key, int state=Map::NonWorking) {
        if (state == Map::None)
            IntMapType::remove(key);
        else
            IntMapType::insert(key, state); }

    virtual int state(int key) {
        IntMapType::iterator it = IntMapType::find(key);
        if (it == IntMapType::end()) return 0;
        else return it.data();
    }

    bool operator==(const IntMap &m) const { 
        return keys() == m.keys() && values() == m.values(); 
    }
    bool operator!=(const IntMap &m) const { 
        return keys() != m.keys() || values() != m.values(); 
    }
    
    // boolean use
    void toggle(int key, int state=Map::NonWorking) { IntMapType::contains(key) ? remove(key) : insert(key, state); }
    void toggleClear(int key, int state=Map::NonWorking) {
        bool s =contains(key);
        clear();
        if (!s) insert(key, state);
    }
};

class WeekMap : public IntMap
{
public:
    bool contains(int week, int year) { return IntMap::contains(week*10000 + year); }
    bool contains(TQPair<int,int> week) { return contains(week.first,  week.second); }

    void insert(int week, int year, int state=Map::NonWorking) {
        if (week < 1 || week > 53) { kdError()<<k_funcinfo<<"Illegal week number: "<<week<<endl; return; }
        IntMap::insert(week*10000 + year, state);
    }
    void insert(TQPair<int,int> week, int state=Map::NonWorking) { insert(week.first, week.second, state); }

    void insert(WeekMap::iterator it, int state) { insert(week(it.key()), state); }

    void remove(TQPair<int,int> week) { IntMap::remove(week.first*10000 + week.second); }

    static TQPair<int, int> week(int key) { return TQPair<int, int>(key/10000, key%10000); }

    int state(TQPair<int, int> week) { return IntMap::state(week.first*10000 + week.second); }
    int state(int week, int year) { return state(TQPair<int, int>(week, year)); }

    void toggle(TQPair<int,int> week, int state=Map::NonWorking) {
        if (week.first < 1 || week.first > 53) { kdError()<<k_funcinfo<<"Illegal week number: "<<week.first<<endl; return; }
        IntMap::toggle(week.first*10000 + week.second, state);
    }
    void toggleClear(TQPair<int,int> week, int state=Map::NonWorking) {
        if (week.first < 1 || week.first > 53) { kdError()<<k_funcinfo<<"Illegal week number: "<<week.first<<endl; return; }
        IntMap::toggleClear(week.first*10000 + week.second, state);
    }
};

}  //KPlato namespace

#endif
