/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  begin:     Mit Aug 7 2002
  copyright: (C) 2002 by Dario Abatianni
  email:     eisfuchs@tigress.com
*/
// Copyright (C) 2004-2008 Shintaro Matsuoka <shin@shoegazed.org>

#include "dcctransferpanel.h"
#include "konversationapplication.h"
#include "dcctransferdetailedinfopanel.h"
#include "dcctransfermanager.h"
#include "dcctransferpanelitem.h"
#include "dcctransfersend.h"

#include <tqhbox.h>
#include <tqheader.h>
#include <tqpushbutton.h>
#include <tqtooltip.h>
#include <tqvbox.h>

#include <kdebug.h>
#include <tdeversion.h>
#include <kdialog.h>
#include <tdeglobal.h>
#include <tdeglobalsettings.h>
#include <kiconloader.h>
#include <tdelistview.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <tdepopupmenu.h>
#include <krun.h>
#include <tdeapplication.h>


DccTransferPanel::DccTransferPanel(TQWidget* parent) : ChatWindow(parent)
{
    setType(ChatWindow::DccTransferPanel);
    setName(i18n("DCC Status"));

    initGUI();

    connect( KonversationApplication::instance()->getDccTransferManager(), TQ_SIGNAL( newTransferAdded( DccTransfer* ) ), this, TQ_SLOT( slotNewTransferAdded( DccTransfer* ) ) );
}

DccTransferPanel::~DccTransferPanel()
{
    // remember column widths
    TQValueList<int> columnWidths;
    for ( uint i = 0 ; i < Column::COUNT ; ++i )
        columnWidths.push_back( m_listView->columnWidth( i ) );
    Preferences::setDccColumnWidths( columnWidths );
}

void DccTransferPanel::initGUI()
{
    setSpacing( 0 );

    m_listView = new TDEListView(this,"dcc_control_panel");

    m_listView->setSelectionMode(TQListView::Extended);
    m_listView->setDragEnabled(true);
    m_listView->setAcceptDrops(true);
    m_listView->setSorting(-1,false);
    m_listView->setAllColumnsShowFocus(true);

    for(unsigned int i=0 ; i < Column::COUNT ; ++i)
        m_listView->addColumn("");

    //m_listView->setColumnText(Column::TypeIcon,      "");
    m_listView->setColumnText(Column::OfferDate,     i18n("Started at"));
    m_listView->setColumnText(Column::Status,        i18n("Status"));
    m_listView->setColumnText(Column::FileName,      i18n("File"));
    m_listView->setColumnText(Column::PartnerNick,   i18n("Partner"));
    m_listView->setColumnText(Column::Progress,      i18n("Progress"));
    m_listView->setColumnText(Column::Position,      i18n("Position"));
    m_listView->setColumnText(Column::TimeLeft,      i18n("Remaining"));
    m_listView->setColumnText(Column::CurrentSpeed,  i18n("Speed"));
    m_listView->setColumnText(Column::SenderAddress, i18n("Sender Address"));

    TQValueList<int> columnWidths = Preferences::dccColumnWidths();
    for ( uint i = 0 ; i < Column::COUNT && i < columnWidths.count() ; ++i )
        m_listView->setColumnWidth( i, columnWidths[i] );

    m_listView->setColumnWidthMode(Column::FileName, TQListView::Manual);

    m_listView->setColumnAlignment(Column::OfferDate,     AlignHCenter);
    m_listView->setColumnAlignment(Column::Progress,      AlignHCenter);
    m_listView->setColumnAlignment(Column::Position,      AlignHCenter);
    m_listView->setColumnAlignment(Column::TimeLeft,      AlignHCenter);
    m_listView->setColumnAlignment(Column::CurrentSpeed,  AlignHCenter);

    m_listView->setSorting(Column::OfferDate, false);

    connect(m_listView,TQ_SIGNAL (selectionChanged()),this,TQ_SLOT (updateButton()) );

    // detailed info panel
    m_detailPanel = new DccTransferDetailedInfoPanel(this);

    // button

    TQHBox* buttonsBox=new TQHBox(this);
    buttonsBox->setSpacing(spacing());

    // convenience, undeffed below again to avoid name clashes
    #define icon(s) TDEGlobal::iconLoader()->loadIconSet( s, TDEIcon::Small )

    m_buttonAccept = new TQPushButton(icon("media-playback-start"), i18n("Accept"), buttonsBox, "start_dcc");
    m_buttonAbort  = new TQPushButton(icon("process-stop"),        i18n("Abort"),  buttonsBox, "abort_dcc");
    m_buttonClear  = new TQPushButton(icon("edit-delete"),  i18n("Clear"),  buttonsBox, "clear_dcc");
    m_buttonOpen   = new TQPushButton(icon("application-x-executable"),        i18n("Open File"),   buttonsBox, "open_dcc_file");
    m_buttonDetail = new TQPushButton(icon("view_text"),   i18n("Details"), buttonsBox, "detail_dcc");
    m_buttonDetail->setToggleButton( true );

    TQToolTip::add( m_buttonAccept, i18n( "Start receiving" ) );
    TQToolTip::add( m_buttonAbort,  i18n( "Abort the transfer(s)" ) );
    TQToolTip::add( m_buttonOpen,   i18n( "Run the file" ) );
    TQToolTip::add( m_buttonDetail, i18n( "View DCC transfer details" ) );

    connect( m_buttonAccept, TQ_SIGNAL(clicked()), this, TQ_SLOT(acceptDcc()) );
    connect( m_buttonAbort,  TQ_SIGNAL(clicked()), this, TQ_SLOT(abortDcc()) );
    connect( m_buttonClear,  TQ_SIGNAL(clicked()), this, TQ_SLOT(clearDcc()) );
    connect( m_buttonOpen,   TQ_SIGNAL(clicked()), this, TQ_SLOT(runDcc()) );
    //connect( m_buttonDetail, TQ_SIGNAL(clicked()), this, TQ_SLOT(openDetail()) );
    connect( m_buttonDetail, TQ_SIGNAL(toggled(bool)), m_detailPanel, TQ_SLOT(setShown(bool)) );
    m_buttonDetail->setOn(true);


    // popup menu

    m_popup = new TDEPopupMenu(this);
    m_popup->insertItem(                         i18n("&Select All Items"),           Popup::SelectAll);
    m_popup->insertItem(                         i18n("S&elect All Completed Items"), Popup::SelectAllCompleted);
    m_popup->insertSeparator();                   // -----
    m_popup->insertItem(icon("media-playback-start"),     i18n("&Accept"),                     Popup::Accept);
    m_popup->insertItem(icon("process-stop"),            i18n("A&bort"),                      Popup::Abort);
    m_popup->insertSeparator();                   // -----
    // FIXME: make it neat
    m_popup->insertItem(icon("edit-redo"),       i18n("Resend"),                      Popup::Resend);
    m_popup->insertItem(icon("edit-delete"),      i18n("&Clear"),                      Popup::Clear);
    m_popup->insertSeparator();                   // -----
    m_popup->insertItem(icon("application-x-executable"),            i18n("&Open File"),                  Popup::Open);
    m_popup->insertItem(icon("messagebox_info"), i18n("File &Information"),           Popup::Info);

    #undef icon

    connect(m_listView, TQ_SIGNAL(contextMenuRequested(TQListViewItem*,const TQPoint&,int)), this, TQ_SLOT(popupRequested(TQListViewItem*,const TQPoint&,int)));
    connect(m_popup, TQ_SIGNAL(activated(int)), this, TQ_SLOT(popupActivated(int)));

    // misc.
    connect(m_listView, TQ_SIGNAL(doubleClicked(TQListViewItem*,const TQPoint&,int)), this, TQ_SLOT(doubleClicked(TQListViewItem*,const TQPoint&,int)));

    connect(m_listView, TQ_SIGNAL(currentChanged(TQListViewItem*)), this, TQ_SLOT(setDetailPanelItem(TQListViewItem*)));

    updateButton();
}

void DccTransferPanel::slotNewTransferAdded( DccTransfer* transfer )
{
    DccTransferPanelItem* item = new DccTransferPanelItem( this, transfer );
    connect( transfer, TQ_SIGNAL( statusChanged( DccTransfer*, int, int ) ), this, TQ_SLOT( slotTransferStatusChanged() ) );
    if ( m_listView->childCount() == 1 )
    {
        m_listView->clearSelection();
        m_listView->setSelected( item, true );
        m_listView->setCurrentItem( item );
        updateButton();
        setDetailPanelItem( item );
    }
}

void DccTransferPanel::slotTransferStatusChanged()
{
    updateButton();
    activateTabNotification(Konversation::tnfSystem);
}

void DccTransferPanel::updateButton()
{
    bool accept             = true,
         abort              = false,
         clear              = false,
         info               = true,
         open               = true,
         resend             = false,
         selectAll          = false,
         selectAllCompleted = false;

    int selectedItems = 0;
    TQListViewItemIterator it( m_listView );

    while( it.current() )
    {
        DccTransferPanelItem* item = static_cast<DccTransferPanelItem*>( it.current() );

        DccTransfer::DccType type = item->transfer()->getType();
        DccTransfer::DccStatus status = item->transfer()->getStatus();

        selectAll = true;
        selectAllCompleted |= ( status >= DccTransfer::Done );

        if( it.current()->isSelected() )
        {
            ++selectedItems;

            accept &= ( status == DccTransfer::Queued );

            abort  |= ( status < DccTransfer::Done );

            clear  |= ( status >= DccTransfer::Done );

            info   &= ( type == DccTransfer::Send ||
                status == DccTransfer::Done );

            open   &= ( type == DccTransfer::Send ||
                status == DccTransfer::Done );

            resend |= ( type == DccTransfer::Send &&
                status >= DccTransfer::Done );
        }
        ++it;
    }

    if( !selectedItems )
    {
        accept = false;
        abort = false;
        clear = false;
        info = false;
        open = false;
        resend = false;
    }

    if (!tdeApp->authorize("allow_downloading"))
    {
        accept = false;
    }

    m_buttonAccept->setEnabled( accept );
    m_buttonAbort->setEnabled( abort );
    m_buttonClear->setEnabled( clear );
    m_buttonOpen->setEnabled( open );

    m_popup->setItemEnabled( Popup::SelectAll,          selectAll );
    m_popup->setItemEnabled( Popup::SelectAllCompleted, selectAllCompleted );
    m_popup->setItemEnabled( Popup::Accept,             accept );
    m_popup->setItemEnabled( Popup::Abort,              abort );
    m_popup->setItemEnabled( Popup::Clear,              clear );
    m_popup->setItemEnabled( Popup::Open,               open );
    m_popup->setItemEnabled( Popup::Resend,             resend );
    m_popup->setItemEnabled( Popup::Info,               info );
}

void DccTransferPanel::setDetailPanelItem(TQListViewItem* item_)
{
    if ( item_ )
    {
        DccTransferPanelItem* item = static_cast< DccTransferPanelItem* >( item_ );
        m_detailPanel->setItem( item );
    }
}

void DccTransferPanel::acceptDcc()
{
    TQListViewItemIterator it( m_listView );
    while( it.current() )
    {
        if( it.current()->isSelected() )
        {
            DccTransferPanelItem* item=static_cast<DccTransferPanelItem*>( it.current() );
            DccTransfer* transfer = item->transfer();
            if( transfer->getType() == DccTransfer::Receive && transfer->getStatus() == DccTransfer::Queued )
                transfer->start();
        }
        ++it;
    }
}

void DccTransferPanel::abortDcc()
{
    TQListViewItemIterator it( m_listView );
    while( it.current() )
    {
        if( it.current()->isSelected() )
        {
            DccTransferPanelItem* item=static_cast<DccTransferPanelItem*>( it.current() );
            DccTransfer* transfer = item->transfer();
            if( transfer->getStatus() < DccTransfer::Done )
                transfer->abort();
        }
        ++it;
    }
}

void DccTransferPanel::resendFile()
{
    TQListViewItemIterator it( m_listView );
    while( it.current() )
    {
        if( it.current()->isSelected() )
        {
            DccTransferPanelItem* item=static_cast<DccTransferPanelItem*>( it.current() );
            DccTransfer* transfer = item->transfer();
            if( transfer->getType() == DccTransfer::Send && transfer->getStatus() >= DccTransfer::Done )
            {
                DccTransferSend* newTransfer = KonversationApplication::instance()->getDccTransferManager()->newUpload();

                newTransfer->setConnectionId( transfer->getConnectionId() );
                newTransfer->setPartnerNick( transfer->getPartnerNick() );
                newTransfer->setFileURL( transfer->getFileURL() );
                newTransfer->setFileName( transfer->getFileName() );
                // FIXME
                newTransfer->setOwnIp( transfer->getOwnIp() );

                if ( newTransfer->queue() )
                    newTransfer->start();
            }
        }
        ++it;
    }
}

void DccTransferPanel::clearDcc()
{
    TQPtrList<TQListViewItem> lst;
    TQListViewItemIterator it( m_listView );
    while( it.current() )
    {
        DccTransferPanelItem* item = static_cast<DccTransferPanelItem*>( it.current() );
        // should we check that [item] is not null?
        if( it.current()->isSelected() && item->transfer()->getStatus() >= DccTransfer::Done )
            lst.append( it.current() );
        ++it;
    }

    // Figure out the first 'gap' in the selection and select that item, 
    // or, if there are no gaps, select first item below the selection
    TQPtrListIterator<TQListViewItem> selected( lst );
    bool itemSelected = false;
    while( selected.current() )
    {
        if (selected.current()->itemBelow() && !lst.containsRef(selected.current()->itemBelow()))
        {
            m_listView->setSelected(selected.current()->itemBelow(),true);
            m_listView->setCurrentItem(selected.current()->itemBelow());
            itemSelected = true;
            break; 
        }
        ++selected;
    }

    // When there are neither gaps in nor items below the selection, select the first item
    if (!itemSelected)
    {
        m_listView->setSelected(m_listView->firstChild(),true);
        m_listView->setCurrentItem(m_listView->firstChild());
    }

    lst.setAutoDelete( true );
    while( lst.remove() ) ;
    updateButton();
}

void DccTransferPanel::runDcc()
{
    TQListViewItemIterator it( m_listView );
    while( it.current() )
    {
        if( it.current()->isSelected() )
        {
            DccTransferPanelItem* item=static_cast<DccTransferPanelItem*>( it.current() );
            DccTransfer* transfer = item->transfer();
            if( transfer->getType() == DccTransfer::Send || transfer->getStatus() == DccTransfer::Done )
                item->runFile();
        }
        ++it;
    }
}

void DccTransferPanel::showFileInfo()
{
    TQListViewItemIterator it( m_listView );
    while( it.current() )
    {
        if( it.current()->isSelected() )
        {
            DccTransferPanelItem* item=static_cast<DccTransferPanelItem*>( it.current() );
            if( item->transfer()->getType() == DccTransfer::Send || item->transfer()->getStatus() == DccTransfer::Done )
                item->openFileInfoDialog();
        }
        ++it;
    }
}

void DccTransferPanel::selectAll()
{
    TQListViewItemIterator it( m_listView );
    while ( it.current() )
    {
        m_listView->setSelected( *it, true );
        ++it;
    }
    updateButton();
}

void DccTransferPanel::selectAllCompleted()
{
    TQListViewItemIterator it( m_listView );
    while ( it.current() )
    {
        DccTransferPanelItem* item=static_cast<DccTransferPanelItem*>( it.current() );
        m_listView->setSelected( *it, item->transfer()->getStatus() >= DccTransfer::Done );
        ++it;
    }
    updateButton();
}

void DccTransferPanel::popupRequested(TQListViewItem* /* item */, const TQPoint& pos, int /* col */)  // slot
{
    updateButton();
    m_popup->popup(pos);
}

void DccTransferPanel::popupActivated( int id ) // slot
{
    if ( id == Popup::Abort )                    abortDcc();
    else if ( id == Popup::Accept )              acceptDcc();
    else if ( id == Popup::Clear )               clearDcc();
    else if ( id == Popup::Info )                showFileInfo();
    else if ( id == Popup::Open )                runDcc();
    else if ( id == Popup::SelectAll )           selectAll();
    else if ( id == Popup::SelectAllCompleted )  selectAllCompleted();
    else if ( id == Popup::Resend )              resendFile();
}

void DccTransferPanel::doubleClicked(TQListViewItem* _item, const TQPoint& /* _pos */, int /* _col */)
{
    DccTransferPanelItem* item = static_cast<DccTransferPanelItem*>(_item);
    item->runFile();
}

// virtual
void DccTransferPanel::childAdjustFocus()
{
}

TDEListView* DccTransferPanel::getListView() 
{ 
  return m_listView; 
}

#include "dcctransferpanel.moc"
