#!/usr/bin/env python
"""
This is a rough Python translation of the ideas presented in this KMDI tutorial:

    http://web.tiscali.it/andreabergia/tdemditutorial.html

What does work:

    IDEAlMode - yay!

    Adding and closing child views

    Two-way syncing between a tool widget and a matching child view

All is not rosy, however:

    Instances of the KmdiExample maintain a dictionary of child views.  Values
    cannot be deleted from this dictionary during a window close (causes an
    immediate segfault).

    Child views created after initialization aren't numbered correctly; given
    the first problem, it's harder to do this than it's really worth.

    The example segfaults at shutdown if the tool (on the left) is is open but
    is not in overlap-mode.

"""
import os
import sys

from PyTQt.tqt import TQ_SIGNAL, TQVBoxLayout, TQLabel
from tdecore import i18n, TDEAboutData, TDEApplication, TDEGlobal, TDEIcon, TDECmdLineArgs
from tdeui import KDockWidget, TDEListBox, KStdAction

try:
    from tdemdi import KMdi, KMdiMainFrm, KMdiChildView
except (ImportError, ):
    print('Exception importing KMDI; check your PyTDE installation')
    sys.exit(1)


sigChildCloseRequest = TQ_SIGNAL('childWindowCloseRequest(KMdiChildView *)')
sigChildViewActivated = TQ_SIGNAL('viewActivated(KMdiChildView *)')
sigBoxSelectionChanged = TQ_SIGNAL('selectionChanged(TQListBoxItem *)')


def getIcon(name, group=TDEIcon.NoGroup, size=TDEIcon.SizeSmall):
    """ returns a kde icon by name

    """
    return TDEGlobal.instance().iconLoader().loadIcon(name, group, size)


class KmdiExample(KMdiMainFrm):
    """ KmdiExample(parent=None) -> an example KMdiMainFrm window

    """
    uifilebase = 'uitdemdi.rc'
    viewIcons = ('network', 'email', 'stop', 'back', 'forward', )
    toolIcons = ('view_icon', 'configure')

    def __init__(self, parent=None):
        KMdiMainFrm.__init__(self, parent, 'KmdiExample', KMdi.IDEAlMode)

        xmlfile = os.path.join('.', self.uifilebase)
        self.setXMLFile(os.path.abspath(xmlfile))
        actions = self.actionCollection()
        self.openNewAction = KStdAction.openNew(self.newView, actions)
        self.quitAction = KStdAction.quit(self.close, actions)
        self.closeAction = KStdAction.close(self.closeActiveChild, actions)
        self.createGUI(None)
        self.statusBar()
        self.resize(400, 300)

        self.tools = {}
        for idx, ico in enumerate(self.toolIcons):
            wid = TDEListBox(self, 'list%s' % idx)
            self.makeTool(wid, 'Tool %s' % idx, ico)
        ## smells
        self.mainToolWidget = maintool = self.tools['Tool 0'][0]

        self.childs = {}
        for idx, ico in enumerate(self.viewIcons):
            self.makeView('View %s' % idx, ico, ico)


        self.connect(self, sigChildViewActivated, self.activatedMessage)
        self.connect(self, sigChildViewActivated, self.syncFromChildView)
        self.connect(maintool, sigBoxSelectionChanged, self.syncFromMainTool)
        self.syncFromChildView(self.activeWindow())


    def syncFromMainTool(self, item):
        """ activate the view that matches the item text

        """
        try:
            self.activateView(self.findWindow(item.text()))
        except (RuntimeError, ):
            pass

    def syncFromChildView(self, child):
        """ sync the main tool to the indicated child

        """
        maintool = self.mainToolWidget
        item = maintool.findItem(child.tabCaption())
        if item:
            maintool.setSelected(item, True)

    def makeTool(self, widget, caption, icon, percent=50):
        """ makes a tool from the widget

        """
        tip = i18n('%s Tool Tip' % caption)
        dock = KDockWidget.DockLeft
        maindock = self.getMainDockWidget()
        widget.setIcon(getIcon(icon))
        tool = self.addToolWindow(widget, dock, maindock, percent, tip, caption)
        self.tools[caption] = (widget, tool)

    def makeView(self, label, icon, text):
        """ makes a child view with a text label and a pixmap label

        """
        view = KMdiChildView(label, self)
        self.childs[label] = view
        view.setIcon(getIcon(icon))
        layout = TQVBoxLayout(view)
        layout.setAutoAdd(True)

        lbl = i18n('Label for a view with an icon named %s' % text)
        lbl = TQLabel(lbl, view)
        pxm = TQLabel('', view)
        pxm.setPixmap(getIcon(icon, size=TDEIcon.SizeLarge))

        self.addWindow(view)
        self.mainToolWidget.insertItem(label)
        self.connect(view, sigChildCloseRequest, self.closeChild)

    def removeMainToolItem(self, view):
        """ remove item from the main list tool that corresponds to the view

        """
        maintool = self.mainToolWidget
        maintool.takeItem(maintool.findItem(view.tabCaption(), 0))

    def newView(self):
        """ make a view when the user invokes the new action

        """
        self.makeView('View %s' % len(self.childs), 'network', 'A Fresh View')
        self.syncFromChildView(self.activeWindow())

    def closeActiveChild(self):
        """ close the current view

        """
        self.removeMainToolItem(self.activeWindow())
        self.closeActiveView()
        self.syncFromChildView(self.activeWindow())

    def closeChild(self, which):
        """ called to close a view from its tab close button

        """
        try:
            caption = which.tabCaption()
        except (AttributeError, ):
            ## probably None; bug in tdemdi?
            return
        self.removeMainToolItem(which)
        which.close()
        self.statusBar().message(i18n('%s closed' % caption))
        self.syncFromChildView(self.activeWindow())

    def activatedMessage(self, view):
        """ updates the status bar with the caption of the current view

        """
        try:
            self.statusBar().message(i18n('%s activated' % view.tabCaption()))
        except (RuntimeError, ):
            ## sometimes the status bar or the current object is already gone...
            pass


if __name__ == '__main__':
    aname = b'PyTDE KMDI Sample'
    desc = b'A Simple PyTDE KMDI Sample'
    ver = b'1.0'
    lic = TDEAboutData.License_GPL
    author = b'Troy Melhase'
    authormail = b'troy@gci.net'

    about = TDEAboutData(aname, aname, ver, desc, lic, b'troy@gci.net (c) 2004')
    about.addAuthor(author, b'hi, mom!', authormail)
    about.addAuthor (b'Jim Bublitz', b'For PyTDE', b'jbublitz@nwinternet.com')
    TDECmdLineArgs.init(sys.argv, about)
    app = TDEApplication()
    mainWindow = KmdiExample()
    mainWindow.show()
    app.exec_loop()
