/*
 *  Copyright (C) 2001-2003, Richard J. Moore <rich@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include <tqfile.h>
#include <tqhbox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqpushbutton.h>

#include <kdebug.h>
#include <kdialog.h>
#include <tdeglobal.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <tdelocale.h>
#include <tdepopupmenu.h>
#include <tdeprocess.h>
#include <ktextedit.h>
#include <twin.h>

#include <kjs/interpreter.h>
#include <kjs/ustring.h>
#include <kjs/types.h>

#include "jsbinding.h"
#include "kjsembedpart.h"

#include "jsconsolewidget.h"
#include "jsconsolewidget.moc"

namespace KJSEmbed {

class JSConsoleWidgetPrivate
{
};

JSConsoleWidget::JSConsoleWidget( KJSEmbedPart *jspart, TQWidget *parent, const char *name )
    : TQFrame( parent, name ? name : "jsconsole_widget" ),
      js(jspart), proc(0), d(0)
{
    setFocusPolicy( TQWidget::StrongFocus );
    createView();
}

JSConsoleWidget::~JSConsoleWidget()
{
}

void JSConsoleWidget::createView()
{
    TQPixmap px( TDEGlobal::iconLoader()->loadIcon("konsole", TDEIcon::NoGroup, TDEIcon::SizeSmall) );
    TQPixmap pxl( TDEGlobal::iconLoader()->loadIcon("konsole", TDEIcon::NoGroup, TDEIcon::SizeLarge) );
    setIcon( px );
    KWin::setIcons( winId(), pxl, px );

    ttl = new TDEPopupTitle( this, "title" );
    ttl->setText( i18n("JavaScript Console") );
    ttl->setIcon( px );

    log = new KTextEdit( this, "log_widget" );
    log->setReadOnly( true );
    log->setUndoRedoEnabled( false );
    log->setTextFormat( TQt::RichText );
    log->setWrapPolicy( TQTextEdit::Anywhere );
    log->setText( "<qt><pre>" );
    log->setFocusPolicy( TQWidget::NoFocus );

    // Command entry section
    cmdBox = new TQHBox( this, "cmd_box" );
    cmdBox->setSpacing( KDialog::spacingHint() );

    TQLabel *prompt = new TQLabel( i18n("&KJS>"), cmdBox, "prompt" );
    cmd = new KLineEdit( cmdBox, "cmd_edit" );
    cmd->setFocusPolicy( TQWidget::StrongFocus );
    cmd->setFocus();
    prompt->setBuddy( cmd );

    go = new TQPushButton( i18n("&Run"), cmdBox, "run_button" );
    go->setFixedSize( go->sizeHint() );

    connect( cmd, TQ_SIGNAL(returnPressed(const TQString&)), go, TQ_SLOT( animateClick() ) );
    connect( go, TQ_SIGNAL( clicked() ), TQ_SLOT( invoke() ) );

    // Setup completion
    TDECompletion *comp = cmd->completionObject();
    connect( cmd, TQ_SIGNAL(returnPressed(const TQString&)), comp, TQ_SLOT(addItem(const TQString&)) );

    // Layout
    TQVBoxLayout *vert = new TQVBoxLayout( this, KDialog::marginHint(), KDialog::spacingHint() );
    vert->addWidget( ttl );
    vert->addWidget( log );
    vert->addWidget( cmdBox );
}

void JSConsoleWidget::invoke()
{
    TQString code( cmd->text() );
    println( TQString( "<b><font color=\"#888888\">KJS&gt;</font> %1</b>" ).arg( code ) );
    execute( code );
}

bool JSConsoleWidget::execute( const TQString &cmd )
{
    return execute( cmd, KJS::Null() );
}

bool JSConsoleWidget::execute( const TQString &cmd, const KJS::Value &self )
{
    KJS::Completion jsres;
    bool ok = js->execute( jsres, cmd, self );

    // Executed ok
    if ( ok ) {

	// No return value
	if ( !jsres.isValueCompletion() )
	    return ok;

	// Return value
	KJS::Value ret = jsres.value();
	KJS::UString s = ret.toString( js->globalExec() );

	if ( s.isNull() ) {
	    warn( i18n("Success, but return value cannot be displayed") );
	    return ok;
	}

	TQString txt = s.qstring();
	txt = txt.replace( TQChar('\n'), "<br>" );
	println( txt );

	return ok;
    }

    // Handle errors
    KJS::ComplType ct = jsres.complType();
    if ( (ct == KJS::Throw) || (ct == KJS::Break) || ct == KJS::Continue ) {

	KJS::UString s = jsres.value().toString( js->globalExec() );
	if ( !s.isNull() )
	    warn( s.qstring() );
	else
	    warn( i18n("Unspecified error") );
    }
    else {
	kdDebug(80001) << "jsconsolewidget: Unknown completion error, " << ct << endl;
	warn( i18n("Unknown error returned, completion type %1").arg(ct) );
    }

    return ok;
}

void JSConsoleWidget::println( const TQString &msg )
{
    log->append( msg );
    log->scrollToBottom();
}

void JSConsoleWidget::warn( const TQString &msg )
{
    TQString err( "<font color=\"red\"><b>%1</b></font>" );
    println( err.arg(msg) );
}

//
// Process Handling
//

bool JSConsoleWidget::run( const TQString &cmd )
{
    kdDebug(80001) << "JSConsoleWidget::run(" << cmd << ")" << endl;

    if ( proc )
	return false;

    proc = new KShellProcess("/bin/sh");
    *proc << cmd;

    connect( proc, TQ_SIGNAL( processExited(TDEProcess *) ), TQ_SLOT( childExited() ) );
    connect( proc, TQ_SIGNAL( receivedStdout(TDEProcess *, char *, int) ),
	     this, TQ_SLOT( receivedStdOutput(TDEProcess *, char *, int) ) );
    connect( proc, TQ_SIGNAL( receivedStderr(TDEProcess *, char *, int) ),
	     this, TQ_SLOT( receivedStdError(TDEProcess *, char *, int) ) );

    return proc->start( TDEProcess::NotifyOnExit,
			TDEProcess::Communication( TDEProcess::Stdout|TDEProcess::Stderr ));
}


void JSConsoleWidget::childExited()
{
    TQString s;
    if ( proc->normalExit() ) {
	if ( proc->exitStatus() )
	    s = i18n( "<b>[Exited with status %1]</b>\n" ).arg( proc->exitStatus() );
	else
	    s = i18n( "<b>[Finished]</b>\n" );
	println( s );
    }
    else {
        s = i18n("[Aborted]\n");
	warn( s );
    }

    delete proc;
    proc = 0;
}

void JSConsoleWidget::receivedStdOutput( TDEProcess *, char *buffer, int buflen )
{
    TQCString buf = TQCString( buffer, buflen+1 );
    println( TQString(buf) );
}

void JSConsoleWidget::receivedStdError( TDEProcess *, char *buffer, int buflen )
{
    TQCString buf = TQCString( buffer, buflen+1 );
    warn( TQString(buf) );
}


} // namespace KJSEmbed
