/*
 * Copyright (C) 2004, Mart Kelder (mart.kde@hccnet.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "tdeio_count.h"

#include "tdeio.h"
#include "tdeio_proto.h"
#include "tdeio_single_subject.h"
#include "mailsubject.h"
#include "sortedmailsubject.h"

#include <kdebug.h>
#include <tdeio/jobclasses.h>
#include <tdeio/scheduler.h>
#include <tdeio/slave.h>
#include <tdeio/global.h>
#include <kurl.h>
#include <tqvaluelist.h>

#include <tqstring.h>

TDEIO_Count::TDEIO_Count( TQObject * parent, const char * name )
	: TQObject ( parent, name ),
	_kurl( 0 ),
	_metadata( 0 ),
	_protocol( 0 ),
	_valid( true ),
	_new_mailurls( 0 ),
	_subjects_pending( 0 ),
	_total_new_messages( 0 ),
	_popup_subjects( 0 )
{
}

TDEIO_Count::~TDEIO_Count()
{
	// Delete copies of urls.
	delete _kurl;
	delete _metadata;
}

void TDEIO_Count::count( KKioDrop *drop )
{
	if( _new_mailurls )
		return; //A counting is pending, so no new one is started.

	delete _kurl;
	delete _metadata;
	_tdeio = drop;

	/*
	 * Saving current settings: all actions are asynchroon, so if someone
	 * use slow servers, settings could been changed before this class is
	 * finished with counten. To be able to track back te staring values;
	 * these are saved in the class.
	 */
	_kurl = new KURL( *_tdeio->_kurl );
	_metadata = new TDEIO::MetaData( *_tdeio->_metadata );
	_protocol = _tdeio->_protocol;

	KURL kurl = *_kurl;
	TDEIO::MetaData metadata = *_metadata;

	// Serup a connection
	if( _protocol->connectionBased( ) )
	{
		_protocol->recheckConnectKURL( kurl, metadata );

		if( kurl.port() == 0 )
			kurl.setPort( _protocol->defaultPort( _tdeio->_ssl ) );

		if( ! ( _slave = TDEIO::Scheduler::getConnectedSlave( kurl, metadata ) ) ) //Forcing reload
		{
			kdWarning() << i18n( "Not able to open a tdeio slave for %1." ).arg( _protocol->configName() ) << endl;
			_tdeio->emitShowPassivePopup( i18n( "Not able to open a tdeio slave for %1." ).arg( _protocol->configName() ) );
			_valid = false;
			_tdeio->emitValidChanged();
			_slave = 0;
			//delete _new_mailurls; _new_mailurls = 0; //No connection pending
			return;
		}

		connect( _slave, TQ_SIGNAL( error( int, const TQString& ) ), _tdeio, TQ_SLOT( slotConnectionError( int, const TQString& ) ) );
		connect( _slave, TQ_SIGNAL( warning( const TQString& ) ), _tdeio, TQ_SLOT( slotConnectionWarning( const TQString& ) ) );
		connect( _slave, TQ_SIGNAL( infoMessage( const TQString& ) ), _tdeio, TQ_SLOT( slotConnectionInfoMessage( const TQString& ) ) );

		/*
		 * _protocol->recheckConnectKURL could have change kurl and metadata in order to have the right
		 * settings to connect. But some other functions assumed unmodified settings,
		 * so the settings are set back to his originals.
		 */
		kurl = *_kurl;
		metadata = *_metadata;
	}
	else
	{
		_slave = 0; //Prevent disconnecting not-existing slave
	}

	/* Blocking this function: no new counts can be started from now */
	_new_mailurls = new TQValueList< KKioDrop::FileInfo >;

	_protocol->recheckKURL( kurl, metadata );

	if( kurl.port() == 0 )
		kurl.setPort( _protocol->defaultPort( _tdeio->_ssl ) );

	//Making job to fetch file-list
	
	_job = TDEIO::listDir( kurl, false );
	_job->addMetaData( metadata );

	connect( _job, TQ_SIGNAL( result( TDEIO::Job* ) ), this, TQ_SLOT( result( TDEIO::Job* ) ) );
	connect( _job, TQ_SIGNAL( entries( TDEIO::Job*, const TDEIO::UDSEntryList& ) ),
	         this, TQ_SLOT( entries( TDEIO::Job*, const TDEIO::UDSEntryList& ) ) );

	if( _protocol->connectionBased() )
		TDEIO::Scheduler::assignJobToSlave( _slave, _job );
	else
		TDEIO::Scheduler::scheduleJob( _job );
}

void TDEIO_Count::stopActiveCount()
{
	if( !_new_mailurls )
		return;

	disconnect( _job, TQ_SIGNAL( result( TDEIO::Job* ) ), this, TQ_SLOT( result( TDEIO::Job* ) ) );
	disconnect( _job, TQ_SIGNAL( entries( TDEIO::Job*, const TDEIO::UDSEntryList& ) ),
	            this, TQ_SLOT( entries( TDEIO::Job*, const TDEIO::UDSEntryList& ) ) );

	TDEIO::Scheduler::cancelJob( _job );

	if( _slave )
	{
		//Slave seems to be disconnected by canceling the last job of the slave
		//TDEIO::Scheduler::disconnectSlave( _slave );
		_slave = 0;
	}

	//Deletings settings
	delete _kurl; _kurl = 0;
	delete _metadata; _metadata = 0;

	delete _new_mailurls; _new_mailurls = 0;
}

void TDEIO_Count::showPassive( const TQString& id )
{
	KURL kurl = *_tdeio->_kurl;
	TDEIO::MetaData metadata = *_tdeio->_metadata;
	kurl = id;
	//TDEIO::Slave *slave = 0;

	_tdeio->_protocol->readSubjectKURL( kurl, metadata );
	if( kurl.port() == 0 )
		kurl.setPort( _tdeio->_protocol->defaultPort( _tdeio->_ssl ) );

	TDEIO_Single_Subject *subject = new TDEIO_Single_Subject( this, id.latin1(), kurl, metadata, _tdeio->_protocol, _slave, id, 0 );

	_subjects_pending++;

	connect( subject, TQ_SIGNAL( readSubject( KornMailSubject* ) ), this, TQ_SLOT( addtoPassivePopup( KornMailSubject* ) ) );
        connect( subject, TQ_SIGNAL( finished( TDEIO_Single_Subject* ) ), this, TQ_SLOT( deleteSingleSubject( TDEIO_Single_Subject* ) ) );
}

void TDEIO_Count::disconnectSlave()
{
	if( _subjects_pending > 0 )
		return; //Still getting data

	if( !_protocol->connectionBased() )
		return; //Protocol doesn't have a connection

	if( !_slave )
		return; //Slave doens't exist

	//Disconnect slave
	TDEIO::Scheduler::disconnectSlave( _slave );
	_slave = 0;
	_protocol = 0;
}

//This function is called when fetching is over
void TDEIO_Count::result( TDEIO::Job* job )
{
	//job should be the latest job; elsewise: print an error.
	if( job != _job )
		kdError() << i18n( "Got unknown job; something must be wrong..." ) << endl;

	//look of an error occurred. If there is, print the error.
	//This could be very useful by resolving bugs.
	if( job->error() )
	{
		kdError() << i18n( "The next TDEIO-error occurred by counting: %1" ).arg( job->errorString() ) << endl;
		_tdeio->emitShowPassivePopup( i18n( "The next TDEIO-error occurred by counting: %1" ).arg( job->errorString() ) );
		_valid = false;
		_tdeio->emitValidChanged();
	}

	disconnect( job, TQ_SIGNAL( result( TDEIO::Job* ) ), this, TQ_SLOT( result( TDEIO::Job* ) ) );
	disconnect( job, TQ_SIGNAL( entries( TDEIO::Job*, const TDEIO::UDSEntryList& ) ),
	            this, TQ_SLOT( entries( TDEIO::Job*, const TDEIO::UDSEntryList& ) ) );

	disconnectSlave();

	//Deletings settings
	delete _kurl; _kurl = 0;
	delete _metadata; _metadata = 0;

	if( _tdeio->_mailurls->count() != _new_mailurls->count() )
	{
		*_tdeio->_mailurls = *_new_mailurls;
		_tdeio->emitChanged(_tdeio->_mailurls->count());
	}
	else
	{
		*_tdeio->_mailurls = *_new_mailurls;
	}
	delete _new_mailurls; _new_mailurls = 0;

	_valid = true;
	_tdeio->emitValidChanged();
	_tdeio->emitRechecked();
}

//An file list is ready; now save it in _tdeio->_mailurls.
void TDEIO_Count::entries( TDEIO::Job* job, const TDEIO::UDSEntryList &list )
{
	TQStringList old_list;
	TDEIO::UDSEntryListConstIterator  it1 ;
	TDEIO::UDSEntry::ConstIterator it2 ;
	TDEIO::MetaData metadata;
	KURL kurl;
	bool isFile;

	//job should be the latest job
	if( job != _job )
		kdError() << i18n( "Got unknown job; something must be wrong..." ) << endl;

	for( TQValueListConstIterator<KKioDrop::FileInfo> it = _tdeio->_mailurls->begin(); it != _tdeio->_mailurls->end(); ++it )
		old_list.append( (*it).name );

	for ( it1 = list.begin() ; it1 != list.end() ; it1++ )
	{
		/*
		 * The list contains multiple objects. Each object could be a file.
		 * Settings about it are saved in this scope until it is added to the list.
		 */
		isFile=false;
		KKioDrop::FileInfo fileinfo;
		fileinfo.name = TQString();
		fileinfo.size = 0;

		for ( it2 = (*it1).begin() ; it2 != (*it1).end() ; it2++ )
		{
			if( (*it2).m_uds == TDEIO::UDS_FILE_TYPE &&
			    ((long)(*it2).m_long & S_IFREG ) )
				isFile=true;
			else if( (*it2).m_uds == TDEIO::UDS_URL )
				fileinfo.name = (*it2).m_str;
			else if( (*it2).m_uds == TDEIO::UDS_NAME )
			{ //The file tdeioslave doesn't return UDS_URL.
				kurl = *_kurl;
				metadata = *_metadata;
				_protocol->recheckKURL( kurl, metadata );
				kurl.setPath ( kurl.path() + '/' + (*it2).m_str );
				fileinfo.name = kurl.url();
			}
			else if( (*it2).m_uds == TDEIO::UDS_SIZE )
			{
				fileinfo.size = (*it2).m_long;
			}
		}

		//Add the entry.
		if( ! fileinfo.name.isNull() && isFile )
		{
			_new_mailurls->append( fileinfo );
			if( ! old_list.contains( fileinfo.name ) && _tdeio->passivePopup() )
				showPassive( fileinfo.name );
		}
	}
}

void TDEIO_Count::addtoPassivePopup( KornMailSubject* subject )
{
	if( ! _popup_subjects )
	{
		_popup_subjects = new SortedMailSubject;
		_popup_subjects->setAutoDelete( true );
	}

	_popup_subjects->inSort( subject );
	if( _popup_subjects->count() > 5 )
		_popup_subjects->removeFirst(); //Overhead: subject is downloaded

	_subjects_pending--;
	_total_new_messages++;
	if( _subjects_pending == 0 )
	{
		_tdeio->emitShowPassivePopup( dynamic_cast< TQPtrList<KornMailSubject>* >( _popup_subjects ), _total_new_messages );
		delete _popup_subjects; _popup_subjects = 0;
		_total_new_messages = 0;

		disconnectSlave();
	}
}

void TDEIO_Count::deleteSingleSubject( TDEIO_Single_Subject* single_subject )
{
	delete single_subject;
}

#include "tdeio_count.moc"
