/*
    This file is part of tdepim.

    Copyright (c) 2004 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "groupwisewizard.h"

#include "groupwiseconfig.h"
#include "kmailchanges.h"

#include "tderesources/groupwise/tdeabc_groupwiseprefs.h"
#include "tderesources/groupwise/tdeabc_resourcegroupwise.h"
#include "tderesources/groupwise/kcal_groupwiseprefsbase.h"
#include "tderesources/groupwise/kcal_resourcegroupwise.h"

#include <libkcal/resourcecalendar.h>
#include <libemailfunctions/email.h>

#include <klineedit.h>
#include <tdelocale.h>

#include <tqcheckbox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqspinbox.h>
#include <tqgroupbox.h>

TQString serverUrl()
{
  TQString url;
  if ( GroupwiseConfig::self()->useHttps() ) url = "https";
  else url = "http";
  url += "://" + GroupwiseConfig::self()->host() + ":" +
    TQString::number( GroupwiseConfig::self()->port() ) + GroupwiseConfig::self()->path();
  return url;
}

class CreateGroupwiseKcalResource : public TDEConfigPropagator::Change
{
  public:
    CreateGroupwiseKcalResource()
      : TDEConfigPropagator::Change( i18n("Create GroupWise Calendar Resource") )
    {
    }

    void apply()
    {
      KCal::CalendarResourceManager m( "calendar" );
      m.readConfig();

      KCal::ResourceGroupwise *r = new KCal::ResourceGroupwise();

      r->setResourceName( i18n("GroupWise") );
      r->prefs()->setUrl( serverUrl() );
      r->prefs()->setUser( GroupwiseConfig::self()->user() );
      r->prefs()->setPassword( GroupwiseConfig::self()->password() );
      r->setSavePolicy( KCal::ResourceCached::SaveDelayed );
      r->setReloadPolicy( KCal::ResourceCached::ReloadInterval );
      r->setReloadInterval( 20 );
      m.add( r );

      m.writeConfig();

      GroupwiseConfig::self()->setKcalResource( r->identifier() );
    }
};

class UpdateGroupwiseKcalResource : public TDEConfigPropagator::Change
{
  public:
    UpdateGroupwiseKcalResource()
      : TDEConfigPropagator::Change( i18n("Update GroupWise Calendar Resource") )
    {
    }

    void apply()
    {
      KCal::CalendarResourceManager m( "calendar" );
      m.readConfig();

      KCal::CalendarResourceManager::Iterator it;
      for ( it = m.begin(); it != m.end(); ++it ) {
        if ( (*it)->identifier() == GroupwiseConfig::kcalResource() ) {
          KCal::ResourceGroupwise *r = static_cast<KCal::ResourceGroupwise *>( *it );
          r->prefs()->setUrl( serverUrl() );
          r->prefs()->setUser( GroupwiseConfig::self()->user() );
          r->prefs()->setPassword( GroupwiseConfig::self()->password() );
          r->setSavePolicy( KCal::ResourceCached::SaveDelayed );
          r->setReloadPolicy( KCal::ResourceCached::ReloadInterval );
          r->setReloadInterval( 20 );
        }
      }
      m.writeConfig();
    }
};

class CreateGroupwiseTdeabcResource : public TDEConfigPropagator::Change
{
  public:
    CreateGroupwiseTdeabcResource()
      : TDEConfigPropagator::Change( i18n("Create GroupWise Addressbook Resource") )
    {
    }

    void apply()
    {
      KRES::Manager<TDEABC::Resource> m( "contact" );
      m.readConfig();

      TQString url = serverUrl();
      TQString user( GroupwiseConfig::self()->user() );
      TQString password( GroupwiseConfig::self()->password() );

      TDEABC::ResourceGroupwise *r = new TDEABC::ResourceGroupwise( url, user,
                                                                password,
                                                                TQStringList(),
                                                                TQString() );
      r->setResourceName( i18n("GroupWise") );
      m.add( r );
      m.writeConfig();

      GroupwiseConfig::self()->setTdeabcResource( r->identifier() );
    }
};

class UpdateGroupwiseTdeabcResource : public TDEConfigPropagator::Change
{
  public:
    UpdateGroupwiseTdeabcResource()
      : TDEConfigPropagator::Change( i18n("Update GroupWise Addressbook Resource") )
    {
    }

    void apply()
    {
      KRES::Manager<TDEABC::Resource> m( "contact" );
      m.readConfig();

      KRES::Manager<TDEABC::Resource>::Iterator it;
      for ( it = m.begin(); it != m.end(); ++it ) {
        if ( (*it)->identifier() == GroupwiseConfig::tdeabcResource() ) {
          TDEABC::ResourceGroupwise *r = static_cast<TDEABC::ResourceGroupwise *>( *it );
          r->prefs()->setUrl( serverUrl() );
          r->prefs()->setUser( GroupwiseConfig::self()->user() );
          r->prefs()->setPassword( GroupwiseConfig::self()->password() );
        }
      }
      m.writeConfig();
    }
};


class GroupwisePropagator : public TDEConfigPropagator
{
  public:
    GroupwisePropagator()
      : TDEConfigPropagator( GroupwiseConfig::self(), "groupwise.kcfg" )
    {
    }

    ~GroupwisePropagator()
    {
      GroupwiseConfig::self()->writeConfig();
    }

  protected:
    class Writer : public CreateDisconnectedImapAccount::CustomWriter
    {
      public:
	void writeFolder( TDEConfig &, int ) {}
	void writeIds( int accountId, int transportId )
	{
          GroupwiseConfig::setKMailAccountId( accountId );
	  GroupwiseConfig::setKMailTransportId( transportId );
	}
    };

    void addCustomChanges( Change::List &changes )
    {
      ChangeConfig *c = new ChangeConfig;
      c->file = "korganizerrc";
      c->group = "FreeBusy";
      c->name = "FreeBusyRetrieveUrl";
      c->value = "groupwise://" + GroupwiseConfig::self()->host() + GroupwiseConfig::self()->path() +
        "/freebusy/";
      changes.append( c );

      KCal::CalendarResourceManager m1( "calendar" );
      m1.readConfig();
      KCal::CalendarResourceManager::Iterator it;
      for ( it = m1.begin(); it != m1.end(); ++it ) {
        if ( (*it)->type() == "groupwise" ) break;
      }
      if ( it == m1.end() ) {
        changes.append( new CreateGroupwiseKcalResource );
      } else {
        if ( (*it)->identifier() == GroupwiseConfig::kcalResource() ) {
          KCal::GroupwisePrefsBase *prefs =
            static_cast<KCal::ResourceGroupwise *>( *it )->prefs();
          if ( prefs->url() != serverUrl() ||
               prefs->port() != GroupwiseConfig::self()->port() ||
               prefs->user() != GroupwiseConfig::user() ||
               prefs->password() != GroupwiseConfig::password() ) {
            changes.append( new UpdateGroupwiseKcalResource );
          }
        }
      }

      KRES::Manager<TDEABC::Resource> m2( "contact" );
      m2.readConfig();
      KRES::Manager<TDEABC::Resource>::Iterator it2;
      for ( it2 = m2.begin(); it2 != m2.end(); ++it2 ) {
        if ( (*it2)->type() == "groupwise" ) break;
      }
      if ( it2 == m2.end() ) {
        changes.append( new CreateGroupwiseTdeabcResource );
      } else {
        if ( (*it2)->identifier() == GroupwiseConfig::tdeabcResource() ) {
          TDEABC::GroupwisePrefs *prefs = static_cast<TDEABC::ResourceGroupwise *>( *it2 )->prefs();
          if ( prefs->url() != serverUrl() ||
               prefs->user() != GroupwiseConfig::user() ||
               prefs->password() != GroupwiseConfig::password() ) {
            changes.append( new UpdateGroupwiseTdeabcResource );
          }
        }
      }

      if ( GroupwiseConfig::createEmailAccount() ) {
        CreateDisconnectedImapAccount *ca =
          new CreateDisconnectedImapAccount( i18n("GroupWise") );

        ca->setServer( GroupwiseConfig::host() );
        ca->setUser( GroupwiseConfig::user() );
        ca->setPassword( GroupwiseConfig::password() );
        ca->setRealName( GroupwiseConfig::fullName() );
        TQString email = GroupwiseConfig::email();
        if ( !email.isEmpty() ) ca->setEmail( email );
        ca->enableSavePassword( true );
        ca->enableSieve( false );
        ca->setEncryption( CreateDisconnectedImapAccount::TLS );
        ca->setAuthenticationSend( CreateDisconnectedImapAccount::LOGIN );
        ca->setSmtpPort( 25 );

        ca->setExistingAccountId( GroupwiseConfig::kMailAccountId() );
        ca->setExistingTransportId( GroupwiseConfig::kMailTransportId() );

        ca->setCustomWriter( new Writer );

        changes.append( ca );
      }
    }
};

GroupwiseWizard::GroupwiseWizard() : TDEConfigWizard( new GroupwisePropagator )
{
  TQFrame *page = createWizardPage( i18n("Novell GroupWise") );

  TQGridLayout *topLayout = new TQGridLayout( page );
  topLayout->setSpacing( spacingHint() );

  TQLabel *label = new TQLabel( i18n("Server name:"), page );
  topLayout->addWidget( label, 0, 0 );
  mServerEdit = new KLineEdit( page );
  topLayout->addWidget( mServerEdit, 0, 1 );

  label = new TQLabel( i18n("Path to SOAP interface:"), page );
  topLayout->addWidget( label, 1, 0 );
  mPathEdit = new KLineEdit( page );
  topLayout->addWidget( mPathEdit, 1, 1 );

  label = new TQLabel( i18n("Port:"), page );
  topLayout->addWidget( label, 2, 0 );
  mPortEdit = new TQSpinBox( 1, 65536, 1, page );
  topLayout->addWidget( mPortEdit, 2, 1 );

  label = new TQLabel( i18n("User name:"), page );
  topLayout->addWidget( label, 3, 0 );
  mUserEdit = new KLineEdit( page );
  topLayout->addWidget( mUserEdit, 3, 1 );

  label = new TQLabel( i18n("Password:"), page );
  topLayout->addWidget( label, 4, 0 );
  mPasswordEdit = new KLineEdit( page );
  mPasswordEdit->setEchoMode( KLineEdit::Password );
  topLayout->addWidget( mPasswordEdit, 4, 1 );

  mSavePasswordCheck = new TQCheckBox( i18n("Save password"), page );
  topLayout->addMultiCellWidget( mSavePasswordCheck, 5, 5, 0, 1 );

  mSecureCheck = new TQCheckBox( i18n("Encrypt communication with server"),
                                page );
  topLayout->addMultiCellWidget( mSecureCheck, 5, 5, 0, 1 );

  topLayout->setRowStretch( 6, 1 );


  mEmailPage = createWizardPage( i18n("Mail") );

  topLayout = new TQGridLayout( mEmailPage );
  topLayout->setSpacing( spacingHint() );

  mEmailBox = new TQGroupBox( 1, TQt::Horizontal,
    i18n("Create Mail Account"), mEmailPage );
  mEmailBox->setCheckable( true );
  topLayout->addWidget( mEmailBox, 0, 0 );

  mEmailWidget = new TQWidget( mEmailBox );
  connect( mEmailBox, TQ_SIGNAL( toggled( bool ) ), mEmailWidget,
    TQ_SLOT( setEnabled( bool ) ) );

  TQGridLayout *accountLayout= new TQGridLayout( mEmailWidget );
  accountLayout->setSpacing( spacingHint() );

  label = new TQLabel( i18n("Email address:"), mEmailWidget );
  accountLayout->addWidget( label, 0, 0 );

  mEmailEdit = new KLineEdit( mEmailWidget );
  accountLayout->addWidget( mEmailEdit, 0, 1 );

  label = new TQLabel( i18n("Full name:"), mEmailWidget );
  accountLayout->addWidget( label, 1, 0 );

  mFullNameEdit = new KLineEdit( mEmailWidget );
  accountLayout->addWidget( mFullNameEdit, 1, 1 );

  accountLayout->setRowStretch( 2, 1 );

  connect( this, TQ_SIGNAL( aboutToShowPage( TQWidget * ) ),
    TQ_SLOT( slotAboutToShowPage( TQWidget * ) ) );


  setupRulesPage();
  setupChangesPage();

  resize( 600, 400 );
}

GroupwiseWizard::~GroupwiseWizard()
{
}

TQString GroupwiseWizard::validate()
{
  if( mServerEdit->text().isEmpty() ||
      mPathEdit->text().isEmpty() ||
      mPortEdit->text().isEmpty() ||
      mUserEdit->text().isEmpty() ||
      mPasswordEdit->text().isEmpty() )
    return i18n( "Please fill in all fields." );

  if ( mEmailBox->isChecked() ) {
    if( !KPIM::isValidSimpleEmailAddress( mEmailEdit->text() ) )
      return i18n("Invalid email address entered.");
    if( mFullNameEdit->text().isEmpty() )
      return i18n( "Please fill in all fields." );
  }

  return TQString();
}

void GroupwiseWizard::usrReadConfig()
{
  mServerEdit->setText( GroupwiseConfig::self()->host() );
  mPathEdit->setText( GroupwiseConfig::self()->path() );
  mPortEdit->setValue( GroupwiseConfig::self()->port() );
  mUserEdit->setText( GroupwiseConfig::self()->user() );
  mPasswordEdit->setText( GroupwiseConfig::self()->password() );
  mSavePasswordCheck->setChecked( GroupwiseConfig::self()->savePassword() );
  mSecureCheck->setChecked( GroupwiseConfig::self()->useHttps() );
  mEmailEdit->setText( GroupwiseConfig::self()->email() );
  mFullNameEdit->setText( GroupwiseConfig::fullName() );
  mEmailBox->setChecked( GroupwiseConfig::createEmailAccount() );
}

void GroupwiseWizard::usrWriteConfig()
{
  GroupwiseConfig::self()->setHost( mServerEdit->text() );
  GroupwiseConfig::self()->setPath( mPathEdit->text() );
  GroupwiseConfig::self()->setPort( mPortEdit->value() );
  GroupwiseConfig::self()->setUser( mUserEdit->text() );
  GroupwiseConfig::self()->setPassword( mPasswordEdit->text() );
  GroupwiseConfig::self()->setSavePassword( mSavePasswordCheck->isChecked() );
  GroupwiseConfig::self()->setUseHttps( mSecureCheck->isChecked() );
  GroupwiseConfig::setEmail( mEmailEdit->text() );
  GroupwiseConfig::setFullName( mFullNameEdit->text() );
  GroupwiseConfig::setCreateEmailAccount( mEmailBox->isChecked() );
}

void GroupwiseWizard::slotAboutToShowPage( TQWidget *page )
{
  if ( page == mEmailPage ) {
    if ( mEmailEdit->text().isEmpty() ) {
      TQString host = GroupwiseConfig::host();
      int pos = host.findRev( "." );
      if ( pos > 0 ) pos = host.findRev( ".", pos - 1 );
      if ( pos > 0 ) host = host.mid( pos + 1 );
      TQString email = GroupwiseConfig::user() + "@" + host;
      mEmailEdit->setText( email );
    }
  }
}

#include "groupwisewizard.moc"
