/*
  This file is part of the Groupware/KOrganizer integration.

  Requires the TQt and KDE widget libraries, available at no cost at
  http://www.trolltech.com and http://www.kde.org respectively

  Copyright (c) 2002-2004 Klarälvdalens Datakonsult AB
        <info@klaralvdalens-datakonsult.se>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
  MA  02110-1301, USA.

  In addition, as a special exception, the copyright holders give
  permission to link the code of this program with any edition of
  the TQt library by Trolltech AS, Norway (or with modified versions
  of TQt that use the same license as TQt), and distribute linked
  combinations including the two.  You must obey the GNU General
  Public License in all respects for all of the code used other than
  TQt.  If you modify this file, you may extend this exception to
  your version of the file, but you are not obligated to do so.  If
  you do not wish to do so, delete this exception statement from
  your version.
*/
#ifndef FREEBUSYMANAGER_H
#define FREEBUSYMANAGER_H

#include <kurl.h>
#include <libkcal/icalformat.h>
#include <libkcal/freebusycache.h>
#include <tqstring.h>
#include <tqobject.h>

namespace TDEIO { class Job; }
namespace KCal {
class Calendar;
class FreeBusy;
}
class FreeBusyManager;

/**
 * Class for downloading FreeBusy Lists
 */
class FreeBusyDownloadJob : public TQObject
{
    TQ_OBJECT
  
  public:
    FreeBusyDownloadJob( const TQString &email, const KURL &url,
                   FreeBusyManager *manager, const char *name = 0 );

    virtual ~FreeBusyDownloadJob();

  protected slots:
    void slotResult( TDEIO::Job * );
    void slotData(  TDEIO::Job *, const TQByteArray &data );

  signals:
    void freeBusyDownloaded( KCal::FreeBusy *, const TQString& email );
    void freeBusyDownloadError( const TQString& email );

  private:
    FreeBusyManager *mManager;
    TQString mEmail;

    TQCString mFreeBusyData;
};

class FreeBusyManager : public TQObject, public KCal::FreeBusyCache
{
    TQ_OBJECT
  
  public:
    FreeBusyManager( TQObject *parent, const char *name );

    void setCalendar( KCal::Calendar * );

    /// KOrganizer publishes the free/busy list
    void publishFreeBusy();

    /**
      KOrganizer downloads somebody else's free/busy list
      The call is asynchronous, and upon download, the
      receivers slot specified by member will be called.
      The slot should be of type "member(const TQString&, KCal::FreeBusy*)"

      Return true if a download is initiated, and false otherwise
    */
    bool retrieveFreeBusy( const TQString &email, bool forceDownload );

    void cancelRetrieval();

    KCal::FreeBusy *iCalToFreeBusy( const TQCString &data );

    /**
      Load freebusy information belonging to email.
    */
    KCal::FreeBusy *loadFreeBusy( const TQString &email );
    /**
      Store freebusy information belonging to email.
    */
    bool saveFreeBusy( KCal::FreeBusy *freebusy, const KCal::Person &person );
//    bool saveFreeBusy( KCal::FreeBusy *, const TQString &email );

    /**
      Return URL of freeBusy information for given email address.
    */
    KURL freeBusyUrl( const TQString &email );

    /**
      Return directory used for storing free/busy information.
    */
    TQString freeBusyDir();

    /**
      Change the broken Url status
      mBrokenUrl is used to show the 'broken url popup' only once
     */
    void setBrokenUrl( bool isBroken );

  public slots:
    // When something changed in the calendar, we get this called
    void slotPerhapsUploadFB();

  signals:
    /**
      This signal is emitted to return results of free/busy requests.
    */
    void freeBusyRetrieved( KCal::FreeBusy *, const TQString &email );

  protected:
    void timerEvent( TQTimerEvent* );

    /**
      Return free/busy list of calendar owner as iCalendar string.
    */
    TQString ownerFreeBusyAsString();

    /**
      Return free/busy list of calendar owner.
    */
    KCal::FreeBusy *ownerFreeBusy();

    /**
      Convert free/busy object to iCalendar string.
    */
    TQString freeBusyToIcal( KCal::FreeBusy * );

  protected slots:
    bool processRetrieveQueue();

  private slots:
    void slotUploadFreeBusyResult( TDEIO::Job * );
    void slotFreeBusyDownloadError( const TQString& email );

  private:
    KCal::Calendar *mCalendar;
    KCal::ICalFormat mFormat;

    TQStringList mRetrieveQueue;

    // Free/Busy uploading
    TQDateTime mNextUploadTime;
    int mTimerID;
    bool mUploadingFreeBusy;
    bool mBrokenUrl;
};

#endif
