/*
    This file is part of libkcal.

    Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>
    Copyright (C) 2005 Reinhold Kainhofer <reinhold@kainhofe.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "calendarlocal.h"

extern "C" {
#include "libical/icaltimezone.h"
}

#include <tdeaboutdata.h>
#include <tdeapplication.h>
#include <kdebug.h>
#include <tdecmdlineargs.h>

#include <tqfile.h>



using namespace KCal;


static const TDECmdLineOptions options[] =
{
  { "verbose", "Verbose output", 0 },
  { "+input", "Name of input file", 0 },
  { "[+output]", "optional name of output file for the recurrence dates", 0 },
  TDECmdLineLastOption
};


int main( int argc, char **argv )
{
  TDEAboutData aboutData( "testrecurson", "Tests all dates from 2002 to 2010 to test if the event recurs on each individual date. This is meant to test the Recurrence::recursOn method for errors.", "0.1" );
  TDECmdLineArgs::init( argc, argv, &aboutData );
  TDECmdLineArgs::addCmdLineOptions( options );

  TDEApplication app( false, false );

  TDECmdLineArgs *args = TDECmdLineArgs::parsedArgs();

  if ( args->count() < 1 ) {
    args->usage( "Wrong number of arguments." );
  }

  TQString input = TQFile::decodeName( args->arg( 0 ) );
  kdDebug(5800) << "Input file: " << input << endl;

  TQTextStream *outstream;
  outstream = 0;
  TQString fn("");
  if ( args->count() > 1 ) {
    fn = args->arg( 1 );
    kdDebug() << "We have a file name given: " << fn << endl;
  }
  TQFile outfile( fn );
  if ( !fn.isEmpty() && outfile.open( IO_WriteOnly ) ) {
    kdDebug() << "Opened output file!!!" << endl;
    outstream = new TQTextStream( &outfile );
  }

  CalendarLocal cal( TQString::fromLatin1("UTC") );

  if ( !cal.load( input ) ) return 1;
	TQString tz = cal.nonKDECustomProperty( "X-LibKCal-Testsuite-OutTZ" );
	if ( !tz.isEmpty() ) {
	  cal.setTimeZoneIdViewOnly( tz );
	}

  Incidence::List inc = cal.incidences();

  for ( Incidence::List::Iterator it = inc.begin(); it != inc.end(); ++it ) {
    Incidence *incidence = *it;
    kdDebug(5800) << "*+*+*+*+*+*+*+*+*+*" << endl;
    kdDebug(5800) << " -> " << incidence->summary() << " <- " << endl;

    incidence->recurrence()->dump();

    TQDate dt( 1996, 7, 1 );
    if ( outstream ) {
      // Output to file for testing purposes
      while ( dt.year() <= 2010 ) {
        if ( incidence->recursOn( dt ) )
          (*outstream) << dt.toString( TQt::ISODate ) << endl;
        dt = dt.addDays( 1 );
      }
    } else {
      dt = TQDate( 2005, 1, 1 );
      while ( dt.year() < 2007 ) {
        if ( incidence->recursOn( dt ) )
          kdDebug(5800) << dt.toString( TQt::ISODate ) << endl;
        dt = dt.addDays( 1 );
      }
    }
  }

  delete outstream;
  outfile.close();
  return 0;
}
