/*
    This file is part of tdepim.

    Copyright (c) 2004 Cornelius Schumacher <schumacher@kde.org>
    Copyright (c) 2004 Till Adam <adam@kde.org>
    Copyright (c) 2005 Reinhold Kainhofer <reinhold@kainhofer.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "groupwaredataadaptor.h"
#include <kdebug.h>
#include <tdeio/job.h>
#include <libemailfunctions/idmapper.h>

using namespace KPIM;

GroupwareUploadItem::GroupwareUploadItem( UploadType type ) : mItemType( KPIM::FolderLister::Unknown ), mType( type )
{
}

KURL GroupwareUploadItem::adaptNewItemUrl( GroupwareDataAdaptor *adaptor,
                                           const KURL &baseurl )
{
kdDebug()<<"GroupwareUploadItem::adaptNewItemUrl, baseurl=" << baseurl.url() << endl;
  if ( adaptor ) {
    TQString path( adaptor->defaultNewItemName( this ) );
kdDebug() << "path=" << path << endl;
    KURL u( baseurl );
    if ( path.isEmpty() ) return u;
    else {
      u.addPath( path );
kdDebug() << "Final Path for new item: " << u.url() << endl;
      return u;
    }
  } else return baseurl;
}

TDEIO::TransferJob *GroupwareUploadItem::createRawUploadJob(
                       GroupwareDataAdaptor *adaptor, const KURL &/*baseurl*/ )
{
  Q_ASSERT( adaptor );
  if ( !adaptor ) return 0;
  const TQString dta = data();
  //kdDebug(7000) << "Uploading: " << data << endl;
  KURL upUrl( url() );
  if ( adaptor )
    adaptor->adaptUploadUrl( upUrl );
  kdDebug(7000) << "Uploading to: " << upUrl.prettyURL() << endl;
  TDEIO::TransferJob *job = TDEIO::storedPut( dta.utf8(), upUrl, -1, true,
                                          false, false );
  job->addMetaData( "PropagateHttpHeader", "true" );
  if ( adaptor ) {
    job->addMetaData( "customHTTPHeader", "Content-Type: " + adaptor->mimeType() );
  }
  return job;
}

TDEIO::TransferJob *GroupwareUploadItem::createUploadNewJob(
      GroupwareDataAdaptor *adaptor, const KURL &baseurl )
{
kdDebug()<<"GroupwareUploadItem::createUploadNewJob, baseurl=" << baseurl.url() << endl;
  setUrl( adaptNewItemUrl( adaptor, baseurl ) );
  TDEIO::TransferJob *job = createRawUploadJob( adaptor, baseurl );
  if ( job ) {
    kdDebug() << "Adding If-None-Match " << endl;
    TQString header;
    if ( job->outgoingMetaData().contains("customHTTPHeader") ) {
      header = job->outgoingMetaData()["customHTTPHeader"];
      header += "\r\n";
    }
    header += "If-None-Match: *";
    job->addMetaData( "customHTTPHeader", header );
  }
  return job;
}

TDEIO::TransferJob *GroupwareUploadItem::createUploadJob(
                           GroupwareDataAdaptor *adaptor, const KURL &baseurl )
{
kdDebug()<<"GroupwareUploadItem::createUploadJob" << endl;
  TDEIO::TransferJob *job = createRawUploadJob( adaptor, baseurl );
  if ( job && adaptor ) {
  kdDebug()<<"Adding If-Match header: " << adaptor->idMapper()->fingerprint( uid() ) << endl;
    TQString header;
    if ( job->outgoingMetaData().contains("customHTTPHeader") ) {
      header = job->outgoingMetaData()["customHTTPHeader"];
      header += "\r\n";
    }
kdDebug()<<"old HEADER: " << header << endl;
    header += "If-Match: " + adaptor->idMapper()->fingerprint( uid() );
kdDebug()<<"new HEADER: " << header << endl;
    job->addMetaData( "customHTTPHeader", header );
  }
  return job;
}


GroupwareDataAdaptor::GroupwareDataAdaptor()
  : TQObject(), mFolderLister( 0 ), mIdMapper( 0 )
{
}

GroupwareDataAdaptor::~GroupwareDataAdaptor()
{
}

void GroupwareDataAdaptor::setUserPassword( KURL &url )
{
  kdDebug(5800) << "GroupwareDataAdaptor::setUserPassword, mUser="
                << mUser << endl;
  url.setUser( mUser );
  url.setPass( mPassword );
}

FolderLister::Entry::List GroupwareDataAdaptor::defaultFolders()
{
  return FolderLister::Entry::List();
}

TDEIO::TransferJob *GroupwareDataAdaptor::createUploadJob( const KURL &url,
                                                     GroupwareUploadItem *item )
{
  if ( item ) {
    TDEIO::TransferJob *job = item->createUploadJob( this, url );
    setUidForJob( job, item->uid() );
    return job;
  } else return 0;
}

TDEIO::TransferJob *GroupwareDataAdaptor::createUploadNewJob( const KURL &url,
                                                     GroupwareUploadItem *item )
{
kdDebug()<<"GroupwareDataAdaptor::createUploadNewJob, url=" << url.url() << endl;
  if ( item ) {
    TDEIO::TransferJob *job = item->createUploadNewJob( this, url );
    setUidForJob( job, item->uid() );
    return job;
  } else return 0;
}

void GroupwareDataAdaptor::processDownloadListItem( const KURL &entry,
        const TQString &newFingerprint, KPIM::FolderLister::ContentType type )
{
  bool download = false;
  const TQString &location = entry.path();

  emit itemOnServer( entry );
  // if not locally present, download
  const TQString &localId = idMapper()->localId( location );
  kdDebug(5800) << "Looking up remote: " << location
                << " found: " << localId << endl;
  if ( localId.isEmpty() || !localItemExists( localId ) ) {
    //kdDebug(7000) << "Not locally present, download: " << location << endl;
    download = true;
  } else {
    kdDebug(5800) << "Locally present " << endl;
    // locally present, let's check if it's newer than what we have
    const TQString &oldFingerprint = idMapper()->fingerprint( localId );
      if ( oldFingerprint != newFingerprint ) {
      kdDebug(5800) << "Fingerprint changed old: " << oldFingerprint <<
        " new: " << newFingerprint << endl;
      // something changed on the server, check if we also changed it locally
      if ( localItemHasChanged( localId ) ) {
        // TODO conflict resolution
        kdDebug(5800) << "TODO conflict resolution" << endl;
        download = true;
      } else {
        download = true;
      }
    } else {
      kdDebug(5800) << "Fingerprint not changed, don't download this " << endl;
    }
  }
  if ( download ) {
    emit itemToDownload( entry, type );
  }
}

bool GroupwareDataAdaptor::interpretRemoveJob( TDEIO::Job *job, const TQString &/*jobData*/ )
{
  if ( !job ) return false;
  TDEIO::DeleteJob *deljob = dynamic_cast<TDEIO::DeleteJob*>(job);
  bool error = job->error();
  const TQString err = job->errorString();

  if ( deljob ) {
    KURL::List urls( deljob->urls() );
    for ( KURL::List::Iterator it = urls.begin(); it != urls.end(); ++it ) {
      if ( error ) {
        emit itemDeletionError( *it, err );
      } else {
        // FIXME: Don't use TQString() here
        emit itemDeleted( TQString(), *it );
      }
    }
    return true;
  } else {
    return false;
  }
}


bool GroupwareDataAdaptor::interpretUploadJob( TDEIO::Job *job, const TQString &/*jobData*/ )
{
  kdDebug(7000) << "GroupwareDataAdaptor::interpretUploadJob " << endl;
  TDEIO::TransferJob *trfjob = dynamic_cast<TDEIO::TransferJob*>(job);
  bool error = job->error();
  const TQString err = job->errorString();

  if ( trfjob ) {
    KURL url( trfjob->url() );
    if ( error ) {
      emit itemUploadError( url, err );
    } else {
      // We don't know the local id here (and we don't want to extract it from
      // the idMapper, that's the task of the receiver
      emit itemUploaded( uidFromJob( job ), url );
    }
    return true;
  } else {
    return false;
  }
}

bool GroupwareDataAdaptor::interpretUploadNewJob( TDEIO::Job *job, const TQString &/*jobData*/ )
{
// TODO: How does the incidence mapper know the old/new ids???
  kdDebug(7000) << "GroupwareDataAdaptor::interpretUploadNewJob " << endl;
  TDEIO::TransferJob *trfjob = dynamic_cast<TDEIO::TransferJob*>(job);
  bool error = job->error();
  const TQString err = job->errorString();

  if ( trfjob ) {
    KURL url( trfjob->url() );
    if ( error ) {
      emit itemUploadNewError( idMapper()->localId( url.path() ), err );
    } else {
      // We don't know the local id here (and we don't want to extract it from
      // the idMapper, that's the task of the receiver
      emit itemUploadedNew( uidFromJob( job ), url );
    }
    return true;
  } else {
    return false;
  }
}

TQString GroupwareDataAdaptor::uidFromJob( TDEIO::Job *job ) const
{
kdDebug()<<"GroupwareDataAdaptor::uidFromJob( "<<job<<")"<<endl;
  if ( mJobUIDMap.contains( job ) ) {
    kdDebug()<<"  Contained: "<< mJobUIDMap[job] << endl;
    return mJobUIDMap[ job ];
  } else {
    return TQString();
  }
}

void GroupwareDataAdaptor::setUidForJob( TDEIO::Job *job, const TQString &uid )
{
  if ( uid.isEmpty() ) {
    mJobUIDMap.remove( job );
  } else {
    mJobUIDMap[ job ] = uid;
  }
}


#include "groupwaredataadaptor.moc"
