/***************************************************************************
 *   Copyright (C) 2006-2012 by Thomas Schweitzer                          *
 *   thomas-schweitzer(at)arcor.de                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2.0 as   *
 *   published by the Free Software Foundation.                            *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program in the file LICENSE.GPL; if not, write to the *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "config.h"
#include "MainWindow.h"

#include "stdlib.h"

#include "AboutDialog.h"
#include "IndentHandler.h"
#include "SettingsPaths.h"
#include "UiGuiInfoDialog.h"
#include "UiGuiSettings.h"
#include "UiGuiSettingsDialog.h"
#include "UiGuiVersion.h"
#include "ToolBarWidget.h"
#include "UiGuiHighlighter.h"

#include <tqaction.h>
#include <tqapplication.h>
#include <tqcheckbox.h>
#include <tqcursor.h>
#include <tqdragobject.h>
#include <tqfile.h>
#include <tqfiledialog.h>
#include <tqfileinfo.h>
#include <tqlabel.h>
#include <tqlocale.h>
#include <tqmessagebox.h>
#include <tqpixmap.h>
#include <tqpopupmenu.h>
#include <tqpushbutton.h>
#include <tqsplitter.h>
#include <tqstatusbar.h>
#include <tqtextcodec.h>
#include <tqtooltip.h>
#include <tqtranslator.h>
#include <tqwidget.h>

#include <tqextscintilla.h>
///-- #include <Qsci/qsciprinter.h>


/*
    \brief Main window of UniversalIndentGUI-tqt

    The MainWindow class is responsible for generating and displaying most of the gui elements.
    Its look is set in the file "mainwindow.ui". An object for the indent handler is generated here
    and user actions are being controlled. It is responsible for file open dialogs and indenter selection.
 */

/*
    \brief Constructs the main window.
 */
MainWindow::MainWindow(TQString file2OpenOnStart, TQWidget *parent) :
        MainWindowBase(parent), m_aboutDialog(nullptr), m_qSciSourceCodeEditor(nullptr),
        m_uiGuiTranslator(nullptr), m_textEditLineColumnInfoLabel(nullptr),
        m_oldLinesNumber(0), m_openEncodingActions(), m_saveEncodingActions(),
        m_encodingActionGroup(nullptr), m_saveEncodedActionGroup(nullptr), m_infoDialog(nullptr),
        m_highlighterActionGroup(nullptr), m_documentModified(false), m_previewToggled(true),
        m_indentHandler(nullptr), m_centralSplitter(nullptr), m_settingsDialog(nullptr),
        m_highlighter(nullptr), m_highlightingActions(), m_toolBarWidget(nullptr)
///--      m_textEditVScrollBar(nullptr)
{
	// Init of some variables.
	m_sourceCodeChanged = false;

	// Create the settings object, which loads all UiGui settings from a file.
	m_settings = UiGuiSettings::getInstance();

	// Initialize the language of the application.
	initApplicationLanguage();

  // Creates the main window and initializes it.
	initMainWindow();

  // Create toolbar and insert it into the main window.
	initToolBar();

	// Create the text edit component using the TQScintilla widget.
	initTextEditor();

	// Create and init the syntax highlighter.
	initSyntaxHighlighter();

	// Create and init the indenter.
	initIndenter();

	// Create some menus.
	createEncodingMenu();
	createHighlighterMenu();

	// Generate about dialog box
	m_aboutDialog = new AboutDialog(this);
	connect(actionAboutUniversalIndentGUITQt, TQ_SIGNAL(activated()), this, TQ_SLOT(showAboutDialog()));

	// Generate info dialog box
	m_infoDialog = new UiGuiInfoDialog(this, m_indentHandler);
	connect(actionShowInfo, TQ_SIGNAL(activated()), m_infoDialog, TQ_SLOT(showDialog()));

	// Generate settings dialog box
	m_settingsDialog = new UiGuiSettingsDialog(this, m_settings);
	connect(actionShowSettings, TQ_SIGNAL(activated()), m_settingsDialog, TQ_SLOT(showDialog()));

	if (TQFile::exists(file2OpenOnStart))
	{
		// If a file that should be opened on start exists, load it
		openSourceFileDialog(file2OpenOnStart);
	}
	else
	{
		// Otherwise load the last opened file, if this is enabled in the settings.
		loadLastOpenedFile();
	}

	updateSyntaxHighlighting();

	// Enable accept dropping of files.
  setAcceptDrops(true);
}

MainWindow::~MainWindow()
{
	UiGuiSettings::deleteInstance();
}

/*
    \brief Initializes the main window by creating the main gui and make some settings.
 */
void MainWindow::initMainWindow()
{
	// For icon setup
	const TQString ICONS_PATH(APP_ICONS_PATH);
	// Application icon
	setIcon(TQPixmap(ICONS_PATH + "universalIndentGUI_64x64.png"));
	// Menu icons
	// - File menu
	actionOpenSourceFile->setIconSet(TQPixmap(ICONS_PATH + "document-open.png"));
	actionMenuRecentlyOpenedFiles->setIconSet(TQPixmap(ICONS_PATH + "document-open.png"));
	actionClearRecentlyOpenedList->setIconSet(TQPixmap(ICONS_PATH + "edit-clear.png"));
	actionMenuEncoding->setIconSet(TQPixmap(ICONS_PATH + "document-open.png"));
	actionSaveSourceFile->setIconSet(TQPixmap(ICONS_PATH + "document-save.png"));
	actionSaveSourceFileAs->setIconSet(TQPixmap(ICONS_PATH + "document-save-as.png"));
	actionMenuSaveEncoded->setIconSet(TQPixmap(ICONS_PATH + "document-save-as.png"));
	actionMenuExport->setIconSet(TQPixmap(ICONS_PATH + "exporthtml.png"));
	actionExportPDF->setIconSet(TQPixmap(ICONS_PATH + "exportpdf.png"));
	actionExportHTML->setIconSet(TQPixmap(ICONS_PATH + "exporthtml.png"));
	actionShowInfo->setIconSet(TQPixmap(ICONS_PATH + "info.png"));
	actionExit->setIconSet(TQPixmap(ICONS_PATH + "system-log-out.png"));
	// - Indenter menu
	actionLoadIndenterConfigFile->setIconSet(TQPixmap(ICONS_PATH + "load_indent_cfg.png"));
	actionSaveIndenterConfigFile->setIconSet(TQPixmap(ICONS_PATH + "save_indent_cfg.png"));
	actionCreateShellScript->setIconSet(TQPixmap(ICONS_PATH + "shell.png"));
	actionResetIndenterParameters->setIconSet(TQPixmap(ICONS_PATH + "view-refresh.png"));
	// - Setting menu
	actionLiveIndentPreview->setIconSet(TQPixmap(ICONS_PATH + "live-preview.png"));
	actionEnableSyntaxHighlighting->setIconSet(TQPixmap(ICONS_PATH + "syntax-highlight.png"));
	actionIndenterParameterTooltipsEnabled->setIconSet(TQPixmap(ICONS_PATH + "tooltip.png"));
	actionShowSettings->setIconSet(TQPixmap(ICONS_PATH + "preferences-system.png"));
	// - Help menu
	actionAboutUniversalIndentGUITQt->setIconSet(TQPixmap(ICONS_PATH + "info.png"));
	// Menu ids
	m_actionClearRecentlyOpenedListId = popupMenuRecentlyOpenedFiles->idAt(
			    popupMenuRecentlyOpenedFiles->count() - 1);

	// Central splitter
	m_centralSplitter = new TQSplitter(this);
	m_centralSplitter->setChildrenCollapsible(true);
	setCentralWidget(m_centralSplitter);

	// Handle last opened window size
	// ------------------------------
	bool    maximized = m_settings->getValueByName("WindowIsMaximized").toBool();
	TQPoint pos       = m_settings->getValueByName("WindowPosition").toPoint();
	TQSize  size      = m_settings->getValueByName("WindowSize").toSize();
	resize(size);
	move(pos);
	if (maximized)
	{
		showMaximized();
	}

	// Get last selected file encoding
	// -------------------------------
	m_currentEncoding = m_settings->getValueByName("FileEncoding").toString();

	// Register the syntax highlighting setting in the menu to the settings object.
	connect(actionEnableSyntaxHighlighting, TQ_SIGNAL(toggled(bool)),
	        m_settings, TQ_SLOT(handleValueChangeFromExtern(bool)));
	connect(m_settings, TQ_SIGNAL(syntaxHighlightingEnabled(bool)),
	        actionEnableSyntaxHighlighting, TQ_SLOT(setOn(bool)));
	actionEnableSyntaxHighlighting->setOn(
	        m_settings->getValueByName("SyntaxHighlightingEnabled").toBool());
	// Tell the highlighter if it has to be enabled or disabled.
	connect(m_settings, TQ_SIGNAL(syntaxHighlightingEnabled(bool)), this, TQ_SLOT(turnHighlightOnOff(bool)));

	// Register the load last file setting in the menu to the settings object
	connect(actionLoadLastOpenedFileOnStartup, TQ_SIGNAL(toggled(bool)),
	        m_settings, TQ_SLOT(handleValueChangeFromExtern(bool)));
	connect(m_settings, TQ_SIGNAL(loadLastOpenedFileOnStartup(bool)),
	        actionLoadLastOpenedFileOnStartup, TQ_SLOT(setOn(bool)));
	actionLoadLastOpenedFileOnStartup->setOn(
	        m_settings->getValueByName("LoadLastOpenedFileOnStartup").toBool());

	// Register the white space setting in the menu to the settings object.
	connect(actionWhiteSpaceIsVisible, TQ_SIGNAL(toggled(bool)),
	        m_settings, TQ_SLOT(handleValueChangeFromExtern(bool)));
	connect(m_settings, TQ_SIGNAL(whiteSpaceIsVisible(bool)),
	        actionWhiteSpaceIsVisible, TQ_SLOT(setOn(bool)));
	actionWhiteSpaceIsVisible->setOn(m_settings->getValueByName("WhiteSpaceIsVisible").toBool());
	// Tell the TQScintilla editor if it has to show white space.
	connect(m_settings, TQ_SIGNAL(whiteSpaceIsVisible(bool)), this, TQ_SLOT(setWhiteSpaceVisibility(bool)));

	// Register the indenter parameter tooltip setting in the menu to the settings object
	connect(actionIndenterParameterTooltipsEnabled, TQ_SIGNAL(toggled(bool)),
	        m_settings, TQ_SLOT(handleValueChangeFromExtern(bool)));
	connect(m_settings, TQ_SIGNAL(indenterParameterTooltipsEnabled(bool)),
	        actionIndenterParameterTooltipsEnabled, TQ_SLOT(setOn(bool)));
	actionIndenterParameterTooltipsEnabled->setOn(
	        m_settings->getValueByName("IndenterParameterTooltipsEnabled").toBool());

	// Connect the remaining menu items.
	connect(actionOpenSourceFile,   TQ_SIGNAL(activated()), this, TQ_SLOT(openSourceFileDialog()));
	connect(actionSaveSourceFile,   TQ_SIGNAL(activated()), this, TQ_SLOT(saveSourceFile()));
	connect(actionSaveSourceFileAs, TQ_SIGNAL(activated()), this, TQ_SLOT(saveasSourceFileDialog()));
	connect(actionExportPDF, TQ_SIGNAL(activated()), this, TQ_SLOT(exportToPDF()));
	connect(actionExportHTML, TQ_SIGNAL(activated()), this, TQ_SLOT(exportToHTML()));
	connect(m_settings, TQ_SIGNAL(indenterParameterTooltipsEnabled(bool)),
	        this, TQ_SLOT(setIndenterParameterTooltips(bool)));

	// Init the menu for selecting one of the recently opened files.
	initRecentlyOpenedList();
	updateRecentlyOpenedList();
	connect(popupMenuRecentlyOpenedFiles, TQ_SIGNAL(highlighted(int)),
	        this, TQ_SLOT(recentlyOpenedFileHighlighted(int)));
	connect(popupMenuRecentlyOpenedFiles, TQ_SIGNAL(activated(int)),
	        this, TQ_SLOT(openFileFromRecentlyOpenedList(int)));
	connect(m_settings, TQ_SIGNAL(recentlyOpenedListSize(int)), this, TQ_SLOT(updateRecentlyOpenedList()));

	updateWindowTitle();
}

/*
    \brief Creates and inits the tool bar. It is added to the main window.
 */
void MainWindow::initToolBar()
{
	// For icon setup
	const TQString ICONS_PATH(APP_ICONS_PATH);

	// Create the tool bar and add it to the main window.
	m_toolBarWidget = new ToolBarWidget(toolBar);

	// Connect the tool bar widgets to their functions.
	m_toolBarWidget->cbEnableSyntaxHL->hide();

	m_toolBarWidget->pbOpenSourceFile->setIconSet(TQPixmap(ICONS_PATH + "document-open.png"));
	connect(m_toolBarWidget->pbOpenSourceFile, TQ_SIGNAL(clicked()), this, TQ_SLOT(openSourceFileDialog()));

	m_toolBarWidget->pbAbout->setIconSet(TQPixmap(ICONS_PATH + "info.png"));
	connect(m_toolBarWidget->pbAbout, TQ_SIGNAL(clicked()), this, TQ_SLOT(showAboutDialog()));

	m_toolBarWidget->pbExit->setIconSet(TQPixmap(ICONS_PATH + "system-log-out.png"));
	connect(m_toolBarWidget->pbExit, TQ_SIGNAL(clicked()), this, TQ_SLOT(close()));

	// Settings a pixmap hides the text in TQt3
	//m_toolBarWidget->cbLivePreview->setPixmap(TQPixmap(ICONS_PATH + "live-preview.png"));
	connect(m_toolBarWidget->cbLivePreview, TQ_SIGNAL(toggled(bool)), this, TQ_SLOT(previewTurnedOnOff(bool)));
	connect(m_toolBarWidget->cbLivePreview, TQ_SIGNAL(toggled(bool)),
	        actionLiveIndentPreview, TQ_SLOT(setOn(bool)));
}

/*
    \brief Create and initialize the text editor component. It uses the TQScintilla widget.
 */
void MainWindow::initTextEditor()
{
	// Create the TQScintilla widget and add it to the layout.
	try
	{
		m_qSciSourceCodeEditor = new TQextScintilla(m_centralSplitter);
	}
	catch (...)
	{
		TQMessageBox::critical(this, "Error creating TQScintilla text editor component!",
		        "While trying to create the text editor component (based on TQScintilla), an error "
		        "occurred. Please make sure that TQScintilla is installed and that release and debug "
		        "versions are not mixed.");
		exit(1);
	}

	// Make some settings for the TQScintilla widget.
	m_qSciSourceCodeEditor->setUtf8(true);
	m_qSciSourceCodeEditor->setMarginLineNumbers(1, true);
	m_qSciSourceCodeEditor->setMarginWidth(1, TQString("10000"));
	m_qSciSourceCodeEditor->setBraceMatching(m_qSciSourceCodeEditor->SloppyBraceMatch);
	m_qSciSourceCodeEditor->setMatchedBraceForegroundColor(TQColor("red"));
	m_qSciSourceCodeEditor->setFolding(TQextScintilla::BoxedTreeFoldStyle);
	m_qSciSourceCodeEditor->setAutoCompletionSource(TQextScintilla::AcsAll);
	m_qSciSourceCodeEditor->setAutoCompletionThreshold(3);

	// Handle if white space is set to be visible
	bool whiteSpaceIsVisible = m_settings->getValueByName("WhiteSpaceIsVisible").toBool();
	setWhiteSpaceVisibility(whiteSpaceIsVisible);

	// Handle the width of tabs in spaces
	int tabWidth = m_settings->getValueByName("TabWidth").toInt();
	m_qSciSourceCodeEditor->setTabWidth(tabWidth);
	connect(m_settings, TQ_SIGNAL(tabWidth(int)), m_qSciSourceCodeEditor, TQ_SLOT(setTabWidth(int)));

	// TODO not available in TQScintilla 1.71
	// Remember a pointer to the scrollbar of the TQScintilla widget used to keep
	// on the same line as before when turning preview on/off.
	//m_textEditVScrollBar = m_qSciSourceCodeEditor->verticalScrollBar();

	// Add a column row indicator to the status bar.
	m_textEditLineColumnInfoLabel = new TQLabel(tr("Line %1, Column %2").arg(1).arg(1), this);
	statusBar()->addWidget(m_textEditLineColumnInfoLabel, 0, true);
	connect(m_qSciSourceCodeEditor, TQ_SIGNAL(cursorPositionChanged(int,int)),
	        this, TQ_SLOT(setStatusBarCursorPosInfo(int, int)));

	// Connect the text editor to dependent functions.
	connect(m_qSciSourceCodeEditor, TQ_SIGNAL(textChanged()), this, TQ_SLOT(sourceCodeChangedHelperSlot()));
	// TODO signal 'linesChanged' is not available in TQScintilla 1.71.
	// Use textChanged for now and check for line number changes in the slot
	//connect(m_qSciSourceCodeEditor, TQ_SIGNAL(linesChanged()), this, TQ_SLOT(numberOfLinesChanged()));
	connect(m_qSciSourceCodeEditor, TQ_SIGNAL(textChanged()), this, TQ_SLOT(numberOfLinesChanged()));
	numberOfLinesChanged();
}

void MainWindow::updateSyntaxHighlighting()
{
  turnHighlightOnOff(actionEnableSyntaxHighlighting->isOn());
}

/*
    \brief Create and init the syntax highlighter and set it to use the TQScintilla edit component.
 */
void MainWindow::initSyntaxHighlighter()
{
	// Create the highlighter.
	m_highlighter = new UiGuiHighlighter(m_qSciSourceCodeEditor);

	// Handle if syntax highlighting is enabled
	bool syntaxHighlightningEnabled = m_settings->getValueByName("SyntaxHighlightningEnabled").toBool();
	turnHighlightOnOff(syntaxHighlightningEnabled);
}

/*
    \brief Initializes the language of UniversalIndentGUI.

    If the program language is defined in the m_settings, the corresponding language
    file will be loaded and set for the application. If not set there, the system
    default language will be set, if a translation file for that language exists.
    Returns true, if the translation file could be loaded. Otherwise it returns
    false and uses the default language, which is English.
 */
bool MainWindow::initApplicationLanguage()
{
	TQString languageShort;

	// Get the language settings from the settings object.
	int languageIndex = m_settings->getValueByName("Language").toInt();

	// If no language was set, indicated by a negative index, use the system language.
	if (languageIndex < 0)
	{
		languageShort = TQLocale::system().name();

		// Chinese and Japanese language consist of country and language code.
		// For all others the language code will be cut off.
		if (languageShort.left(2) != "zh" && languageShort.left(2) != "ja")
		{
			languageShort = languageShort.left(2);
		}

		// If no translation file for the systems local language exist, fall back to English.
		if (m_settings->getAvailableTranslations().findIndex(languageShort) < 0)
		{
			languageShort = "en";
		}

		// Set the language setting to the new language.
		m_settings->setValueByName("Language",
		        m_settings->getAvailableTranslations().findIndex(languageShort));
	}
	// If a language was defined in the m_settings, get this language mnemonic.
	else
	{
		languageShort = m_settings->getAvailableTranslations()[languageIndex];
	}

	// Load the TQt own translation file and set it for the application.
	// Load the uigui translation file and set it for the application.
	m_uiGuiTranslator     = new TQTranslator();
	bool translationFileLoaded = m_uiGuiTranslator->load(SettingsPaths::getGlobalFilesPath() +
	        "/translations/universalindent_" + languageShort + ".qm");
	if (translationFileLoaded)
	{
		tqApp->installTranslator(m_uiGuiTranslator);
	}

	connect(m_settings, TQ_SIGNAL(language(int)), this, TQ_SLOT(languageChanged(int)) );

	return translationFileLoaded;
}

/*
    \brief Creates and initializes the indenter.
 */
void MainWindow::initIndenter()
{
	// Get Id of last selected indenter.
	m_currentIndenterID = m_settings->getValueByName("SelectedIndenter").toInt();

	// Create the indenter widget with the ID and add it to the layout.
	m_indentHandler = new IndentHandler(m_currentIndenterID, this, m_centralSplitter);
	m_centralSplitter->moveToFirst(m_indentHandler);

	// If settings for the indenter have changed, let the main window know aboud it.
	connect(m_indentHandler, TQ_SIGNAL(indenterSettingsChanged()), this,
	        TQ_SLOT(indentSettingsChangedSlot()));

	// Set this true, so the indenter is called at first program start
	m_indentSettingsChanged = true;
	m_previewToggled = true;
}

/*
    \brief Tries to load the by \a filePath defined file and returns its content as TQString.

    If the file could not be loaded a error dialog will be shown.
 */
TQString MainWindow::loadFile(const TQString &filePath)
{
	TQFile   inSrcFile(filePath);
	TQString fileContent = "";

	if (!inSrcFile.open(IO_ReadOnly | IO_Translate))
	{
		TQMessageBox::warning(nullptr, tr("Error opening file"),
		        tr("Cannot read the file ") + "\"" + filePath + "\".");
	}
	else
	{
		TQTextStream inSrcStream(&inSrcFile);
		TQApplication::setOverrideCursor(TQt::WaitCursor);
		inSrcStream.setCodec(TQTextCodec::codecForName(m_currentEncoding.ascii()));
		fileContent = inSrcStream.read();
		TQApplication::restoreOverrideCursor();
		inSrcFile.close();

		TQFileInfo fileInfo(filePath);
		m_currentSourceFileExtension = fileInfo.extension(false);
		int indexOfHighlighter = m_highlighter->setLexer(m_currentSourceFileExtension);
		// Mark the selected highlighter in the corresponding menu
		int idx = 0;
		for (TQAction *action : m_highlightingActions)
		{
			if (idx == indexOfHighlighter)
			{
				action->setOn(true);
				break; 
			}
			++idx;
		}
	}
	return fileContent;
}

/*
    \brief Calls the source file open dialog to load a source file for the formatting preview.
    If the file was successfully loaded, the indenter will be called to generate the formatted source code.
 */
void MainWindow::openSourceFileDialog(const TQString &fileName)
{
	// If the source code file is changed and the shown dialog for saving the file
	// is canceled, also stop opening another source file.
	if (!maybeSave())
	{
		return;
	}
	TQString openedSourceFileContent = "";
	TQString fileExtensions = tr("Supported by indenter") + " (" +
	        m_indentHandler->getPossibleIndenterFileExtensions() + ");;" + tr("All files") + " (*.*)";

  TQString fileToOpen = fileName;
	if (fileToOpen.isEmpty())
	{
		fileToOpen = TQFileDialog::getOpenFileName(m_currentSourceFile, fileExtensions, this, nullptr,
		        tr("Choose source code file"));
	}

	if (!fileToOpen.isEmpty())
	{
		m_currentSourceFile = fileToOpen;
		TQFileInfo fileInfo(fileToOpen);
		m_currentSourceFileExtension = fileInfo.extension(false);

		openedSourceFileContent = loadFile(fileToOpen);
		m_sourceFileContent      = openedSourceFileContent;
		if (m_toolBarWidget->cbLivePreview->isChecked())
		{
			callIndenter();
		}
		m_sourceCodeChanged = true;
		m_previewToggled    = true;
		updateSourceView();
		updateWindowTitle();
		updateRecentlyOpenedList();
///-- 		_textEditLastScrollPos = 0;
///-- 		m_textEditVScrollBar->setValue(_textEditLastScrollPos);

		m_savedSourceContent = openedSourceFileContent;
		m_qSciSourceCodeEditor->setModified(false);
		m_documentModified = false;
	}
}

/*
    \brief Calls the source file save as dialog to save a source file under a chosen name.
    If the file already exists and it should be overwritten, a warning is shown before.
 */
bool MainWindow::saveasSourceFileDialog(TQAction *chosenEncodingAction)
{
	TQString fileExtensions = tr("Supported by indenter") + " (" +
	        m_indentHandler->getPossibleIndenterFileExtensions() + ");;" + tr("All files") + " (*.*)";

	TQString fileName	= TQFileDialog::getSaveFileName(m_currentSourceFile, fileExtensions, this, nullptr,
	        tr("Save source code file"));

	// Saving has been canceled if the filename is empty
	if (fileName.isEmpty())
	{
		return false;
	}

	m_savedSourceContent = m_qSciSourceCodeEditor->text();

	m_currentSourceFile = fileName;
	TQFile::remove(fileName);
	TQFile outSrcFile(fileName);
	outSrcFile.open(IO_ReadWrite | IO_Translate);

	// Get current encoding.
	TQString encoding;
	if (chosenEncodingAction)
	{
		encoding = chosenEncodingAction->text();
	}
	else
	{
		encoding = m_currentEncoding;
	}
	TQTextStream outSrcStrm(&outSrcFile);
	outSrcStrm.setCodec(TQTextCodec::codecForName(encoding.ascii()));
	outSrcStrm << m_savedSourceContent;
	outSrcFile.close();

	TQFileInfo fileInfo(fileName);
	m_currentSourceFileExtension = fileInfo.extension(false);

	m_qSciSourceCodeEditor->setModified(false);
	m_documentModified = false;

	updateRecentlyOpenedList();
	updateWindowTitle();
	return true;
}

/*
    \brief Saves the currently shown source code to the last save or opened source file.
    If no source file has been opened, because only the static example has been loaded,
    the save as file dialog will be shown.
 */
bool MainWindow::saveSourceFile()
{
	if (m_currentSourceFile.isEmpty())
	{
		return saveasSourceFileDialog();
	}
	else
	{
		TQFile::remove(m_currentSourceFile);
		TQFile outSrcFile(m_currentSourceFile);
		m_savedSourceContent = m_qSciSourceCodeEditor->text();
		outSrcFile.open(IO_ReadWrite | IO_Translate);

		// Get current encoding.
		TQTextStream outSrcStrm(&outSrcFile);
		outSrcStrm.setCodec(TQTextCodec::codecForName(m_currentEncoding.ascii()));
		outSrcStrm << m_savedSourceContent;
		outSrcFile.close();

		m_qSciSourceCodeEditor->setModified(false);
		m_documentModified = false;
	}
	updateWindowTitle();
	return true;
}

/*
    \brief Updates the displaying of the source code.

    Updates the text edit field, which is showing the loaded, and if preview is enabled formatted, source code.
    Reassigns the line numbers and in case of switch between preview and none preview keeps the text field
    at the same line number.
 */
void MainWindow::updateSourceView()
{
///-- 	_textEditLastScrollPos = m_textEditVScrollBar->value();

	if (m_toolBarWidget->cbLivePreview->isOn())
	{
		m_sourceViewContent = m_sourceFormattedContent;
	}
	else
	{
		m_sourceViewContent = m_sourceFileContent;
	}

	if (m_previewToggled)
	{
		disconnect(m_qSciSourceCodeEditor, TQ_SIGNAL(textChanged()),
		        this, TQ_SLOT(sourceCodeChangedHelperSlot()));
		m_qSciSourceCodeEditor->setText(m_sourceViewContent);
		m_previewToggled = false;
		connect(m_qSciSourceCodeEditor, TQ_SIGNAL(textChanged()),
		        this, TQ_SLOT(sourceCodeChangedHelperSlot()));
	}

///-- 	m_textEditVScrollBar->setValue(_textEditLastScrollPos);
}

/*
    \brief Calls the selected indenter with the currently loaded source code to retrieve the formatted source code.

    The original loaded source code file will not be changed.
 */
void MainWindow::callIndenter()
{
	TQApplication::setOverrideCursor(TQt::WaitCursor);
	m_sourceFormattedContent = m_indentHandler->callIndenter(m_sourceFileContent,
	        m_currentSourceFileExtension);
	updateSourceView();
	TQApplication::restoreOverrideCursor();
}

/*
    \brief Switches the syntax highlighting corresponding to the value \a turnOn either on or off.
 */
void MainWindow::turnHighlightOnOff(bool turnOn)
{
	if (turnOn)
	{
		m_highlighter->turnHighlightOn();
	}
	else
	{
		m_highlighter->turnHighlightOff();
	}
	m_previewToggled = true;
	updateSourceView();
}

/*
    \brief Added this slot to avoid multiple calls because of changed text.
 */
void MainWindow::sourceCodeChangedHelperSlot()
{
	TQTimer::singleShot(0, this, TQ_SLOT(sourceCodeChangedSlot()));
}

/*
    \brief Is emitted whenever the text inside the source view window changes. Calls the indenter
    to format the changed source code.
 */
void MainWindow::sourceCodeChangedSlot()
{
 	TQChar   enteredCharacter;
 	int      cursorPos, cursorPosAbsolut, cursorLine;
 	TQString text;
 
 	m_sourceCodeChanged = true;
 
 	// Get the content text of the text editor.
 	m_sourceFileContent = m_qSciSourceCodeEditor->text();
 
 	// Get the position of the cursor in the unindented text.
 	if (m_sourceFileContent.isEmpty())
 	{
 		// Add this line feed, because AStyle has problems with a totally empty file.
 		m_sourceFileContent += "\n";
 		cursorPosAbsolut    = 0;
 		cursorPos        = 0;
 		cursorLine       = 0;
 		enteredCharacter = m_sourceFileContent.at(cursorPosAbsolut);
 	}
 	else
 	{
 		m_qSciSourceCodeEditor->getCursorPosition(&cursorLine, &cursorPos);
 		cursorPosAbsolut = m_qSciSourceCodeEditor->SendScintilla(TQextScintillaBase::SCI_GETCURRENTPOS);
 		text = m_qSciSourceCodeEditor->text(cursorLine);
 		if (cursorPosAbsolut > 0)
 		{
 			cursorPosAbsolut--;
 		}
 		if (cursorPos > 0)
 		{
 			cursorPos--;
 		}
		// TODO: TQScintilla 1.7 returns the position as byte number within a C string.
		// In case of UTF-8 characters, the index could increase by more than one per
		// character. Therefore change the text to a C string and access the local
		// 8 bit representation to get the character that was entered.
 		enteredCharacter = m_qSciSourceCodeEditor->text().local8Bit().at(cursorPosAbsolut);
 	}
 
 	// Call the indenter to reformat the text.
 	if (m_toolBarWidget->cbLivePreview->isChecked())
 	{
 		callIndenter();
 		m_previewToggled = true;
 	}
 
 	// Update the text editor.
 	updateSourceView();
 
 	if (m_toolBarWidget->cbLivePreview->isChecked() && !enteredCharacter.isNull() &&
 	        enteredCharacter != TQChar(10))
 	{
 		int saveCursorLine = cursorLine;
 		int saveCursorPos  = cursorPos;
 
 		bool charFound = false;
 
 		// Search forward
 		for (cursorLine = saveCursorLine;
 		        cursorLine - saveCursorLine < 6 && cursorLine < m_qSciSourceCodeEditor->lines();
 		        cursorLine++)
 		{
 			text = m_qSciSourceCodeEditor->text(cursorLine);
 			while (cursorPos < text.length() && enteredCharacter != text.at(cursorPos))
 			{
 				cursorPos++;
 			}
 			if (cursorPos >= text.length())
 			{
 				cursorPos = 0;
 			}
 			else
 			{
 				charFound = true;
 				break;
 			}
 		}
 
 		// If forward search did not find the character, search backward
 		if (!charFound)
 		{
 			text      = m_qSciSourceCodeEditor->text(saveCursorLine);
 			cursorPos = saveCursorPos;
 			if (cursorPos >= text.length())
 			{
 				cursorPos = text.length() - 1;
 			}
 
 			for (cursorLine = saveCursorLine; saveCursorLine - cursorLine < 6 && cursorLine >= 0;
 			        cursorLine--)
 			{
 				text = m_qSciSourceCodeEditor->text(cursorLine);
 				while (cursorPos >= 0 && enteredCharacter != text.at(cursorPos))
 				{
 					cursorPos--;
 				}
 				if (cursorPos < 0)
 				{
 					cursorPos = m_qSciSourceCodeEditor->lineLength(cursorLine - 1) - 1;
 				}
 				else
 				{
 					charFound = true;
 					break;
 				}
 			}
 		}
 
 		// If the character was found set its new cursor position...
 		if (charFound)
 		{
 			m_qSciSourceCodeEditor->setCursorPosition(cursorLine, cursorPos + 1);
 		}
 		// ...if it was not found, set the previous cursor position.
 		else
 		{
 			m_qSciSourceCodeEditor->setCursorPosition(saveCursorLine, saveCursorPos + 1);
 		}
 	}
 	// set the previous cursor position.
 	else if (enteredCharacter == TQChar(10))
 	{
 		m_qSciSourceCodeEditor->setCursorPosition(cursorLine, cursorPos);
 	}
 
 	if (m_toolBarWidget->cbLivePreview->isChecked())
 	{
 		m_sourceCodeChanged = false;
 	}
 
 	if (m_savedSourceContent == m_qSciSourceCodeEditor->text())
 	{
 		m_qSciSourceCodeEditor->setModified(false);
 		m_documentModified = false;
 	}
 	else
 	{
 		m_qSciSourceCodeEditor->setModified(true); // Has no effect according to TQScintilla docs.
 		m_documentModified = true;
 	}
}

/*
    \brief This slot is called whenever one of the indenter settings are changed.

    It calls the selected indenter if the preview is turned on. If preview
    is not active, set a flag to indicate that the settings have changed.
 */
void MainWindow::indentSettingsChangedSlot()
{
	m_indentSettingsChanged = true;

	int cursorLine, cursorPos;
	m_qSciSourceCodeEditor->getCursorPosition(&cursorLine, &cursorPos);

	if (m_toolBarWidget->cbLivePreview->isChecked())
	{
		callIndenter();
		m_previewToggled = true;

		updateSourceView();
		if (m_sourceCodeChanged)
		{
			m_sourceCodeChanged = false;
		}
		m_indentSettingsChanged = false;
	}
	else
	{
		updateSourceView();
	}

	if (m_savedSourceContent == m_qSciSourceCodeEditor->text())
	{
		m_qSciSourceCodeEditor->setModified(false);
		m_documentModified = false;
	}
	else
	{
		m_qSciSourceCodeEditor->setModified(true); // Has no effect according to TQScintilla docs.
		m_documentModified = true;
	}
}

/*
    \brief This slot is called whenever the preview button is turned on or off.

    It calls the selected indenter to format the current source code if
    the code has been changed since the last indenter call.
 */
void MainWindow::previewTurnedOnOff(bool turnOn)
{
	m_previewToggled = true;

	int cursorLine, cursorPos;
	m_qSciSourceCodeEditor->getCursorPosition(&cursorLine, &cursorPos);

	if (turnOn && (m_indentSettingsChanged || m_sourceCodeChanged))
	{
		callIndenter();
	}
	updateSourceView();
	if (m_sourceCodeChanged)
	{
		m_sourceCodeChanged = false;
	}
	m_indentSettingsChanged = false;

	if (m_savedSourceContent == m_qSciSourceCodeEditor->text())
	{
		m_qSciSourceCodeEditor->setModified(false);
		m_documentModified = false;
	}
	else
	{
		m_qSciSourceCodeEditor->setModified(true);
		m_documentModified = true;
	}
}

/*
    \brief This slot updates the main window title to show the currently opened
    source code filename.
 */
void MainWindow::updateWindowTitle()
{
	// TQt3 does not support the 'modified [*]' property present in Qt4's QWidget
	setCaption("UniversalIndentGUI (TQt) " + TQString(PROGRAM_VERSION_STRING) +
	        //" [*] " + m_currentSourceFile);
	        m_currentSourceFile);
}

/*
    \brief Opens a dialog to save the current source code as a PDF document.
 */
void MainWindow::exportToPDF()
{
	// Not supported in TQt3
	TQMessageBox::warning(this, tr("Not supported"),
	        tr("Export to PDF is currently not supported."));

///-- 	TQString fileExtensions = tr("PDF Document") + " (*.pdf)";
///-- 
///-- 	TQString   fileName = m_currentSourceFile;
///-- 	TQFileInfo fileInfo(fileName);
///-- 	TQString   fileExtension = fileInfo.suffix();
///-- 
///-- 	fileName.replace(fileName.length() - fileExtension.length(), fileExtension.length(), "pdf");
///-- 	fileName = TQFileDialog::getSaveFileName(this, tr(
///-- 						"Export source code file"), fileName, fileExtensions);
///-- 
///-- 	if (!fileName.isEmpty())
///-- 	{
///-- 		QsciPrinter printer(TQPrinter::HighResolution);
///-- 		printer.setOutputFormat(TQPrinter::PdfFormat);
///-- 		printer.setOutputFileName(fileName);
///-- 		printer.printRange(m_qSciSourceCodeEditor);
///-- 	}
}

/*
    \brief Opens a dialog to save the current source code as a HTML document.
 */
void MainWindow::exportToHTML()
{
	// Not supported in TQt3
	TQMessageBox::warning(this, tr("Not supported"),
	        tr("Export to HTML is currently not supported."));

///-- 	TQString fileExtensions = tr("HTML Document") + " (*.html)";
///-- 
///-- 	TQString   fileName = m_currentSourceFile;
///-- 	TQFileInfo fileInfo(fileName);
///-- 	TQString   fileExtension = fileInfo.suffix();
///-- 
///-- 	fileName.replace(fileName.length() - fileExtension.length(), fileExtension.length(), "html");
///-- 	fileName = TQFileDialog::getSaveFileName(this, tr(
///-- 						"Export source code file"), fileName, fileExtensions);
///-- 
///-- 	if (!fileName.isEmpty())
///-- 	{
///-- 		// Create a document from which HTML code can be generated.
///-- 		TQTextDocument sourceCodeDocument(m_qSciSourceCodeEditor->text());
///-- 		sourceCodeDocument.setDefaultFont(TQFont("Courier", 12, TQFont::Normal));
///-- 		TQString sourceCodeAsHTML = sourceCodeDocument.toHtml();
///-- 		// To ensure that empty lines are kept in the HTML code make this replacement.
///-- 		sourceCodeAsHTML.replace("\"></p>", "\"><br /></p>");
///-- 
///-- 		// Write the HTML file.
///-- 		TQFile::remove(fileName);
///-- 		TQFile outSrcFile(fileName);
///-- 		outSrcFile.open(TQFile::ReadWrite | TQFile::Text);
///-- 		outSrcFile.write(sourceCodeAsHTML.toAscii());
///-- 		outSrcFile.close();
///-- 	}
}

/*
    \brief Loads the last opened file if this option is enabled in the settings.

    If the file does not exist, the default example file is tried to be loaded.
		If even that fails, a very small code example is shown.
    If the setting for opening the last file is disabled, the editor is empty on startup.
*/
void MainWindow::loadLastOpenedFile()
{
	// Get setting for last opened source code file.
	m_loadLastSourceCodeFileOnStartup =
	        m_settings->getValueByName("LoadLastOpenedFileOnStartup").toBool();

	// Only load last source code file if set to do so
	if (m_loadLastSourceCodeFileOnStartup)
	{
		// From the list of last opened files get the first one.
		TQString sourceFiles = m_settings->getValueByName("LastOpenedFiles").toString();
		m_currentSourceFile = TQStringList::split("|", sourceFiles).first();

		if (TQFile::exists(m_currentSourceFile))
		{
			// If source file exist load it.
			TQFileInfo fileInfo(m_currentSourceFile);
			m_currentSourceFile = fileInfo.absFilePath();
			m_sourceFileContent = loadFile(m_currentSourceFile);
		}
		else if (TQFile::exists(SettingsPaths::getIndenterPath() + "/example.cpp"))
		{
			// If the last opened source code file does not exist, try to load the default example.cpp file.
			TQFileInfo fileInfo(SettingsPaths::getIndenterPath() + "/example.cpp");
			m_currentSourceFile = fileInfo.absFilePath();
			m_sourceFileContent = loadFile(m_currentSourceFile);
		}
		else
		{
			// If neither the example source code file exists show some small code example.
			m_currentSourceFile = "untitled.cpp";
			m_currentSourceFileExtension = "cpp";
			m_sourceFileContent = "if(x==\"y\"){x=z;}";
		}
	}
	else
	{
		// if last opened source file should not be loaded make some default settings.
		m_currentSourceFile = "untitled.cpp";
		m_currentSourceFileExtension = "cpp";
		m_sourceFileContent = "";
	}
	m_savedSourceContent = m_sourceFileContent;

	// Update the mainwindow title to show the name of the loaded source code file.
	updateWindowTitle();
}

/*
    \brief Saves the m_settings for the main application to the file "UniversalIndentGUI.ini".

    Settings are for example last selected indenter, last loaded config file and so on.
*/
void MainWindow::saveSettings()
{
	m_settings->setValueByName("VersionInSettingsFile", PROGRAM_VERSION_STRING);
	m_settings->setValueByName("WindowIsMaximized", isMaximized());
	if (!isMaximized())
	{
		m_settings->setValueByName("WindowPosition", pos());
		m_settings->setValueByName("WindowSize", size());
	}
	m_settings->setValueByName("FileEncoding", m_currentEncoding);

	m_settings->setValueByName("LastOpenedFiles", m_recentlyOpenedList.join("|"));

///-- 	m_settings->setValueByName("MainWindowState", saveState());

	// Also save the syntax highlight style for all lexers.
	m_highlighter->writeCurrentSettings();
}

/*
    \brief Called when the program exits. Calls the saveSettings function before really quits.
*/
void MainWindow::closeEvent(TQCloseEvent *event)
{
	if (maybeSave())
	{
		saveSettings();
		event->accept();
	}
	else
	{
		event->ignore();
	}
}

/*
    \brief Is called at application exit and asks whether to save the source code file,
		if it has been changed.
 */
bool MainWindow::maybeSave()
{
	// TQt3 does not support the 'isWindowModified()' method present in Qt4's QWidget
///-- 	if (isWindowModified())
///-- 	{
///-- 		int ret = TQMessageBox::warning(this, tr("Modified code"), tr(
///-- 							"The source code has been modified.\nDo you want to save your changes?"), TQMessageBox::Yes | TQMessageBox::Default, TQMessageBox::No,
///-- 		        TQMessageBox::Cancel | TQMessageBox::Escape);
///-- 		if (ret == TQMessageBox::Yes)
///-- 		{
///-- 			return saveSourceFile();
///-- 		}
///-- 		else if (ret == TQMessageBox::Cancel)
///-- 		{
///-- 			return false;
///-- 		}
///-- 	}
	return true;
}

/*
    \brief This slot is called whenever a language is selected in the menu. It tries to find the
    corresponding action in the languageInfoList and sets the language.
 */
void MainWindow::languageChanged(int languageIndex)
{
	if (languageIndex >= 0 && languageIndex < m_settings->getAvailableTranslations().size())
	{
		// Get the mnemonic of the new selected language.
		TQString languageShort = m_settings->getAvailableTranslations()[languageIndex];

		// Remove the old uigui translation.
		tqApp->removeTranslator(m_uiGuiTranslator);

		// Load the uigui translation file and set it for the application.
		bool translationFileLoaded = m_uiGuiTranslator->load(SettingsPaths::getGlobalFilesPath() +
		        "/translations/universalindent_" + languageShort + ".qm");
		if (translationFileLoaded)
		{
			tqApp->installTranslator(m_uiGuiTranslator);
		}
	}
}

/*
    \brief Creates menu entries in the file menu for opening and saving a file with different encodings.
*/
void MainWindow::createEncodingMenu()
{
	TQStringList encodingsList;
	encodingsList << "UTF-8" << "ISO-10646-UCS-2" << "Apple Roman" << "Big5" << "Big5-HKSCS" << "CP 874" <<
	        "EUC-JP" << "EUC-KR" << "GB18030-0" << "IBM 850" << "IBM 866" << "ISO 2022-JP" <<
	        "ISO 8859-1" << "ISO 8859-13" << "Iscii-Bng" << "KOI8-R" << "KOI8-U" << "MuleLao-1" <<
	        "ROMAN8" << "Shift-JIS" << "TIS-620" << "TSCII" << "Windows-1250" << "WINSAMI2";

	m_encodingActionGroup    = new TQActionGroup(this);
	m_saveEncodedActionGroup = new TQActionGroup(this);

	// Loop for each available encoding
	for (const TQString &encodingName : encodingsList)
	{
		// Create actions for the "reopen" menu
		TQAction *encodingAction = new TQAction(m_encodingActionGroup);
		encodingAction->setText(encodingName);
		encodingAction->setStatusTip(tr(
		        "Reopen the currently opened source code file by using the text encoding scheme ") +
		        encodingName);
		encodingAction->setToggleAction(true);
		if (encodingName == m_currentEncoding)
		{
			encodingAction->setOn(true);
		}
		m_openEncodingActions.append(encodingAction);

		// Create actions for the "save as encoded" menu
		encodingAction = new TQAction(m_saveEncodedActionGroup);
		encodingAction->setText(encodingName);
		encodingAction->setStatusTip(tr(
		        "Save the currently opened source code file by using the text encoding scheme ") +
		        encodingName);
		encodingAction->setToggleAction(true);
		m_saveEncodingActions.append(encodingAction);
	}

	m_encodingActionGroup->addTo(popupMenuEncoding);
	connect(m_encodingActionGroup, TQ_SIGNAL(selected(TQAction*)),
	        this, TQ_SLOT(encodingChanged(TQAction*)));

	m_saveEncodedActionGroup->addTo(popupMenuSaveEncoded);
	connect(m_saveEncodedActionGroup, TQ_SIGNAL(selected(TQAction*)),
	        this, TQ_SLOT(saveAsOtherEncoding(TQAction*)));
}

/*
    \brief This slot calls the save dialog to save the current source file with another encoding.

    If the saving is successful and not aborted, the currently used encoding, visible in the
    "reopen" menu, is also changed to the new encoding.
*/
void MainWindow::saveAsOtherEncoding(TQAction *chosenEncodingAction)
{
	bool fileWasSaved = saveasSourceFileDialog(chosenEncodingAction);

	// If the file was save with another encoding, change the selected encoding in the reopen menu.
	if (fileWasSaved)
	{
		for (TQAction *action : m_openEncodingActions)
		{
			if (action->text() == chosenEncodingAction->text())
			{
				action->setOn(true);
				return;
			}
		}
	}
}

/*
    \brief This slot is called whenever an encoding is selected in the settings menu.
*/
void MainWindow::encodingChanged(TQAction *encodingAction)
{
	m_currentEncoding = encodingAction ? encodingAction->text() : TQString::null;
	if (maybeSave())
	{
		TQFile   inSrcFile(m_currentSourceFile);
		TQString fileContent = "";

		if (!inSrcFile.open(IO_ReadOnly | IO_Translate))
		{
			TQMessageBox::warning(NULL, tr("Error opening file"), tr(
			        "Cannot read the file ") + "\"" + m_currentSourceFile + "\".");
		}
		else
		{
			TQTextStream inSrcStrm(&inSrcFile);
			TQApplication::setOverrideCursor(TQt::WaitCursor);
			TQString encodingName = encodingAction->text();
			m_currentEncoding = encodingName;
			inSrcStrm.setCodec(TQTextCodec::codecForName(encodingName.ascii()));
			fileContent = inSrcStrm.read();
			TQApplication::restoreOverrideCursor();
			inSrcFile.close();
			m_qSciSourceCodeEditor->setText(fileContent);
			m_qSciSourceCodeEditor->setModified(false);
		}
	}
}

/*
    \brief Creates a menu entry under the settings menu for all available text encodings.
*/
void MainWindow::createHighlighterMenu()
{
	m_highlighterActionGroup = new TQActionGroup(this);

	// Loop for each known highlighter
	for (const TQString &highlighterName : m_highlighter->getAvailableHighlighters())
	{
		TQAction *highlighterAction = new TQAction(m_highlighterActionGroup);
		highlighterAction->setText(highlighterName);
		highlighterAction->setStatusTip(tr("Set the syntax highlighting to ") + highlighterName);
		highlighterAction->setToggleAction(true);
		m_highlightingActions.append(highlighterAction);
	}

	m_highlighterActionGroup->addTo(popupMenuHighlighter);
	connect(m_highlighterActionGroup , TQ_SIGNAL(selected(TQAction*)),
	        m_highlighter, TQ_SLOT(setHighlighterByAction(TQAction*)));
}

/*
    \brief Is called whenever the white space visibility is being changed in the menu.
 */
void MainWindow::setWhiteSpaceVisibility(bool visible)
{
	if (m_qSciSourceCodeEditor)
	{
		if (visible)
		{
			m_qSciSourceCodeEditor->setWhitespaceVisibility(TQextScintilla::WsVisible);
		}
		else
		{
			m_qSciSourceCodeEditor->setWhitespaceVisibility(TQextScintilla::WsInvisible);
		}
	}
}

/*
    \brief Is called whenever the tooltip visibility is being changed in the settings.
 */
void MainWindow::setIndenterParameterTooltips(bool enabled)
{
  TQToolTip::setGloballyEnabled(enabled);
}

/*
    \brief This slot is called whenever the number of lines in the editor changes
    and adapts the margin for the displayed line numbers.
*/
void MainWindow::numberOfLinesChanged()
{
	int lines = m_qSciSourceCodeEditor->lines();
	if (lines != m_oldLinesNumber)
	{
		m_oldLinesNumber = lines;
		TQString lineNumbers;
		lineNumbers.setNum(lines * 10);
		m_qSciSourceCodeEditor->setMarginWidth(1, lineNumbers);
	}
}

///-- /*
///--     \brief Catches language change events and retranslates all needed widgets.
///--  */
///-- void MainWindow::changeEvent(TQEvent *event)
///-- {
///-- 	int i = 0;
///-- 
///-- 	if (event->type() == TQEvent::LanguageChange)
///-- 	{
///-- 		TQString languageName;
///-- 
///-- 		// Translate the main window.
///-- 		_mainWindowForm->retranslateUi(this);
///-- 		updateWindowTitle();
///-- 
///-- 		// Translate the toolbar.
///-- 		_toolBarWidget->retranslateUi(_mainWindowForm->toolBar);
///-- 
///-- 		// Translate the load encoding menu.
///-- 		TQList<TQAction*> encodingActionList = m_encodingActionGroup->actions();
///-- 		for (i = 0; i < encodingActionList.size(); i++)
///-- 		{
///-- 			encodingActionList.at(i)->setStatusTip(tr(
///-- 								"Reopen the currently opened source code file by using the text encoding scheme ") + m_encodingsList.at(
///-- 								i));
///-- 		}
///-- 
///-- 		// Translate the save encoding menu.
///-- 		encodingActionList = m_saveEncodedActionGroup->actions();
///-- 		for (i = 0; i < encodingActionList.size(); i++)
///-- 		{
///-- 			encodingActionList.at(i)->setStatusTip(tr(
///-- 								"Save the currently opened source code file by using the text encoding scheme ") + m_encodingsList.at(
///-- 								i));
///-- 		}
///-- 
///-- 		// Translate the _highlighter menu.
///-- 		TQList<TQAction*> actionList = _mainWindowForm->popupMenuHighlighter->actions();
///-- 		i = 0;
///-- 		foreach(TQString highlighterName, _highlighter->getAvailableHighlighters())
///-- 		{
///-- 			TQAction *highlighterAction = actionList.at(i);
///-- 			highlighterAction->setStatusTip(tr("Set the syntax highlighting to ") + highlighterName);
///-- 			i++;
///-- 		}
///-- 
///-- 		// Translate the line and column indicators in the statusbar.
///-- 		int line, column;
///-- 		m_qSciSourceCodeEditor->getCursorPosition(&line, &column);
///-- 		setStatusBarCursorPosInfo(line, column);
///-- 	}
///-- 	else
///-- 	{
///-- 		TQWidget::changeEvent(event);
///-- 	}
///-- }

/*
    \brief Initialize the list of recently opened files and
		the menu using the saved settings.
 */
void MainWindow::initRecentlyOpenedList()
{
	m_recentlyOpenedListMaxSize = m_settings->getValueByName("RecentlyOpenedListSize").toInt();
	TQString lastOpenedFilesString = m_settings->getValueByName("LastOpenedFiles").toString();
	TQStringList recentlyOpenedList = TQStringList::split("|", lastOpenedFilesString);

	// Append only existing files and as long as the max limit has not been reached
	int index = 0;
	for (const TQString &file : recentlyOpenedList)
	{
		TQFileInfo fileInfo(file);
		if (fileInfo.exists())
		{
			if (index < m_recentlyOpenedListMaxSize)
			{
				m_recentlyOpenedList.append(file);
				int id = popupMenuRecentlyOpenedFiles->insertItem(fileInfo.fileName(), -1, index);
				popupMenuRecentlyOpenedFiles->setWhatsThis(id, file);
				++index;
			}
			else
			{
				break;
			}
		}
	}

	// Enable or disable "actionClearRecentlyOpenedList"
	actionClearRecentlyOpenedList->setEnabled(!m_recentlyOpenedList.isEmpty());
}

/*
    \brief Updates the list of recently opened files.
    The currently open file is set at the top of the list.
    Files in excess of maximum list length will be discarded.
    The new list will be updated into the m_settings object and
    the recently opened menu will be updated too.
 */
void MainWindow::updateRecentlyOpenedList()
{
	// Make sure to read the currnt max size settings
	m_recentlyOpenedListMaxSize = m_settings->getValueByName("RecentlyOpenedListSize").toInt();

	TQString fileName;
	TQString filePath;

	// Check if the currently open file is in the list of recently opened.
	int currentFileIndex = m_recentlyOpenedList.findIndex(m_currentSourceFile);
	if (!m_currentSourceFile.isEmpty())
	{
		// If it is in the list of recently opened files and not at the first position,
		// move it to the first pos.
		if (currentFileIndex > 0)
		{
			TQStringList::iterator currentFileIt = m_recentlyOpenedList.find(m_currentSourceFile);
			m_recentlyOpenedList.remove(currentFileIt);
			m_recentlyOpenedList.prepend(m_currentSourceFile);
			popupMenuRecentlyOpenedFiles->removeItemAt(currentFileIndex);
			int id = popupMenuRecentlyOpenedFiles->insertItem(TQFileInfo(m_currentSourceFile).fileName(),
			        -1, 0);
			popupMenuRecentlyOpenedFiles->setWhatsThis(id, m_currentSourceFile);
		}
		// Put the current file at the first position if it was not in the list
		else if (currentFileIndex == -1)
		{
			m_recentlyOpenedList.prepend(m_currentSourceFile);
			int id = popupMenuRecentlyOpenedFiles->insertItem(TQFileInfo(m_currentSourceFile).fileName(),
			        -1, 0);
			popupMenuRecentlyOpenedFiles->setWhatsThis(id, m_currentSourceFile);
		}
	}

	// Trim excessive elements if the size has shrinked
	while (m_recentlyOpenedList.size() > m_recentlyOpenedListMaxSize)
	{
		m_recentlyOpenedList.pop_back();
		popupMenuRecentlyOpenedFiles->removeItemAt(m_recentlyOpenedListMaxSize);
	}

	// Enable or disable "actionClearRecentlyOpenedList"
	actionClearRecentlyOpenedList->setEnabled(!m_recentlyOpenedList.isEmpty());
}

/*
    \brief This slot empties the list of recently opened files.
 */
void MainWindow::clearRecentlyOpenedList()
{
	int numItems = m_recentlyOpenedList.size();
	m_recentlyOpenedList.clear();
	for (int i = 0; i < numItems; ++i)
	{
		popupMenuRecentlyOpenedFiles->removeItemAt(0);
	}

	// Disable "actionClearRecentlyOpenedList"
	actionClearRecentlyOpenedList->setEnabled(false);
}

/*
    \brief When a recent file action is highlighted, set the full path
		in the status bar
 */
void MainWindow::recentlyOpenedFileHighlighted(int recentlyOpenedActionId)
{
	statusBar()->message(popupMenuRecentlyOpenedFiles->whatsThis(recentlyOpenedActionId));
}

/*
    \brief This slot is called if an entry from the list of recently opened files is
    being selected.
 */
void MainWindow::openFileFromRecentlyOpenedList(int recentlyOpenedActionId)
{
	// If the selected action from the recently opened list menu is the clear action
	// call the slot to clear the list and then leave.
	if (recentlyOpenedActionId == m_actionClearRecentlyOpenedListId)
	{
		clearRecentlyOpenedList();
		return;
	}

	TQString   filePath = popupMenuRecentlyOpenedFiles->whatsThis(recentlyOpenedActionId);
	TQFileInfo fileInfo(filePath);

	if (fileInfo.exists())
	{
		// If the file exists, open it.
		openSourceFileDialog(filePath);
	}
	else
	{
		// If it does not exist, show a warning message and update the list of recently opened files.
		TQMessageBox::warning(this, tr("File no longer exists"),
		        tr("The file %1 no longer exists.").arg(filePath));
		int fileIndex = m_recentlyOpenedList.findIndex(filePath);
		if (fileIndex >= 0)
		{
			TQStringList::iterator fileIt = m_recentlyOpenedList.find(filePath);
			m_recentlyOpenedList.remove(fileIt);
			popupMenuRecentlyOpenedFiles->removeItemAt(fileIndex);

			// Enable or disable "actionClearRecentlyOpenedList"
			actionClearRecentlyOpenedList->setEnabled(!m_recentlyOpenedList.isEmpty());
		}
	}
}

/*
    \brief If the dragged in object contains urls/paths to a file, accept the drag.
 */
void MainWindow::dragEnterEvent(TQDragEnterEvent *event)
{
  event->accept(TQUriDrag::canDecode(event));
}

/*
    \brief If the dropped in object contains urls/paths to a file, open that file.
 */
void MainWindow::dropEvent(TQDropEvent *event)
{
	TQStringList droppedFileNames;
	if (TQUriDrag::decodeLocalFiles(event, droppedFileNames))
	{
		openSourceFileDialog(droppedFileNames[0]);
	}
}

/*
    \brief Show the About dialog.
*/
void MainWindow::showAboutDialog()
{
	//TQPixmap originalPixmap = TQPixmap::grabWindow(TQApplication::desktop()->screen()->winId());
	//tqDebug("in main pixmap width %d, numScreens = %d", originalPixmap.size().width(),
	// TQApplication::desktop()->availableGeometry().width());
	//_aboutDialogGraphicsView->setScreenshotPixmap( originalPixmap );
	//----_aboutDialogGraphicsView->show();
	m_aboutDialog->show();
}

/*
    \brief Sets the label in the status bar to show the \a line and \a column number.
*/
void MainWindow::setStatusBarCursorPosInfo(int line, int column)
{
	m_textEditLineColumnInfoLabel->setText(tr("Line %1, Column %2").arg(line + 1).arg(column + 1));
}

#include "MainWindow.moc"
