/***************************************************************************
*   Copyright (C) 2002 by Roberto Raggi                                   *
*   roberto@kdevelop.org                                                 *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
***************************************************************************/

#ifndef BACKGROUNDPARSER_H
#define BACKGROUNDPARSER_H

#include "ast.h"
#include "driver.h"

#include <tqthread.h>
#include <tqwaitcondition.h>
#include <tqmutex.h>
#include <tqmap.h>
#include <kdebug.h>

class CppSupportPart;
class TranslationUnitAST;
class SynchronizedFileList;
class BackgroundKDevDriver;
class Unit
{
public:
	Unit() : translationUnit( 0 )
	{}
	~Unit()
	{
		translationUnit = 0;
	}

	TQString fileName;
	TQValueList<Problem> problems;
	ParsedFilePointer translationUnit;

protected:
	Unit( const Unit& source );
	void operator = ( const Unit& source );
};

class BackgroundParser: public TQThread
{
public:
	BackgroundParser( CppSupportPart*, TQWaitCondition* consumed );
	virtual ~BackgroundParser();

	TQMutex& mutex()
	{
		return m_mutex;
	}
	void lock ()
	{
		m_mutex.lock();
	}
	void unlock()
	{
		m_mutex.unlock();
	}

	TQWaitCondition& canParse()
	{
		return m_canParse;
	}
	TQWaitCondition& isEmpty()
	{
		return m_isEmpty;
	}

	bool filesInQueue();

    int countInQueue( const TQString& file ) const;

    void addFile( const TQString& fileName, bool readFromDisk = false );
    void addFileFront( const TQString& fileName, bool readFromDisk = false );
    void removeFile( const TQString& fileName );
	void removeAllFiles();

    bool hasTranslationUnit( const TQString& fileName );
	ParsedFilePointer translationUnit( const TQString& fileName );
	TQValueList<Problem> problems( const TQString& fileName , bool readFromDisk = false, bool forceParse = false );
	void updateParserConfiguration();

    ///Should be run on a regular basis(every X minutes). It reduces some caches etc. BackgroundParser must be locked before.
    void saveMemory();
  
	void close();

	virtual void run();

protected:
    friend class BackgroundKDevDriver;
    void fileParsed( ParsedFile& fileName );
	Unit* findUnit( const TQString& fileName );
	void parseFile( const TQString& fileName, bool readFromDisk, bool lock = false )
		;

private:
	class KDevDriver* m_driver;
	TQString m_currentFile;
	TQWaitCondition m_canParse;
	TQWaitCondition m_isEmpty;
	TQWaitCondition* m_consumed;
	TQMutex m_mutex;
	SynchronizedFileList* m_fileList;
	CppSupportPart* m_cppSupport;
	bool m_close;
	TQMap<TQString, Unit*> m_unitDict;
    bool m_saveMemory; //used to prevent blocking

    //State of parseFile(..):
    bool m_readFromDisk;
};

#endif 
