/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Reginald Stadlbauer <reggie@kde.org>
   Copyright (C) 2005 Thorsten Zachmann <zachmann@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <tdeapplication.h>
#include <KoUnitWidgets.h>
#include <tdelocale.h>
#include <tdeconfig.h>
#include <kdialogbase.h>
#include <kiconloader.h>
#include <knuminput.h>
#include <kcolorbutton.h>
#include "KPrVariableCollection.h"
#include "KPrCanvas.h"
#include <tkcoloractions.h>

#include <KoSpeaker.h>

#include <tqgroupbox.h>
#include <tqvgroupbox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqvbox.h>
#include <tqcheckbox.h>
#include <tqcombobox.h>
#include <tqtabwidget.h>
#include <tqwhatsthis.h>

#include "KPrPenStyleWidget.h"
#include "KPrBrushProperty.h"
#include "KPrPieProperty.h"
#include "KPrRectProperty.h"
#include "KPrPolygonProperty.h"

#include "KPrConfig.h"
#include "KPrView.h"
#include "KPrDocument.h"
#include "KPrPage.h"
#include <KoUnit.h>

#include <float.h>
#include <knumvalidator.h>
#include <tqlineedit.h>
#include "KPrCommand.h"
#include <tqvgroupbox.h>
#include <tdefontdialog.h>
#include <klineedit.h>
#include <KoRect.h>
#include <tdemessagebox.h>
#include <tdeversion.h>
#include <kurlrequesterdlg.h>
#include <tdelistview.h>
#include <tdefiledialog.h>
#include <KoEditPath.h>

#include <tdespell2/configwidget.h>
#include <tdespell2/settings.h>
#include <tdespell2/broker.h>
using namespace KSpell2;

KPrConfig::KPrConfig( KPrView* parent )
    : KDialogBase(KDialogBase::IconList,i18n("Configure KPresenter") ,
                  KDialogBase::Ok | KDialogBase::Apply | KDialogBase::Cancel| KDialogBase::Default,
                  KDialogBase::Ok, parent)

{
    m_doc = parent->kPresenterDoc();
    TQVBox *page = addVBoxPage( i18n("Interface"), i18n("Interface"),
                               BarIcon("misc", TDEIcon::SizeMedium) );
    _interfacePage=new KPrConfigureInterfacePage( parent, page );
    page = addVBoxPage( i18n("Color"), i18n("Color"),
                        BarIcon("colorize", TDEIcon::SizeMedium) );
    _colorBackground = new KPrConfigureColorBackground( parent, page );

    page = addVBoxPage( i18n("Spelling"), i18n("Spellchecker Behavior"),
                        BarIcon("tools-check-spelling", TDEIcon::SizeMedium) );
    _spellPage=new KPrConfigureSpellPage(parent, page);

    page = addVBoxPage( i18n("Misc"), i18n("Misc"),
                        BarIcon("misc", TDEIcon::SizeMedium) );
    _miscPage=new KPrConfigureMiscPage(parent, page);

    page = addVBoxPage( i18n("Document"), i18n("Document Settings"),
                        BarIcon("application-x-kpresenter", TDEIcon::SizeMedium) );

    _defaultDocPage=new KPrConfigureDefaultDocPage(parent, page);

    page = addVBoxPage( i18n("Tools"), i18n("Default Tools Settings"),
                        BarIcon("configure", TDEIcon::SizeMedium) );

    _toolsPage=new KPrConfigureToolsPage(parent, page);

    page = addVBoxPage( i18n("Paths"), i18n("Path Settings"),
                        BarIcon("path") );

    m_pathPage=new KPrConfigurePathPage(parent, page);

    if (KoSpeaker::isKttsdInstalled()) {
        page = addVBoxPage( i18n("Abbreviation for Text-to-Speech", "TTS"), i18n("Text-to-Speech Settings"),
                            BarIcon("access", TDEIcon::SizeMedium) );
        m_ttsPage=new KPrConfigureTTSPage(parent, page);
    } else m_ttsPage = 0;

    connect( this, TQ_SIGNAL( okClicked() ),this, TQ_SLOT( slotApply() ) );
}

void KPrConfig::openPage(int flags)
{
    if(flags & KP_INTERFACE)
        showPage( 0 );
    else if(flags & KP_COLOR)
        showPage(1 );
    else if(flags & KP_TDESPELL)
        showPage(2);
    else if(flags & KP_MISC)
        showPage(3 );
    else if(flags & KP_DOC)
        showPage(4 );
    else if(flags & KP_TOOLS)
        showPage(5);
    else if(flags & KP_PATH)
        showPage(6);
}

void KPrConfig::slotApply()
{
    KMacroCommand *macro = 0L;
    _interfacePage->apply();
    _colorBackground->apply();
    if (_spellPage) _spellPage->apply();
    m_pathPage->apply();
    KCommand *cmd = _miscPage->apply();
    if ( cmd )
    {
        if ( !macro )
            macro = new KMacroCommand(i18n("Change Config") );
        macro->addCommand( cmd );
    }
    cmd = _defaultDocPage->apply();
    if ( cmd )
    {
        if ( !macro )
            macro = new KMacroCommand(i18n("Change Config") );

        macro->addCommand( cmd );
    }

    _toolsPage->apply();
    if (m_ttsPage) m_ttsPage->apply();
    if ( macro )
        m_doc->addCommand( macro);
}

void KPrConfig::slotDefault()
{
    switch( activePageIndex() ) {
    case 0:
        _interfacePage->slotDefault();
        break;
    case 1:
        _colorBackground->slotDefault();
        break;
    case 2:
        if (_spellPage) _spellPage->slotDefault();
        break;
    case 3:
        _miscPage->slotDefault();
        break;
    case 4:
        _defaultDocPage->slotDefault();
        break;
    case 5:
        _toolsPage->slotDefault();
        break;
    case 6:
        m_pathPage->slotDefault();
        break;
    case 7:
        m_ttsPage->slotDefault();
    default:
        break;
    }
}

KPrConfigureInterfacePage::KPrConfigureInterfacePage( KPrView *_view, TQWidget *parent , char *name )
    :TQWidget ( parent,name )
{
    TQVBoxLayout *box = new TQVBoxLayout( this, 0, 0 );

    m_pView=_view;
    config = KPrFactory::global()->config();

    KoUnit::Unit unit = m_pView->kPresenterDoc()->unit();

    oldNbRecentFiles=10;
    double ptIndent = MM_TO_POINT(10.0);
    bool bShowRuler=true;
    bool oldShowStatusBar = true;

    if( config->hasGroup("Interface") ) {
        config->setGroup( "Interface" );
        oldNbRecentFiles=config->readNumEntry("NbRecentFile",oldNbRecentFiles);
        ptIndent = config->readDoubleNumEntry("Indent", ptIndent);
        bShowRuler=config->readBoolEntry("Rulers",true);
        oldShowStatusBar = config->readBoolEntry( "ShowStatusBar" , true );

    }

    showRuler= new TQCheckBox(i18n("Show rulers"),this);
    TQWhatsThis::add(showRuler, i18n( "When checked, both vertical and horizontal rulers are shown on the KPresenter slide (this is the default). When unchecked, the rulers are not shown on any slide." ) );
    showRuler->setChecked(bShowRuler);
    box->addWidget(showRuler);

    showStatusBar = new TQCheckBox(i18n("Show status bar"),this);
    TQWhatsThis::add(showStatusBar, i18n( "Toggle the statusbar, which is shown by default." ) );
    showStatusBar->setChecked(oldShowStatusBar);
    box->addWidget(showStatusBar);


    recentFiles=new KIntNumInput( oldNbRecentFiles, this);
    recentFiles->setRange(1, 20, 1);
    recentFiles->setLabel(i18n("Number of recent files:"));
    TQWhatsThis::add(recentFiles, i18n( "Set the number of recent files which will be opened using the File->Open Recent menu. Default is to remember 10 filenames. The maximum you can set is 20 and the minimum is 1." ) );

    box->addWidget(recentFiles);

    TQString suffix = KoUnit::unitName( unit ).prepend(' ');
    indent = new KDoubleNumInput( this );
    indent->setValue( KoUnit::toUserValue( ptIndent, unit ) );
    indent->setRange(KoUnit::toUserValue( 0.1, unit ), KoUnit::toUserValue( 50, unit ), KoUnit::toUserValue( 0.1, unit ));


    indent->setSuffix( suffix );
    indent->setLabel(i18n("Text indentation depth:"));
    TQWhatsThis::add(indent, i18n( "This setting is used by Increase Depth and Decrease Depth menu items (in the Text menu) to change the indentation depth. The Default is 1 centimeter." ) );

    box->addWidget(indent);

    TQSpacerItem* spacer = new TQSpacerItem( 20, 20, TQSizePolicy::Minimum, TQSizePolicy::Expanding );
    box->addItem( spacer);
}

void KPrConfigureInterfacePage::apply()
{
    bool ruler=showRuler->isChecked();
    bool statusBar=showStatusBar->isChecked();

    KPrDocument * doc = m_pView->kPresenterDoc();

    config->setGroup( "Interface" );

    double newIndent = KoUnit::fromUserValue( indent->value(), doc->unit() );
    if( newIndent != doc->getIndentValue() )
    {
        config->writeEntry( "Indent", newIndent, true, false, 'g', DBL_DIG /* 6 is not enough */ );
        doc->setIndentValue( newIndent );
    }
    int nbRecent=recentFiles->value();
    if(nbRecent!=oldNbRecentFiles)
    {
        config->writeEntry( "NbRecentFile", nbRecent);
        m_pView->changeNbOfRecentFiles(nbRecent);
        oldNbRecentFiles=nbRecent;
    }
    bool refreshGUI=false;
    if(ruler != doc->showRuler())
    {
        config->writeEntry( "Rulers", ruler );
        doc->setShowRuler( ruler );
        refreshGUI=true;

    }
    if( statusBar != doc->showStatusBar() )
    {
        config->writeEntry( "ShowStatusBar", statusBar );
        doc->setShowStatusBar( statusBar );
        refreshGUI=true;
    }

    if( refreshGUI )
        doc->reorganizeGUI();

}

void KPrConfigureInterfacePage::slotDefault()
{
    double newIndent = KoUnit::toUserValue( MM_TO_POINT( 10 ), m_pView->kPresenterDoc()->unit() );
    indent->setValue( newIndent );
    recentFiles->setValue(10);
    showRuler->setChecked(true);
    showStatusBar->setChecked(true);
}

KPrConfigureColorBackground::KPrConfigureColorBackground( KPrView* _view, TQWidget *parent , char *name )
    :TQWidget ( parent,name )
{
    m_pView = _view;
    config = KPrFactory::global()->config();

    oldBgColor = m_pView->kPresenterDoc()->txtBackCol();
    oldGridColor = m_pView->kPresenterDoc()->gridColor();
    TQVBoxLayout *box = new TQVBoxLayout( this, 0, 0 );

    TQLabel *lab = new TQLabel( this, "label20" );
    lab->setText( i18n( "Background object color:" ) );
    TQWhatsThis::add(lab, i18n( "Change the background color of the text box. The background is white by default. If you have a dark background color and you want to put some white text on it, you can change the color of the text box so that you can see what you are typing. When you have finished, the area around the text will revert to the background color. The Defaults button restores the original settings." ) );
    box->addWidget(lab);

    bgColor = new KColorButton( oldBgColor,
                                oldBgColor,
                                this );
    bgColor->setColor( oldBgColor );
    box->addWidget(bgColor);


    lab = new TQLabel( this, "label20" );
    lab->setText( i18n( "Grid color:" ) );
    TQWhatsThis::add(lab, i18n( "Here you can change the grid color, which is black by default." ) );
    box->addWidget(lab);

    gridColor = new KColorButton( oldGridColor,
                                  oldGridColor,
                                  this );
    box->addWidget(gridColor);
    TQSpacerItem* spacer = new TQSpacerItem( 20, 20, TQSizePolicy::Minimum, TQSizePolicy::Expanding );
    box->addItem( spacer);

}

void KPrConfigureColorBackground::apply()
{
    KPrDocument * doc = m_pView->kPresenterDoc();
    bool repaintNeeded = false;
    TQColor _col = bgColor->color();
    if( oldBgColor != _col ) {
        config->setGroup( "KPresenter Color" );
        config->writeEntry( "BackgroundColor", _col );
        doc->setTxtBackCol( _col );
        doc->replaceObjs();
        oldBgColor=_col;
        repaintNeeded = true;
    }
    _col = gridColor->color();
    if( oldGridColor != _col ) {
        config->setGroup( "KPresenter Color" );
        config->writeEntry( "GridColor", _col );
        doc->repaint( false );
        doc->setGridColor( _col );
        oldGridColor=_col;
        repaintNeeded = true;
    }
    if (repaintNeeded)
        doc->repaint( false );
}

void KPrConfigureColorBackground::slotDefault()
{
    bgColor->setColor( TQt::white );
    gridColor->setColor( TQt::black );
}



KPrConfigureSpellPage::KPrConfigureSpellPage( KPrView *_view, TQWidget *parent, char *name )
    : TQWidget( parent, name )
{
    m_pView=_view;
    config = KPrFactory::global()->config();
    m_spellConfigWidget = new ConfigWidget( _view->broker(), parent );
    m_spellConfigWidget->setBackgroundCheckingButtonShown( true );
}

void KPrConfigureSpellPage::apply()
{

    KPrDocument* doc = m_pView->kPresenterDoc();
    m_spellConfigWidget->save();


    m_pView->kPresenterDoc()->setSpellCheckIgnoreList(
        m_pView->broker()->settings()->currentIgnoreList() );
    //FIXME reactivate just if there are changes.
    doc->enableBackgroundSpellCheck( m_pView->broker()->settings()->backgroundCheckerEnabled() );
    doc->reactivateBgSpellChecking();
}

void KPrConfigureSpellPage::slotDefault()
{
    m_spellConfigWidget->slotDefault();
}

KPrConfigureMiscPage::KPrConfigureMiscPage( KPrView *_view, TQWidget *parent, char *name )
    : TQWidget( parent, name )
{
    TQVBoxLayout *box = new TQVBoxLayout( this, 0, 0 );

    m_pView=_view;
    config = KPrFactory::global()->config();

    TQGroupBox* tmpTQGroupBox = new TQGroupBox( 0, TQt::Vertical, i18n("Misc"), this, "GroupBox" );
    tmpTQGroupBox->layout()->setSpacing(KDialog::spacingHint());
    tmpTQGroupBox->layout()->setMargin(KDialog::marginHint());
    TQGridLayout *grid = new TQGridLayout( tmpTQGroupBox->layout(), 8, 1 );

    m_oldNbRedo=30;
    m_printNotes=true;
    if( config->hasGroup("Misc") )
    {
        config->setGroup( "Misc" );
        m_oldNbRedo=config->readNumEntry("UndoRedo",m_oldNbRedo);
        m_printNotes = config->readBoolEntry("PrintNotes", true);
    }

    m_undoRedoLimit=new KIntNumInput( m_oldNbRedo, tmpTQGroupBox );
    m_undoRedoLimit->setLabel(i18n("Undo/redo limit:"));
    m_undoRedoLimit->setRange(10, 60, 1);
    TQWhatsThis::add(m_undoRedoLimit, i18n( "Set the number of actions you can undo and redo (how many actions KPresenter keeps in its Undo buffer). This ranges from a minimum of 10 to a maximum of 60 (the default is 30). Once the number of actions reaches the number set here, earlier actions will be forgotten." ) );
    grid->addWidget(m_undoRedoLimit,0,0);

    KPrDocument* doc = m_pView->kPresenterDoc();

    m_displayLink=new TQCheckBox(i18n("Display links"),tmpTQGroupBox);
    TQWhatsThis::add(m_displayLink, i18n( "When you want to include a link in your slide, you can use the Insert->Link... menu, which allows you to insert URL, mail or file links. If the option Display links is checked, all links will be active and displayed in a different color (this is the default behavior). If the option is unchecked, the links will be inactive and the same color as the text. This affects both the edited slides and the slide show." ) );
    grid->addWidget(m_displayLink,3,0);
    m_displayLink->setChecked(doc->getVariableCollection()->variableSetting()->displayLink());

    m_underlineLink=new TQCheckBox(i18n("&Underline all links"),tmpTQGroupBox);
    m_underlineLink->setChecked(doc->getVariableCollection()->variableSetting()->underlineLink());
    TQWhatsThis::add(m_underlineLink, i18n( "If this is checked, all links will be underlined (this is the default). If it is not checked, the links will not be underlined." ) );
    grid->addWidget(m_underlineLink,4,0);


    m_displayComment=new TQCheckBox(i18n("Display comments"),tmpTQGroupBox);
    m_displayComment->setChecked(doc->getVariableCollection()->variableSetting()->displayComment());
    TQWhatsThis::add(m_displayComment, i18n( "Comments are inserted in the text at the cursor using the Insert->Comment... menu. Comments can only be viewed in edit mode and not in the slide show. If this option is checked (default) then each comment will be shown as a small yellow rectangle. You can then right-click on them to edit them, remove them or copy the text." ) );
    grid->addWidget(m_displayComment,5,0);

    m_displayFieldCode=new TQCheckBox(i18n("Display field code"),tmpTQGroupBox);
    m_displayFieldCode->setChecked(doc->getVariableCollection()->variableSetting()->displayFieldCode());
    TQWhatsThis::add(m_displayFieldCode, i18n( "In editor mode (not in slide show) this option will display all the variable codes as well as Link at links location. This is very useful to see what variable is displayed. Variables are inserted using the Insert -> Variable menu." ) );
    grid->addWidget(m_displayFieldCode,6,0);

    m_cbPrintNotes=new TQCheckBox(i18n("Print slide notes"),tmpTQGroupBox);
    m_cbPrintNotes->setChecked(m_printNotes);
    TQWhatsThis::add(m_cbPrintNotes, i18n( "If checked, all notes will be printed on paper. The notes will all be printed separately on the last page, from the first slide to the last and finally the Master Page Note. You can see the notes for each slide using the View->Show notebar menu." ) );
    grid->addWidget(m_cbPrintNotes,7,0);

    box->addWidget(tmpTQGroupBox);

    tmpTQGroupBox = new TQGroupBox( 0, TQt::Vertical, i18n("Grid"), this, "GroupBox" );
    tmpTQGroupBox->layout()->setSpacing(KDialog::spacingHint());
    tmpTQGroupBox->layout()->setMargin(KDialog::marginHint());
    grid = new TQGridLayout( tmpTQGroupBox->layout(), 8, 1 );

    KoRect rect = doc->masterPage()->getPageRect();
    TQLabel *lab=new TQLabel(i18n("Horizontal grid size:"), tmpTQGroupBox);
    TQWhatsThis::add(lab, i18n( "Set the space in millimeters between two horizontal points on the grid. Default is 5 millimeters." ) );
    grid->addWidget(lab,0,0);
    KoUnit::Unit unit = doc->unit();
    resolutionX = new KoUnitDoubleSpinBox(tmpTQGroupBox, 5.0,rect.width(),1,doc->getGridX() );
    resolutionX->setUnit( unit );

    grid->addWidget(resolutionX,1,0);

    lab=new TQLabel(i18n("Vertical grid size:"), tmpTQGroupBox);
    TQWhatsThis::add(lab, i18n( "Set the space in millimeters between two vertical points on the grid. Default is 5 millimeters." ) );
    grid->addWidget(lab,2,0);

    resolutionY = new KoUnitDoubleSpinBox(tmpTQGroupBox, 5.0,rect.height(),1,doc->getGridY() );
    resolutionY->setUnit( unit );

    grid->addWidget(resolutionY, 3, 0);
    box->addWidget(tmpTQGroupBox);

    TQSpacerItem* spacer = new TQSpacerItem( 20, 20, TQSizePolicy::Minimum, TQSizePolicy::Expanding );
    box->addItem( spacer);


}

KCommand * KPrConfigureMiscPage::apply()
{
    config->setGroup( "Misc" );
    int newUndo=m_undoRedoLimit->value();
    KPrDocument* doc = m_pView->kPresenterDoc();
    if(newUndo!=m_oldNbRedo)
    {
        config->writeEntry("UndoRedo",newUndo);
        doc->setUndoRedoLimit(newUndo);
        m_oldNbRedo=newUndo;
    }
    config->writeEntry("PrintNotes", m_cbPrintNotes->isChecked());

    KMacroCommand * macroCmd=0L;
    bool b=m_displayLink->isChecked();
    bool b_new=doc->getVariableCollection()->variableSetting()->displayLink();
    if(b_new!=b)
    {
        if(!macroCmd)
            macroCmd=new KMacroCommand(i18n("Change Display Link Command"));

        KPrChangeVariableSettingsCommand *cmd=new KPrChangeVariableSettingsCommand(
            i18n("Change Display Link Command"), doc, b_new, b, KPrChangeVariableSettingsCommand::VS_DISPLAYLINK);
        cmd->execute();
        macroCmd->addCommand(cmd);
    }

    b=m_underlineLink->isChecked();
    if(doc->getVariableCollection()->variableSetting()->underlineLink()!=b)
    {
        if(!macroCmd)
            macroCmd=new KMacroCommand(i18n("Change Display Link Command"));

        KPrChangeVariableSettingsCommand *cmd=new KPrChangeVariableSettingsCommand(
            i18n("Change Display Link Command"), doc, doc->getVariableCollection()->variableSetting()->underlineLink(),
            b, KPrChangeVariableSettingsCommand::VS_UNDERLINELINK);
        cmd->execute();
        macroCmd->addCommand(cmd);
    }

    b=m_displayComment->isChecked();
    if(doc->getVariableCollection()->variableSetting()->displayComment()!=b)
    {
        if(!macroCmd)
            macroCmd=new KMacroCommand(i18n("Change Display Link Command"));

        KPrChangeVariableSettingsCommand *cmd=new KPrChangeVariableSettingsCommand(
            i18n("Change Display Link Command"), doc, doc->getVariableCollection()->variableSetting()->displayComment(),
            b, KPrChangeVariableSettingsCommand::VS_DISPLAYCOMMENT);
        cmd->execute();
        macroCmd->addCommand(cmd);
    }

    b=m_displayFieldCode->isChecked();
    if(doc->getVariableCollection()->variableSetting()->displayFieldCode()!=b)
    {
        if(!macroCmd)
            macroCmd=new KMacroCommand(i18n("Change Display Field Code Command"));

        KPrChangeVariableSettingsCommand *cmd=new KPrChangeVariableSettingsCommand(
            i18n("Change Display Field Code Command"), doc, doc->getVariableCollection()->variableSetting()->displayComment(),
            b, KPrChangeVariableSettingsCommand::VS_DISPLAYFIELDCODE);
        cmd->execute();
        macroCmd->addCommand(cmd);
    }

    doc->setGridValue( resolutionX->value(),
                       resolutionY->value(), true);
    doc->repaint( false );

    config->sync();

    return macroCmd;
}

void KPrConfigureMiscPage::slotDefault()
{
    m_undoRedoLimit->setValue(30);
    m_displayLink->setChecked(true);
    m_displayComment->setChecked(true);
    m_underlineLink->setChecked(true);
    m_displayFieldCode->setChecked( false );
    m_cbPrintNotes->setChecked(true);
    //KPrDocument* doc = m_pView->kPresenterDoc();

    resolutionY->setValue( MM_TO_POINT( 5.0 ));
    resolutionX->setValue( MM_TO_POINT( 5.0 ));
}

KPrConfigureDefaultDocPage::KPrConfigureDefaultDocPage(KPrView *_view, TQWidget *parent, char *name )
    : TQWidget( parent, name )
{
    TQVBoxLayout *box = new TQVBoxLayout( this, 0, 0 );

    m_pView=_view;
    config = KPrFactory::global()->config();
    KPrDocument *doc = m_pView->kPresenterDoc();
    oldAutoSaveValue =  doc->defaultAutoSave()/60;
    m_oldBackupFile = true;
    m_oldLanguage = doc->globalLanguage();
    m_oldHyphenation = doc->globalHyphenation();
    if( config->hasGroup("Interface") ) {
        config->setGroup( "Interface" );
        oldAutoSaveValue = config->readNumEntry( "AutoSave", oldAutoSaveValue );
        m_oldBackupFile=config->readBoolEntry("BackupFile",m_oldBackupFile);
        m_oldLanguage = config->readEntry( "language", m_oldLanguage );
        m_oldHyphenation = config->readBoolEntry( "hyphenation", m_oldHyphenation);
    }

    TQVGroupBox* gbDocumentDefaults = new TQVGroupBox( i18n("Document Defaults"), this, "GroupBox" );
    gbDocumentDefaults->setMargin( KDialog::marginHint() );
    gbDocumentDefaults->setInsideSpacing( 5 );

    TQWidget *fontContainer = new TQWidget(gbDocumentDefaults);
    TQGridLayout * fontLayout = new TQGridLayout(fontContainer, 1, 3);

    fontLayout->setColStretch(0, 0);
    fontLayout->setColStretch(1, 1);
    fontLayout->setColStretch(2, 0);

    TQLabel *fontTitle = new TQLabel(i18n("Default font:"), fontContainer);

    font= new TQFont( doc->defaultFont() );

    TQString labelName = font->family() + ' ' + TQString::number(font->pointSize());
    fontName = new TQLabel(labelName, fontContainer);
    fontName->setFont(*font);
    fontName->setFrameStyle(TQFrame::StyledPanel | TQFrame::Sunken);

    TQPushButton *chooseButton = new TQPushButton(i18n("Choose..."), fontContainer);
    TQWhatsThis::add(chooseButton, i18n( "Click here if you want to set a new font. The TDE default Select Font dialog will then be displayed." ) );
    connect(chooseButton, TQ_SIGNAL(clicked()), this, TQ_SLOT(selectNewDefaultFont()));

    fontLayout->addWidget(fontTitle, 0, 0);
    fontLayout->addWidget(fontName, 0, 1);
    fontLayout->addWidget(chooseButton, 0, 2);

    TQWidget *languageContainer = new TQWidget(gbDocumentDefaults);
    TQGridLayout * languageLayout = new TQGridLayout(languageContainer, 1, 3);

    languageLayout->setColStretch(0, 0);
    languageLayout->setColStretch(1, 1);

    TQLabel *languageTitle = new TQLabel(i18n("Global language:"), languageContainer);
    TQWhatsThis::add(languageTitle, i18n( "Use this drop down box to determine the default language for the document. This setting is used by the hyphenation and spelling tools." ) );

    m_globalLanguage = new TQComboBox( languageContainer );
    m_globalLanguage->insertStringList( KoGlobal::listOfLanguages() );
    m_globalLanguage->setCurrentText( KoGlobal::languageFromTag( doc->globalLanguage() ) );

    languageLayout->addWidget(languageTitle, 0, 0);
    languageLayout->addWidget(m_globalLanguage, 0, 1);

    m_autoHyphenation = new TQCheckBox( i18n("Automatic hyphenation"), gbDocumentDefaults);
    TQWhatsThis::add(m_autoHyphenation, i18n( "Check this box if you want KPresenter to automatically hyphenate long words when it determines the word wrap in text frames. This is not set by default." ) );
    m_autoHyphenation->setChecked( m_oldHyphenation );

    box->addWidget(gbDocumentDefaults);

    TQVGroupBox* gbDocumentSettings = new TQVGroupBox( i18n("Document Settings"), this );
    gbDocumentSettings->setMargin( KDialog::marginHint() );
    gbDocumentSettings->setInsideSpacing( KDialog::spacingHint() );

    m_createBackupFile = new TQCheckBox( i18n("Create backup file"), gbDocumentSettings);
    TQWhatsThis::add(m_createBackupFile, i18n( "If checked, this will create a .<name>.kpr.autosave.kpr in the folder where your file is. This backup file can then be used in case of a problem.\nThe backup file is updated every time you save your document or every time there is an autosave." ) );
    m_createBackupFile->setChecked( m_oldBackupFile );


    autoSave = new KIntNumInput( oldAutoSaveValue, gbDocumentSettings );
    autoSave->setRange( 0, 60, 1 );
    autoSave->setLabel( i18n("Autosave (min):") );
    autoSave->setSpecialValueText( i18n("No autosave") );
    autoSave->setSuffix( i18n("min") );
    TQWhatsThis::add(autoSave, i18n( "You can use this to adjust how often KPresenter saves a temporary file. If you set this value to No autosave, KPresenter will not autosave. You can adjust the autosave from 1 to 60 minutes." ) );

    new TQLabel(i18n("Starting page number:"), gbDocumentSettings);
    m_oldStartingPage=doc->getVariableCollection()->variableSetting()->startingPageNumber();
    m_variableNumberOffset=new KIntNumInput(gbDocumentSettings);
    m_variableNumberOffset->setRange(1, 9999, 1, false);
    m_variableNumberOffset->setValue(m_oldStartingPage);
    TQWhatsThis::add(m_variableNumberOffset, i18n( "Here you can change the number for the first page. It is set to 1 by default.\nTip: this is helpful if you have split a single document into multiple files." ) );

    new TQLabel(i18n("Tab stop:"), gbDocumentSettings);
    m_oldTabStopWidth = doc->tabStopValue();
    KoRect rect = doc->masterPage()->getPageRect();
    m_tabStopWidth = new KoUnitDoubleSpinBox( gbDocumentSettings , MM_TO_POINT(2), rect.width() ,0.1, m_oldTabStopWidth );

    m_tabStopWidth->setUnit( doc->unit() );

    TQWhatsThis::add(m_tabStopWidth, i18n( "Each KPresenter document has a default set of tab stops. If you add tab stops to your document, the newly added tab stops override the default ones. You can use this text box to define the spacing between default tab stops. As an example, if you enter 1.5 in this text box, and the unit of measurement is in centimeters, the first default tab stop will be located 1.5 cm to the right of the frame's left-hand margin. The second default tab stop will be located at 3 cm from the left-hand margin, and so on." ) );

    box->addWidget(gbDocumentSettings);
    TQVGroupBox* gbDocumentCursor = new TQVGroupBox( i18n("Cursor"), this );
    gbDocumentCursor->setMargin( KDialog::marginHint() );
    gbDocumentCursor->setInsideSpacing( KDialog::spacingHint() );

    m_cursorInProtectedArea= new TQCheckBox(i18n("Cursor in protected area"),gbDocumentCursor);
    m_cursorInProtectedArea->setChecked(doc->cursorInProtectedArea());
    TQWhatsThis::add(m_cursorInProtectedArea, i18n( "When this box is checked and you click in a protected frame within your document, a cursor will appear. When this box is unchecked, and you click in a protected frame, no cursor will be visible." ) );

    m_directInsertCursor= new TQCheckBox(i18n("Direct insert cursor"),gbDocumentCursor);
    m_directInsertCursor->setChecked(doc->insertDirectCursor());
    TQWhatsThis::add(m_directInsertCursor, i18n( "When this box is checked, you can select a section of text using your mouse. Move the mouse to a new area in your document and click once with the middle mouse button and a copy of the selected text will be copied and pasted to the new location in the document.\nWhen this box is unchecked, in order to copy text from one section to another, you must select the text, manually copy the text to the clipboard, then manually paste the text in the new location." ) );
    box->addWidget(gbDocumentCursor);

    TQSpacerItem* spacer = new TQSpacerItem( 20, 20, TQSizePolicy::Minimum, TQSizePolicy::Expanding );
    box->addItem(spacer);

}

KPrConfigureDefaultDocPage::~KPrConfigureDefaultDocPage()
{
    delete font;
}

KCommand *KPrConfigureDefaultDocPage::apply()
{
    config->setGroup( "Document defaults" );
    KPrDocument* doc = m_pView->kPresenterDoc();
    config->writeEntry("DefaultFont",TQString(font->toString()));

    config->setGroup( "Interface" );
    int autoSaveVal = autoSave->value();
    if( autoSaveVal != oldAutoSaveValue ) {
        config->writeEntry( "AutoSave", autoSaveVal );
        m_pView->kPresenterDoc()->setAutoSave( autoSaveVal*60 );
        oldAutoSaveValue=autoSaveVal;
    }
    bool state =m_createBackupFile->isChecked();

    if(state!=m_oldBackupFile)
    {
        config->writeEntry( "BackupFile", state );
        doc->setBackupFile( state);
        m_oldBackupFile=state;
    }

    state = m_cursorInProtectedArea->isChecked();
    if ( state != doc->cursorInProtectedArea() )
    {
        config->writeEntry( "cursorInProtectArea", state );
        m_pView->kPresenterDoc()->setCursorInProtectedArea( state );
    }

    state = m_directInsertCursor->isChecked();
    if ( state != doc->insertDirectCursor() )
        doc->setInsertDirectCursor( state );

    //Laurent Todo add a message box to inform user that
    //global language will change after re-launch kword
    const TQString lang = KoGlobal::tagOfLanguage( m_globalLanguage->currentText() );
    config->writeEntry( "language" , lang);
    m_oldLanguage = lang;
    //don't call this function otherwise we can have a textobject with
    // a default language and other textobject with other default language.
    //doc->setGlobalLanguage( lang );


    state = m_autoHyphenation->isChecked();
    config->writeEntry( "hyphenation", state  );
    m_oldHyphenation = state;

    KMacroCommand *macro = 0L;
    int newStartingPage=m_variableNumberOffset->value();
    if(newStartingPage!=m_oldStartingPage)
    {
        macro = new KMacroCommand( i18n("Change Starting Page Number") );
        KPrChangeStartingPageCommand *cmd = new KPrChangeStartingPageCommand( i18n("Change Starting Page Number"), doc, m_oldStartingPage,newStartingPage );
        cmd->execute();
        macro->addCommand( cmd);
        m_oldStartingPage=newStartingPage;
    }
    double newTabStop = m_tabStopWidth->value();
    if ( newTabStop != m_oldTabStopWidth)
    {
        if ( !macro )
            macro = new KMacroCommand( i18n("Change Tab Stop Value") );
        KPrChangeTabStopValueCommand *cmd = new KPrChangeTabStopValueCommand( i18n("Change Tab Stop Value"), m_oldTabStopWidth, newTabStop, doc);
        cmd->execute();
        macro->addCommand( cmd );
        m_oldTabStopWidth = newTabStop;
    }
    return macro;
}

void KPrConfigureDefaultDocPage::slotDefault()
{
    autoSave->setValue( m_pView->kPresenterDoc()->defaultAutoSave()/60 );
    m_variableNumberOffset->setValue(1);
    m_cursorInProtectedArea->setChecked(true);
    m_tabStopWidth->setValue( MM_TO_POINT(15));
    m_createBackupFile->setChecked( true );
    m_directInsertCursor->setChecked( false );
    m_globalLanguage->setCurrentText( KoGlobal::languageFromTag( TDEGlobal::locale()->language() ) );
    m_autoHyphenation->setChecked( false );
}

void KPrConfigureDefaultDocPage::selectNewDefaultFont() {
    TQStringList list;
    TDEFontChooser::getFontList(list, TDEFontChooser::SmoothScalableFonts);
    TDEFontDialog dlg( this, "Font Selector", false, true, list, true );
    dlg.setFont(*font);
    int result = dlg.exec();
    if (KDialog::Accepted == result) {
        delete font;
        font = new TQFont(dlg.font());
        fontName->setText(font->family() + ' ' + TQString::number(font->pointSize()));
        fontName->setFont(*font);
        m_pView->kPresenterDoc()->setDefaultFont( *font );
    }
}

KPrConfigureToolsPage::KPrConfigureToolsPage( KPrView *_view, TQWidget *parent, char *name )
    : TQWidget( parent, name )
{
    TQVBoxLayout *box = new TQVBoxLayout( this, 0, 0 );

    m_pView = _view;
    config = KPrFactory::global()->config();
    m_pView->getCanvas()->deSelectAllObj();

    TQTabWidget *tab = new TQTabWidget(this);

    KoPenCmd::Pen pen( m_pView->getPen(), m_pView->getLineBegin(), m_pView->getLineEnd() );
    m_confPenDia = new KPrPenStyleWidget(tab, 0, pen, true );
    tab->addTab( m_confPenDia, i18n( "Outl&ine" ) );

    KPrBrushCmd::Brush brush( m_pView->getBrush(),
                           m_pView->getGColor1(),
                           m_pView->getGColor2(),
                           m_pView->getGType(),
                           m_pView->getFillType(),
                           m_pView->getGUnbalanced(),
                           m_pView->getGXFactor(),
                           m_pView->getGYFactor() );
    m_brushProperty = new KPrBrushProperty( this, 0, brush );
    tab->addTab( m_brushProperty, i18n( "&Fill" ) );

    KPrRectValueCmd::RectValues rectValues;
    rectValues.xRnd = m_pView->getRndX();
    rectValues.yRnd = m_pView->getRndY();
    m_rectProperty = new KPrRectProperty( this, 0, rectValues );
    tab->addTab( m_rectProperty, i18n( "&Rectangle" ) );

    KPrPolygonSettingCmd::PolygonSettings polygonSettings;
    polygonSettings.checkConcavePolygon = m_pView->getCheckConcavePolygon();
    polygonSettings.cornersValue = m_pView->getCornersValue();
    polygonSettings.sharpnessValue = m_pView->getSharpnessValue();
    m_polygonProperty = new KPrPolygonProperty( this, 0, polygonSettings );
    tab->addTab( m_polygonProperty, i18n( "Polygo&n" ) );

    KPrPieValueCmd::PieValues pieValues;
    pieValues.pieType = m_pView->getPieType();
    pieValues.pieAngle = m_pView->getPieAngle();
    pieValues.pieLength = m_pView->getPieLength();
    m_pieProperty = new KPrPieProperty( this, 0, pieValues );
    tab->addTab( m_pieProperty, i18n( "&Pie" ) );

    box->addWidget(tab);
}

KPrConfigureToolsPage::~KPrConfigureToolsPage()
{
}

void KPrConfigureToolsPage::apply()
{
    KoPenCmd::Pen pen = m_confPenDia->getPen();
    m_pView->setPen( pen.pen );
    m_pView->setLineBegin( pen.lineBegin );
    m_pView->setLineEnd( pen.lineEnd );
    m_pView->getActionPenColor()->setCurrentColor( pen.pen.color() );

    KPrBrushCmd::Brush brush = m_brushProperty->getBrush();
    m_pView->setBrush( brush.brush );
    m_pView->setFillType( brush.fillType );
    m_pView->setGColor1( brush.gColor1 );
    m_pView->setGColor2( brush.gColor2 );
    m_pView->setGType( brush.gType );
    m_pView->setGUnbalanced( brush.unbalanced );
    m_pView->setGXFactor( brush.xfactor );
    m_pView->setGYFactor( brush.yfactor );
    m_pView->getActionBrushColor()->setCurrentColor( brush.brush.color() );

    KPrRectValueCmd::RectValues rectValues = m_rectProperty->getRectValues();
    m_pView->setRndX( rectValues.xRnd );
    m_pView->setRndY( rectValues.yRnd );

    KPrPolygonSettingCmd::PolygonSettings polygonSettings = m_polygonProperty->getPolygonSettings();
    m_pView->setCheckConcavePolygon( polygonSettings.checkConcavePolygon );
    m_pView->setCornersValue( polygonSettings.cornersValue );
    m_pView->setSharpnessValue( polygonSettings.sharpnessValue );

    KPrPieValueCmd::PieValues pieValues = m_pieProperty->getPieValues();
    m_pView->setPieType( pieValues.pieType );
    m_pView->setPieAngle( pieValues.pieAngle );
    m_pView->setPieLength( pieValues.pieLength );

    //TODO set pen brush in m_rectProperty
    //TODO set pen brush in m_polygonProperty
    //TODO set pen brush in m_pieProperty
}

void KPrConfigureToolsPage::slotDefault()
{
    KoPenCmd::Pen pen( KoPen(black, 1.0, TQt::SolidLine), L_NORMAL, L_NORMAL );
    m_confPenDia->setPen( pen );
    m_pView->getActionPenColor()->setCurrentColor( pen.pen.color() );

    KPrBrushCmd::Brush brush( TQBrush( white, SolidPattern ), red, green,
                           BCT_GHORZ, FT_BRUSH, false, 100, 100 );
    m_brushProperty->setBrush( brush );
    m_pView->getActionBrushColor()->setCurrentColor( brush.brush.color() );

    KPrRectValueCmd::RectValues rectValues;
    rectValues.xRnd = 0;
    rectValues.yRnd = 0;
    m_rectProperty->setRectValues( rectValues );

    KPrPolygonSettingCmd::PolygonSettings polygonSettings;
    polygonSettings.checkConcavePolygon = false;
    polygonSettings.cornersValue = 3;
    polygonSettings.sharpnessValue = 0;
    m_polygonProperty->setPolygonSettings( polygonSettings );

    KPrPieValueCmd::PieValues pieValues;
    pieValues.pieType = PT_PIE;
    pieValues.pieAngle = 45 * 16;
    pieValues.pieLength = 270 * 16;
}

KPrConfigurePathPage::KPrConfigurePathPage( KPrView *_view, TQWidget *parent, char *name )
    : TQWidget( parent, name )
{
    TQVBoxLayout *box = new TQVBoxLayout( this, 0, 0 );

    m_pView=_view;
    KPrDocument* doc = m_pView->kPresenterDoc();
    config = KPrFactory::global()->config();

    m_pPathView = new TDEListView( this );
    m_pPathView->setResizeMode(TQListView::NoColumn);
    m_pPathView->addColumn( i18n( "Type" ) );
    m_pPathView->addColumn( i18n( "Path" ) );
    (void) new TQListViewItem( m_pPathView, i18n("Picture Path"),doc->picturePath() );
    (void) new TQListViewItem( m_pPathView, i18n("Backup Path"),doc->backupPath() );
    TQWhatsThis::add(m_pPathView, i18n( "There are two paths that are set here: the Backup Path and the Picture Path. The Backup path is the folder where your backup files are saved and the Picture Path is the folder where your pictures are saved." ) );
    box->addWidget(m_pPathView);

    m_modifyPath = new TQPushButton( i18n("Modify Path..."), this);
    connect( m_modifyPath, TQ_SIGNAL( clicked ()), this, TQ_SLOT( slotModifyPath()));
    connect( m_pPathView, TQ_SIGNAL( doubleClicked (TQListViewItem *, const TQPoint &, int )),
             this, TQ_SLOT( slotModifyPath()));
    connect( m_pPathView, TQ_SIGNAL( selectionChanged ( TQListViewItem * )),
             this, TQ_SLOT( slotSelectionChanged(TQListViewItem * )));
    slotSelectionChanged(m_pPathView->currentItem());
    TQWhatsThis::add(m_modifyPath, i18n( "When you click this button, a small dialog will appear and, if Default path is unchecked, you can either enter a path yourself or choose one using the standard TDE file dialog." ) );
    box->addWidget(m_modifyPath);

}

void KPrConfigurePathPage::slotSelectionChanged(TQListViewItem * item)
{
    m_modifyPath->setEnabled( item );
}

void KPrConfigurePathPage::slotModifyPath()
{
    TQListViewItem *item = m_pPathView->currentItem ();
    if ( item )
    {
        if ( item->text(0)==i18n("Picture Path"))
        {
            KURLRequesterDlg * dlg = new KURLRequesterDlg( item->text(1), 0L,
                                                           "picture path dlg");
            dlg->fileDialog()->setMode(KFile::Directory | KFile::LocalOnly);
            if ( dlg->exec() )
                item->setText( 1, dlg->selectedURL().path());
            delete dlg;
        }
        else if ( item->text(0)==i18n("Backup Path"))
        {
            KoChangePathDia *dlg = new KoChangePathDia( item->text(1), 0L,
                                                        "backup path" );
            if (dlg->exec() )
                item->setText(1, dlg->newPath());
            delete dlg;
        }
    }
}

void KPrConfigurePathPage::slotDefault()
{
    TQListViewItem * item = m_pPathView->findItem(i18n("Picture Path"), 0);
    if ( item )
        item->setText(1, TDEGlobalSettings::documentPath());
    item = m_pPathView->findItem(i18n("Backup Path"), 0);
    if ( item )
        item->setText(1, TQString() );
}

void KPrConfigurePathPage::apply()
{
    TQListViewItem *item = m_pPathView->findItem(i18n("Backup Path"), 0);
    if ( item )
    {
        TQString res = item->text(1 );
        if ( res != m_pView->kPresenterDoc()->backupPath())
        {
            config->setGroup( "Kpresenter Path" );
            m_pView->kPresenterDoc()->setBackupPath( res );
#if KDE_IS_VERSION(3,1,3)
            config->writePathEntry( "backup path",res );
#else
            config->writeEntry( "backup path",res );
#endif
        }
    }
    item = m_pPathView->findItem(i18n("Picture Path"), 0);
    if ( item )
    {
        TQString res = item->text(1 );
        if ( res != m_pView->kPresenterDoc()->picturePath())
        {
            config->setGroup( "Kpresenter Path" );
            m_pView->kPresenterDoc()->setPicturePath( res );
#if KDE_IS_VERSION(3,1,3)
            config->writePathEntry( "picture path",res );
#else
            config->writeEntry( "picture path",res );
#endif
        }
    }
}

////

KPrConfigureTTSPage::KPrConfigureTTSPage( KPrView *_view, TQWidget *parent, char *name )
    : TQWidget( parent, name )
{
    Q_UNUSED(_view);

    TQVBoxLayout *box = new TQVBoxLayout( this, 0, 0 );

    m_cbSpeakPointerWidget = new TQCheckBox(i18n("Speak widget under &mouse pointer"), this);
    m_cbSpeakFocusWidget = new TQCheckBox(i18n("Speak widget with &focus"), this);
    m_gbScreenReaderOptions = new TQVGroupBox("", this);
    box->add(m_cbSpeakPointerWidget);
    box->add(m_cbSpeakFocusWidget);
    box->add(m_gbScreenReaderOptions);
    m_gbScreenReaderOptions->setMargin( KDialog::marginHint() );
    m_gbScreenReaderOptions->setInsideSpacing( KDialog::spacingHint() );
    m_cbSpeakTooltips = new TQCheckBox(i18n("Speak &tool tips"), m_gbScreenReaderOptions);
    m_cbSpeakWhatsThis = new TQCheckBox(i18n("Speak &What's This?"), m_gbScreenReaderOptions);
    m_cbSpeakDisabled = new TQCheckBox(i18n("Verbal indication if widget is disabled (grayed)",
        "&Say whether disabled"), m_gbScreenReaderOptions);
    m_cbSpeakAccelerators = new TQCheckBox(i18n("Spea&k accelerators"), m_gbScreenReaderOptions);
    TQHBox* hbAcceleratorPrefix = new TQHBox(m_gbScreenReaderOptions);
    TQWidget* spacer = new TQWidget(hbAcceleratorPrefix);
    spacer->setMinimumWidth(2 * KDialog::marginHint());
    m_lblAcceleratorPrefix = 
        new TQLabel(i18n("A word spoken before another word", "Pr&efaced by the word:"),
        hbAcceleratorPrefix);
    m_leAcceleratorPrefixWord = new TQLineEdit(i18n("Keyboard accelerator, such as Alt+F", "Accelerator"),
        hbAcceleratorPrefix);
    m_lblAcceleratorPrefix->setBuddy(m_leAcceleratorPrefixWord);
    TQHBox* hbPollingInterval = new TQHBox(m_gbScreenReaderOptions);
    hbPollingInterval->setMargin( 0 );
    TQLabel* lblPollingInterval = new TQLabel(i18n("&Polling interval:"), hbPollingInterval);
    m_iniPollingInterval = new KIntNumInput(hbPollingInterval);
    m_iniPollingInterval->setSuffix(" ms");
    m_iniPollingInterval->setRange(100, 5000, 100, true);
    lblPollingInterval->setBuddy(m_iniPollingInterval);

    config = KPrFactory::global()->config();
    config->setGroup("TTS");
    m_cbSpeakPointerWidget->setChecked(config->readBoolEntry("SpeakPointerWidget", false));
    m_cbSpeakFocusWidget->setChecked(config->readBoolEntry("SpeakFocusWidget", false));
    m_cbSpeakTooltips->setChecked(config->readBoolEntry("SpeakTooltips", true));
    m_cbSpeakWhatsThis->setChecked(config->readBoolEntry("SpeakWhatsThis", false));
    m_cbSpeakDisabled->setChecked(config->readBoolEntry("SpeakDisabled", true));
    m_cbSpeakAccelerators->setChecked(config->readBoolEntry("SpeakAccelerators", true));
    m_leAcceleratorPrefixWord->setText(config->readEntry("AcceleratorPrefixWord",
        i18n("Keyboard accelerator, such as Alt+F", "Accelerator")));
    m_iniPollingInterval->setValue(config->readNumEntry("PollingInterval", 600));

    screenReaderOptionChanged();
    connect(m_cbSpeakPointerWidget, TQ_SIGNAL(toggled(bool)), this, TQ_SLOT(screenReaderOptionChanged()));
    connect(m_cbSpeakFocusWidget, TQ_SIGNAL(toggled(bool)), this, TQ_SLOT(screenReaderOptionChanged()));
    connect(m_cbSpeakAccelerators, TQ_SIGNAL(toggled(bool)), this, TQ_SLOT(screenReaderOptionChanged()));
}

void KPrConfigureTTSPage::slotDefault()
{
    m_cbSpeakPointerWidget->setChecked(false);
    m_cbSpeakFocusWidget->setChecked(false);
    m_cbSpeakTooltips->setChecked(true);
    m_cbSpeakWhatsThis->setChecked(false);
    m_cbSpeakDisabled->setChecked(true);
    m_cbSpeakAccelerators->setChecked(true);
    m_leAcceleratorPrefixWord->setText(i18n("Keyboard accelerator, such as Alt+F", "Accelerator"));
    m_iniPollingInterval->setValue(600);
}

void KPrConfigureTTSPage::apply()
{
    config->setGroup("TTS");
    config->writeEntry("SpeakPointerWidget", m_cbSpeakPointerWidget->isChecked());
    config->writeEntry("SpeakFocusWidget", m_cbSpeakFocusWidget->isChecked());
    config->writeEntry("SpeakTooltips", m_cbSpeakTooltips->isChecked());
    config->writeEntry("SpeakWhatsThis", m_cbSpeakWhatsThis->isChecked());
    config->writeEntry("SpeakDisabled", m_cbSpeakDisabled->isChecked());
    config->writeEntry("SpeakAccelerators", m_cbSpeakAccelerators->isChecked());
    config->writeEntry("AcceleratorPrefixWord", m_leAcceleratorPrefixWord->text());
    config->writeEntry("PollingInterval", m_iniPollingInterval->value());
    if (kospeaker) kospeaker->readConfig(config);
}

void KPrConfigureTTSPage::screenReaderOptionChanged()
{
    m_gbScreenReaderOptions->setEnabled(
        m_cbSpeakPointerWidget->isChecked() | m_cbSpeakFocusWidget->isChecked());
    m_leAcceleratorPrefixWord->setEnabled(m_cbSpeakAccelerators->isChecked());
    m_lblAcceleratorPrefix->setEnabled(m_cbSpeakAccelerators->isChecked());
}

#include "KPrConfig.moc"
