/***************************************************************************
 *   Copyright (C) 2012-2019 by Timothy Pearson                            *
 *   kb9vqf@pearsoncomputing.net                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef TDEKRBSERVERSOCKET_H
#define TDEKRBSERVERSOCKET_H

#include <tqsocket.h>

#define N_CALLBACKS 3

class TQBuffer;
class SASLDataPrivate;

class TDEKerberosServerSocket : public TQSocket
{
	Q_OBJECT

	public:
		enum KerberosStatus {
			KerberosNotRequested,
			KerberosFailure,
			KerberosInitializing,
			KerberosInUse
		};

	public:
		TDEKerberosServerSocket(TQObject *parent=0, const char *name=0);
		virtual ~TDEKerberosServerSocket();

		bool open(int mode);
		void close();
		void flush(int hidebasehack=0);
		Offset size() const;
		Offset at() const;
		bool at(Offset off, int hidebasehack=0);
		bool atEnd() const;

		int getch();
		int putch(int);
		int ungetch(int);

		TQ_LONG readBlock(char *data, TQ_ULONG maxlen);
		TQ_LONG writeBlock(const char *data, TQ_ULONG len);
		TQ_LONG readLine(char *data, TQ_ULONG maxlen);
		TQString readLine();
		void writeLine(TQString);
		TQ_ULONG bytesAvailable() const;

		int setUsingKerberos(bool krbactive);
		void setServiceName(TQString name);
		void setServerFQDN(TQString name);
		
		void setDataTimeout(int timeoutms);

		KerberosStatus kerberosStatus() const;
		bool canReadData();
		void clearIncomingData();

		int writeEndOfFrame();
		bool canReadFrame(bool processPendingData=true);
		void clearFrameTail();
		void writeBufferedData();

	public slots:
		int processPendingData();

	private:
		int initializeKerberosInterface();
		void freeKerberosConnection();
		void sendSASLDataToNetwork(const char *buffer, unsigned length);
		int getSASLDataFromNetwork(char *buf, int trunclen, bool shouldblock=true);
		int transmitEncryptedData(const char* readbuf, int cc);
		int receiveEncryptedData(char *buf, unsigned int trunclen, bool shouldblock=true);
		void setStatusMessage(TQString message);

	private slots:
		void continueKerberosInitialization();

	signals:
		void statusMessageUpdated(const TQString&);
		void newDataReceived();

	protected:
		TQString m_authenticatedUserName;
		TQString m_authenticatedRealmName;

	private:
		bool m_kerberosRequested;
		TQString m_serviceName;
		TQString m_serverFQDN;
		int m_criticalSection;
		TQBuffer* m_readBuffer;
		long m_readBufferLength;
		long m_readBufferReadPointer;
		TQBuffer* m_writeBuffer;
		long m_writeBufferLength;
		bool m_krbInitRunning;
		int m_krbInitState;
		TQString m_prevStatusMessage;
		int m_dataTimeout;
		TQTimer* kerberosInitLoopTimer;
		bool* m_canary;

	private:
		SASLDataPrivate *saslData;
		unsigned int m_negotiatedMaxBufferSize;
		int m_krbInitResult;
		int m_krbInitServerLast;
		unsigned int m_krbInitLastLen;
		const char *m_krbInitData;
};

#endif // TDEKRBSERVERSOCKET_H