/*
 * Remote Laboratory FPGA Viewer Part
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * (c) 2012-2014 Timothy Pearson
 * Raptor Engineering
 * http://www.raptorengineeringinc.com
 */

#ifndef REMOTELAB_FPGAVIEWPART_H
#define REMOTELAB_FPGAVIEWPART_H

#include <tdekrbclientsocket.h>

#include <tqcstring.h>
#include <tqimage.h>

#include <ksimpleconfig.h>

#include <tdeparts/browserextension.h>
#include <tdeparts/statusbarextension.h>
#include <tdeparts/part.h>
#include <kurl.h>
#include <kled.h>

#include <tdemdichildview.h>

#include <tqtrla.h>

#include <sevensegment.h>

class TDEAboutData;
using KParts::StatusBarExtension;
class TraceWidget;
class TQSocket;
class TQTimer;
class TQMutex;
class TQFile;
class FPGAViewBase;

class FPGALed : public KLed
{
	Q_OBJECT

	public:
		FPGALed(TQWidget *parent=0, const char *name=0);
		void setClickable(bool clickable);

	protected:
		virtual void mouseReleaseEvent(TQMouseEvent *);

	signals:
		void clicked();

	private:
		bool m_clickable;
};

class FPGAPushButton : public KLed
{
	Q_OBJECT

	public:
		FPGAPushButton(TQWidget *parent=0, const char *name=0);

	protected:
		virtual void enterEvent(TQEvent *);
		virtual void leaveEvent(TQEvent *);
		virtual void mousePressEvent(TQMouseEvent *);
		virtual void mouseReleaseEvent(TQMouseEvent *);

	signals:
		void buttonPressed();
		void buttonReleased();
		void changed();

	private:
		bool mouseDown;
};

class ImageViewerWindow : public KMdiChildView
{
	Q_OBJECT

	public:
		ImageViewerWindow(TQString caption, TQWidget* parent=0, const char* name=0);
		~ImageViewerWindow();

	public:
		void setPixmap(TQPixmap);

	protected:
		void resizeEvent(TQResizeEvent *);

	private:
		TQLabel* m_label;
};

typedef TQValueList<unsigned int> UnsignedIntegerList;

namespace RemoteLab
{
	class FPGAViewPart : public KParts::RemoteInstrumentPart
	{
		Q_OBJECT

		public:
			enum InterfaceMode {
				BasicInterfaceMode,
				IntermediateInterfaceMode,
				AdvancedInterfaceMode
			};
	
		public:
			FPGAViewPart(TQWidget *, const char *, TQObject *, const char *, const TQStringList&);
			~FPGAViewPart();
	
			virtual bool openFile() { return false; }	// pure virtual in the base class
			virtual bool closeURL();
			static TDEAboutData *createAboutData();
			virtual TQPtrList<TDEAction> menuActionList();

		public slots:
			virtual bool openURL(const KURL &url);
		
		private slots:
			void postInit();
			void resizeToHint();
			void updateDisplay();
			void processLockouts();
			void connectionClosed();
			void connectionFinishedCallback();
			void disconnectFromServerCallback();
			void connectionStatusChangedCallback();
			void switchToBasicMode();
			void switchToIntermediateMode();
			void switchToAdvancedMode();
			void processAllGraphicsUpdates();
			void update8BitInputLEDs();

			void process4BitInputChanges();
			void process4BitOutputChanges();
			void process8BitInputChanges();
			void process8BitOutputChanges();
			void process16BitInputChanges();
			void process16BitOutputChanges();
			void processLCDOutputChanges();
			void process7SegmentLEDOutputChanges();

			void groupGlobalControlResetButtonClicked();
			void batchTestRunButtonClicked();
			void dataProcessingRunButtonClicked();

		private:
			void sendInputStatesToRemoteFPGA();
			void receiveInputStatesFromRemoteFPGA();
	
		private:
			FPGAViewBase* m_base;
			TQMutex* m_connectionMutex;
			KSimpleConfig* m_config;
			TQTimer* m_updateTimer;
			TQTimer* m_timeoutTimer;

			TQPtrList<TDEAction> m_menuActionList;
			TDEActionMenu* m_modeSubMenu;
			TDEToggleAction* m_modeBasicEnabled;
			TDEToggleAction* m_modeIntermediateEnabled;
			TDEToggleAction* m_modeAdvancedEnabled;

			InterfaceMode m_interfaceMode;

			int m_commHandlerState;
			int m_commHandlerMode;
			int m_commHandlerNextState;
			int m_commHandlerNextMode;
			bool m_connectionActiveAndValid;
			unsigned char m_tickerState;
			bool m_remoteInputModeEnabled;
			unsigned char m_4bitInputValue;
			unsigned char m_4bitOutputValue;
			unsigned char m_8bitInputValue;
			unsigned char m_8bitOutputValue;
			unsigned int m_16bitInputValue;
			unsigned int m_16bitOutputValue;
			TQCString m_LCDOutputValue;
			unsigned char m_7segDigit3OutputValue;
			unsigned char m_7segDigit2OutputValue;
			unsigned char m_7segDigit1OutputValue;
			unsigned char m_7segDigit0OutputValue;
			bool m_sendingUserLogicReset;

			UnsignedIntegerList m_batchInputValueList;
			UnsignedIntegerList m_batchOutputValueList;
			unsigned int m_batchCurrentValueIndex;
			bool m_batchUsing16Bit;
			TQFile* m_batchOutputFile;
			bool m_dataGenerateValidationString;
			bool m_dataIsImage;
			TQImage m_dataInputImage;
			TQFile* m_dataOutputFile;
			TQByteArray m_dataByteArray;
			unsigned int m_dataMemorySize;
			unsigned int m_dataMemorySizePrev;
			unsigned int m_dataMemoryImageWidth;
			unsigned int m_dataMemoryImageHeight;

			ImageViewerWindow* m_inputImageViewer;
			ImageViewerWindow* m_outputImageViewer;
	};
}

#endif
