/*
 * Floating-point SpinBox Widget
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * (c) 2012 Timothy Pearson
 * Raptor Engineering
 * http://www.raptorengineeringinc.com
 *
 * Original Author: Ewald R. de Wit
 * From Qt-Interest mailing list
 * http://lists.trolltech.com/qt-interest/1999-07/thread00400-0.html
 */

#include <tqvalidator.h>
#include <math.h>

#include "floatspinbox.h"

#define ROUND(x) ((int)(0.5 + (x)))

FloatSpinBox::FloatSpinBox(double fmin, double fmax, double fvalue, TQWidget *parent) : TQSpinBox(parent) {
	init(fmin, fmax, fvalue);

	connect( this, SIGNAL(valueChanged(int)), SLOT(acceptValueChanged(int)) );
}

FloatSpinBox::FloatSpinBox(TQWidget *parent , const char* name) : TQSpinBox(parent, name) {
	init(0, 0, 0);

	connect( this, SIGNAL(valueChanged(int)), SLOT(acceptValueChanged(int)) );
}

void FloatSpinBox::init(double fmin, double fmax, double fvalue, int precision) {
	min = fmin;
	max = fmax;
	value = fvalue;
	
	// How many decimals after the floating point?
	if (precision < 0) {
		dec = ((fmax - fmin) == 0) ? 2 : 2 - (int)( log10(fabs(fmax - fmin)) );
		if (dec < 0) dec = 0;
	}
	else {
		dec = precision;
	}
	
	int intmax = (int)((max - min) * pow( 10, dec ));
	int intval = ROUND( (value - min) * pow( 10, dec ) );
	setRange( 0, intmax );
	setValue( intval );
	setSteps( 10, 100 );

	TQDoubleValidator *validator = new TQDoubleValidator(min, max, dec, this);
	setValidator(validator);
}

void FloatSpinBox::setFloatMin(double fmin) {
	init(fmin, max, value, dec);
}

void FloatSpinBox::setFloatMax(double fmax) {
	init(min, fmax, value, dec);
}

void FloatSpinBox::setPrecision(int precision) {
	init(min, max, value, precision);
}

TQString FloatSpinBox::mapValueToText(int ival) {
	TQString str;
	value = min + (double)ival * pow(10, -dec);
	str.sprintf("%.*f", dec, value);	// This can hang as 'value' may (randomly) have an insanely high precision that is very difficult to convert to text
	return( str );
}

int FloatSpinBox::mapTextToValue (bool * ok) {
	TQString str = cleanText();
	double tryValue = str.toDouble( ok );
	if (*ok) {
		value = tryValue;
	}
	return ROUND( (value - min) * pow( 10, dec ) );
}

void FloatSpinBox::setFloatValue(double d) {
	value = d;
	setValue( ROUND( (value - min) * pow( 10, dec )) );
}

double FloatSpinBox::floatValue() {
	return value;
}

void FloatSpinBox::setEnabled(bool enable) {
	if (enable != isEnabled()) {
		TQSpinBox::setEnabled(enable);
	}
}

void FloatSpinBox::acceptValueChanged(int ival) {
	Q_UNUSED(ival);
	emit floatValueChanged( value );
}

FloatSpinBox::~FloatSpinBox()
{
	//
}

#include "floatspinbox.moc"