//Author:    Timothy Pearson <kb9vqf@pearsoncomputing.net>, (C) 2012
//Copyright: See COPYING file that comes with this distribution

#include "instrumentview.h"

#include <tqtimer.h>

#include <klibloader.h>
#include <kiconloader.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <tdemdimainfrm.h>

namespace RemoteLab {

InstrumentView::InstrumentView(const TQString &library, TQWidget *parentWidget, const char *name, WFlags f)
	: KMdiChildView(parentWidget, name, f)
	, m_libraryName(library)
	, m_instrumentPart( 0 )
	, m_fixedSize( false )
	, m_canary( NULL )
{
	m_mainForm = dynamic_cast<KMdiMainFrm*>(parentWidget);
	init();
}

InstrumentView::InstrumentView(const TQString &library, const TQString &caption, TQWidget *parentWidget, const char *name, WFlags f)
	: KMdiChildView(caption, parentWidget, name, f)
	, m_libraryName(library)
	, m_instrumentPart( 0 )
	, m_fixedSize( false )
	, m_canary( NULL )
{
	m_mainForm = dynamic_cast<KMdiMainFrm*>(parentWidget);
	init();
}

InstrumentView::~InstrumentView() {
	if (m_canary) {
		*m_canary = true;
	}
}

void InstrumentView::init() {
	KLibFactory *factory = KLibLoader::self()->factory(m_libraryName.ascii());
	
	if (!factory) {
		KMessageBox::error( this, i18n("TDE could not find the %1 part, or the %2 part could not be started.").arg(m_libraryName).arg(m_libraryName) );
		TQTimer::singleShot(0, this, SLOT(close()));
	}
	else {
		m_instrumentPart = (InstrumentPart *)factory->create(TQT_TQOBJECT(this), "part", "KParts::RemoteInstrumentPart");
		m_instrumentPart->setMDIMainForm(m_mainForm);
		setIcon(SmallIcon(m_libraryName));
		connect(m_instrumentPart, SIGNAL(statusMessageSet(const TQString&)), this, SLOT(setStatusMessage(const TQString&)));
		connect(m_instrumentPart, SIGNAL(usingFixedSizeChanged(bool)), this, SLOT(setUsingFixedSize(bool)));
		TQWidget *childPartWidget = m_instrumentPart->widget();
		if (childPartWidget) {
			childPartWidget->installEventFilter(this);
		}
	}
}

bool InstrumentView::eventFilter(TQObject *o, TQEvent *e) {
	if (m_instrumentPart) {
		TQWidget *childPartWidget = m_instrumentPart->widget();
		if (childPartWidget) {
			if (o == childPartWidget) {
				if ((e->type() == TQEvent::Resize) || (e->type() == TQEvent::LayoutHint)) {
					setChildSizeData();
				}
			}
		}
	}

	// Allow event processing by other routines
	return FALSE;
}

void InstrumentView::setChildSizeData() {
	if (m_instrumentPart) {
		TQWidget *childPartWidget = m_instrumentPart->widget();
		if (childPartWidget) {
			if (m_fixedSize) {
				setFixedSize(childPartWidget->sizeHint());
			}
			else {
				TQSize minimumSizeHint;
				if (childPartWidget->layout()) {
					minimumSizeHint = childPartWidget->layout()->minimumSize();
				}
				else {
					minimumSizeHint = childPartWidget->minimumSize();
				}
				setMinimumSize(minimumSizeHint.width(), minimumSizeHint.height());
				resize(childPartWidget->size());
			}
		}
	}
}

void InstrumentView::setUsingFixedSize(bool fixed) {
	m_fixedSize = fixed;
	if (!fixed) {
		setMinimumSize(0, 0);
		setMaximumSize(TQWIDGETSIZE_MAX, TQWIDGETSIZE_MAX);
	}
	setChildSizeData();
}

void InstrumentView::resizeEvent(TQResizeEvent *) {
	setChildSizeData();
}

TQPtrList<TDEAction> InstrumentView::menuActionList() {
	if (m_instrumentPart) {
		return m_instrumentPart->menuActionList();
	}
	else {
		return TQPtrList<TDEAction>();
	}
}

void InstrumentView::setStatusMessage(const TQString& message) {
	emit(statusMessageSet(message));
}

bool InstrumentView::queryExit() {
	if (!m_instrumentPart) {	// apparently std::exit() still calls this function, and abort() causes a crash..
		return true;
	}

	m_instrumentPart->closeURL();
	return true;
}

void InstrumentView::closeConnections() {
	queryExit();
}

void InstrumentView::connectServer(TQString server) {
	if (!m_canary) {
		m_canary = new bool;
		*m_canary = false;
	}
	bool* canary = m_canary;

	if (m_instrumentPart) {
		if (m_instrumentPart->openURL(KURL(server))) {		// This can call processEvents, therefore this object may not exist when it returns!
			if (*canary == true) {
				delete canary;
				return;
			}
			TQTimer::singleShot(0, this, SLOT(close()));
		}
	}

	delete m_canary;
	m_canary = NULL;
}

/**********************************************
  SESSION MANAGEMENT
 **********************************************/

void InstrumentView::saveProperties( TDEConfig *config ) {
}

void InstrumentView::readProperties( TDEConfig *config ) {
}

} //namespace RemoteLab
