/*
 * Remote Laboratory User Management Part
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * (c) 2013-2019 Timothy Pearson
 * Raptor Engineering
 * http://www.raptorengineeringinc.com
 */

#include "define.h"
#include "part.h"

#include <tdeaboutdata.h>   //::createAboutData()
#include <tdeaction.h>
#include <tdelocale.h>
#include <ktextedit.h>
#include <tdefiledialog.h>
#include <tdemessagebox.h>  //::start()
#include <tdeparts/genericfactory.h>
#include <kstatusbar.h>
#include <kstdaction.h>
#include <knuminput.h>
#include <kmdcodec.h>
#include <kurlrequester.h>
#include <tqfile.h>        //encodeName()
#include <tqtimer.h>       //postInit() hack
#include <tqvbox.h>
#include <tqsocket.h>
#include <tqmutex.h>
#include <tqeventloop.h>
#include <tqapplication.h>
#include <tqgroupbox.h>
#include <tqcheckbox.h>
#include <tqpushbutton.h>
#include <tqprogressbar.h>
#include <tqlistview.h>
#include <tdelistview.h>
#include <unistd.h>       //access()
#include <stdint.h>

#include <tqpainter.h>

#include "tracewidget.h"
#include "floatspinbox.h"
#include "layout.h"
#include "groupauthdlg.h"

#define NETWORK_COMM_TIMEOUT_MS 2500

enum connectionModes {
	ModeIdle						= 0,
	ModeUpdate						= 1,
	ModeDelete						= 2
};

enum connectionStates {
	ModeIdle_None						= 0,
	ModeIdle_StateTerminalListRequest			= 1,
	ModeIdle_StateProcessTerminalList			= 2,
	ModeIdle_StateWorkspaceListRequest			= 3,
	ModeIdle_StateProcessWorkspaceList			= 4,
	ModeIdle_StateTerminalAuthGroupListRequest		= 5,
	ModeIdle_StateProcessTerminalAuthGroupList		= 6,
	ModeIdle_StateWorkspaceServiceAuthGroupListRequest	= 7,
	ModeIdle_StateProcessWorkspaceServiceAuthGroupList	= 8,
	ModeIdle_StateDelay					= 9,
	ModeUpdate_TerminalServices				= 10,
	ModeUpdate_LaboratoryWorkspace				= 11,
	ModeUpdate_ProcessResponse				= 12,
	ModeDelete_TerminalServices				= 13,
	ModeDelete_LaboratoryWorkspace				= 14,
	ModeDelete_ProcessResponse				= 15
};

namespace RemoteLab {

typedef KParts::GenericFactory<RemoteLab::UserManagementPart> Factory;
#define CLIENT_LIBRARY "libremotelab_adminusermgmt"
K_EXPORT_COMPONENT_FACTORY(libremotelab_adminusermgmt, RemoteLab::Factory)

UserManagementPart::UserManagementPart(TQWidget *parentWidget, const char *widgetName, TQObject *parent, const char *name, const TQStringList&)
	: RemoteInstrumentPart( parent, name ), m_base(NULL), m_commHandlerState(0), m_connectionActiveAndValid(false), m_tickerState(0)
{
	// Initialize important base class variables
	m_clientLibraryName = CLIENT_LIBRARY;

	// Initialize mutex
	m_connectionMutex = new TQMutex(false);

	// Initialize kpart
	setInstance(Factory::instance());
	setWidget(new TQVBox(parentWidget, widgetName));

	// Create timers
	m_forcedUpdateTimer = new TQTimer(this);
	connect(m_forcedUpdateTimer, SIGNAL(timeout()), this, SLOT(mainEventLoop()));
	m_updateTimeoutTimer = new TQTimer(this);
	connect(m_updateTimeoutTimer, SIGNAL(timeout()), this, SLOT(mainEventLoop()));
	m_pingDelayTimer = new TQTimer(this);
	connect(m_pingDelayTimer, SIGNAL(timeout()), this, SLOT(mainEventLoop()));

	// Create widgets
	m_base = new UserManagementBase(widget());

	// Initialize widgets
	connect(m_base->ts_buttonAddGroup, SIGNAL(clicked()), this, SLOT(terminalServiceAddGroupButtonClicked()));
	connect(m_base->ts_buttonModifyGroup, SIGNAL(clicked()), this, SLOT(terminalServiceModifyGroupButtonClicked()));
	connect(m_base->ts_buttonDeleteGroup, SIGNAL(clicked()), this, SLOT(terminalServiceDeleteGroupButtonClicked()));
	connect(m_base->workspace_buttonAddGroup, SIGNAL(clicked()), this, SLOT(workspaceAddGroupButtonClicked()));
	connect(m_base->workspace_buttonModifyGroup, SIGNAL(clicked()), this, SLOT(workspaceModifyGroupButtonClicked()));
	connect(m_base->workspace_buttonDeleteGroup, SIGNAL(clicked()), this, SLOT(workspaceDeleteGroupButtonClicked()));
	connect(m_base->ts_list, SIGNAL(doubleClicked(TQListViewItem*, const TQPoint&, int)), this, SLOT(terminalServiceModifyGroupButtonClicked()));
	connect(m_base->workspace_list, SIGNAL(doubleClicked(TQListViewItem*, const TQPoint&, int)), this, SLOT(workspaceModifyGroupButtonClicked()));

	m_base->ts_list->setAllColumnsShowFocus(true);
	connect(m_base->ts_list, SIGNAL(selectionChanged()), this, SLOT(terminalServiceListSelect()));
	m_base->workspace_list->setAllColumnsShowFocus(true);
	connect(m_base->workspace_list, SIGNAL(selectionChanged()), this, SLOT(workspaceListSelect()));

	TQTimer::singleShot(0, this, TQT_SLOT(postInit()));
}

UserManagementPart::~UserManagementPart() {
	if (m_connectionMutex->locked()) {
		printf("[WARNING] Exiting when data transfer still in progress!\n\r"); fflush(stdout);
	}

	disconnectFromServer();
	delete m_connectionMutex;
}

void UserManagementPart::processLockouts() {
	TQWidget* mainWidget = widget();
	if (mainWidget) {
		if ((m_socket) && (m_socket->state() == TQSocket::Connected) && (connToServerState > 0) && (connToServerConnecting == false) && (m_commHandlerNextMode == ModeIdle)) {
			mainWidget->setEnabled(true);
		}
		else {
			mainWidget->setEnabled(false);
		}
	}

	if (m_connectionActiveAndValid == true) {
		m_base->ts_buttonAddGroup->setEnabled(true);
		m_base->workspace_buttonAddGroup->setEnabled(true);
	}
	else {
		m_base->ts_buttonAddGroup->setEnabled(false);
		m_base->workspace_buttonAddGroup->setEnabled(false);
	}

	if ((m_connectionActiveAndValid == true) && (m_base->ts_list->selectedItem())) {
		m_base->ts_buttonModifyGroup->setEnabled(true);
		m_base->ts_buttonDeleteGroup->setEnabled(true);
	}
	else {
		m_base->ts_buttonModifyGroup->setEnabled(false);
		m_base->ts_buttonDeleteGroup->setEnabled(false);
	}

	if ((m_connectionActiveAndValid == true) && (m_base->workspace_list->selectedItem())) {
		m_base->workspace_buttonModifyGroup->setEnabled(true);
		m_base->workspace_buttonDeleteGroup->setEnabled(true);
	}
	else {
		m_base->workspace_buttonModifyGroup->setEnabled(false);
		m_base->workspace_buttonDeleteGroup->setEnabled(false);
	}
}

void UserManagementPart::resizeToHint() {
	resize(widget()->sizeHint());
}

void UserManagementPart::connectionClosed() {
	closeURL();
}

void UserManagementPart::postInit() {
	setUsingFixedSize(false);
}

bool UserManagementPart::openURL(const KURL &url) {
	int ret;
	ret = connectToServer(url.url());
	processLockouts();
	return (ret != 0);
}

bool UserManagementPart::closeURL() {
	disconnectFromServer();
	m_url = KURL();
	return true;
}

void UserManagementPart::disconnectFromServerCallback() {
	m_forcedUpdateTimer->stop();
	m_updateTimeoutTimer->stop();
}

void UserManagementPart::connectionFinishedCallback() {
	connect(m_socket, SIGNAL(readyRead()), m_socket, SLOT(processPendingData()));
	m_socket->processPendingData();
	connect(m_socket, SIGNAL(newDataReceived()), this, SLOT(mainEventLoop()));
	m_tickerState = 0;
	m_commHandlerState = ModeIdle_StateTerminalListRequest;
	m_commHandlerMode = ModeIdle;
	m_commHandlerNextState = ModeIdle_None;
	m_commHandlerNextMode = ModeIdle;
	m_socket->setDataTimeout(NETWORK_COMM_TIMEOUT_MS);
	m_updateTimeoutTimer->start(NETWORK_COMM_TIMEOUT_MS, TRUE);
	processLockouts();
	mainEventLoop();
	return;
}

void UserManagementPart::connectionStatusChangedCallback() {
	processLockouts();
}

void UserManagementPart::terminalServiceAddGroupButtonClicked() {
	GroupPermissionsDialog groupauthdlg(0);
	groupauthdlg.setPermissionsSelectorLabel(i18n("Allowed Servers"));
	{
		TQKeyedStringList keyedServerList;
		for (TQStringList::Iterator it2 = m_availableTerminalServersList.begin(); it2 != m_availableTerminalServersList.end(); ++it2) {
			keyedServerList.append(TQKeyedStringPair(TQString("%1").arg(*it2), 0));
		}
		groupauthdlg.setAvailableServers(keyedServerList);
		groupauthdlg.setSessionLimit(0, true);
	}
	if (groupauthdlg.exec() == TQDialog::Accepted) {
		m_updateTerminalServiceAuthGroupList.clear();
		TerminalServiceAuthGroupType agt;
		agt.groupName = groupauthdlg.groupName();

		TQKeyedStringList selectedServers = groupauthdlg.selectedServers();
		agt.allowedServerNames.clear();
		for (TQKeyedStringList::Iterator it = selectedServers.begin(); it != selectedServers.end(); ++it) {
			agt.allowedServerNames.append((*it).first);
		}
		agt.maximumActiveSessionCount = groupauthdlg.sessionLimit();
		m_updateTerminalServiceAuthGroupList.append(agt);
		m_commHandlerNextState = ModeUpdate_TerminalServices;
		m_commHandlerNextMode = ModeUpdate;
	}

	processLockouts();
}

void UserManagementPart::terminalServiceModifyGroupButtonClicked() {
	TQListViewItem* item = m_base->ts_list->selectedItem();

	GroupPermissionsDialog groupauthdlg(0);
	groupauthdlg.setPermissionsSelectorLabel(i18n("Allowed Servers"));
	{
		TQKeyedStringList keyedServerList;
		for (TQStringList::Iterator it2 = m_availableTerminalServersList.begin(); it2 != m_availableTerminalServersList.end(); ++it2) {
			keyedServerList.append(TQKeyedStringPair(TQString("%1").arg(*it2), 0));
		}
		groupauthdlg.setAvailableServers(keyedServerList);
	}
	TerminalServiceAuthGroupList::iterator it = m_terminalServiceAuthGroupList.findByName(item->text(0));
	if (it != m_terminalServiceAuthGroupList.end()) {
		TQKeyedStringList keyedServerList;
		for (TQStringList::Iterator it2 = (*it).allowedServerNames.begin(); it2 != (*it).allowedServerNames.end(); ++it2) {
			keyedServerList.append(TQKeyedStringPair(TQString("%1").arg(*it2), 0));
		}
		groupauthdlg.setSelectedServers(keyedServerList);
		groupauthdlg.setSessionLimit((*it).maximumActiveSessionCount, true);
	}
	groupauthdlg.setGroupName(item->text(0), false);
	if (groupauthdlg.exec() == TQDialog::Accepted) {
		m_updateTerminalServiceAuthGroupList.clear();
		TerminalServiceAuthGroupType agt;
		agt.groupName = groupauthdlg.groupName();
		TQKeyedStringList selectedServers = groupauthdlg.selectedServers();
		agt.allowedServerNames.clear();
		for (TQKeyedStringList::Iterator it = selectedServers.begin(); it != selectedServers.end(); ++it) {
			agt.allowedServerNames.append((*it).first);
		}
		agt.maximumActiveSessionCount = groupauthdlg.sessionLimit();
		m_updateTerminalServiceAuthGroupList.append(agt);
		m_commHandlerNextState = ModeUpdate_TerminalServices;
		m_commHandlerNextMode = ModeUpdate;
	}

	processLockouts();
}

void UserManagementPart::terminalServiceDeleteGroupButtonClicked() {
	TQListViewItem* item = m_base->ts_list->selectedItem();

	if (KMessageBox::warningYesNo(0, i18n("<qt><b>You are attempting to delete the group '%1'</b><br>You will not be able to undo this action<p>Do you wish to proceed?</qt>").arg(item->text(0)), i18n("Delete group?")) == KMessageBox::Yes) {
		m_updateTerminalServiceAuthGroupList.clear();
		TerminalServiceAuthGroupType agt;
		agt.groupName = item->text(0);
		m_updateTerminalServiceAuthGroupList.append(agt);
		m_commHandlerNextState = ModeDelete_TerminalServices;
		m_commHandlerNextMode = ModeDelete;
	}

	processLockouts();
}

void UserManagementPart::workspaceAddGroupButtonClicked() {
	GroupPermissionsDialog groupauthdlg(0);
	groupauthdlg.setPermissionsSelectorLabel(i18n("Allowed Stations"));
	groupauthdlg.setSessionLimit(-1, false);
	{
		TQKeyedStringList friendlyStations;
		for (StationList::iterator it = m_availableWorkspaceStationsList.begin(); it != m_availableWorkspaceStationsList.end(); ++it) {
			friendlyStations.append(TQKeyedStringPair(TQString("%1").arg((*it).name), (*it).id));
		}
		groupauthdlg.setAvailableServers(friendlyStations);
	}
	if (groupauthdlg.exec() == TQDialog::Accepted) {
		m_updateWorkspaceServiceAuthGroupList.clear();
		WorkspaceServiceAuthGroupType agt;
		agt.groupName = groupauthdlg.groupName();
		TQKeyedStringList selectedStations = groupauthdlg.selectedServers();
		agt.allowedStationIDs.clear();
		for (TQKeyedStringList::Iterator it = selectedStations.begin(); it != selectedStations.end(); ++it) {
			agt.allowedStationIDs.append((*it).second);
		}
		m_updateWorkspaceServiceAuthGroupList.append(agt);
		m_commHandlerNextState = ModeUpdate_LaboratoryWorkspace;
		m_commHandlerNextMode = ModeUpdate;
	}

	processLockouts();
}

void UserManagementPart::workspaceModifyGroupButtonClicked() {
	TQListViewItem* item = m_base->workspace_list->selectedItem();

	GroupPermissionsDialog groupauthdlg(0);
	groupauthdlg.setPermissionsSelectorLabel(i18n("Allowed Stations"));
	groupauthdlg.setSessionLimit(-1, false);
	{
		TQKeyedStringList friendlyStations;
		for (StationList::iterator it = m_availableWorkspaceStationsList.begin(); it != m_availableWorkspaceStationsList.end(); ++it) {
			friendlyStations.append(TQKeyedStringPair(TQString("%1").arg((*it).name), (*it).id));
		}
		groupauthdlg.setAvailableServers(friendlyStations);
	}
	WorkspaceServiceAuthGroupList::iterator it = m_workspaceAuthGroupList.findByName(item->text(0));
	if (it != m_workspaceAuthGroupList.end()) {
		TQKeyedStringList friendlyStations;
		for (TQInt32List::Iterator it2 = (*it).allowedStationIDs.begin(); it2 != (*it).allowedStationIDs.end(); ++it2) {
			int stationID = *it2;
			StationList::iterator stationInfoIterator = m_availableWorkspaceStationsList.findByID(stationID);
			if (stationInfoIterator != m_availableWorkspaceStationsList.end()) {
				friendlyStations.append(TQKeyedStringPair(TQString("%1").arg((*stationInfoIterator).name), (*stationInfoIterator).id));
			}
			else {
				friendlyStations.append(TQKeyedStringPair(TQString("<unknown>"), stationID));
			}
		}
		groupauthdlg.setSelectedServers(friendlyStations);
	}
	groupauthdlg.setGroupName(item->text(0), false);
	if (groupauthdlg.exec() == TQDialog::Accepted) {
		m_updateWorkspaceServiceAuthGroupList.clear();
		WorkspaceServiceAuthGroupType agt;
		agt.groupName = groupauthdlg.groupName();
		TQKeyedStringList selectedStations = groupauthdlg.selectedServers();
		agt.allowedStationIDs.clear();
		for (TQKeyedStringList::Iterator it = selectedStations.begin(); it != selectedStations.end(); ++it) {
			agt.allowedStationIDs.append((*it).second);
		}
		m_updateWorkspaceServiceAuthGroupList.append(agt);
		m_commHandlerNextState = ModeUpdate_LaboratoryWorkspace;
		m_commHandlerNextMode = ModeUpdate;
	}

	processLockouts();
}

void UserManagementPart::workspaceDeleteGroupButtonClicked() {
	TQListViewItem* item = m_base->workspace_list->selectedItem();

	if (KMessageBox::warningYesNo(0, i18n("<qt><b>You are attempting to delete the group '%1'</b><br>You will not be able to undo this action<p>Do you wish to proceed?</qt>").arg(item->text(0)), i18n("Delete group?")) == KMessageBox::Yes) {
		m_updateWorkspaceServiceAuthGroupList.clear();
		WorkspaceServiceAuthGroupType agt;
		agt.groupName = item->text(0);
		agt.allowedStationIDs.clear();
		m_updateWorkspaceServiceAuthGroupList.append(agt);
		m_commHandlerNextState = ModeDelete_LaboratoryWorkspace;
		m_commHandlerNextMode = ModeDelete;
	}

	processLockouts();
}

void UserManagementPart::terminalServiceListSelect() {
	// Highlight the matching user session in the workspace list,
	// or deselect all workspace list items if no matching workspace session is found

	TQListViewItem* item = m_base->ts_list->selectedItem();
	if (item) {
		TQString username = item->text(1);

		TQListViewItemIterator it;
		it = TQListViewItemIterator(m_base->workspace_list);
		bool found = false;
		TQListViewItem* item2 = NULL;
		while (it.current()) {
			item2 = *it;
			if (item2->text(1) == username) {
				found = true;
				break;
			}
			++it;
		}
		if (found) {
			m_base->workspace_list->setCurrentItem(item2);
		}
		else {
			m_base->workspace_list->clearSelection();
		}
	}
	else {
		m_base->workspace_list->clearSelection();
	}

	processLockouts();
}

void UserManagementPart::workspaceListSelect() {
	// Highlight the matching user session in the terminal services list,
	// or deselect all workspace list items if no matching terminal services session is found

	TQListViewItem* item = m_base->workspace_list->selectedItem();
	if (item) {
		TQString username = item->text(1);

		TQListViewItemIterator it;
		it = TQListViewItemIterator(m_base->ts_list);
		bool found = false;
		TQListViewItem* item2 = NULL;
		while (it.current()) {
			item2 = *it;
			if (item2->text(1) == username) {
				found = true;
				break;
			}
			++it;
		}
		if (found) {
			m_base->ts_list->setCurrentItem(item2);
		}
		else {
			m_base->ts_list->clearSelection();
		}
	}
	else {
		m_base->ts_list->clearSelection();
	}

	processLockouts();
}

#define UPDATEDISPLAY_TIMEOUT		m_connectionActiveAndValid = false;														\
					m_tickerState = 0;																\
					m_commHandlerState = ModeIdle_StateTerminalListRequest;												\
					m_commHandlerMode = ModeIdle;															\
					m_commHandlerNextState = ModeIdle_None;														\
					m_commHandlerNextMode = ModeIdle;														\
					m_socket->clearIncomingData();															\
					setStatusMessage(i18n("Server ping timeout.  Please verify the status of your network connection."));						\
					m_updateTimeoutTimer->start(NETWORK_COMM_TIMEOUT_MS, TRUE);											\
					m_connectionMutex->unlock();															\
					return;

#define SET_WATCHDOG_TIMER		if (!m_updateTimeoutTimer->isActive()) m_updateTimeoutTimer->start(NETWORK_COMM_TIMEOUT_MS, TRUE);
#define PAT_WATCHDOG_TIMER		m_updateTimeoutTimer->stop(); m_updateTimeoutTimer->start(NETWORK_COMM_TIMEOUT_MS, TRUE);

#define SET_NEXT_STATE(x)		m_commHandlerState = x;	

#define EXEC_NEXT_STATE_IMMEDIATELY	m_forcedUpdateTimer->start(0, TRUE);

void UserManagementPart::setTickerMessage(TQString message) {
	m_connectionActiveAndValid = true;
	TQString tickerChar;
	switch (m_tickerState) {
		case 0:
			tickerChar = "-";
			break;
		case 1:
			tickerChar = "\\";
			break;
		case 2:
			tickerChar = "|";
			break;
		case 3:
			tickerChar = "/";
			break;
	}
	setStatusMessage(message + TQString("... %1").arg(tickerChar));
	m_tickerState++;
	if (m_tickerState > 3) {
		m_tickerState = 0;
	}
}

void UserManagementPart::mainEventLoop() {
	TQDataStream ds(m_socket);
	ds.setPrintableData(true);

	if (!m_connectionMutex->tryLock()) {
		EXEC_NEXT_STATE_IMMEDIATELY
		return;
	}

	if (m_socket) {
		if (m_commHandlerMode == ModeIdle) {
			// Normal operation
			switch (m_commHandlerState) {
				case ModeIdle_StateTerminalListRequest:
					// Get status of remote system
					// Clear buffers to synchronize frames in case of data corruption
					m_socket->clearIncomingData();
					ds << TQString("LIST");
					ds << TQString("TERMINALS");
					m_socket->writeEndOfFrame();

					SET_NEXT_STATE(ModeIdle_StateProcessTerminalList)
					break;
				case ModeIdle_StateProcessTerminalList:
					// Get all data
					if (m_socket->canReadFrame()) {
						PAT_WATCHDOG_TIMER

						TQString status;
						ds >> status;
						if (status == "OK") {
							ds >> m_availableTerminalServersList;
							m_socket->clearFrameTail();

							setTickerMessage(i18n("Connected"));
						}

						SET_NEXT_STATE(ModeIdle_StateWorkspaceListRequest);
						EXEC_NEXT_STATE_IMMEDIATELY
					}
					else {
						if (!m_updateTimeoutTimer->isActive()) {
							UPDATEDISPLAY_TIMEOUT
						}
					}
					break;
				case ModeIdle_StateWorkspaceListRequest:
					// Get status of remote system
					// Clear buffers to synchronize frames in case of data corruption
					m_socket->clearIncomingData();
					ds << TQString("LIST");
					ds << TQString("WORKSPACES");
					m_socket->writeEndOfFrame();

					SET_NEXT_STATE(ModeIdle_StateProcessWorkspaceList)
					EXEC_NEXT_STATE_IMMEDIATELY
					break;
				case ModeIdle_StateProcessWorkspaceList:
					// Get all data
					if (m_socket->canReadFrame()) {
						PAT_WATCHDOG_TIMER

						TQString status;
						ds >> status;
						if (status == "OK") {
							ds >> m_availableWorkspaceStationsList;
							m_socket->clearFrameTail();

							setTickerMessage(i18n("Connected"));
						}

						SET_NEXT_STATE(ModeIdle_StateTerminalAuthGroupListRequest);
						EXEC_NEXT_STATE_IMMEDIATELY
					}
					else {
						if (!m_updateTimeoutTimer->isActive()) {
							UPDATEDISPLAY_TIMEOUT
						}
					}
					break;
				case ModeIdle_StateTerminalAuthGroupListRequest:
					// Get status of remote system
					// Clear buffers to synchronize frames in case of data corruption
					m_socket->clearIncomingData();
					ds << TQString("LISTGROUPACCESS");
					ds << TQString("TERMINALS");
					m_socket->writeEndOfFrame();

					SET_NEXT_STATE(ModeIdle_StateProcessTerminalAuthGroupList)
					break;
				case ModeIdle_StateProcessTerminalAuthGroupList:
					// Get all data
					if (m_socket->canReadFrame()) {
						PAT_WATCHDOG_TIMER

						TQString status;
						ds >> status;
						if (status == "OK") {
							ds >> m_terminalServiceAuthGroupList;
							m_socket->clearFrameTail();

							TerminalServiceAuthGroupList::iterator it;
							TQListViewItemIterator it2;
	
							for (it = m_terminalServiceAuthGroupList.begin(); it != m_terminalServiceAuthGroupList.end(); ++it) {
								TerminalServiceAuthGroupType info = *it;
								it2 = TQListViewItemIterator(m_base->ts_list);
								bool found = false;
								TQListViewItem* item = NULL;
								while (it2.current()) {
									item = *it2;
									if (info.groupName == item->text(0)) {
										found = true;
										break;
									}
									++it2;
								}
								if (!found) {
									item = new TQListViewItem(m_base->ts_list);
								}
								item->setText(0, TQString("%1").arg(info.groupName));
								item->setText(1, info.allowedServerNames.join(", "));
								item->setText(2, TQString("%1").arg(info.maximumActiveSessionCount));
							}
							it2 = TQListViewItemIterator(m_base->ts_list);
							while (it2.current()) {
								TQListViewItem* item = *it2;
								bool found = false;
								for (it = m_terminalServiceAuthGroupList.begin(); it != m_terminalServiceAuthGroupList.end(); ++it) {
									TerminalServiceAuthGroupType info = *it;
									if (info.groupName == item->text(0)) {
										found = true;
									}
								}
								if (!found) {
									delete item;
								}
								++it2;
							}
	
							setTickerMessage(i18n("Connected"));
						}

						SET_NEXT_STATE(ModeIdle_StateWorkspaceServiceAuthGroupListRequest);
						EXEC_NEXT_STATE_IMMEDIATELY
					}
					else {
						if (!m_updateTimeoutTimer->isActive()) {
							UPDATEDISPLAY_TIMEOUT
						}
					}
					break;
				case ModeIdle_StateWorkspaceServiceAuthGroupListRequest:
					// Get status of remote system
					// Clear buffers to synchronize frames in case of data corruption
					m_socket->clearIncomingData();
					ds << TQString("LISTGROUPACCESS");
					ds << TQString("WORKSPACES");
					m_socket->writeEndOfFrame();

					SET_NEXT_STATE(ModeIdle_StateProcessWorkspaceServiceAuthGroupList)
					EXEC_NEXT_STATE_IMMEDIATELY
					break;
				case ModeIdle_StateProcessWorkspaceServiceAuthGroupList:
					// Get all data
					if (m_socket->canReadFrame()) {
						PAT_WATCHDOG_TIMER

						TQString status;
						ds >> status;
						if (status == "OK") {
							ds >> m_workspaceAuthGroupList;
							m_socket->clearFrameTail();

							WorkspaceServiceAuthGroupList::iterator it;
							TQListViewItemIterator it2;
	
							for (it = m_workspaceAuthGroupList.begin(); it != m_workspaceAuthGroupList.end(); ++it) {
								WorkspaceServiceAuthGroupType info = *it;
								it2 = TQListViewItemIterator(m_base->workspace_list);
								bool found = false;
								TQListViewItem* item = NULL;
								while (it2.current()) {
									item = *it2;
									if (info.groupName == item->text(0)) {
										found = true;
										break;
									}
									++it2;
								}
								if (!found) {
									item = new TQListViewItem(m_base->workspace_list);
								}
								item->setText(0, TQString("%1").arg(info.groupName));
								TQString allowedStationsText;
								TQInt32List::iterator it3;
								for (it3 = info.allowedStationIDs.begin(); it3 != info.allowedStationIDs.end(); ++it3) {
									StationList::iterator stationInfoIterator = m_availableWorkspaceStationsList.findByID(*it3);
									if (stationInfoIterator != m_availableWorkspaceStationsList.end()) {
										if (allowedStationsText != "") {
											allowedStationsText.append(", ");
										}
										allowedStationsText.append((*stationInfoIterator).name);
									}
								}
								item->setText(1, allowedStationsText);
							}
							it2 = TQListViewItemIterator(m_base->workspace_list);
							while (it2.current()) {
								TQListViewItem* item = *it2;
								bool found = false;
								for (it = m_workspaceAuthGroupList.begin(); it != m_workspaceAuthGroupList.end(); ++it) {
									WorkspaceServiceAuthGroupType info = *it;
									if (info.groupName == item->text(0)) {
										found = true;
									}
								}
								if (!found) {
									delete item;
								}
								++it2;
							}

							setTickerMessage(i18n("Connected"));
						}

						if (m_commHandlerState == ModeIdle_StateProcessWorkspaceServiceAuthGroupList) {
							m_pingDelayTimer->start(250, TRUE);
							SET_NEXT_STATE(ModeIdle_StateDelay);
						}
					}
					else {
						if (!m_updateTimeoutTimer->isActive()) {
							UPDATEDISPLAY_TIMEOUT
						}
					}
					break;
				case ModeIdle_StateDelay:
					if (m_commHandlerNextMode == ModeIdle) {
						// Let the client and server rest for a bit to lower CPU/network overhead
						if (!m_pingDelayTimer->isActive()) {
							EXEC_NEXT_STATE_IMMEDIATELY
							// Execute query on next event loop
							SET_NEXT_STATE(ModeIdle_StateTerminalListRequest);
						}
						PAT_WATCHDOG_TIMER
					}
					else {
						m_commHandlerMode = m_commHandlerNextMode;
						SET_NEXT_STATE(m_commHandlerNextState);
						EXEC_NEXT_STATE_IMMEDIATELY
						m_commHandlerNextState = ModeIdle_None;
						m_commHandlerNextMode = ModeIdle;
					}
					break;
			}
		}
		else if (m_commHandlerMode == ModeUpdate) {
			switch (m_commHandlerState) {
				case ModeUpdate_TerminalServices:
					ds << TQString("SETGROUPACCESS");
					ds << TQString("TERMINALS");
					ds << m_updateTerminalServiceAuthGroupList;
					m_socket->writeEndOfFrame();

					SET_NEXT_STATE(ModeUpdate_ProcessResponse)
					EXEC_NEXT_STATE_IMMEDIATELY
					break;
				case ModeUpdate_LaboratoryWorkspace:
					ds << TQString("SETGROUPACCESS");
					ds << TQString("WORKSPACES");
					ds << m_updateWorkspaceServiceAuthGroupList;
					m_socket->writeEndOfFrame();

					SET_NEXT_STATE(ModeUpdate_ProcessResponse)
					EXEC_NEXT_STATE_IMMEDIATELY
					break;
				case ModeUpdate_ProcessResponse:
					// Get all data
					if (m_socket->canReadFrame()) {
						PAT_WATCHDOG_TIMER

						TQString status;
						WorkspaceServiceStatusList list;
						ds >> status;

						if (status != "OK") {
							// Command failed!
							KMessageBox::error(0, i18n("Command Failure"), i18n("Unable to execute command!"));
						}

						m_commHandlerMode = ModeIdle;
						SET_NEXT_STATE(ModeIdle_StateTerminalListRequest);
						EXEC_NEXT_STATE_IMMEDIATELY
					}
					else {
						if (!m_updateTimeoutTimer->isActive()) {
							UPDATEDISPLAY_TIMEOUT
						}
					}
					break;
			}
		}
		else if (m_commHandlerMode == ModeDelete) {
			switch (m_commHandlerState) {
				case ModeDelete_TerminalServices:
					ds << TQString("DELETEGROUPACCESS");
					ds << TQString("TERMINALS");
					ds << m_updateTerminalServiceAuthGroupList;
					m_socket->writeEndOfFrame();

					SET_NEXT_STATE(ModeDelete_ProcessResponse)
					EXEC_NEXT_STATE_IMMEDIATELY
					break;
				case ModeDelete_LaboratoryWorkspace:
					ds << TQString("DELETEGROUPACCESS");
					ds << TQString("WORKSPACES");
					ds << m_updateWorkspaceServiceAuthGroupList;
					m_socket->writeEndOfFrame();

					SET_NEXT_STATE(ModeDelete_ProcessResponse)
					EXEC_NEXT_STATE_IMMEDIATELY
					break;
				case ModeDelete_ProcessResponse:
					// Get all data
					if (m_socket->canReadFrame()) {
						PAT_WATCHDOG_TIMER

						TQString status;
						WorkspaceServiceStatusList list;
						ds >> status;

						if (status != "OK") {
							// Command failed!
							KMessageBox::error(0, i18n("Command Failure"), i18n("Unable to execute command!"));
						}

						m_commHandlerMode = ModeIdle;
						SET_NEXT_STATE(ModeIdle_StateTerminalListRequest);
						EXEC_NEXT_STATE_IMMEDIATELY
					}
					else {
						if (!m_updateTimeoutTimer->isActive()) {
							UPDATEDISPLAY_TIMEOUT
						}
					}
					break;
			}
		}

		processLockouts();
		SET_WATCHDOG_TIMER
	}
	else {
		SET_NEXT_STATE(ModeIdle_StateTerminalListRequest);
		m_commHandlerMode = ModeIdle;
	}

	m_connectionMutex->unlock();
}

TDEAboutData* UserManagementPart::createAboutData() {
	return new TDEAboutData( APP_NAME, I18N_NOOP( APP_PRETTYNAME ), APP_VERSION );
}

} //namespace RemoteLab

#include "part.moc"
