/*
 * Remote Laboratory Instrumentation Server
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * (c) 2009 Timothy Pearson
 * Raptor Engineering
 * http://www.raptorengineeringinc.com
 */

#include <ctype.h>
#include <ctime>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <getopt.h>
#include "parameters.h"
#include "gpib_functions.h"
#include "gpib/ib.h"

extern char falpha[1024];
double commanalyzer_raw_trace_data[1024];

unsigned long commanalyzerTraceLength (const char * commanalyzerType) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		return 417;
	}
	else {
		return 1;
	}
}

int commanalyzer_set_date(struct tm * datetime, const char * commanalyzerType, int gpibDevice) {
	char datebuffer [80];
	strftime(datebuffer,80,"CONF:DATE %m%d%y",datetime);

	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting date on communications analyzer\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", datebuffer);
			#endif
			if (gpib_write(gpibDevice, datebuffer) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_time(struct tm * datetime, const char * commanalyzerType, int gpibDevice) {
	char timebuffer [80];
	strftime(timebuffer,80,"CONF:TIME +%H.%M",datetime);	// FIXME wrong format

	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting time on communications analyzer\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", timebuffer);
			#endif
			if (gpib_write(gpibDevice, timebuffer) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_switch_to_spectrum_analyzer_mode(const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting communications analyzer to spectrum anayzer mode\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"DISP SAN");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_lock_screen (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Locking communications analyzer screen\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SPEC:DISP 'LOCKED'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_spectrum_analyzer_set_generator_mode_tracking (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting spectrum analyzer generator to tracking mode\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:RFG 'TRACK'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_spectrum_analyzer_set_generator_mode_fixed (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting spectrum analyzer generator to fixed mode\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:RFG 'FIXED'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_spectrum_analyzer_set_rf_input_dedicated (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting spectrum analyzer RF input to dedicated connector\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:INP 'ANT'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_spectrum_analyzer_set_rf_input_muxed (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting spectrum analyzer RF input to multiplexed connector\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:INP 'RF IN'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_spectrum_analyzer_set_generator_output_dedicated (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting spectrum analyzer generator output to dedicated connector\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:TGEN:DEST 'DUPL'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_spectrum_analyzer_set_generator_output_muxed (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting spectrum analyzer generator output to multiplexed connector\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:TGEN:DEST 'RF OUT'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_get_spectrum_analyzer_trace(const char * commanalyzerType, int gpibDevice) {
	int max_num_bytes = 0;

	char segarray[4194304];
	char floatstring[1024];
	long array_pointer;
	long ai;
	long left_char;
	long right_char;

	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		// Send request
		printf("[INFO] Getting spectrum analyzer trace [Stage 1]\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"MEAS:SAN:TRACE?");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				max_num_bytes = (commanalyzerTraceLength(commanalyzerType)*24);	// Request more bytes than are possible to ensure no bytes are left behind
			}
			else {
				return 2;
			}
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif
	
		ibtmo(gpibDevice, T30s);
		ibeos(gpibDevice, 0x0);
	
		ai = gpib_read_array(gpibDevice, max_num_bytes, segarray);
		if (ai == -1) {
			return 1;
		}
		else {
			if (strcmp("HP8924C", commanalyzerType) == 0) {
				left_char = 0;
				right_char = 0;
				array_pointer = 0;
				while (left_char < ai) {
					for (right_char=left_char;right_char<ai;right_char++) {
						if (segarray[right_char] == ',') {
							break;
						}
					}
					strncpy(floatstring, segarray+left_char, right_char-left_char);
					floatstring[right_char-left_char] = 0;
					commanalyzer_raw_trace_data[array_pointer] = atof(floatstring);
					array_pointer++;
					left_char = right_char+1;
				}
			}
			else {
				return 2;
			}
		}
	
		ibtmo(gpibDevice, T10s);
	
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", array_pointer);
		#endif

		return 0;
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_center_frequency(float desired_frequency, const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		printf("[INFO] Setting spectrum analyzer center frequency to %f\n\r", desired_frequency);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:CFR %E", desired_frequency);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_frequency_span(float desired_frequency, const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		printf("[INFO] Setting spectrum analyzer span frequency to %f\n\r", desired_frequency);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:SPAN %E", desired_frequency);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_input_attenuation(float desired_attenuation, const char * commanalyzerType, int gpibDevice) {
	int instrument_att;

	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			// Only three attenuation values are valid
			if (desired_attenuation < 10) {
				instrument_att = 0;
			}
			else if (desired_attenuation < 30) {
				instrument_att = 20;
			}
			else {
				instrument_att = 40;
			}
		}
		else {
			return 1;
		}
		printf("[INFO] Setting spectrum analyzer attenuation to %d dB\n\r", instrument_att);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:ATT '%d DB'", instrument_att);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_scale(float desired_scale, const char * commanalyzerType, int gpibDevice) {
	int instrument_scale;

	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			// Only three scale values are valid
			if (desired_scale < 1.5) {
				instrument_scale = 1;
			}
			else if (desired_scale < 6) {
				instrument_scale = 2;
			}
			else {
				instrument_scale = 10;
			}
		}
		else {
			return 1;
		}
		printf("[INFO] Setting spectrum analyzer scale to %d dB/div\n\r", instrument_scale);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:DISP:SCAL '%d DB/DIV'", instrument_scale);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_input_attenuator_mode_auto( const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		printf("[INFO] Setting spectrum analyzer attenuation mode to automatic\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:ATT:MODE 'AUTO'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_input_attenuator_mode_fixed( const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		printf("[INFO] Setting spectrum analyzer attenuation mode to fixed\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:ATT:MODE 'HOLD'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_generator_power(float desired_power, const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		printf("[INFO] Setting spectrum analyzer generator power to %f\n\r", desired_power);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:TGEN:AMPL %E", desired_power);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_generator_frequency(float desired_frequency, const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		printf("[INFO] Setting spectrum analyzer generator frequency to %f\n\r", desired_frequency);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:TGEN:OFR %E", desired_frequency);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_spectrum_analyzer_set_generator_sweep_ascending (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting spectrum analyzer generator output sweep to ascending\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:TGEN:SWE 'NORM'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_spectrum_analyzer_set_generator_sweep_descending (const char * commanalyzerType, int gpibDevice) {
	if (strcmp("HP8924C", commanalyzerType) == 0) {
		printf("[INFO] Setting spectrum analyzer generator output sweep to descending\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:TGEN:SWE 'INVERT'");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_trace_averaging(float desired_avg_samples, const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		int instrument_samples;
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			// Only certain values are valid, and I am interpreting 0 as no pk/avg and -1 as peak hold
			instrument_samples = desired_avg_samples;
			if (desired_avg_samples > 5) {
				if (desired_avg_samples < 5) {
					instrument_samples = desired_avg_samples;
				}
				else if (desired_avg_samples < 7) {
					instrument_samples = 5;
				}
				else if (desired_avg_samples < 15) {
					instrument_samples = 10;
				}
				else if (desired_avg_samples < 35) {
					instrument_samples = 20;
				}
				else if (desired_avg_samples < 75) {
					instrument_samples = 50;
				}
				else {
					instrument_samples = 100;
				}
			}
		}
		else {
			return 1;
		}
		printf("[INFO] Setting spectrum analyzer generator trace averaging to %d\n\r", instrument_samples);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			if (instrument_samples > 0) {
				sprintf(falpha,"SAN:TRAC:MHOL 'AVG %d'", instrument_samples);
			}
			else if (instrument_samples == -1) {
				sprintf(falpha,"SAN:TRAC:MHOL 'PK HOLD'");
			}
			else {
				sprintf(falpha,"SAN:TRAC:MHOL 'NO PK/AVG'");
			}
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_set_spectrum_analyzer_reference_power_level(float desired_reflevel, const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		printf("[INFO] Setting spectrum analyzer generator reference level to %f dBm\n\r", desired_reflevel);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:RLEV %E", desired_reflevel);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

int commanalyzer_get_spectrum_analyzer_number_of_vertical_divisions( const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		return 8;
	}
	else {
		return -1;
	}
}

int commanalyzer_get_spectrum_analyzer_number_of_horizontal_divisions( const char * commanalyzerType, int gpibDevice) {
	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		return 10;
	}
	else {
		return -1;
	}
}

double commanalyzer_get_spectrum_analyzer_reference_power_level(double * retval, const char * commanalyzerType, int gpibDevice) {
	char segarray[4194304];
	char floatstring[1024];
	long array_pointer;
	long ai;
	long left_char;
	long right_char;

	int max_num_bytes = 0;

	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		// Send request
		printf("[INFO] Getting spectrum analyzer reference power level\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:RLEV?");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif
	
		ibtmo(gpibDevice, T10s);
		ibeos(gpibDevice, 0x0);
	
		ai = gpib_read_array(gpibDevice, max_num_bytes, segarray);
		if (ai == -1) {
			return 1;
		}
		else {
			if (strcmp("HP8924C", commanalyzerType) == 0) {
				segarray[ai] = 0;
				*retval = atof(segarray);
			}
			else {
				return 2;
			}
		}
	
		ibtmo(gpibDevice, T10s);
	
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

double commanalyzer_get_spectrum_analyzer_scale(double * retval, const char * commanalyzerType, int gpibDevice) {
	char segarray[4194304];
	char floatstring[1024];
	long array_pointer;
	long ai;
	long left_char;
	long right_char;

	int max_num_bytes = 0;

	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		// Send request
		printf("[INFO] Getting spectrum analyzer scale\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:DISP:SCAL?");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif
	
		ibtmo(gpibDevice, T10s);
		ibeos(gpibDevice, 0x0);
	
		ai = gpib_read_array(gpibDevice, max_num_bytes, segarray);
		if (ai == -1) {
			return 1;
		}
		else {
			if (strcmp("HP8924C", commanalyzerType) == 0) {
				left_char = 1;
				right_char = 0;
				array_pointer = 0;
				for (right_char=left_char;right_char<ai;right_char++) {
					if (segarray[right_char] == ' ') {
						break;
					}
				}
				strncpy(floatstring, segarray+left_char, right_char-left_char);
				floatstring[right_char-left_char] = 0;
				*retval = atof(floatstring);
			}
			else {
				return 2;
			}
		}
	
		ibtmo(gpibDevice, T10s);
	
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

int commanalyzer_set_display_brightness(float percent, const char * commanalyzerType, int gpibDevice) {
	int instrument_scale;

	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			instrument_scale = ((percent/100.0)*7.0)+1;
		}
		else {
			return 1;
		}
		printf("[INFO] Setting display brightness to %d\n\r", instrument_scale);
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"CONF:INT %d", instrument_scale);
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				return 0;
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}
	}
	else {
		return 1;
	}
}

double commanalyzer_get_spectrum_analyzer_center_frequency(double * retval, const char * commanalyzerType, int gpibDevice) {
	char segarray[4194304];
	char floatstring[1024];
	long array_pointer;
	long ai;
	long left_char;
	long right_char;

	int max_num_bytes = 0;

	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		// Send request
		printf("[INFO] Getting spectrum analyzer center frequency\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:CFR?");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
			}
			else {
				return 2;
			}
		}
		else {
			return 1;
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif
	
		ibtmo(gpibDevice, T10s);
		ibeos(gpibDevice, 0x0);
	
		ai = gpib_read_array(gpibDevice, max_num_bytes, segarray);
		if (ai == -1) {
			return 1;
		}
		else {
			if (strcmp("HP8924C", commanalyzerType) == 0) {
				left_char = 1;
				right_char = 0;
				array_pointer = 0;
				for (right_char=left_char;right_char<ai;right_char++) {
					if (segarray[right_char] == ' ') {
						break;
					}
				}
				strncpy(floatstring, segarray+left_char, right_char-left_char);
				floatstring[right_char-left_char] = 0;
				*retval = atof(floatstring);
			}
			else {
				return 2;
			}
		}
	
		ibtmo(gpibDevice, T10s);
	
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}

double commanalyzer_get_spectrum_analyzer_span(double * retval, const char * commanalyzerType, int gpibDevice) {
	char segarray[4194304];
	char floatstring[1024];
	long array_pointer;
	long ai;
	long left_char;
	long right_char;

	int max_num_bytes = 0;

	if ((strcmp("HP8924C", commanalyzerType) == 0)) {
		// Send request
		printf("[INFO] Getting spectrum analyzer frequency span\n\r");
		if (strcmp("HP8924C", commanalyzerType) == 0) {
			sprintf(falpha,"SAN:SPAN?");
			#ifdef ENABLE_EXTRA_DEBUGGING
			printf("[DEBG] Writing: %s\n\r", falpha);
			#endif
			if (gpib_write(gpibDevice, falpha) == 0) {
				max_num_bytes = 24;	// Request more bytes than are possible to ensure no bytes are left behind
			}
			else {
				return 2;
			}
		}

		// Read response
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Trying to read %i bytes from GPIB device...\n", max_num_bytes);
		#endif
	
		ibtmo(gpibDevice, T10s);
		ibeos(gpibDevice, 0x0);
	
		ai = gpib_read_array(gpibDevice, max_num_bytes, segarray);
		if (ai == -1) {
			return 1;
		}
		else {
			if (strcmp("HP8924C", commanalyzerType) == 0) {
				left_char = 1;
				right_char = 0;
				array_pointer = 0;
				for (right_char=left_char;right_char<ai;right_char++) {
					if (segarray[right_char] == ' ') {
						break;
					}
				}
				strncpy(floatstring, segarray+left_char, right_char-left_char);
				floatstring[right_char-left_char] = 0;
				*retval = atof(floatstring);
			}
			else {
				return 2;
			}
		}
	
		ibtmo(gpibDevice, T10s);
	
		#ifdef ENABLE_EXTRA_DEBUGGING
		printf("[DEBG] Read %li bytes from GPIB device\n", ai);
		#endif

		return 0;
	}
	else {
		return -1;
	}
}