/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <algorithm>
#include <cstring>
#include <iterator>
#include <vector>

#include <boost/archive/iterators/binary_from_base64.hpp>
#include <boost/archive/iterators/remove_whitespace.hpp>
#include <boost/archive/iterators/transform_width.hpp>
#include <boost/range/iterator_range.hpp>

#include <libwpd/WPXBinaryData.h>
#include <libwpd/WPXDocumentInterface.h>
#include <libwpd/WPXPropertyList.h>
#include <libwpd/WPXString.h>

#include "FB2Content.h"
#include "FB2ContentCollector.h"
#include "FB2ContentMap.h"
#include "FB2Style.h"

namespace libebook
{

FB2ContentCollector::FB2ContentCollector(
  WPXDocumentInterface *document, const WPXPropertyList &metadata,
  const FB2ContentMap &notes, const FB2ContentMap &bitmaps)
  : m_document(document)
  , m_metadata(metadata)
  , m_notes(notes)
  , m_bitmaps(bitmaps)
  , m_currentFootnote(1)
{
}

void FB2ContentCollector::defineMetadataEntry(const char *, const char *)
{
}

void FB2ContentCollector::openMetadataEntry(const char *)
{
}

void FB2ContentCollector::closeMetadataEntry()
{
}

void FB2ContentCollector::defineID(const char *)
{
}

void FB2ContentCollector::openPageSpan()
{
  m_document->openPageSpan(WPXPropertyList());
}

void FB2ContentCollector::closePageSpan()
{
  m_document->closePageSpan();
}

void FB2ContentCollector::openBlock()
{
}

void FB2ContentCollector::closeBlock()
{
}

void FB2ContentCollector::openParagraph(const FB2BlockFormat &format)
{
  m_document->openParagraph(makePropertyList(format), WPXPropertyListVector());
}

void FB2ContentCollector::closeParagraph()
{
  m_document->closeParagraph();
}

void FB2ContentCollector::openSpan(const FB2Style &style)
{
  m_document->openSpan(makePropertyList(style));
}

void FB2ContentCollector::closeSpan()
{
  m_document->closeSpan();
}

void FB2ContentCollector::insertText(const char *text)
{
  m_document->insertText(WPXString(text));
}

void FB2ContentCollector::openTable(const FB2BlockFormat &format)
{
  m_document->openTable(makePropertyList(format), WPXPropertyListVector());
}

void FB2ContentCollector::closeTable()
{
  m_document->closeTable();
}

void FB2ContentCollector::openTableRow(const FB2BlockFormat &format)
{
  WPXPropertyList props;
  if (format.headerRow)
    props.insert("fo:is-header-row", true);

  m_document->openTableRow(props);
}

void FB2ContentCollector::closeTableRow()
{
  m_document->closeTableRow();
}

void FB2ContentCollector::openTableCell(int rowspan, int colspan)
{
  WPXPropertyList props;
  if (colspan > 0)
    props.insert("table:number-columns-spanned", colspan);
  if (rowspan > 0)
    props.insert("table:number-rows-spanned", rowspan);

  m_document->openTableCell(props);
}

void FB2ContentCollector::closeTableCell()
{
  m_document->closeTableCell();
}

void FB2ContentCollector::insertCoveredTableCell()
{
  m_document->insertCoveredTableCell(WPXPropertyList());
}

void FB2ContentCollector::insertFootnote(const char *id)
{
  const FB2Content *const note = m_notes.get(id);
  if (note)
  {
    WPXPropertyList props;
    props.insert("libwpd:number", m_currentFootnote);
    ++m_currentFootnote;

    m_document->openFootnote(props);
    note->unfold(*this);
    m_document->closeFootnote();
  }
}

void FB2ContentCollector::insertBitmap(const char *id)
{
  const FB2Content *const bitmap = m_bitmaps.get(id);
  if (bitmap)
    bitmap->unfold(*this);
}

void FB2ContentCollector::insertBitmapData(const char *contentType, const char *base64Data)
{
  std::string base64String(base64Data, std::strlen(base64Data));
  const unsigned numPadding = std::count(base64String.begin(), base64String.end(), '=');
  std::replace(base64String.begin(), base64String.end(), '=', 'A'); // replace '=' by base64 encoding of '\0'

  namespace bit = boost::archive::iterators;
  typedef
  bit::transform_width<bit::binary_from_base64<bit::remove_whitespace<std::string::const_iterator> >, 8, 6>
  base64_decoder;

  std::vector<unsigned char> buffer;
  std::copy(base64_decoder(base64String.begin()), base64_decoder(base64String.end()), std::back_inserter(buffer));

  if (!buffer.empty())
  {
    buffer.erase(buffer.end() - numPadding, buffer.end());  // erase padding '\0' characters
    if (!buffer.empty())
    {
      WPXPropertyList props;
      props.insert("libwpd:mimetype", WPXString(contentType));

      WPXBinaryData data(&buffer[0], buffer.size());
      m_document->insertBinaryObject(props, data);
    }
  }
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
