/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <libwpd/WPXDocumentInterface.h>
#include <libwpd/WPXPropertyList.h>
#include <libwpd/WPXPropertyListVector.h>
#include <libwpd/WPXString.h>

#include "libebook_utils.h"
#include "LRFCollector.h"

using std::string;

namespace libebook
{

namespace
{

void insert(WPXPropertyList &props, const char *const name, const unsigned value)
{
  props.insert(name, static_cast<int>(value));
}

void insert(WPXPropertyList &props, const char *const name, const string &value)
{
  props.insert(name, value.c_str());
}

void insert(WPXPropertyList &props, const char *const name, const LRFColor &value)
{
  WPXString str;
  str.sprintf("#%02x%02x%02x", value.r, value.g, value.b);
  props.insert(name, str.cstr());
}

WPXPropertyList makeCharacterProperties(const LRFAttributes &attributes)
{
  WPXPropertyList props;

  // TODO: what units LRF uses?
  if (attributes.fontSize)
    insert(props, "fo:font-size", get(attributes.fontSize));
  if (attributes.fontWeight && (0 < get(attributes.fontWeight)))
    insert(props, "fo:font-weight", "bold");
  if (attributes.fontFacename)
    insert(props, "style:font-name", get(attributes.fontFacename));
  if (attributes.textColor)
    insert(props, "fo:color", get(attributes.textColor));
  if (attributes.textBgColor)
    insert(props, "fo:background-color", get(attributes.textBgColor));
  if (attributes.italic && get(attributes.italic))
    insert(props, "fo:font-style", "italic");

  return props;
}

WPXPropertyList makePageProperties(const LRFAttributes &attributes)
{
  WPXPropertyList props;

  if (attributes.width)
    insert(props, "fo:page-width", get(attributes.width));
  if (attributes.height)
    insert(props, "fo:page-height", get(attributes.height));

  return props;
}

WPXPropertyList makeParagraphProperties(const LRFAttributes &attributes)
{
  WPXPropertyList props;

  if (attributes.parIndent)
    insert(props, "fo:text-indent", get(attributes.parIndent));

  return props;
}

void merge(LRFAttributes &merged, const LRFAttributes &other)
{
  if (other.fontSize)
    merged.fontSize = other.fontSize;
  if (other.fontWidth)
    merged.fontWidth = other.fontWidth;
  if (other.fontEscapement)
    merged.fontEscapement = other.fontEscapement;
  if (other.fontOrientation)
    merged.fontOrientation = other.fontOrientation;
  if (other.fontWeight)
    merged.fontWeight = other.fontWeight;
  if (other.fontFacename)
    merged.fontFacename = other.fontFacename;
  if (other.textColor)
    merged.textColor = other.textColor;
  if (other.textBgColor)
    merged.textBgColor = other.textBgColor;
  if (other.wordSpace)
    merged.wordSpace = other.wordSpace;
  if (other.letterSpace)
    merged.letterSpace = other.letterSpace;
  if (other.baseLineSkip)
    merged.baseLineSkip = other.baseLineSkip;
  if (other.lineSpace)
    merged.lineSpace = other.lineSpace;
  if (other.parIndent)
    merged.parIndent = other.parIndent;
  if (other.parSkip)
    merged.parSkip = other.parSkip;
  if (other.height)
    merged.height = other.height;
  if (other.width)
    merged.width = other.width;
  if (other.locationX)
    merged.locationX = other.locationX;
  if (other.locationY)
    merged.locationY = other.locationY;
  if (other.italic)
    merged.italic = other.italic;
  if (other.sup)
    merged.sup = other.sup;
  if (other.sub)
    merged.sub = other.sub;
}

}

LRFCollector::LRFCollector(WPXDocumentInterface *const document)
  : m_bookAttributes()
  , m_textAttributeMap()
  , m_blockAttributeMap()
  , m_pageAttributeMap()
  , m_paragraphAttributeMap()
  , m_document(document)
  , m_currentAttributes()
  , m_imageMap()
{
}

LRFCollector::~LRFCollector()
{
  for (ImageMap_t::const_iterator it = m_imageMap.begin(); it != m_imageMap.end(); ++it)
    delete it->second.image;
}

void LRFCollector::startDocument()
{
  m_document->startDocument();
  m_currentAttributes.push(m_bookAttributes);
}

void LRFCollector::endDocument()
{
  m_document->endDocument();
  m_currentAttributes.pop();
}

void LRFCollector::openPage(const unsigned pageAtrID, const LRFAttributes &attributes)
{
  openBlock(pageAtrID, attributes, &m_pageAttributeMap);
  m_document->openPageSpan(makePageProperties(m_currentAttributes.top()));
}

void LRFCollector::closePage()
{
  m_document->closePageSpan();
  closeBlock();
}

void LRFCollector::openBlock(unsigned atrID, const LRFAttributes &attributes)
{
  openBlock(atrID, attributes, &m_blockAttributeMap);
}

void LRFCollector::closeBlock()
{
  m_currentAttributes.pop();
}

void LRFCollector::openTextBlock(unsigned atrID, const LRFAttributes &attributes)
{
  openBlock(atrID, attributes, &m_textAttributeMap);
}

void LRFCollector::closeTextBlock()
{
  closeBlock();
}

void LRFCollector::openParagraph(unsigned atrID, const LRFAttributes &attributes)
{
  openBlock(atrID, attributes, &m_paragraphAttributeMap);
  m_document->openParagraph(makeParagraphProperties(m_currentAttributes.top()), WPXPropertyListVector());
}

void LRFCollector::closeParagraph()
{
  m_document->closeParagraph();
  closeBlock();
}

void LRFCollector::collectText(const std::string &text, const LRFAttributes &attributes)
{
  openBlock(0, attributes, 0);
  m_document->openSpan(makeCharacterProperties(m_currentAttributes.top()));
  m_document->insertText(WPXString(text.c_str()));
  m_document->closeSpan();
  closeBlock();
}

void LRFCollector::insertLineBreak()
{
  m_document->insertLineBreak();
}

void LRFCollector::collectMetadata(const LRFMetadata &metadata)
{
  // TODO: implement me
  (void) metadata;

  WPXPropertyList props;

  m_document->setDocumentMetaData(props);
}

void LRFCollector::collectBookAttributes(const LRFAttributes &attributes)
{
  m_bookAttributes = attributes;
}

void LRFCollector::collectTextAttributes(const unsigned id, const LRFAttributes &attributes)
{
  collectAttributes(id, attributes, m_textAttributeMap);
}

void LRFCollector::collectBlockAttributes(const unsigned id, const LRFAttributes &attributes)
{
  collectAttributes(id, attributes, m_blockAttributeMap);
}

void LRFCollector::collectPageAttributes(const unsigned id, const LRFAttributes &attributes)
{
  collectAttributes(id, attributes, m_pageAttributeMap);
}

void LRFCollector::collectParagraphAttributes(const unsigned id, const LRFAttributes &attributes)
{
  collectAttributes(id, attributes, m_paragraphAttributeMap);
}

void LRFCollector::collectImage(const unsigned id)
{
  if (0 == id)
  {
    EBOOK_DEBUG_MSG(("invalid image id 0\n"));
    return;
  }

  const ImageMap_t::const_iterator it = m_imageMap.find(id);
  if (it != m_imageMap.end())
  {
    const char *mimetype = 0;
    switch (it->second.type)
    {
    case IMAGE_TYPE_JPEG :
      mimetype = "image/jpeg";
      break;
    case IMAGE_TYPE_PNG :
      mimetype = "image/png";
      break;
    case IMAGE_TYPE_BMP :
      mimetype = "image/bmp";
      break;
    default :
      EBOOK_DEBUG_MSG(("unknown image type %x\n", it->second.type));
      return;
    }

    WPXPropertyList props;
    props.insert("libwpd:mimetype", mimetype);

    WPXInputStream *const image = it->second.image;
    image->seek(0, WPX_SEEK_END);
    const unsigned long length = image->tell();
    image->seek(0, WPX_SEEK_SET);
    const unsigned char *const bytes = readNBytes(image, length);
    WPXBinaryData data(bytes, length);

    m_document->insertBinaryObject(props, data);
  }
  else
  {
    EBOOK_DEBUG_MSG(("image stream with id %x not found\n", id));
  }
}

void LRFCollector::collectImageData(const unsigned id, const ImageType type, WPXInputStream *const image)
{
  ImageData data;
  data.image = image;
  data.type = type;

  if (!m_imageMap.insert(ImageMap_t::value_type(id, data)).second)
  {
    EBOOK_DEBUG_MSG(("cannot insert image stream %x: already present\n", id));
  }
}

void LRFCollector::collectAttributes(const unsigned id, const LRFAttributes &attributes, LRFAttributeMap_t &attributeMap)
{
  if (!attributeMap.insert(LRFAttributeMap_t::value_type(id, attributes)).second)
  {
    EBOOK_DEBUG_MSG(("cannot insert attributes for object %x: already present\n", id));
  }
}

void LRFCollector::openBlock(unsigned atrID, const LRFAttributes &attributes, const LRFAttributeMap_t *attributeMap)
{
  LRFAttributes mergedAttributes(m_currentAttributes.top());

  if ((0 != atrID) && attributeMap)
  {
    const LRFAttributeMap_t::const_iterator it = attributeMap->find(atrID);
    if (it != attributeMap->end())
      merge(mergedAttributes, it->second);
    else
    {
      EBOOK_DEBUG_MSG(("attributes for object %x not found\n", atrID));
    }
  }

  merge(mergedAttributes, attributes);

  m_currentAttributes.push(mergedAttributes);
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
