static char rcsid[] = "$Id: splice.c 224745 2021-12-04 15:54:54Z twu $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "splice.h"

#include <stdio.h>
#include "mem.h"
#include "assert.h"
#include "sense.h"
#include "genome128_hr.h"
#include "genome_sites.h"
#include "substring.h"
#include "maxent.h"
#include "maxent_hr.h"
#include "stage3hr.h"
#include "indel.h"
#include "univcoord.h"


#define LOWPROB_SUPPORT 20
#define MIN_SUPPORT_INDEL 0
#define MIN_SUPPORT_SPLICE 6	/* First level in sufficient_support_p */
#define MIN_SUPPORT_SPLICE_PLUS_INDEL 12

#define MISMATCHES_SLOP 1

/* #define MIN_SPLICE_PROB 0.4 */	/* Skip if both probs are less than this */
#define MIN_SPLICE_PLUS_INDEL_PROB 0.5 /* Skip if either prob is less than this */

#define AUTO_SPLICE_PROB_HIGH 0.9
#define AUTO_SPLICE_PROB_LOW 0.2


#define END_SPLICESITE_SEARCH_MM 1 /* Amount to search in the trimmed area */
#define END_SPLICESITE_SEARCH 10
#define MIN_EXON_LENGTH 20

#if 0
/* Creates issues with ambiguous substrings */
#define LOCALSPLICING_NMATCHES_SLOP 1
#else
#define LOCALSPLICING_NMATCHES_SLOP 0
#endif
#define LOCALSPLICING_PROB_SLOP 0.05

#define SLOP 1


/* Splice_resolve_sense and Splice_resolve_antisense */
#ifdef DEBUG1
#define debug1(x) x
#else
#define debug1(x)
#endif

/* Splice_resolve_distant */
#ifdef DEBUG2
#define debug2(x) x
#else
#define debug2(x)
#endif

/* sufficient_support_p */
#ifdef DEBUG3
#define debug3(x) x
#else
#define debug3(x)
#endif

/* Trim novel splice ends */
#ifdef DEBUG13
#define debug13(x) x
#else
#define debug13(x)
#endif


static bool novelsplicingp;

static Genome_T genomebits;
static Genome_T genomebits_alt;

static int min_shortend;
static int min_shortend_distant = 20;


void
Splice_setup (Genome_T genomebits_in, Genome_T genomebits_alt_in,
	      bool novelsplicingp_in, int min_shortend_in) {
  genomebits = genomebits_in;
  genomebits_alt = genomebits_alt_in;
  novelsplicingp = novelsplicingp_in;
  min_shortend = min_shortend_in;
  /* max_end_length = 2*(index1part_in + index1interval_in - 1); */
  return;
}


void
Spliceinfo_free (Spliceinfo_T *old) {
  FREE((*old)->double_memory);
  FREE((*old)->int_memory);
  FREE(*old);
  return;
}


#define MAX_NSPLICEENDS 3

Spliceinfo_T
Spliceinfo_new (int querylength) {
  Spliceinfo_T new = (Spliceinfo_T) MALLOC(sizeof(*new));
  int a, b = (querylength + 1);

  new->int_memory = (int *) MALLOC((12*b + 6*querylength + 2*MAX_NSPLICEENDS) * sizeof(int));

  /* 12 arrays of (querylength+1) ints */
  new->donor_positions_alloc = &(new->int_memory[0]); a = b;
  new->donor_knowni_alloc = &(new->int_memory[a]); a += b;
  new->acceptor_positions_alloc = &(new->int_memory[a]); a += b;
  new->acceptor_knowni_alloc = &(new->int_memory[a]); a += b;

  new->segmenti_donor_knownpos = &(new->int_memory[a]); a += b;
  new->segmentj_acceptor_knownpos = &(new->int_memory[a]); a += b;
  new->segmentj_antidonor_knownpos = &(new->int_memory[a]); a += b;
  new->segmenti_antiacceptor_knownpos = &(new->int_memory[a]); a += b;

  new->segmenti_donor_knowni = &(new->int_memory[a]); a += b;
  new->segmentj_acceptor_knowni = &(new->int_memory[a]); a += b;
  new->segmentj_antidonor_knowni = &(new->int_memory[a]); a += b;
  new->segmenti_antiacceptor_knowni = &(new->int_memory[a]); a += b;

  /* 6 arrays of (querylength) ints */
  /* Needed for Splice_resolve_sense and Splice_resolve_antisense */
  new->saved_knowni_i = &(new->int_memory[a]); a += querylength;
  new->saved_knowni_j = &(new->int_memory[a]); a += querylength;
  new->saved_segmenti_nmismatches = &(new->int_memory[a]); a += querylength;
  new->saved_segmentj_nmismatches = &(new->int_memory[a]); a += querylength;
  new->saved_segmenti_ref_nmismatches = &(new->int_memory[a]); a += querylength;
  new->saved_segmentj_ref_nmismatches = &(new->int_memory[a]); a += querylength;

  /* 2 arrays of (MAX_NSPLICEENDS) ints */
  new->ambig_qstarts = &(new->int_memory[a]); a += MAX_NSPLICEENDS;
  new->ambig_qends = &(new->int_memory[a]); /*a += MAX_NSPLICEENDS;*/


  new->double_memory = (double *) MALLOC((2*b + 2*querylength + 2*MAX_NSPLICEENDS) * sizeof(double));

  /* 2 arrays of (querylength+1) doubles */
  new->donor_probs_alloc = &(new->double_memory[0]); a = b;
  new->acceptor_probs_alloc = &(new->double_memory[a]); a += b;

  /* 2 arrays of (querylength) doubles */
  /* Needed for Splice_resolve_sense and Splice_resolve_antisense */
  new->saved_probi = &(new->double_memory[a]); a += querylength;
  new->saved_probj = &(new->double_memory[a]); a += querylength;

  /* 2 arrays of (MAX_NSPLICEENDS) doubles */
  new->ambig_probs_5 = &(new->double_memory[a]); a += MAX_NSPLICEENDS;
  new->ambig_probs_3 = &(new->double_memory[a]); /*a += MAX_NSPLICEENDS;*/

  return new;
}

/* Note: contents of spliceinfo are filled in by kmer-search and path-solve procedures */


static bool
sufficient_support_p (int adj_support, double splice_prob) {
  debug3(printf("Checking for sufficient splice prob, based on adj_support %d and splice prob %.2f\n",
		adj_support,splice_prob));

  if (splice_prob > 0.95) {
    return (adj_support >= 6) ? true : false; /* threshold set to MIN_SUPPORT_SPLICE */

  } else if (splice_prob > 0.90) {
    return (adj_support >= 8) ? true : false;

  } else if (splice_prob > 0.80) {
    return (adj_support >= 12) ? true : false;

  } else if (splice_prob > 0.50) {
    return (adj_support >= 15) ? true : false;

  } else {
    return (adj_support >= 20) ? true : false;
  }
}


#if 0
/* Do not compare against true or false */
/* Want loose criterion, otherwise, we incur slowdown from having to
   run GSNAP algorithm */
static int
sufficient_splice_prob_local (int support, int nmismatches, double spliceprob) {
  support -= 3*nmismatches;
  if (support <= 9) {
    return (spliceprob > 0.80);
  } else if (support <= 12) {
    return (spliceprob > 0.70);
  } else if (support <= 15) {
    return (spliceprob > 0.60);
  } else if (support <= 25) {
    return (spliceprob > 0.50);
  } else {
    return (spliceprob > 0.40);
  }
}
#endif




/* Note: knowni holds joffset + j + 1, so 0 represents no known site
   and values greater than 0 represent a known site.  Need to subtract
   1 to obtain joffset + j. */

int
Splice_resolve_sense (int *best_nindels, int *best_indel_pos, int *best_knowni_i, int *best_knowni_j,
		      int *best_nmismatches_i, int *best_nmismatches_j, int *best_nmismatches_indel,
		      int *best_ref_nmismatches_i, int *best_ref_nmismatches_j,
		      int *best_ref_nmismatches_indel, double *best_prob_i, double *best_prob_j,

		      Univcoord_T segmenti_left, Univcoord_T segmentj_left, Univcoord_T chroffset,
		     
		      int pos5, int pos3, int querylength, Compress_T query_compress,
		      Indelinfo_T indelinfo, Spliceinfo_T spliceinfo,
		      bool plusp, int genestrand, int max_deletionlen, int max_insertionlen,
		      bool allow_indel_p) {
  int best_splice_qpos = -1, splice_qpos_start, splice_qpos_end, splice_qpos, i, j;

  int best_nmismatches, nmismatches, nmismatches1, nmismatches2;
  int best_ref_nmismatches, ref_nmismatches, ref_nmismatches1, ref_nmismatches2; 
  int segmenti_nmismatches, segmentj_nmismatches;
  double best_prob, probi, probj;
  /* bool sufficient1p, sufficient2p; */

  int donori_nsites, acceptorj_nsites, antiacceptori_nsites, antidonorj_nsites;
  int *donori_positions, *acceptorj_positions, *antiacceptori_positions, *antidonorj_positions;
  int *donori_knowni, *acceptorj_knowni, *antiacceptori_knowni, *antidonorj_knowni;
  double *donori_probs, *acceptorj_probs, *antiacceptori_probs, *antidonorj_probs;

  int adj_supporti, adj_supportj, supporti, supportj, support_indel;
  int nindels, indel_pos;
  int nmismatches_indel, nmismatches_i, nmismatches_j;

  Intlist_T candidates = NULL, p;
#if 0
  int *intspace, *m;
  int *saved_knowni_i, *saved_knowni_j, *saved_segmenti_nmismatches, *saved_segmentj_nmismatches,
    *saved_segmenti_ref_nmismatches, *saved_segmentj_ref_nmismatches;
  double *saved_probi, *saved_probj;
#endif
  double prob;


  debug1(printf("Splice_resolve_sense: Getting genome at lefti %u [%u] and leftj %u [%u] (diff: %d), range %d..%d, max_mismatches_allowed %d\n",
		segmenti_left,segmenti_left - chroffset,segmentj_left,segmentj_left - chroffset,
		segmentj_left-segmenti_left,pos5,pos3,max_mismatches_allowed));
  debug1(printf("novelsplicingp %d\n",novelsplicingp));

#if 0
  /* Now allocated in Spliceinfo_T */
  saved_probi = MALLOC(querylength * sizeof(double));
  saved_probj = MALLOC(querylength * sizeof(double));
  intspace = MALLOC(6 * querylength * sizeof(int));
  m = saved_knowni_i = &(intspace[0]);
  m = saved_knowni_j = &(m[querylength]);
  m = saved_segmenti_nmismatches = &(m[querylength]);
  m = saved_segmentj_nmismatches = &(m[querylength]);
  m = saved_segmenti_ref_nmismatches = &(m[querylength]);
  m = saved_segmentj_ref_nmismatches = &(m[querylength]);
#endif

  *best_nindels = 0;
  *best_indel_pos = -1;
  *best_knowni_i = *best_knowni_j = -1;
  *best_nmismatches_i = *best_nmismatches_j = *best_nmismatches_indel = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
  *best_ref_nmismatches_i = *best_ref_nmismatches_j = *best_ref_nmismatches_indel = -1;
  *best_prob_i = *best_prob_j = 0.0;


  /* Require separation from endpoints */
  splice_qpos_start = pos5 + 1;
  splice_qpos_end = pos3 - 1;
  if (splice_qpos_start < min_shortend) {
    splice_qpos_start = min_shortend;
  }
  if (splice_qpos_end > querylength - min_shortend) {
    splice_qpos_end = querylength - min_shortend;
  }

  if (plusp == true) {
    /* Originally from plus strand.  No complement.  */
    /* Sense (End 1 to End 2) or Antisense (End 5 to End 6) */
    if (novelsplicingp == true && segmenti_left + splice_qpos_start >= DONOR_MODEL_LEFT_MARGIN) {
      donori_nsites = Genome_donor_positions(spliceinfo->donor_positions_alloc,spliceinfo->donor_knowni_alloc,
					     spliceinfo->segmenti_donor_knownpos,
					     spliceinfo->segmenti_donor_knowni,
					     segmenti_left,splice_qpos_start,splice_qpos_end);
      donori_positions = spliceinfo->donor_positions_alloc;
      donori_knowni = spliceinfo->donor_knowni_alloc;
    } else {
      donori_nsites = spliceinfo->segmenti_donor_nknown;
      donori_positions = spliceinfo->segmenti_donor_knownpos;
      donori_knowni = spliceinfo->segmenti_donor_knowni;
    }

#ifdef DEBUG1
    printf("Found %d donori sites:",donori_nsites);
    for (i = 0; i < donori_nsites; i++) {
      printf(" %d",donori_positions[i]);
      if (donori_knowni[i] >= 0) {
	printf(" (%d)",donori_knowni[i]);
      } else {
	probi = Maxent_hr_donor_prob(segmenti_left + donori_positions[i],chroffset);
	printf(" (%.6f)",probi);
      }
    }
    printf("\n");
#endif

    if (novelsplicingp == true && segmentj_left + splice_qpos_start >= ACCEPTOR_MODEL_LEFT_MARGIN) {
      acceptorj_nsites = Genome_acceptor_positions(spliceinfo->acceptor_positions_alloc,spliceinfo->acceptor_knowni_alloc,
						   spliceinfo->segmentj_acceptor_knownpos,
						   spliceinfo->segmentj_acceptor_knowni,
						   segmentj_left,splice_qpos_start,splice_qpos_end);
      acceptorj_positions = spliceinfo->acceptor_positions_alloc;
      acceptorj_knowni = spliceinfo->acceptor_knowni_alloc;
    } else {
      acceptorj_nsites = spliceinfo->segmentj_acceptor_nknown;
      acceptorj_positions = spliceinfo->segmentj_acceptor_knownpos;
      acceptorj_knowni = spliceinfo->segmentj_acceptor_knowni;
    }

#ifdef DEBUG1
    printf("Found %d acceptorj sites:",acceptorj_nsites);
    for (i = 0; i < acceptorj_nsites; i++) {
      printf(" %d",acceptorj_positions[i]);
      if (acceptorj_knowni[i] >= 0) {
	printf(" (%d)",acceptorj_knowni[i]);
      } else {
	probj = Maxent_hr_acceptor_prob(segmentj_left + acceptorj_positions[i],chroffset);
	printf(" (%.6f)",probj);
      }
    }
    printf("\n");
#endif

    i = j = 0;
    while (i < donori_nsites && j < acceptorj_nsites) {
      supporti = donori_positions[i] - pos5;
      supportj = pos3 - acceptorj_positions[j];

      if ((splice_qpos = donori_positions[i]) < acceptorj_positions[j]) {
	i++;
      } else if (splice_qpos > acceptorj_positions[j]) {
	j++;
      } else if (supporti < MIN_SUPPORT_SPLICE || supportj < MIN_SUPPORT_SPLICE) {
	/* Skip */
	i++; j++;
      } else {
	debug1(printf("splice matches at %d\n",splice_qpos));
	segmenti_nmismatches = spliceinfo->saved_segmenti_nmismatches[splice_qpos] = 
	  Genome_count_mismatches_substring(&(spliceinfo->saved_segmenti_ref_nmismatches[splice_qpos]),
					    genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
					    pos5,/*pos3*/splice_qpos,plusp,genestrand);
	segmentj_nmismatches = spliceinfo->saved_segmentj_nmismatches[splice_qpos] =
	  Genome_count_mismatches_substring(&(spliceinfo->saved_segmentj_ref_nmismatches[splice_qpos]),
					    genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
					    /*pos5*/splice_qpos,pos3,plusp,genestrand);

	debug1(printf("%d mismatches on segmenti (%d..%d)\n",spliceinfo->saved_segmenti_nmismatches[splice_qpos],pos5,splice_qpos));
	debug1(printf("%d mismatches on segmentj (%d..%d)\n",spliceinfo->saved_segmentj_nmismatches[splice_qpos],splice_qpos,pos3));

	if ((spliceinfo->saved_knowni_i[splice_qpos] = donori_knowni[i]) >= 0) {
	  probi = spliceinfo->saved_probi[splice_qpos] = 1.0; /* Needs to be 1.0 for output */
	} else {
	  probi = spliceinfo->saved_probi[splice_qpos] = Maxent_hr_donor_prob(segmenti_left + splice_qpos,chroffset);
	}

	if ((spliceinfo->saved_knowni_j[splice_qpos] = acceptorj_knowni[j]) >= 0) {
	  probj = spliceinfo->saved_probj[splice_qpos] = 1.0; /* Needs to be 1.0 for output */
	} else {
	  probj = spliceinfo->saved_probj[splice_qpos] = Maxent_hr_acceptor_prob(segmentj_left + splice_qpos,chroffset);
	}

	adj_supporti = supporti - 3*segmenti_nmismatches;
	adj_supportj = supportj - 3*segmentj_nmismatches;
	if (sufficient_support_p(adj_supporti,probi) == false) {
	  debug1(printf("Skipping, because too many mismatches %d in segmenti\n",segmenti_nmismatches));
	} else if (sufficient_support_p(adj_supportj,probj) == false) {
	  debug1(printf("Skipping, because too many mismatches %d in segmentj\n",segmentj_nmismatches));
	} else {
	  candidates = Intlist_push(candidates,splice_qpos);
	  debug1(
		 if (plusp == true) {
		   printf("plus sense splice_qpos  %d, i.donor %f, j.acceptor %f\n",
			  splice_qpos,spliceinfo->saved_probi[splice_qpos],spliceinfo->saved_probj[splice_qpos]);
		 } else {
		   printf("minus antisense splice_qpos  %d, i.donor %f, j.acceptor %f\n",
			  splice_qpos,spliceinfo->saved_probi[splice_qpos],spliceinfo->saved_probj[splice_qpos]);
		 });
	}

	i++; j++;
      }
    }

  } else {
    /* minus */
    /* Originally from minus strand.  Complement. */
    /* Antisense (End 7 to End 8) or Sense (End 3 to End 4) */
    if (novelsplicingp == true && segmenti_left + splice_qpos_start >= ACCEPTOR_MODEL_RIGHT_MARGIN) {
      antiacceptori_nsites = Genome_antiacceptor_positions(spliceinfo->acceptor_positions_alloc,spliceinfo->acceptor_knowni_alloc,
							   spliceinfo->segmenti_antiacceptor_knownpos,
							   spliceinfo->segmenti_antiacceptor_knowni,
							   segmenti_left,splice_qpos_start,splice_qpos_end);
      antiacceptori_positions = spliceinfo->acceptor_positions_alloc;
      antiacceptori_knowni = spliceinfo->acceptor_knowni_alloc;
    } else {
      antiacceptori_nsites = spliceinfo->segmenti_antiacceptor_nknown;
      antiacceptori_positions = spliceinfo->segmenti_antiacceptor_knownpos;
      antiacceptori_knowni = spliceinfo->segmenti_antiacceptor_knowni;
    }

#ifdef DEBUG1
    printf("Found %d antiacceptori sites:",antiacceptori_nsites);
    for (i = 0; i < antiacceptori_nsites; i++) {
      printf(" %d",antiacceptori_positions[i]);
      if (antiacceptori_knowni[i] >= 0) {
	printf(" (%d)",antiacceptori_knowni[i]);
      } else {
	probi = Maxent_hr_antiacceptor_prob(segmenti_left + antiacceptori_positions[i],chroffset);
	printf(" (%.6f)",probi);
      }
    }
    printf("\n");
#endif

    if (novelsplicingp == true && segmentj_left + splice_qpos_start >= DONOR_MODEL_RIGHT_MARGIN) {
      antidonorj_nsites = Genome_antidonor_positions(spliceinfo->donor_positions_alloc,spliceinfo->donor_knowni_alloc,
						     spliceinfo->segmentj_antidonor_knownpos,
						     spliceinfo->segmentj_antidonor_knowni,
						     segmentj_left,splice_qpos_start,splice_qpos_end);
      antidonorj_positions = spliceinfo->donor_positions_alloc;
      antidonorj_knowni = spliceinfo->donor_knowni_alloc;
    } else {
      antidonorj_nsites = spliceinfo->segmentj_antidonor_nknown;
      antidonorj_positions = spliceinfo->segmentj_antidonor_knownpos;
      antidonorj_knowni = spliceinfo->segmentj_antidonor_knowni;
    }

#ifdef DEBUG1
    printf("Found %d antidonorj sites:",antidonorj_nsites);
    for (i = 0; i < antidonorj_nsites; i++) {
      printf(" %d",antidonorj_positions[i]);
      if (antidonorj_knowni[i] >= 0) {
	printf(" (%d)",antidonorj_knowni[i]);
      } else {
	probj = Maxent_hr_antidonor_prob(segmentj_left + antidonorj_positions[i],chroffset);
	printf(" (%.6f)",probj);
      }
    }
    printf("\n");
#endif

    i = j = 0;
    while (i < antiacceptori_nsites && j < antidonorj_nsites) {
      supporti = antiacceptori_positions[i] - pos5;
      supportj = pos3 - antidonorj_positions[j];

      if ((splice_qpos = antiacceptori_positions[i]) < antidonorj_positions[j]) {
	i++;
      } else if (splice_qpos > antidonorj_positions[j]) {
	j++;
      } else if (supporti < MIN_SUPPORT_SPLICE || supportj < MIN_SUPPORT_SPLICE) {
	/* Skip */
	i++; j++;
      } else {
	debug1(printf("splice matches at %d\n",splice_qpos));
	segmenti_nmismatches = spliceinfo->saved_segmenti_nmismatches[splice_qpos] =
	  Genome_count_mismatches_substring(&(spliceinfo->saved_segmenti_ref_nmismatches[splice_qpos]),
					    genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
					    pos5,/*pos3*/splice_qpos,plusp,genestrand);
	segmentj_nmismatches = spliceinfo->saved_segmentj_nmismatches[splice_qpos] =
	  Genome_count_mismatches_substring(&(spliceinfo->saved_segmentj_ref_nmismatches[splice_qpos]),
					    genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
					    /*pos5*/splice_qpos,pos3,plusp,genestrand);
	debug1(printf("%d mismatches on segmenti (%d..%d)\n",segmenti_nmismatches,pos5,splice_qpos));
	debug1(printf("%d mismatches on segmentj (%d..%d)\n",segmentj_nmismatches,splice_qpos,pos3));

	if ((spliceinfo->saved_knowni_i[splice_qpos] = antiacceptori_knowni[i]) >= 0) {
	  probi = spliceinfo->saved_probi[splice_qpos] = 1.0; /* Needs to be 1.0 for output */
	} else {
	  probi = spliceinfo->saved_probi[splice_qpos] = Maxent_hr_antiacceptor_prob(segmenti_left + splice_qpos,chroffset);
	}

	if ((spliceinfo->saved_knowni_j[splice_qpos] = antidonorj_knowni[j]) >= 0) {
	  probj = spliceinfo->saved_probj[splice_qpos] = 1.0; /* Needs to be 1.0 for output */
	} else {
	  probj = spliceinfo->saved_probj[splice_qpos] = Maxent_hr_antidonor_prob(segmentj_left + splice_qpos,chroffset);
	}

	adj_supporti = supporti - 3*segmenti_nmismatches;
	adj_supportj = supportj - 3*segmentj_nmismatches;
	if (sufficient_support_p(adj_supporti,probi) == false) {
	  debug1(printf("Skipping, because too many mismatches %d in segmenti\n",segmenti_nmismatches));
	} else if (supportj - 3*segmentj_nmismatches < MIN_SUPPORT_SPLICE) {
	  debug1(printf("Skipping, because too many mismatches %d in segmentj\n",segmentj_nmismatches));
	} else {
	  candidates = Intlist_push(candidates,splice_qpos);
	  debug1(
		 if (plusp == true) {
		   printf("plus antisense splice_qpos  %d, j.donor %f, i.acceptor %f\n",splice_qpos,probj,probi);
		 } else {
		   printf("minus sense splice_qpos  %d, j.donor %f, i.acceptor %f\n",splice_qpos,probj,probi);
		 });
	}

	i++; j++;
      }
    }
  }


  /* Iterate through candidates */
  best_nmismatches = best_ref_nmismatches = querylength;  /* was max_mismatches_allowed */;
  best_prob = 0.0;

  for (p = candidates; p != NULL; p = Intlist_next(p)) {
    splice_qpos = Intlist_head(p);
    if ((nmismatches = spliceinfo->saved_segmenti_nmismatches[splice_qpos] + spliceinfo->saved_segmentj_nmismatches[splice_qpos]) < best_nmismatches) {
      best_nmismatches = nmismatches;
    }
  }
  debug1(printf("best_nmismatches is %d\n",best_nmismatches));

  for (p = candidates; p != NULL; p = Intlist_next(p)) {
    splice_qpos = Intlist_head(p);
    if (spliceinfo->saved_segmenti_nmismatches[splice_qpos] + spliceinfo->saved_segmentj_nmismatches[splice_qpos] > best_nmismatches + MISMATCHES_SLOP) {
      /* Too many mismatches */
    } else if ((prob = spliceinfo->saved_probi[splice_qpos] + spliceinfo->saved_probj[splice_qpos]) > best_prob) {
      best_prob = prob;
      best_splice_qpos = splice_qpos;
    }
  }

  if (best_splice_qpos >= 0) {
    /* Success */
    debug1(printf("Success at splice_qpos %d\n",best_splice_qpos));
    *best_knowni_i = spliceinfo->saved_knowni_i[best_splice_qpos];
    *best_knowni_j = spliceinfo->saved_knowni_j[best_splice_qpos];
    *best_prob_i = spliceinfo->saved_probi[best_splice_qpos];
    *best_prob_j = spliceinfo->saved_probj[best_splice_qpos];
    *best_nmismatches_i = spliceinfo->saved_segmenti_nmismatches[best_splice_qpos];
    *best_nmismatches_j = spliceinfo->saved_segmentj_nmismatches[best_splice_qpos];
    *best_ref_nmismatches_i = spliceinfo->saved_segmenti_ref_nmismatches[best_splice_qpos];
    *best_ref_nmismatches_j = spliceinfo->saved_segmentj_ref_nmismatches[best_splice_qpos];
  }

  Intlist_free(&candidates);
#if 0
  FREE(intspace);
  FREE(saved_probi);
  FREE(saved_probj);
#endif


  debug1(printf("best_knowni_i is %d and best_knowni_j is %d\n",*best_knowni_i,*best_knowni_j));

  if (*best_prob_i > AUTO_SPLICE_PROB_HIGH && *best_prob_j > AUTO_SPLICE_PROB_LOW) {
    debug1(printf("Automatic accept: returning %d with probi %f and probj %f\n",best_splice_qpos,*best_prob_i,*best_prob_j));
    debug1(printf("nmismatches %d and %d\n",*best_nmismatches_i,*best_nmismatches_j));
    return best_splice_qpos;
  } else if (*best_prob_j > AUTO_SPLICE_PROB_HIGH && *best_prob_i > AUTO_SPLICE_PROB_LOW) {
    debug1(printf("Automatic accept: returning %d with probi %f and probj %f\n",best_splice_qpos,*best_prob_i,*best_prob_j));
    debug1(printf("nmismatches %d and %d\n",*best_nmismatches_i,*best_nmismatches_j));
    return best_splice_qpos;
  } else if (allow_indel_p == false) {
    debug1(printf("Not returning %d with probi %f and probj %f because allow_indel_p is false\n",best_splice_qpos,*best_prob_i,*best_prob_j));
    *best_nmismatches_i = *best_nmismatches_j = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
    return -1;
  } else {
    /* Find an indel below to see if something is better */
  }


  if (plusp == true) {
    /* Genomic sense (inferred from antisense code) */
    if (donori_nsites == 0 || acceptorj_nsites == 0) {
      debug1(printf("Not returning %d with probi %f and probj %f because nsites is 0\n",best_splice_qpos,*best_prob_i,*best_prob_j));
      *best_nmismatches_i = *best_nmismatches_j = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
      return -1;
    } else {
      donori_probs = spliceinfo->donor_probs_alloc;
      for (i = 0; i < donori_nsites; i++) {
	donori_probs[i] = Maxent_hr_donor_prob(segmenti_left + donori_positions[i],chroffset);
      }

      acceptorj_probs = spliceinfo->acceptor_probs_alloc;
      for (i = 0; i < acceptorj_nsites; i++) {
	acceptorj_probs[i] = Maxent_hr_acceptor_prob(segmentj_left + acceptorj_positions[i],chroffset);
      }

      i = j = 0;
      best_nmismatches = querylength;  /* was max_mismatches_allowed + 1 */
      while (i < donori_nsites && donori_positions[i] - pos5 < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	i++;
      }

      while (i < donori_nsites) {
	supporti = donori_positions[i] - pos5;

	/* Backup */
	while (j >= 0 && acceptorj_positions[j] + max_deletionlen > donori_positions[i]) {
	  j--;
	}
	j++;			/* Finish backup */

	/* Advance */
	while (j < acceptorj_nsites && acceptorj_positions[j] + max_deletionlen <= donori_positions[i]) {
	  j++;
	}

	/* Deletions */
	while (j < acceptorj_nsites && acceptorj_positions[j] < donori_positions[i] &&
	       (supportj = pos3 - acceptorj_positions[j]) >= MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	  debug1(printf("(A) Deletion: %d (%.6f) -- %d (%.6f).  Support: %d and %d\n",
			donori_positions[i],donori_probs[i],acceptorj_positions[j],acceptorj_probs[j],
			donori_positions[i] - pos5,pos3 - acceptorj_positions[j]));
	  probi = donori_probs[i];
	  probj = acceptorj_probs[j];
	  if (probi > MIN_SPLICE_PLUS_INDEL_PROB && probj > MIN_SPLICE_PLUS_INDEL_PROB) {
	    nindels = acceptorj_positions[j] - donori_positions[i]; /* Should be negative */

	    /* Try deletion on segmenti */
	    splice_qpos = acceptorj_positions[j];
	    if ((indel_pos = Indel_resolve_middle_deletion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							   segmenti_left,nindels,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							   pos5,/*pos3*/splice_qpos,querylength,
							   indelinfo,/*plusp:true*/true,genestrand,
							   /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find deletion on segmenti\n"));
	    } else {
	      assert(indel_pos > pos5);
	      assert(indel_pos < splice_qpos);
	      support_indel = splice_qpos - indel_pos;
	      nmismatches_i = nmismatches1;
	      nmismatches_indel = nmismatches2; /* From indel to splice */
	      nmismatches_j =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,
						  /*left*/segmentj_left,/*pos5*/splice_qpos,pos3,plusp,genestrand);
	      debug1(printf("  => (1) deletion indel_pos %d on segmenti with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supporti - support_indel - 3*nmismatches_i < MIN_SUPPORT_SPLICE || supportj - 3*nmismatches_j < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supporti - 3*(nmismatches_indel + nmismatches_i) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_indel + nmismatches_i + nmismatches_j) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches1;
		*best_ref_nmismatches_indel = ref_nmismatches2;
		*best_ref_nmismatches_j = ref_nmismatches;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }

	    /* Try deletion on segmentj */
	    splice_qpos = donori_positions[i];
	    if ((indel_pos = Indel_resolve_middle_deletion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							   segmentj_left + nindels,nindels,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							   /*pos5*/splice_qpos,pos3,querylength,
							   indelinfo,/*plusp:true*/true,genestrand,
							   /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find deletion on segmentj\n"));
	    } else {
	      assert(indel_pos > splice_qpos);
	      assert(indel_pos < pos3);
	      support_indel = indel_pos - splice_qpos;
	      nmismatches_i =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
						  pos5,/*pos3*/splice_qpos,plusp,genestrand);
	      nmismatches_indel = nmismatches1;	  /* From splice to indel */
	      nmismatches_j = nmismatches2;
	      debug1(printf("  => (2) deletion indel_pos %d on segmentj with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supportj - support_indel - 3*nmismatches_j < MIN_SUPPORT_SPLICE || supporti - 3*nmismatches_i < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supportj - 3*(nmismatches_indel + nmismatches_j) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_i + nmismatches_j + nmismatches_indel) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches;
		*best_ref_nmismatches_indel = ref_nmismatches1;
		*best_ref_nmismatches_j = ref_nmismatches2;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }
	  }
	  j++;
	}

	if (j < acceptorj_nsites && acceptorj_positions[j] == donori_positions[i]) {
	  /* Not an indel */
	  j++;
	}

	/* Insertions */
	while (j < acceptorj_nsites && acceptorj_positions[j] <= donori_positions[i] + max_insertionlen &&
	       (supportj = pos3 - acceptorj_positions[j]) >= MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	  debug1(printf("(A) Insertion: %d (%.6f) -- %d (%.6f).  Support: %d and %d\n",
			donori_positions[i],donori_probs[i],acceptorj_positions[j],acceptorj_probs[j],
			donori_positions[i] - pos5,pos3 - acceptorj_positions[j]));
	  probi = donori_probs[i];
	  probj = acceptorj_probs[j];
	  if (probi > MIN_SPLICE_PLUS_INDEL_PROB && probj > MIN_SPLICE_PLUS_INDEL_PROB) {
	    nindels = acceptorj_positions[j] - donori_positions[i]; /* Should be positive */

	    /* Try insertion on segmenti */
	    splice_qpos = acceptorj_positions[j];
	    if ((indel_pos = Indel_resolve_middle_insertion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							    segmenti_left,nindels,
							    /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							    /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							    /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							    pos5,/*pos3*/splice_qpos,querylength,
							    indelinfo,/*plusp:true*/true,genestrand,
							    /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find insertion on segmenti\n"));
	    } else {
	      assert(indel_pos > pos5);
	      assert(indel_pos + nindels < splice_qpos);
	      support_indel = splice_qpos - (indel_pos + nindels);
	      nmismatches_i = nmismatches1;
	      nmismatches_indel = nmismatches2; /* From indel to splice */
	      nmismatches_j =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
						  /*pos5*/splice_qpos,pos3,plusp,genestrand);
	      debug1(printf("  => (1) insertion indel_pos %d on segmenti with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supporti - support_indel - 3*nmismatches_i < MIN_SUPPORT_SPLICE || supportj - 3*nmismatches_j < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supporti - 3*(nmismatches_indel + nmismatches_i) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_indel + nmismatches_i + nmismatches_j) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches1;
		*best_ref_nmismatches_indel = ref_nmismatches2;
		*best_ref_nmismatches_j = ref_nmismatches;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }

	    /* Try insertion on segmentj */
	    splice_qpos = donori_positions[i];
	    if ((indel_pos = Indel_resolve_middle_insertion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							    segmentj_left + nindels,nindels,
							    /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							    /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							    /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							    /*pos5*/splice_qpos,pos3,querylength,
							    indelinfo,/*plusp:true*/true,genestrand,
							    /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find insertion on segmentj\n"));
	    } else {
	      assert(indel_pos > splice_qpos);
	      assert(indel_pos + nindels < pos3);
	      support_indel = indel_pos - splice_qpos;
	      nmismatches_i =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
						  pos5,/*pos3*/splice_qpos,plusp,genestrand);
	      nmismatches_indel = nmismatches1;	  /* From splice to indel */
	      nmismatches_j = nmismatches2;
	      debug1(printf("  => (2) insertion indel_pos %d on segmentj with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));
	      
	      if (supportj - support_indel - 3*nmismatches_j < MIN_SUPPORT_SPLICE || supporti - 3*nmismatches_i < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supportj - 3*(nmismatches_indel + nmismatches_j) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_i + nmismatches_j + nmismatches_indel) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches;
		*best_ref_nmismatches_indel = ref_nmismatches1;
		*best_ref_nmismatches_j = ref_nmismatches2;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }
	  }
	  j++;
	}
	j--;			/* Finish advance */

	i++;
      }
    }

  } else {
    /* Genomic antisense (verified) */
    if (antiacceptori_nsites == 0 || antidonorj_nsites == 0) {
      debug1(printf("Not returning %d with probi %f and probj %f\n",best_splice_qpos,*best_prob_i,*best_prob_j));
      *best_nmismatches_i = *best_nmismatches_j = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
      return -1;
    } else {
      antiacceptori_probs = spliceinfo->acceptor_probs_alloc;
      for (i = 0; i < antiacceptori_nsites; i++) {
	antiacceptori_probs[i] = Maxent_hr_antiacceptor_prob(segmenti_left + antiacceptori_positions[i],chroffset);
      }

      antidonorj_probs = spliceinfo->donor_probs_alloc;
      for (i = 0; i < antidonorj_nsites; i++) {
	antidonorj_probs[i] = Maxent_hr_antidonor_prob(segmentj_left + antidonorj_positions[i],chroffset);
      }

      i = j = 0;
      best_nmismatches = querylength;  /* was max_mismatches_allowed + 1 */
      while (i < antiacceptori_nsites && antiacceptori_positions[i] - pos5 < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	i++;
      }

      while (i < antiacceptori_nsites) {
	supporti = antiacceptori_positions[i] - pos5;

	/* Backup */
	while (j >= 0 && antidonorj_positions[j] + max_deletionlen > antiacceptori_positions[i]) {
	  j--;
	}
	j++;			/* Finish backup */

	/* Advance */
	while (j < antidonorj_nsites && antidonorj_positions[j] + max_deletionlen <= antiacceptori_positions[i]) {
	  j++;
	}

	/* Deletions */
	while (j < antidonorj_nsites && antidonorj_positions[j] < antiacceptori_positions[i] &&
	       (supportj = pos3 - antidonorj_positions[j]) >= MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	  debug1(printf("(B) Deletion: %d (%.6f) -- %d (%.6f).  Support: %d and %d\n",
			antiacceptori_positions[i],antiacceptori_probs[i],antidonorj_positions[j],antidonorj_probs[j],
			antiacceptori_positions[i] - pos5,pos3 - antidonorj_positions[j]));
	  probi = antiacceptori_probs[i];
	  probj = antidonorj_probs[j];
	  if (probi > MIN_SPLICE_PLUS_INDEL_PROB && probj > MIN_SPLICE_PLUS_INDEL_PROB) {
	    nindels = antidonorj_positions[j] - antiacceptori_positions[i]; /* Should be negative */

	    /* Try deletion on segmenti */
	    splice_qpos = antidonorj_positions[j];
	    if ((indel_pos = Indel_resolve_middle_deletion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							   segmenti_left,nindels,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							   pos5,/*pos3*/splice_qpos,querylength,
							   indelinfo,/*plusp:true*/true,genestrand,
							   /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find deletion on segmenti\n"));
	    } else {
	      assert(indel_pos > pos5);
	      assert(indel_pos < splice_qpos);
	      support_indel = splice_qpos - indel_pos;
	      nmismatches_i = nmismatches1;
	      nmismatches_indel = nmismatches2; /* From indel to splice */
	      nmismatches_j =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
						  /*pos5*/splice_qpos,pos3,plusp,genestrand);
	      debug1(printf("  => (3) deletion indel_pos %d on segmenti with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supporti - support_indel - 3*nmismatches_i < MIN_SUPPORT_SPLICE || supportj - 3*nmismatches_j < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supporti - 3*(nmismatches_indel + nmismatches_i) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_indel + nmismatches_i + nmismatches_j) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches1;
		*best_ref_nmismatches_indel = ref_nmismatches2;
		*best_ref_nmismatches_j = ref_nmismatches;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }

	    /* Try deletion on segmentj */
	    splice_qpos = antiacceptori_positions[i];
	    if ((indel_pos = Indel_resolve_middle_deletion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							   segmentj_left + nindels,nindels,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							   /*pos5*/splice_qpos,pos3,querylength,
							   indelinfo,/*plusp:true*/true,genestrand,
							   /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find deletion on segmentj\n"));
	    } else {
	      assert(indel_pos > splice_qpos);
	      assert(indel_pos < pos3);
	      support_indel = indel_pos - splice_qpos;
	      nmismatches_i =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
						  pos5,/*pos3*/splice_qpos,plusp,genestrand);
	      nmismatches_indel = nmismatches1; /* From splice to indel */
	      nmismatches_j = nmismatches2;
	      debug1(printf("  => (4) deletion indel_pos %d on segmentj with support %d+%d+%d and mismatches %d+%d+%d,  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));
	      
	      if (supportj - support_indel - 3*nmismatches_j < MIN_SUPPORT_SPLICE || supporti - 3*nmismatches_i < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supportj - 3*(nmismatches_indel + nmismatches_j) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_i + nmismatches_j + nmismatches_indel) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches;
		*best_ref_nmismatches_indel = ref_nmismatches1;
		*best_ref_nmismatches_j = ref_nmismatches2;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }
	  }
	  j++;
	}

	if (j < antidonorj_nsites && antidonorj_positions[j] == antiacceptori_positions[i]) {
	  /* Not an indel */
	  j++;
	}

	/* Insertions */
	while (j < antidonorj_nsites && antidonorj_positions[j] <= antiacceptori_positions[i] + max_insertionlen &&
	       (supportj = pos3 - antidonorj_positions[j]) >= MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	  debug1(printf("(B) Insertion: %d (%.6f) -- %d (%.6f).  Support: %d and %d\n",
			antiacceptori_positions[i],antiacceptori_probs[i],antidonorj_positions[j],antidonorj_probs[j],
			antiacceptori_positions[i] - pos5,pos3 - antidonorj_positions[j]));
	  probi = antiacceptori_probs[i];
	  probj = antidonorj_probs[j];
	  if (probi > MIN_SPLICE_PLUS_INDEL_PROB && probj > MIN_SPLICE_PLUS_INDEL_PROB) {
	    nindels = antidonorj_positions[j] - antiacceptori_positions[i]; /* Should be positive */

	    /* Try insertion on segmenti */
	    splice_qpos = antidonorj_positions[j];
	    if ((indel_pos = Indel_resolve_middle_insertion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							    segmenti_left,nindels,
							    /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							    /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							    /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							    pos5,/*pos3*/splice_qpos,querylength,
							    indelinfo,/*plusp:true*/true,genestrand,
							    /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find insertion on segmenti\n"));
	    } else {
	      assert(indel_pos > pos5);
	      assert(indel_pos + nindels < splice_qpos);
	      support_indel = splice_qpos - (indel_pos + nindels);
	      nmismatches_i = nmismatches1;
	      nmismatches_indel = nmismatches2; /* From indel to splice */
	      nmismatches_j =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
						  /*pos5*/splice_qpos,pos3,plusp,genestrand);
	      debug1(printf("  => (3) insertion indel_pos %d on segmenti with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supporti - support_indel - 3*nmismatches_i < MIN_SUPPORT_SPLICE || supportj - 3*nmismatches_j < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supporti - 3*(nmismatches_indel + nmismatches_i) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_indel + nmismatches_i + nmismatches_j) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches1;
		*best_ref_nmismatches_indel = ref_nmismatches2;
		*best_ref_nmismatches_j = ref_nmismatches;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }

	    /* Try insertion on segmentj */
	    splice_qpos = antiacceptori_positions[i];
	    if ((indel_pos = Indel_resolve_middle_insertion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							    segmentj_left + nindels,nindels,
							    /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							    /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							    /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							    /*pos5*/splice_qpos,pos3,querylength,
							    indelinfo,/*plusp:true*/true,genestrand,
							    /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find insertion on segmentj\n"));
	    } else {
	      assert(indel_pos > splice_qpos);
	      assert(indel_pos + nindels < pos3);
	      support_indel = indel_pos - splice_qpos;
	      nmismatches_i =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
						  pos5,/*pos3*/splice_qpos,plusp,genestrand);
	      nmismatches_indel = nmismatches1;	  /* From splice to indel */
	      nmismatches_j = nmismatches2;
	      debug1(printf("  => (4) insertion indel_pos %d on segmentj with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));
	      
	      if (supportj - support_indel - 3*nmismatches_j < MIN_SUPPORT_SPLICE || supporti - 3*nmismatches_i < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supportj - 3*(nmismatches_indel + nmismatches_j) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_i + nmismatches_j + nmismatches_indel) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches;
		*best_ref_nmismatches_indel = ref_nmismatches1;
		*best_ref_nmismatches_j = ref_nmismatches2;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }
	  }
	  j++;
	}
	j--;			/* Finish advance */

	i++;
      }
    }
  }

  if (*best_nindels == 0) {
    debug1(printf("Not returning splice plus indel, but resorting to best splice at %d with probi %f and probj %f\n",
		  best_splice_qpos,*best_prob_i,*best_prob_j));
  } else {
    debug1(printf("Returning splice at %d plus %d indels with probi %f and probj %f\n",
		  best_splice_qpos,*best_nindels,*best_prob_i,*best_prob_j));
  }
  return best_splice_qpos;
}


int
Splice_resolve_antisense (int *best_nindels, int *best_indel_pos, int *best_knowni_i, int *best_knowni_j,
			  int *best_nmismatches_i, int *best_nmismatches_j, int *best_nmismatches_indel,
			  int *best_ref_nmismatches_i, int *best_ref_nmismatches_j,
			  int *best_ref_nmismatches_indel, double *best_prob_i, double *best_prob_j,
			  
			  Univcoord_T segmenti_left, Univcoord_T segmentj_left, Univcoord_T chroffset,
			  
			  int pos5, int pos3, int querylength, Compress_T query_compress,
			  Indelinfo_T indelinfo, Spliceinfo_T spliceinfo,
			  bool plusp, int genestrand, int max_deletionlen, int max_insertionlen,
			  bool allow_indel_p) {
  int best_splice_qpos = -1, splice_qpos_start, splice_qpos_end, splice_qpos, i, j;

  int best_nmismatches, nmismatches, nmismatches1, nmismatches2;
  int best_ref_nmismatches, ref_nmismatches, ref_nmismatches1, ref_nmismatches2;
  int segmenti_nmismatches, segmentj_nmismatches;
  double best_prob, probi, probj;
  /* bool sufficient1p, sufficient2p; */

  int donori_nsites, acceptorj_nsites, antiacceptori_nsites, antidonorj_nsites;
  int *donori_positions, *acceptorj_positions, *antiacceptori_positions, *antidonorj_positions;
  int *donori_knowni, *acceptorj_knowni, *antiacceptori_knowni, *antidonorj_knowni;
  double *donori_probs, *acceptorj_probs, *antiacceptori_probs, *antidonorj_probs;

  int adj_supporti, adj_supportj, supporti, supportj, support_indel;
  int nindels, indel_pos;
  int nmismatches_indel, nmismatches_i, nmismatches_j;

  Intlist_T candidates = NULL, p;
#if 0
  int *intspace, *m;
  int *saved_knowni_i, *saved_knowni_j, *saved_segmenti_nmismatches, *saved_segmentj_nmismatches,
    *saved_segmenti_ref_nmismatches, *saved_segmentj_ref_nmismatches;
  double *saved_probi, *saved_probj;
#endif
  double prob;


  debug1(printf("Splice_resolve_antisense: Getting genome at lefti %u [%u] and leftj %u [%u] (diff: %d), range %d..%d, max_mismatches_allowed %d\n",
		segmenti_left,segmenti_left - chroffset,segmentj_left,segmentj_left - chroffset,
		segmentj_left-segmenti_left,pos5,pos3,max_mismatches_allowed));
  debug1(printf("novelsplicingp %d\n",novelsplicingp));

#if 0
  /* Now allocated in Spliceinfo_T */
  saved_probi = MALLOC(querylength * sizeof(double));
  saved_probj = MALLOC(querylength * sizeof(double));
  intspace = MALLOC(6 * querylength * sizeof(int));
  m = saved_knowni_i = &(intspace[0]);
  m = saved_knowni_j = &(m[querylength]);
  m = saved_segmenti_nmismatches = &(m[querylength]);
  m = saved_segmentj_nmismatches = &(m[querylength]);
  m = saved_segmenti_ref_nmismatches = &(m[querylength]);
  m = saved_segmentj_ref_nmismatches = &(m[querylength]);
#endif

  *best_nindels = 0;
  *best_indel_pos = -1;
  *best_knowni_i = *best_knowni_j = -1;
  *best_nmismatches_i = *best_nmismatches_j = *best_nmismatches_indel = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
  *best_ref_nmismatches_i = *best_ref_nmismatches_j = *best_ref_nmismatches_indel = -1;
  *best_prob_i = *best_prob_j = 0.0;

  /* Require separation from endpoints */
  splice_qpos_start = pos5 + 1;
  splice_qpos_end = pos3 - 1;
  if (splice_qpos_start < min_shortend) {
    splice_qpos_start = min_shortend;
  }
  if (splice_qpos_end > querylength - min_shortend) {
    splice_qpos_end = querylength - min_shortend;
  }

  if (plusp == false) {
    /* minus */
    /* Originally from plus strand.  No complement.  */
    /* Sense (End 1 to End 2) or Antisense (End 5 to End 6) */
    if (novelsplicingp == true && segmenti_left + splice_qpos_start >= DONOR_MODEL_LEFT_MARGIN) {
      donori_nsites = Genome_donor_positions(spliceinfo->donor_positions_alloc,spliceinfo->donor_knowni_alloc,
					     spliceinfo->segmenti_donor_knownpos,
					     spliceinfo->segmenti_donor_knowni,
					     segmenti_left,splice_qpos_start,splice_qpos_end);
      donori_positions = spliceinfo->donor_positions_alloc;
      donori_knowni = spliceinfo->donor_knowni_alloc;
    } else {
      donori_nsites = spliceinfo->segmenti_donor_nknown;
      donori_positions = spliceinfo->segmenti_donor_knownpos;
      donori_knowni = spliceinfo->segmenti_donor_knowni;
    }

#ifdef DEBUG1
    printf("Found %d donori sites:",donori_nsites);
    for (i = 0; i < donori_nsites; i++) {
      printf(" %d",donori_positions[i]);
      if (donori_knowni[i] >= 0) {
	printf(" (%d)",donori_knowni[i]);
      } else {
	probi = Maxent_hr_donor_prob(segmenti_left + donori_positions[i],chroffset);
	printf(" (%.6f)",probi);
      }
    }
    printf("\n");
#endif

    if (novelsplicingp == true && segmentj_left + splice_qpos_start >= ACCEPTOR_MODEL_LEFT_MARGIN) {
      acceptorj_nsites = Genome_acceptor_positions(spliceinfo->acceptor_positions_alloc,spliceinfo->acceptor_knowni_alloc,
						   spliceinfo->segmentj_acceptor_knownpos,
						   spliceinfo->segmentj_acceptor_knowni,
						   segmentj_left,splice_qpos_start,splice_qpos_end);
      acceptorj_positions = spliceinfo->acceptor_positions_alloc;
      acceptorj_knowni = spliceinfo->acceptor_knowni_alloc;
    } else {
      acceptorj_nsites = spliceinfo->segmentj_acceptor_nknown;
      acceptorj_positions = spliceinfo->segmentj_acceptor_knownpos;
      acceptorj_knowni = spliceinfo->segmentj_acceptor_knowni;
    }

#ifdef DEBUG1
    printf("Found %d acceptorj sites:",acceptorj_nsites);
    for (i = 0; i < acceptorj_nsites; i++) {
      printf(" %d",acceptorj_positions[i]);
      if (acceptorj_knowni[i] >= 0) {
	printf(" (%d)",acceptorj_knowni[i]);
      } else {
	probj = Maxent_hr_acceptor_prob(segmentj_left + acceptorj_positions[i],chroffset);
	printf(" (%.6f)",probj);
      }
    }
    printf("\n");
#endif

    i = j = 0;
    while (i < donori_nsites && j < acceptorj_nsites) {
      supporti = donori_positions[i] - pos5;
      supportj = pos3 - acceptorj_positions[j];

      if ((splice_qpos = donori_positions[i]) < acceptorj_positions[j]) {
	i++;
      } else if (splice_qpos > acceptorj_positions[j]) {
	j++;
      } else if (supporti < MIN_SUPPORT_SPLICE || supportj < MIN_SUPPORT_SPLICE) {
	/* Skip */
	i++; j++;
      } else {
	debug1(printf("splice matches at %d\n",splice_qpos));
	segmenti_nmismatches = spliceinfo->saved_segmenti_nmismatches[splice_qpos] =
	  Genome_count_mismatches_substring(&(spliceinfo->saved_segmenti_ref_nmismatches[splice_qpos]),
					    genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
					    pos5,/*pos3*/splice_qpos,plusp,genestrand);
	segmentj_nmismatches = spliceinfo->saved_segmentj_nmismatches[splice_qpos] =
	  Genome_count_mismatches_substring(&(spliceinfo->saved_segmentj_ref_nmismatches[splice_qpos]),
					    genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
					    /*pos5*/splice_qpos,pos3,plusp,genestrand);
	debug1(printf("%d mismatches on segmenti (%d..%d)\n",segmenti_nmismatches,pos5,splice_qpos));
	debug1(printf("%d mismatches on segmentj (%d..%d)\n",segmentj_nmismatches,splice_qpos,pos3));

	if ((spliceinfo->saved_knowni_i[splice_qpos] = donori_knowni[i]) >= 0) {
	  probi = spliceinfo->saved_probi[splice_qpos] = 1.0; /* Needs to be 1.0 for output */
	} else {
	  probi = spliceinfo->saved_probi[splice_qpos] = Maxent_hr_donor_prob(segmenti_left + splice_qpos,chroffset);
	}
	
	if ((spliceinfo->saved_knowni_j[splice_qpos] = acceptorj_knowni[j]) >= 0) {
	  probj = spliceinfo->saved_probj[splice_qpos] = 1.0; /* Needs to be 1.0 for output */
	} else {
	  probj = spliceinfo->saved_probj[splice_qpos] = Maxent_hr_acceptor_prob(segmentj_left + splice_qpos,chroffset);
	}

	adj_supporti = supporti - 3*segmenti_nmismatches;
	adj_supportj = supportj - 3*segmentj_nmismatches;
	if (sufficient_support_p(adj_supporti,probi) == false) {
	  debug1(printf("Skipping, because too many mismatches %d in segmenti\n",segmenti_nmismatches));
	} else if (sufficient_support_p(adj_supportj,probj) == false) {
	  debug1(printf("Skipping, because too many mismatches %d in segmentj\n",segmentj_nmismatches));
	} else {
	  candidates = Intlist_push(candidates,splice_qpos);
	  debug1(
		 if (plusp == true) {
		   printf("plus sense splice_qpos  %d, i.donor %f, j.acceptor %f\n",splice_qpos,probi,probj);
		 } else {
		   printf("minus antisense splice_qpos  %d, i.donor %f, j.acceptor %f\n",splice_qpos,probi,probj);
		 });
	}

	i++; j++;
      }
    }

  } else {
    /* plus */
    /* Originally from minus strand.  Complement. */
    /* Antisense (End 7 to End 8) or Sense (End 3 to End 4) */
    if (novelsplicingp == true && segmenti_left + splice_qpos_start >= ACCEPTOR_MODEL_RIGHT_MARGIN) {
      antiacceptori_nsites = Genome_antiacceptor_positions(spliceinfo->acceptor_positions_alloc,spliceinfo->acceptor_knowni_alloc,
							   spliceinfo->segmenti_antiacceptor_knownpos,
							   spliceinfo->segmenti_antiacceptor_knowni,
							   segmenti_left,splice_qpos_start,splice_qpos_end);
      antiacceptori_positions = spliceinfo->acceptor_positions_alloc;
      antiacceptori_knowni = spliceinfo->acceptor_knowni_alloc;
    } else {
      antiacceptori_nsites = spliceinfo->segmenti_antiacceptor_nknown;
      antiacceptori_positions = spliceinfo->segmenti_antiacceptor_knownpos;
      antiacceptori_knowni = spliceinfo->segmenti_antiacceptor_knowni;
    }

#ifdef DEBUG1
    printf("Found %d antiacceptori sites:",antiacceptori_nsites);
    for (i = 0; i < antiacceptori_nsites; i++) {
      printf(" %d",antiacceptori_positions[i]);
      if (antiacceptori_knowni[i] >= 0) {
	printf(" (%d)",antiacceptori_knowni[i]);
      } else {
	probi = Maxent_hr_antiacceptor_prob(segmenti_left + antiacceptori_positions[i],chroffset);
	printf(" (%.6f)",probi);
      }
    }
    printf("\n");
#endif

    if (novelsplicingp == true && segmentj_left + splice_qpos_start >= DONOR_MODEL_RIGHT_MARGIN) {
      antidonorj_nsites = Genome_antidonor_positions(spliceinfo->donor_positions_alloc,spliceinfo->donor_knowni_alloc,
						     spliceinfo->segmentj_antidonor_knownpos,
						     spliceinfo->segmentj_antidonor_knowni,
						     segmentj_left,splice_qpos_start,splice_qpos_end);
      antidonorj_positions = spliceinfo->donor_positions_alloc;
      antidonorj_knowni = spliceinfo->donor_knowni_alloc;
    } else {
      antidonorj_nsites = spliceinfo->segmentj_antidonor_nknown;
      antidonorj_positions = spliceinfo->segmentj_antidonor_knownpos;
      antidonorj_knowni = spliceinfo->segmentj_antidonor_knowni;
    }

#ifdef DEBUG1
    printf("Found %d antidonorj sites:",antidonorj_nsites);
    for (i = 0; i < antidonorj_nsites; i++) {
      printf(" %d",antidonorj_positions[i]);
      if (antidonorj_knowni[i] >= 0) {
	printf(" (%d)",antidonorj_knowni[i]);
      } else {
	probj = Maxent_hr_antidonor_prob(segmentj_left + antidonorj_positions[i],chroffset);
	printf(" (%.6f)",probj);
      }
    }
    printf("\n");
#endif

    i = j = 0;
    while (i < antiacceptori_nsites && j < antidonorj_nsites) {
      supporti = antiacceptori_positions[i] - pos5;
      supportj = pos3 - antidonorj_positions[j];

      if ((splice_qpos = antiacceptori_positions[i]) < antidonorj_positions[j]) {
	i++;
      } else if (splice_qpos > antidonorj_positions[j]) {
	j++;
      } else if (supporti < MIN_SUPPORT_SPLICE || supportj < MIN_SUPPORT_SPLICE) {
	/* Skip */
	i++; j++;
      } else {
	debug1(printf("splice matches at %d\n",splice_qpos));
	segmenti_nmismatches = spliceinfo->saved_segmenti_nmismatches[splice_qpos] =
	  Genome_count_mismatches_substring(&(spliceinfo->saved_segmenti_ref_nmismatches[splice_qpos]),
					    genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
					    pos5,/*pos3*/splice_qpos,plusp,genestrand);
	segmentj_nmismatches = spliceinfo->saved_segmentj_nmismatches[splice_qpos] =
	  Genome_count_mismatches_substring(&(spliceinfo->saved_segmentj_ref_nmismatches[splice_qpos]),
					    genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
					    /*pos5*/splice_qpos,pos3,plusp,genestrand);

	debug1(printf("%d mismatches on segmenti (%d..%d)\n",spliceinfo->saved_segmenti_nmismatches[splice_qpos],pos5,splice_qpos));
	debug1(printf("%d mismatches on segmentj (%d..%d)\n",spliceinfo->saved_segmentj_nmismatches[splice_qpos],splice_qpos,pos3));

	if ((spliceinfo->saved_knowni_i[splice_qpos] = antiacceptori_knowni[i]) >= 0) {
	  probi = spliceinfo->saved_probi[splice_qpos] = 1.0; /* Needs to be 1.0 for output */
	} else {
	  probi = spliceinfo->saved_probi[splice_qpos] = Maxent_hr_antiacceptor_prob(segmenti_left + splice_qpos,chroffset);
	}

	if ((spliceinfo->saved_knowni_j[splice_qpos] = antidonorj_knowni[j]) >= 0) {
	  probj = spliceinfo->saved_probj[splice_qpos] = 1.0; /* Needs to be 1.0 for output */
	} else {
	  probj = spliceinfo->saved_probj[splice_qpos] = Maxent_hr_antidonor_prob(segmentj_left + splice_qpos,chroffset);
	}

	adj_supporti = supporti - 3*segmenti_nmismatches;
	adj_supportj = supportj - 3*segmentj_nmismatches;
	if (sufficient_support_p(adj_supporti,probi) == false) {
	  debug1(printf("Skipping, because too many mismatches %d in segmenti\n",segmenti_nmismatches));
	} else if (sufficient_support_p(adj_supportj,probj) == false) {
	  debug1(printf("Skipping, because too many mismatches %d in segmentj\n",segmentj_nmismatches));
	} else {
	  candidates = Intlist_push(candidates,splice_qpos);
	  debug1(
		 if (plusp == true) {
		   printf("plus antisense splice_qpos  %d, j.donor %f, i.acceptor %f\n",splice_qpos,probj,probi);
		 } else {
		   printf("minus sense splice_qpos  %d, j.donor %f, i.acceptor %f\n",splice_qpos,probj,probi);
		 });
	}

	i++; j++;
      }
    }
  }


  /* Iterate through candidates */
  best_nmismatches = best_ref_nmismatches = querylength;  /* was max_mismatches_allowed */;
  best_prob = 0.0;

  for (p = candidates; p != NULL; p = Intlist_next(p)) {
    splice_qpos = Intlist_head(p);
    if ((nmismatches = spliceinfo->saved_segmenti_nmismatches[splice_qpos] + spliceinfo->saved_segmentj_nmismatches[splice_qpos]) < best_nmismatches) {
      best_nmismatches = nmismatches;
    }
  }
  debug1(printf("best_nmismatches is %d\n",best_nmismatches));
  
  for (p = candidates; p != NULL; p = Intlist_next(p)) {
    splice_qpos = Intlist_head(p);
    if (spliceinfo->saved_segmenti_nmismatches[splice_qpos] + spliceinfo->saved_segmentj_nmismatches[splice_qpos] > best_nmismatches + MISMATCHES_SLOP) {
      /* Too many mismatches */
    } else if ((prob = spliceinfo->saved_probi[splice_qpos] + spliceinfo->saved_probj[splice_qpos]) > best_prob) {
      best_prob = prob;
      best_splice_qpos = splice_qpos;
    }
  }
  
  if (best_splice_qpos >= 0) {
    /* Success */
    debug1(printf("Success at splice_qpos %d\n",best_splice_qpos));
    /* best_donor_splicecoord = segmenti_left + splice_qpos; */
    /* best_acceptor_splicecoord = segmentj_left + splice_qpos; */
    *best_knowni_i = spliceinfo->saved_knowni_i[best_splice_qpos]; /* antiacceptor */
    *best_knowni_j = spliceinfo->saved_knowni_j[best_splice_qpos]; /* antidonor */
    *best_prob_i = spliceinfo->saved_probi[best_splice_qpos]; /* acceptor_prob */
    *best_prob_j = spliceinfo->saved_probj[best_splice_qpos]; /* donor_prob */
    *best_nmismatches_i = spliceinfo->saved_segmenti_nmismatches[best_splice_qpos];
    *best_nmismatches_j = spliceinfo->saved_segmentj_nmismatches[best_splice_qpos];
    *best_ref_nmismatches_i = spliceinfo->saved_segmenti_ref_nmismatches[best_splice_qpos];
    *best_ref_nmismatches_j = spliceinfo->saved_segmentj_ref_nmismatches[best_splice_qpos];
  }

  Intlist_free(&candidates);
#if 0
  FREE(intspace);
  FREE(saved_probi);
  FREE(saved_probj);
#endif

  debug1(printf("best_knowni_i is %d and best_knowni_j is %d\n",*best_knowni_i,*best_knowni_j));

  if (*best_prob_i > AUTO_SPLICE_PROB_HIGH && *best_prob_j > AUTO_SPLICE_PROB_LOW) {
    debug1(printf("Automatic accept: returning %d with probi %f and probj %f\n",best_splice_qpos,*best_prob_i,*best_prob_j));
    debug1(printf("nmismatches %d and %d\n",*best_nmismatches_i,*best_nmismatches_j));
    return best_splice_qpos;
  } else if (*best_prob_j > AUTO_SPLICE_PROB_HIGH && *best_prob_i > AUTO_SPLICE_PROB_LOW) {
    debug1(printf("Automatic accept: returning %d with probi %f and probj %f\n",best_splice_qpos,*best_prob_i,*best_prob_j));
    debug1(printf("nmismatches %d and %d\n",*best_nmismatches_i,*best_nmismatches_j));
    return best_splice_qpos;
  } else if (allow_indel_p == false) {
    debug1(printf("Not returning %d with probi %f and probj %f because allow_indel_p is false\n",best_splice_qpos,*best_prob_i,*best_prob_j));
    *best_nmismatches_i = *best_nmismatches_j = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
    return -1;
  } else {
    /* Find an indel below to see if something is better */
  }

  if (plusp == false) {
    /* Genomic sense (inferred from antisense code) */
    if (donori_nsites == 0 || acceptorj_nsites == 0) {
      debug1(printf("Not returning %d with probi %f and probj %f because nsites is 0\n",best_splice_qpos,*best_prob_i,*best_prob_j));
      *best_nmismatches_i = *best_nmismatches_j = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
      return -1;
    } else {
      donori_probs = spliceinfo->donor_probs_alloc;
      for (i = 0; i < donori_nsites; i++) {
	donori_probs[i] = Maxent_hr_donor_prob(segmenti_left + donori_positions[i],chroffset);
      }

      acceptorj_probs = spliceinfo->acceptor_probs_alloc;
      for (i = 0; i < acceptorj_nsites; i++) {
	acceptorj_probs[i] = Maxent_hr_acceptor_prob(segmentj_left + acceptorj_positions[i],chroffset);
      }

      i = j = 0;
      best_nmismatches = querylength;  /* was max_mismatches_allowed + 1 */
      while (i < donori_nsites && donori_positions[i] - pos5 < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	i++;
      }

      while (i < donori_nsites) {
	supporti = donori_positions[i] - pos5;

	/* Backup */
	while (j >= 0 && acceptorj_positions[j] + max_deletionlen > donori_positions[i]) {
	  j--;
	}
	j++;			/* Finish backup */

	/* Advance */
	while (j < acceptorj_nsites && acceptorj_positions[j] + max_deletionlen <= donori_positions[i]) {
	  j++;
	}

	/* Deletions */
	while (j < acceptorj_nsites && acceptorj_positions[j] < donori_positions[i] &&
	       (supportj = pos3 - acceptorj_positions[j]) >= MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	  debug1(printf("(C) Deletion: %d (%.6f) -- %d (%.6f).  Support: %d and %d\n",
			donori_positions[i],donori_probs[i],acceptorj_positions[j],acceptorj_probs[j],
			donori_positions[i] - pos5,pos3 - acceptorj_positions[j]));
	  probi = donori_probs[i];
	  probj = acceptorj_probs[j];
	  if (probi > MIN_SPLICE_PLUS_INDEL_PROB && probj > MIN_SPLICE_PLUS_INDEL_PROB) {
	    nindels = acceptorj_positions[j] - donori_positions[i]; /* Should be negative */

	    /* Try deletion on segmenti */
	    splice_qpos = acceptorj_positions[j];
	    if ((indel_pos = Indel_resolve_middle_deletion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							   segmenti_left,nindels,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							   pos5,/*pos3*/splice_qpos,querylength,
							   indelinfo,/*plusp:true*/true,genestrand,
							   /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find deletion on segmenti\n"));
	    } else {
	      assert(indel_pos > pos5);
	      assert(indel_pos < splice_qpos);
	      support_indel = splice_qpos - indel_pos;
	      nmismatches_i = nmismatches1;
	      nmismatches_indel = nmismatches2; /* From indel to splice */
	      nmismatches_j =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
						  /*pos5*/splice_qpos,pos3,plusp,genestrand);
	      debug1(printf("  => (5) deletion indel_pos %d on segmenti with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supporti - support_indel - 3*nmismatches_i < MIN_SUPPORT_SPLICE || supportj - 3*nmismatches_j < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supporti - 3*(nmismatches_indel + nmismatches_i) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_indel + nmismatches_i + nmismatches_j) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches1;
		*best_ref_nmismatches_indel = ref_nmismatches2;
		*best_ref_nmismatches_j = ref_nmismatches;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }

	    /* Try deletion on segmentj */
	    splice_qpos = donori_positions[i];
	    if ((indel_pos = Indel_resolve_middle_deletion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							   segmentj_left + nindels,nindels,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							   /*pos5*/splice_qpos,pos3,querylength,
							   indelinfo,/*plusp:true*/true,genestrand,
							   /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find deletion on segmentj\n"));
	    } else {
	      assert(indel_pos > splice_qpos);
	      assert(indel_pos < pos3);
	      support_indel = indel_pos - splice_qpos;
	      nmismatches_i =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
						  pos5,/*pos3*/splice_qpos,plusp,genestrand);
	      nmismatches_indel = nmismatches1;	  /* From splice to indel */
	      nmismatches_j = nmismatches2;
	      debug1(printf("  => (6) deletion indel_pos %d on segmentj with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supportj - support_indel - 3*nmismatches_j < MIN_SUPPORT_SPLICE || supporti - 3*nmismatches_i < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supportj - 3*(nmismatches_indel + nmismatches_j) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_i + nmismatches_j + nmismatches_indel) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches;
		*best_ref_nmismatches_indel = ref_nmismatches1;
		*best_ref_nmismatches_j = ref_nmismatches2;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }
	  }
	  j++;
	}

	if (j < acceptorj_nsites && acceptorj_positions[j] == donori_positions[i]) {
	  /* Not an indel */
	  j++;
	}

	/* Insertions */
	while (j < acceptorj_nsites && acceptorj_positions[j] <= donori_positions[i] + max_insertionlen &&
	       (supportj = pos3 - acceptorj_positions[j]) >= MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	  debug1(printf("(C) Insertion: %d (%.6f) -- %d (%.6f).  Support: %d and %d\n",
			donori_positions[i],donori_probs[i],acceptorj_positions[j],acceptorj_probs[j],
			donori_positions[i] - pos5,pos3 - acceptorj_positions[j]));
	  probi = donori_probs[i];
	  probj = acceptorj_probs[j];
	  if (probi > MIN_SPLICE_PLUS_INDEL_PROB && probj > MIN_SPLICE_PLUS_INDEL_PROB) {
	    nindels = acceptorj_positions[j] - donori_positions[i]; /* Should be positive */

	    /* Try insertion on segmenti */
	    splice_qpos = acceptorj_positions[j];
	    if ((indel_pos = Indel_resolve_middle_insertion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							    segmenti_left,nindels,
							    /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							    /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							    /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							    pos5,/*pos3*/splice_qpos,querylength,
							    indelinfo,/*plusp:true*/true,genestrand,
							    /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find insertion on segmenti\n"));
	    } else {
	      assert(indel_pos > pos5);
	      assert(indel_pos + nindels < splice_qpos);
	      support_indel = splice_qpos - (indel_pos + nindels);
	      nmismatches_i = nmismatches1;
	      nmismatches_indel = nmismatches2; /* From indel to splice */
	      nmismatches_j =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
						  /*pos5*/splice_qpos,pos3,plusp,genestrand);
	      debug1(printf("  => (5) insertion indel_pos %d on segmenti with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supporti - support_indel - 3*nmismatches_i < MIN_SUPPORT_SPLICE || supportj - 3*nmismatches_j < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supporti - 3*(nmismatches_indel + nmismatches_i) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_indel + nmismatches_i + nmismatches_j) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches1;
		*best_ref_nmismatches_indel = ref_nmismatches2;
		*best_ref_nmismatches_j = ref_nmismatches;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }

	    /* Try insertion on segmentj */
	    splice_qpos = donori_positions[i];
	    if ((indel_pos = Indel_resolve_middle_insertion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							    segmentj_left + nindels,nindels,
							    /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							    /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							    /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							    /*pos5*/splice_qpos,pos3,querylength,
							    indelinfo,/*plusp:true*/true,genestrand,
							    /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find insertion on segmentj\n"));
	    } else {
	      assert(indel_pos > splice_qpos);
	      assert(indel_pos + nindels < pos3);
	      support_indel = indel_pos - splice_qpos;
	      nmismatches_i =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
						  pos5,/*pos3*/splice_qpos,plusp,genestrand);
	      nmismatches_indel = nmismatches1;	  /* From splice to indel */
	      nmismatches_j = nmismatches2;
	      debug1(printf("  => (6) insertion indel_pos %d on segmentj with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));
	      
	      if (supportj - support_indel - 3*nmismatches_j < MIN_SUPPORT_SPLICE || supporti - 3*nmismatches_i < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supportj - 3*(nmismatches_indel + nmismatches_j) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_i + nmismatches_j + nmismatches_indel) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches;
		*best_ref_nmismatches_indel = ref_nmismatches1;
		*best_ref_nmismatches_j = ref_nmismatches2;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }
	  }
	  j++;
	}
	j--;			/* Finish advance */

	i++;
      }
    }

  } else {
    /* Genomic antisense (verified) */
    if (antiacceptori_nsites == 0 || antidonorj_nsites == 0) {
      debug1(printf("Not returning %d with probi %f and probj %f because nsites is 0\n",best_splice_qpos,*best_prob_i,*best_prob_j));
      *best_nmismatches_i = *best_nmismatches_j = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
      return -1;
    } else {
      antiacceptori_probs = spliceinfo->acceptor_probs_alloc;
      for (i = 0; i < antiacceptori_nsites; i++) {
	antiacceptori_probs[i] = Maxent_hr_antiacceptor_prob(segmenti_left + antiacceptori_positions[i],chroffset);
      }

      antidonorj_probs = spliceinfo->donor_probs_alloc;
      for (i = 0; i < antidonorj_nsites; i++) {
	antidonorj_probs[i] = Maxent_hr_antidonor_prob(segmentj_left + antidonorj_positions[i],chroffset);
      }

      i = j = 0;
      best_nmismatches = querylength;  /* was max_mismatches_allowed + 1 */
      while (i < antiacceptori_nsites && antiacceptori_positions[i] - pos5 < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	i++;
      }

      while (i < antiacceptori_nsites) {
	supporti = antiacceptori_positions[i] - pos5;

	/* Backup */
	while (j >= 0 && antidonorj_positions[j] + max_deletionlen > antiacceptori_positions[i]) {
	  j--;
	}
	j++;			/* Finish backup */

	/* Advance */
	while (j < antidonorj_nsites && antidonorj_positions[j] + max_deletionlen <= antiacceptori_positions[i]) {
	  j++;
	}

	/* Deletions */
	while (j < antidonorj_nsites && antidonorj_positions[j] < antiacceptori_positions[i] &&
	       (supportj = pos3 - antidonorj_positions[j]) >= MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	  debug1(printf("(D) Deletion: %d (%.6f) -- %d (%.6f).  Support: %d and %d\n",
			antiacceptori_positions[i],antiacceptori_probs[i],antidonorj_positions[j],antidonorj_probs[j],
			antiacceptori_positions[i] - pos5,pos3 - antidonorj_positions[j]));
	  probi = antiacceptori_probs[i];
	  probj = antidonorj_probs[j];
	  if (probi > MIN_SPLICE_PLUS_INDEL_PROB && probj > MIN_SPLICE_PLUS_INDEL_PROB) {
	    nindels = antidonorj_positions[j] - antiacceptori_positions[i]; /* Should be negative */

	    /* Try deletion on segmenti */
	    splice_qpos = antidonorj_positions[j];
	    if ((indel_pos = Indel_resolve_middle_deletion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							   segmenti_left,nindels,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							   pos5,/*pos3*/splice_qpos,querylength,
							   indelinfo,/*plusp:true*/true,genestrand,
							   /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find deletion on segmenti\n"));
	    } else {
	      assert(indel_pos > pos5);
	      assert(indel_pos < splice_qpos);
	      support_indel = splice_qpos - indel_pos;
	      nmismatches_i = nmismatches1;
	      nmismatches_indel = nmismatches2; /* From indel to splice */
	      nmismatches_j =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
						  /*pos5*/splice_qpos,pos3,plusp,genestrand);
	      debug1(printf("  => (7) deletion indel_pos %d on segmenti with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supporti - support_indel - 3*nmismatches_i < MIN_SUPPORT_SPLICE || supportj - 3*nmismatches_j < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supporti - 3*(nmismatches_indel + nmismatches_i) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_indel + nmismatches_i + nmismatches_j) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches1;
		*best_ref_nmismatches_indel = ref_nmismatches2;
		*best_ref_nmismatches_j = ref_nmismatches;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }

	    /* Try deletion on segmentj */
	    splice_qpos = antiacceptori_positions[i];
	    if ((indel_pos = Indel_resolve_middle_deletion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							   segmentj_left + nindels,nindels,
							   /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							   /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							   /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							   /*pos5*/splice_qpos,pos3,querylength,
							   indelinfo,/*plusp:true*/true,genestrand,
							   /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find deletion on segmentj\n"));
	    } else {
	      assert(indel_pos > splice_qpos);
	      assert(indel_pos < pos3);
	      support_indel = indel_pos - splice_qpos;
	      nmismatches_i =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
						  pos5,/*pos3*/splice_qpos,plusp,genestrand);
	      nmismatches_indel = nmismatches1;	  /* From splice to indel */
	      nmismatches_j = nmismatches2;
	      debug1(printf("  => (8) deletion indel_pos %d on segmentj with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));
	      
	      if (supportj - support_indel - 3*nmismatches_j < MIN_SUPPORT_SPLICE || supporti - 3*nmismatches_i < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supportj - 3*(nmismatches_indel + nmismatches_j) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_i + nmismatches_j + nmismatches_indel) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches;
		*best_ref_nmismatches_indel = ref_nmismatches1;
		*best_ref_nmismatches_j = ref_nmismatches2;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }
	  }
	  j++;
	}

	if (j < antidonorj_nsites && antidonorj_positions[j] == antiacceptori_positions[i]) {
	  /* Not an indel */
	  j++;
	}

	/* Insertions */
	while (j < antidonorj_nsites && antidonorj_positions[j] <= antiacceptori_positions[i] + max_insertionlen &&
	       (supportj = pos3 - antidonorj_positions[j]) >= MIN_SUPPORT_SPLICE_PLUS_INDEL) {
	  debug1(printf("(D) Insertion: %d (%.6f) -- %d (%.6f).  Support: %d and %d\n",
			antiacceptori_positions[i],antiacceptori_probs[i],antidonorj_positions[j],antidonorj_probs[j],
			antiacceptori_positions[i] - pos5,pos3 - antidonorj_positions[j]));
	  probi = antiacceptori_probs[i];
	  probj = antidonorj_probs[j];
	  if (probi > MIN_SPLICE_PLUS_INDEL_PROB && probj > MIN_SPLICE_PLUS_INDEL_PROB) {
	    nindels = antidonorj_positions[j] - antiacceptori_positions[i]; /* Should be positive */

	    /* Try insertion on segmenti */
	    splice_qpos = antidonorj_positions[j];
	    if ((indel_pos = Indel_resolve_middle_insertion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							    segmenti_left,nindels,
							    /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							    /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							    /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							    pos5,/*pos3*/splice_qpos,querylength,
							    indelinfo,/*plusp:true*/true,genestrand,
							    /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find insertion on segmenti\n"));
	    } else {
	      assert(indel_pos > pos5);
	      assert(indel_pos + nindels < splice_qpos);
	      support_indel = splice_qpos - (indel_pos + nindels);
	      nmismatches_i = nmismatches1;
	      nmismatches_indel = nmismatches2; /* From indel to splice */
	      nmismatches_j =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmentj_left,
						  /*pos5*/splice_qpos,pos3,plusp,genestrand);
	      debug1(printf("  => (7) insertion indel_pos %d on segmenti with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));

	      if (supporti - support_indel - 3*nmismatches_i < MIN_SUPPORT_SPLICE || supportj - 3*nmismatches_j < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supporti - 3*(nmismatches_indel + nmismatches_i) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_indel + nmismatches_i + nmismatches_j) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches1;
		*best_ref_nmismatches_indel = ref_nmismatches2;
		*best_ref_nmismatches_j = ref_nmismatches;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }

	    /* Try insertion on segmentj */
	    splice_qpos = antiacceptori_positions[i];
	    if ((indel_pos = Indel_resolve_middle_insertion(&nmismatches1,&nmismatches2,&ref_nmismatches1,&ref_nmismatches2,
							    segmentj_left + nindels,nindels,
							    /*mismatch_positions_left*/NULL,/*nmismatches_left*/0,
							    /*mismatch_positions_right*/NULL,/*nmismatches_right*/0,
							    /*ome*/genomebits,/*ome_alt*/genomebits_alt,query_compress,
							    /*pos5*/splice_qpos,pos3,querylength,
							    indelinfo,/*plusp:true*/true,genestrand,
							    /*want_lowest_coordinate_p*/true)) < 0) {
	      debug1(printf("Could not find insertion on segmentj\n"));
	    } else {
	      assert(indel_pos > splice_qpos);
	      assert(indel_pos + nindels < pos3);
	      support_indel = indel_pos - splice_qpos;
	      nmismatches_i =
		Genome_count_mismatches_substring(&ref_nmismatches,genomebits,genomebits_alt,query_compress,/*left*/segmenti_left,
						  pos5,/*pos3*/splice_qpos,plusp,genestrand);
	      nmismatches_indel = nmismatches1;	  /* From splice to indel */
	      nmismatches_j = nmismatches2;
	      debug1(printf("  => (8) insertion indel_pos %d on segmentj with support %d+%d+%d and mismatches %d+%d+%d.  ",
			    indel_pos,supporti,support_indel,supportj,nmismatches_i,nmismatches_indel,nmismatches_j));
	      
	      if (supportj - support_indel - 3*nmismatches_j < MIN_SUPPORT_SPLICE || supporti - 3*nmismatches_i < MIN_SUPPORT_SPLICE) {
		debug1(printf("Not enough support on ends after mismatches\n"));
	      } else if (supportj - 3*(nmismatches_indel + nmismatches_j) < MIN_SUPPORT_SPLICE_PLUS_INDEL) {
		debug1(printf("Not enough support in splice+indel after mismatches\n"));
	      } else if (support_indel - 3*nmismatches_indel < MIN_SUPPORT_INDEL) {
		debug1(printf("Not enough support in indel after mismatches\n"));
	      } else if ((nmismatches = nmismatches_i + nmismatches_j + nmismatches_indel) >= best_nmismatches) {
		debug1(printf("nmismatches %d >= best_nmismatches %d\n",nmismatches,best_nmismatches));
	      } else {
		debug1(printf("Enough support after mismatches\n"));
		*best_nindels = nindels;
		*best_indel_pos = indel_pos;
		*best_prob_i = probi;
		*best_prob_j = probj;
		*best_nmismatches_i = nmismatches_i;
		*best_nmismatches_indel = nmismatches_indel;
		*best_nmismatches_j = nmismatches_j;
		*best_ref_nmismatches_i = ref_nmismatches;
		*best_ref_nmismatches_indel = ref_nmismatches1;
		*best_ref_nmismatches_j = ref_nmismatches2;
		best_splice_qpos = splice_qpos;
		best_nmismatches = nmismatches;
	      }
	    }
	  }
	  j++;
	}
	j--;			/* Finish advance */

	i++;
      }
    }
  }

  if (*best_nindels == 0) {
    debug1(printf("Not returning splice plus indel, but resorting to best splice at %d with probi %f and probj %f\n",
		  best_splice_qpos,*best_prob_i,*best_prob_j));
  } else {
    debug1(printf("Returning splice at %d plus %d indels with probi %f and probj %f\n",
		  best_splice_qpos,*best_nindels,*best_prob_i,*best_prob_j));
  }
  return best_splice_qpos;
}


static double
splice_prob_eval (int *sensedir, double *best_prob_i, double *best_prob_j, int splice_pos,
		  Univcoord_T segmenti_left, Univcoord_T segmentj_left,
		  Univcoord_T segmenti_chroffset, Univcoord_T segmentj_chroffset,
		  int querylength, bool plusp_i, bool plusp_j) {
  double sense_prob, antisense_prob, probi, probj;

  debug2(printf("\nplusp %d and %d\n",plusp_i,plusp_j));

  /* SENSE_FORWARD */
  if (plusp_i == true) {
    probi = Maxent_hr_donor_prob(segmenti_left + splice_pos,segmenti_chroffset);
  } else {
    probi = Maxent_hr_antidonor_prob(segmenti_left + querylength - splice_pos,segmenti_chroffset); /* correct */
  }
  if (plusp_j == true) {
    probj = Maxent_hr_acceptor_prob(segmentj_left + splice_pos,segmentj_chroffset);
  } else {
    probj = Maxent_hr_antiacceptor_prob(segmentj_left + querylength - splice_pos,segmentj_chroffset); /* correct */
  }
#if 0
  debug2(printf("sense forward: %f + %f at %u..%u\n",
		probi,probj,segmenti_left + querylength - splice_pos,segmentj_left + querylength - splice_pos));
#endif
  *best_prob_i = probi;
  *best_prob_j = probj;
  sense_prob = probi + probj;
  
  /* SENSE_ANTI */
  if (plusp_i == true) {
    probi = Maxent_hr_antiacceptor_prob(segmenti_left + splice_pos,segmenti_chroffset); /* correct */
  } else {
    probi = Maxent_hr_acceptor_prob(segmenti_left + querylength - splice_pos,segmenti_chroffset);
  }
  if (plusp_j == true) {
    probj = Maxent_hr_antidonor_prob(segmentj_left + splice_pos,segmentj_chroffset); /* correct */
  } else {
    probj = Maxent_hr_donor_prob(segmentj_left + querylength - splice_pos,segmentj_chroffset);
  }
#if 0
  debug2(printf("sense anti: %f + %f at %u..%u\n",
		probi,probj,segmenti_left + splice_pos,segmentj_left + splice_pos));
#endif
  antisense_prob = probi + probj;

  if (sense_prob > 1.80 && antisense_prob < 1.60) {
    *sensedir = SENSE_FORWARD;
  } else if (antisense_prob > 1.80 && sense_prob < 1.60) {
    *sensedir = SENSE_ANTI;
  } else {
    *sensedir = SENSE_NULL;
  }

  if (antisense_prob > sense_prob) {
    *best_prob_i = probi;
    *best_prob_j = probj;
    return antisense_prob;
  } else {
    return sense_prob;
  }
}


int
Splice_resolve_distant (int *best_nmismatches_i, int *best_nmismatches_j,
			double *best_prob_i, double *best_prob_j, int *sensedir_distant_guess,

			int *mismatch_positions_i, int segmenti_nmismatches,
			int *mismatch_positions_j, int segmentj_nmismatches,

			Univcoord_T segmenti_left, Univcoord_T segmentj_left,
			Univcoord_T segmenti_chroffset, Univcoord_T segmentj_chroffset,

			int pos5, int pos3, int splice_pos_start, int splice_pos_end, int querylength,
			bool plusp_i, bool plusp_j) {

  int best_splice_pos = -1, prev_splice_pos, next_splice_pos, low_splice_pos, high_splice_pos, splice_pos;
  int lowi, highi, lowj, highj, i, j;
  int supporti, supportj;
  int best_nmismatches, nmismatches, nmismatches_i, nmismatches_j;
  double best_prob, prob, prob_i, prob_j;
  int sensedir;
  bool donep;
#ifdef DEBUG2
  int k;
#endif


#ifdef DEBUG2
  printf("Entering Splice_resolve_distant with splice_pos_start %d and splice_pos_end %d\n",
	 splice_pos_start,splice_pos_end);

  printf("startfrag has %d mismatches at:",segmenti_nmismatches);
  for (k = 0; k <= segmenti_nmismatches; k++) {
    printf(" %d",mismatch_positions_i[k]);
  }
  printf("\n");

  printf("endfrag has %d mismatches at:",segmentj_nmismatches);
  for (k = 0; k <= segmentj_nmismatches; k++) {
    printf(" %d",mismatch_positions_j[k]);
  }
  printf("\n");
  printf("\n");
#endif


  *best_nmismatches_i = *best_nmismatches_j = -1; /* Indicates that calling procedure needs to compute numbers of mismatches  */
  *best_prob_i = *best_prob_j = 0.0;

  /* Require separation from endpoints */
  /* May want a different min_shortend for distant */
  debug2(printf("splice_pos boundaries: %d..%d => ",splice_pos_start,splice_pos_end));
  if (splice_pos_start < pos5 + min_shortend_distant) {
    splice_pos_start = pos5 + min_shortend_distant;
  }
  if (splice_pos_end > pos3 - min_shortend_distant) {
    splice_pos_end = pos3 - min_shortend_distant;
  }
  debug2(printf("%d..%d\n",splice_pos_start,splice_pos_end));

  if (splice_pos_start >= splice_pos_end) {
    debug2(printf("Endpoints do not work\n"));
    return -1;
  }

  /* Initialize to splice_pos_start */
  nmismatches_i = 0;
  nmismatches_j = segmentj_nmismatches + 1;

  i = j = 0;
  lowi = highi = 0;
  lowj = highj = 0;
  while (i <= segmenti_nmismatches && mismatch_positions_i[i] < splice_pos_start) {
    lowi = highi = i;
    nmismatches_i++; i++;
  }
  while (j <= segmentj_nmismatches && mismatch_positions_j[j] < splice_pos_start) {
    lowj = highj = j;
    nmismatches_j--; j++;
  }
  best_nmismatches = querylength;
    

  donep = false;
  while (donep == false && i <= segmenti_nmismatches && j <= segmentj_nmismatches) {
    if ((splice_pos = mismatch_positions_i[i]) < mismatch_positions_j[j]) {
      debug2(printf("(1 i) Considering break pos prev through %d, which would give %d+%d mismatches\n",
		    splice_pos,nmismatches_i,nmismatches_j));
      if (splice_pos > splice_pos_end) {
	donep = true;
      } else if (nmismatches_i + nmismatches_j < best_nmismatches) {
	lowi = highi = i;
	lowj = highj = j;
	best_nmismatches = nmismatches_i + nmismatches_j;
      } else if (nmismatches_i + nmismatches_j == best_nmismatches) {
	highi = i;
	highj = j;
      }
      nmismatches_i++; i++;
	
    } else if ((splice_pos = mismatch_positions_j[j]) < mismatch_positions_i[i]) {
      debug2(printf("(1 j) Considering break pos prev through %d, which would give %d+%d mismatches\n",
		    splice_pos,nmismatches_i,nmismatches_j));
      if (splice_pos > splice_pos_end) {
	donep = true;
      } else if (nmismatches_i + nmismatches_j < best_nmismatches) {
	lowi = highi = i;
	lowj = highj = j;
	best_nmismatches = nmismatches_i + nmismatches_j;
      } else if (nmismatches_i + nmismatches_j == best_nmismatches) {
	highi = i;
	highj = j;
      }
      nmismatches_j--; j++;
	
    } else {
      debug2(printf("(1 equal) Considering break pos prev through %d, which would give %d+%d mismatches\n",
		    splice_pos,nmismatches_i,nmismatches_j));
      if (splice_pos > splice_pos_end) {
	donep = true;
      } else if (nmismatches_i + nmismatches_j < best_nmismatches) {
	lowi = highi = i;
	lowj = highj = j;
	best_nmismatches = nmismatches_i + nmismatches_j;
      } else if (nmismatches_i + nmismatches_j == best_nmismatches) {
	highi = i;
	highj = j;
      }
      nmismatches_i++; i++;
      nmismatches_j--; j++;
    }
  }


  if (i > segmenti_nmismatches && j > segmentj_nmismatches) {
    debug2(printf("Reached both ends\n"));
    
  } else if (j > segmentj_nmismatches) {
    while (donep == false && i <= segmenti_nmismatches) {
      debug2(printf("(2 i) Considering break pos prev through %d, which would give %d+%d mismatches\n",
		    mismatch_positions_i[i],nmismatches_i,nmismatches_j));
      if ((splice_pos = mismatch_positions_i[i]) > splice_pos_end) {
	donep = true;
      } else if (nmismatches_i + nmismatches_j < best_nmismatches) {
	lowi = highi = i;
	best_nmismatches = nmismatches_i + nmismatches_j;	
      } else if (nmismatches_i + nmismatches_j == best_nmismatches) {
	highi = i;
      }
      nmismatches_i++; i++;
    }
    
  } else if (i > segmenti_nmismatches) {
    while (donep == false && j <= segmentj_nmismatches) {
      debug2(printf("(2 j) Considering break pos prev through %d, which would give %d+%d mismatches\n",
		    mismatch_positions_j[j],nmismatches_i,nmismatches_j));
      if ((splice_pos = mismatch_positions_j[j]) > splice_pos_end) {
	donep = true;
      } else if (nmismatches_i + nmismatches_j < best_nmismatches) {
	lowj = highj = j;
	best_nmismatches = nmismatches_i + nmismatches_j;
      } else if (nmismatches_i + nmismatches_j == best_nmismatches) {
	highj = j;
      }
      nmismatches_j--; j++;
    }
  }
  debug2(printf("i range: %d..%d.  j range: %d..%d\n",lowi,highi,lowj,highj));
  
  if (mismatch_positions_i[lowi] < mismatch_positions_j[lowj]) {
    low_splice_pos = mismatch_positions_i[lowi];
  } else {
    low_splice_pos = mismatch_positions_j[lowj];
  }

  if (mismatch_positions_i[highi] > mismatch_positions_j[highj]) {
    high_splice_pos = mismatch_positions_i[highi];
  } else {
    high_splice_pos = mismatch_positions_j[highj];
  }
  debug2(printf("(low_splice_pos %d, high_splice_pos %d]\n",low_splice_pos,high_splice_pos));

  if (novelsplicingp == false) {
    *best_prob_i = *best_prob_j = 0.0;
    *sensedir_distant_guess = SENSE_NULL;
    return (low_splice_pos + high_splice_pos)/2;
  }


  /* Find choices outside of this range */
  nmismatches = best_nmismatches;
  prev_splice_pos = low_splice_pos;
  while ((lowi > 0 || lowj > 0) && nmismatches < best_nmismatches + SLOP) {
    if (lowi == 0) {
      prev_splice_pos = mismatch_positions_j[--lowj];
    } else if (lowj == 0) {
      prev_splice_pos = mismatch_positions_i[--lowi];
    } else if (mismatch_positions_i[lowi] > mismatch_positions_j[lowj]) {
      prev_splice_pos = mismatch_positions_i[--lowi];
    } else if (mismatch_positions_j[lowj] > mismatch_positions_i[lowi]) {
      prev_splice_pos = mismatch_positions_j[--lowj];
    } else {
      prev_splice_pos = mismatch_positions_i[--lowi];
      --lowj;
    }

    debug2(printf("Going back to prev_splice_pos %d, with %d mismatches vs best %d\n",
		  prev_splice_pos,nmismatches,best_nmismatches));

    if (prev_splice_pos == mismatch_positions_i[lowi]) {
      nmismatches--;
    }
    if (prev_splice_pos == mismatch_positions_j[lowj]) {
      nmismatches++;
    }
  }

  assert(highi <= segmenti_nmismatches);
  assert(highj <= segmentj_nmismatches);
  nmismatches = best_nmismatches;
  next_splice_pos = high_splice_pos;
  while ((highi < segmenti_nmismatches || highj < segmentj_nmismatches) && nmismatches < best_nmismatches + SLOP) {
    if (highi == segmenti_nmismatches) {
      next_splice_pos = mismatch_positions_j[++highj];
    } else if (highj == segmentj_nmismatches) {
      next_splice_pos = mismatch_positions_i[++highi];
    } else if (mismatch_positions_i[highi] < mismatch_positions_j[highj]) {
      next_splice_pos = mismatch_positions_i[++highi];
    } else if (mismatch_positions_j[highj] < mismatch_positions_i[highi]) {
      next_splice_pos = mismatch_positions_j[++highj];
    } else {
      next_splice_pos = mismatch_positions_i[++highi];
      ++highj;
    }

    debug2(printf("Going forward to next_splice_pos %d, with %d mismatches vs best %d\n",
		  next_splice_pos,nmismatches,best_nmismatches));
    if (next_splice_pos == mismatch_positions_i[highi]) {
      nmismatches++;
    }
    if (next_splice_pos == mismatch_positions_j[highj]) {
      nmismatches--;
    }
  }

  debug2(printf("(prev_splice_pos %d, next_splice_pos %d]\n",
		prev_splice_pos,next_splice_pos));

  if (prev_splice_pos < splice_pos_start) {
    prev_splice_pos = splice_pos_start;
  }
  if (next_splice_pos > splice_pos_end) {
    next_splice_pos = splice_pos_end;
  }
  debug2(printf("After bounds, (prev_splice_pos %d, next_splice_pos %d]\n",
		prev_splice_pos,next_splice_pos));


  /* Check for best splice probabilities from prev_best to next_best */
  nmismatches_i = 0;
  nmismatches_j = segmentj_nmismatches + 1;

  i = j = 0;
  lowi = highi = 0;
  lowj = highj = 0;
  while (i <= segmenti_nmismatches && mismatch_positions_i[i] <= prev_splice_pos) {
    lowi = highi = i;
    nmismatches_i++; i++;
  }
  while (j <= segmentj_nmismatches && mismatch_positions_j[j] <= prev_splice_pos) {
    lowj = highj = j;
    nmismatches_j--; j++;
  }

  best_prob = 0.0;
  for (splice_pos = prev_splice_pos + 1; splice_pos <= next_splice_pos; splice_pos++) {
    supporti = splice_pos - pos5;
    supportj = pos3 - splice_pos;

    if (nmismatches_i >= supporti/15) {
      /* Skip, because too many mismatches in segmenti */
      debug2(printf("Skipping, because too many mismatches %d in segmenti\n",nmismatches_i));

    } else if (nmismatches_j >= supportj/15) {
      /* Skip, because too many mismatches in segmentj */
      debug2(printf("Skipping, because too many mismatches %d in segmentj\n",nmismatches_j));

    } else if ((prob = splice_prob_eval(&sensedir,&prob_i,&prob_j,splice_pos,
					segmenti_left,segmentj_left,segmenti_chroffset,segmentj_chroffset,
					querylength,plusp_i,plusp_j)) > best_prob) {
      *best_prob_i = prob_i;
      *best_prob_j = prob_j;
      *best_nmismatches_i = nmismatches_i;
      *best_nmismatches_j = nmismatches_j;
      *sensedir_distant_guess = sensedir;
      best_splice_pos = splice_pos;
      best_prob = prob;
    }

    debug2(printf("splice_pos %d, supporti %d and %d, mismatches %d+%d, sensedir %d, prob %f + %f\n",
		  splice_pos,supporti,supportj,nmismatches_i,nmismatches_j,sensedir,prob_i,prob_j));
    
    if (i <= segmenti_nmismatches && mismatch_positions_i[i] == splice_pos) {
      nmismatches_i++; i++;
    }
    if (j <= segmentj_nmismatches && mismatch_positions_j[j] == splice_pos) {
      nmismatches_j--; j++;
    }
  }

  debug2(printf("Returning best break pos at %d, probs %f + %f\n",best_splice_pos,*best_prob_i,*best_prob_j));
  return best_splice_pos;
}


static int
count_spliceends_trim5_sense (double *max_prob, Univcoord_T *end_genomicpos,
			      Univcoord_T start_genomicpos, Univcoord_T middle_genomicpos,
			      int *mismatch_positions, int nmismatches,
			      Univcoord_T left, Univcoord_T chroffset, double end_splicesite_prob_match,
			      int max_nconsecutive, bool plusp) {
  int nspliceends = 0;
  Univcoord_T genomicpos;
  double donor_prob, acceptor_prob;
  int querypos;
  int mismatchi;
  int nconsecutive;

  *max_prob = 0.0;

  if (plusp) {
    debug13(printf("Case 2\n"));
    /* assert(start_genomicpos <= *end_genomicpos); */
    genomicpos = start_genomicpos;
    while (genomicpos < middle_genomicpos) {
      acceptor_prob = Maxent_hr_acceptor_prob(genomicpos,chroffset); /* Case 2 */
      debug13(printf("5', watson, sense forward %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos-left,acceptor_prob));
      if (acceptor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
	if (acceptor_prob > *max_prob) {
	  *max_prob = acceptor_prob;
	}
      }
      genomicpos++;
    }
      
    querypos = genomicpos - left;
    mismatchi = 0;
    while (mismatchi < nmismatches && mismatch_positions[mismatchi] < querypos) {
      mismatchi++;
    }
    nconsecutive = 0;
    /* was (nconsecutive < max_nconsecutive || nspliceends == 0) */
    while (genomicpos < *end_genomicpos && nconsecutive < max_nconsecutive) {
      acceptor_prob = Maxent_hr_acceptor_prob(genomicpos,chroffset); /* Case 2 */
      debug13(printf("5', watson, sense forward %u %u %d %f.  nconsecutive %d\n",
		     genomicpos,genomicpos-chroffset,querypos,acceptor_prob,nconsecutive));
      if (acceptor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
	if (acceptor_prob > *max_prob) {
	  *max_prob = acceptor_prob;
	}
      }
      if (mismatchi < nmismatches && querypos == mismatch_positions[mismatchi]) {
	nconsecutive = 0;
	mismatchi++;
      } else {
	nconsecutive++;
      }
      querypos++;
      genomicpos++;
    }
    *end_genomicpos = genomicpos;
    debug13(printf("\n"));

  } else {
    debug13(printf("Case 6\n"));
    /* assert(start_genomicpos <= *end_genomicpos); */
    genomicpos = start_genomicpos;
    while (genomicpos < middle_genomicpos) {
      donor_prob = Maxent_hr_antidonor_prob(genomicpos,chroffset); /* Case 6 */
      debug13(printf("5', watson, sense anti %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos-left,donor_prob));
      if (donor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
	if (donor_prob > *max_prob) {
	  *max_prob = donor_prob;
	}
      }
      genomicpos++;
    }
    
    querypos = genomicpos - left;
    mismatchi = 0;
    while (mismatchi < nmismatches && mismatch_positions[mismatchi] < querypos) {
      mismatchi++;
    }
    nconsecutive = 0;
    /* was (nconsecutive < max_nconsecutive || nspliceends == 0) */
    while (genomicpos < *end_genomicpos && nconsecutive < max_nconsecutive) {
      donor_prob = Maxent_hr_antidonor_prob(genomicpos,chroffset); /* Case 6 */
      debug13(printf("5', watson, sense anti %u %u %d %f.  nconsecutive %d\n",
		     genomicpos,genomicpos-chroffset,querypos,donor_prob,nconsecutive));
      if (donor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
	if (donor_prob > *max_prob) {
	  *max_prob = donor_prob;
	}
      }
      if (mismatchi < nmismatches && querypos == mismatch_positions[mismatchi]) {
	nconsecutive = 0;
	mismatchi++;
      } else {
	nconsecutive++;
      }
      querypos++;
      genomicpos++;
    }
    *end_genomicpos = genomicpos;
    debug13(printf("\n"));
  }

  return nspliceends;
}


static int
count_spliceends_trim5_anti (double *max_prob, Univcoord_T *end_genomicpos,
			     Univcoord_T start_genomicpos, Univcoord_T middle_genomicpos,
			     int *mismatch_positions, int nmismatches,
			     Univcoord_T left, Univcoord_T chroffset, double end_splicesite_prob_match,
			     int max_nconsecutive, bool plusp) {
  int nspliceends = 0;
  Univcoord_T genomicpos;
  double donor_prob, acceptor_prob;
  int querypos;
  int mismatchi;
  int nconsecutive;

  *max_prob = 0.0;

  if (plusp) {
    debug13(printf("Case 6\n"));
    /* assert(start_genomicpos <= *end_genomicpos); */
    genomicpos = start_genomicpos;
    while (genomicpos < middle_genomicpos) {
      donor_prob = Maxent_hr_antidonor_prob(genomicpos,chroffset); /* Case 6 */
      debug13(printf("5', watson, sense anti %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos-left,donor_prob));
      if (donor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
	if (donor_prob > *max_prob) {
	  *max_prob = donor_prob;
	}
      }
      genomicpos++;
    }
      
    querypos = genomicpos - left;
    mismatchi = 0;
    while (mismatchi < nmismatches && mismatch_positions[mismatchi] < querypos) {
      mismatchi++;
    }
    nconsecutive = 0;
    /* was (nconsecutive < max_nconsecutive || nspliceends == 0) */
    while (genomicpos < *end_genomicpos && nconsecutive < max_nconsecutive) {
      donor_prob = Maxent_hr_antidonor_prob(genomicpos,chroffset); /* Case 6 */
      debug13(printf("5', watson, sense anti %u %u %d %f.  nconsecutive %d\n",
		     genomicpos,genomicpos-chroffset,genomicpos-left,donor_prob,nconsecutive));
      if (donor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
	if (donor_prob > *max_prob) {
	  *max_prob = donor_prob;
	}
      }
      if (mismatchi < nmismatches && querypos == mismatch_positions[mismatchi]) {
	nconsecutive = 0;
	mismatchi++;
      } else {
	nconsecutive++;
      }
      querypos++;
      genomicpos++;
    }
    *end_genomicpos = genomicpos;
    debug13(printf("\n"));

  } else {
    debug13(printf("Case 2\n"));
    /* assert(start_genomicpos <= *end_genomicpos); */
    genomicpos = start_genomicpos;
    while (genomicpos < middle_genomicpos) {
      acceptor_prob = Maxent_hr_acceptor_prob(genomicpos,chroffset); /* Case 2 */
      debug13(printf("5', watson, sense forward %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos-left,acceptor_prob));
      if (acceptor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
	if (acceptor_prob > *max_prob) {
	  *max_prob = acceptor_prob;
	}
      }
      genomicpos++;
    }
      
    querypos = genomicpos - left;
    mismatchi = 0;
    while (mismatchi < nmismatches && mismatch_positions[mismatchi] < querypos) {
      mismatchi++;
    }
    nconsecutive = 0;
    /* was (nconsecutive < max_nconsecutive || nspliceends == 0) */
    while (genomicpos < *end_genomicpos && nconsecutive < max_nconsecutive) {
      acceptor_prob = Maxent_hr_acceptor_prob(genomicpos,chroffset); /* Case 2 */
      debug13(printf("5', watson, sense forward %u %u %d %f.  nconsecutive %d\n",
		     genomicpos,genomicpos-chroffset,querypos,acceptor_prob,nconsecutive));
      if (acceptor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
	if (acceptor_prob > *max_prob) {
	  *max_prob = acceptor_prob;
	}
      }
      if (mismatchi < nmismatches && querypos == mismatch_positions[mismatchi]) {
	nconsecutive = 0;
	mismatchi++;
      } else {
	nconsecutive++;
      }
      querypos++;
      genomicpos++;
    }
    *end_genomicpos = genomicpos;
    debug13(printf("\n"));
  }

  return nspliceends;
}


/* Derived from substring_trim_novel_spliceends in substring.c, which
   was modified from trim_novel_spliceends in stage3.c */
/* Note: If substring does not extend to ends of query, then region
   beyond querystart and queryend might actually be matching, and not
   mismatches.  Could fix in the future. */

/* TODO: Consider whether there is a conflict between the regular prob
   and mismatch prob, and if so, set splicedir to be SENSE_NULL */
int
Splice_trim_novel_spliceends_5 (Splicetype_T *splicetype, Spliceinfo_T spliceinfo,
				int *found_sensedir, int try_sensedir,
				Univcoord_T left, int qstart, int qend,
				int *mismatch_positions, int nmismatches,
				Univcoord_T chroffset, double end_splicesite_prob_match,
				int max_nconsecutive, bool plusp) {
  int nspliceends = 0, n_sense_spliceends, n_anti_spliceends;
  Univcoord_T start_genomicpos, middle_genomicpos, end_genomicpos,
    end_genomicpos_sense, end_genomicpos_anti, genomicpos;
  double max_sense_prob = 0.0, max_antisense_prob = 0.0, donor_prob, acceptor_prob;


  debug13(printf("\nEntered Splice_trim_novel_spliceends_5 with try_sensedir %d\n",try_sensedir));
  /* assert(try_sensedir != SENSE_NULL); -- Not true from Substring_extend_anchor procedures */

  middle_genomicpos = left + qstart;

  if (middle_genomicpos < left + END_SPLICESITE_SEARCH_MM) {
    start_genomicpos = left;
  } else {
    start_genomicpos = middle_genomicpos - END_SPLICESITE_SEARCH_MM;
  }

  if ((end_genomicpos = middle_genomicpos + END_SPLICESITE_SEARCH) > left + qend) {
    end_genomicpos = left + qend;
  }

  if (left + qend < MIN_EXON_LENGTH) {
    /* Skip */
  } else if (end_genomicpos < left + qend - MIN_EXON_LENGTH) {
    end_genomicpos = left + qend - MIN_EXON_LENGTH;
  }

  debug13(printf("\n1 Set end points for 5' trim to be %u..%u..%u\n",
		 start_genomicpos - chroffset,middle_genomicpos - chroffset,end_genomicpos - chroffset));

  if (try_sensedir == SENSE_FORWARD) {
    nspliceends = count_spliceends_trim5_sense(&max_sense_prob,&end_genomicpos,
					       start_genomicpos,middle_genomicpos,
					       mismatch_positions,nmismatches,
					       left,chroffset,end_splicesite_prob_match,
					       max_nconsecutive,plusp);

  } else if (try_sensedir == SENSE_ANTI) {
    nspliceends = count_spliceends_trim5_anti(&max_antisense_prob,&end_genomicpos,
					      start_genomicpos,middle_genomicpos,
					      mismatch_positions,nmismatches,
					      left,chroffset,end_splicesite_prob_match,
					      max_nconsecutive,plusp);
  } else {
    end_genomicpos_sense = end_genomicpos_anti = end_genomicpos;
    n_sense_spliceends = count_spliceends_trim5_sense(&max_sense_prob,&end_genomicpos_sense,
						      start_genomicpos,middle_genomicpos,
						      mismatch_positions,nmismatches,
						      left,chroffset,end_splicesite_prob_match,
						      max_nconsecutive,plusp);
    n_anti_spliceends = count_spliceends_trim5_anti(&max_antisense_prob,&end_genomicpos_anti,
						    start_genomicpos,middle_genomicpos,
						    mismatch_positions,nmismatches,
						    left,chroffset,end_splicesite_prob_match,
						    max_nconsecutive,plusp);
    if (max_sense_prob > max_antisense_prob) {
      nspliceends = n_sense_spliceends;
      end_genomicpos = end_genomicpos_sense;
      *found_sensedir = SENSE_FORWARD;
    } else if (max_antisense_prob > max_sense_prob) {
      nspliceends = n_anti_spliceends;
      end_genomicpos = end_genomicpos_anti;
      *found_sensedir = SENSE_ANTI;
    } else {
      debug13(printf("Cannot distinguish between sense and antisense at end\n"));
      *found_sensedir = SENSE_NULL;
      /* ambig_qstarts and mbig_probs_5 not yet allocated */
      return 0;
    }
  }


  if (nspliceends == 0) {
    debug13(printf("Got no spliceends\n"));
    *found_sensedir = SENSE_NULL;
    return 0;

  } else if (nspliceends > MAX_NSPLICEENDS) {
    debug13(printf("Too many spliceends\n"));
    *found_sensedir = SENSE_NULL;
    return 0;

  } else {
    *found_sensedir = try_sensedir;
    
    debug13(printf("Going from genomicpos %u up to %u\n",start_genomicpos,end_genomicpos));
    nspliceends = 0;

    if (try_sensedir == SENSE_FORWARD) {
      if (plusp) {
	*splicetype = ACCEPTOR;
	genomicpos = start_genomicpos;
	while (genomicpos < end_genomicpos) {
	  acceptor_prob = Maxent_hr_acceptor_prob(genomicpos,chroffset); /* Case 2 */
	  debug13(printf("Assignment: 5', watson, sense forward %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos-left,acceptor_prob));
	  if (acceptor_prob >= end_splicesite_prob_match) {
	    spliceinfo->ambig_qstarts[nspliceends] = (int) (genomicpos - left);
	    spliceinfo->ambig_probs_5[nspliceends] = acceptor_prob;
	    nspliceends += 1;
	  }
	  genomicpos++;
	}
      } else {
	*splicetype = ANTIDONOR;

	genomicpos = start_genomicpos;
	while (genomicpos < end_genomicpos) {
	  donor_prob = Maxent_hr_antidonor_prob(genomicpos,chroffset); /* Case 6 */
	  if (donor_prob >= end_splicesite_prob_match) {
	    spliceinfo->ambig_qstarts[nspliceends] = (int) (genomicpos - left);
	    spliceinfo->ambig_probs_5[nspliceends] = donor_prob;
	    nspliceends += 1;
	  }
	  genomicpos++;
	}
      }

    } else {
      /* SENSE_ANTI */
      if (plusp) {
	*splicetype = ANTIDONOR;

	genomicpos = start_genomicpos;
	while (genomicpos < end_genomicpos) {
	  donor_prob = Maxent_hr_antidonor_prob(genomicpos,chroffset); /* Case 6 */
	  if (donor_prob >= end_splicesite_prob_match) {
	    spliceinfo->ambig_qstarts[nspliceends] = (int) (genomicpos - left);
	    spliceinfo->ambig_probs_5[nspliceends] = donor_prob;
	    nspliceends += 1;
	  }
	  genomicpos++;
	}
      } else {
	*splicetype = ACCEPTOR;
	
	genomicpos = start_genomicpos;
	while (genomicpos < end_genomicpos) {
	  acceptor_prob = Maxent_hr_acceptor_prob(genomicpos,chroffset); /* Case 2 */
	  if (acceptor_prob >= end_splicesite_prob_match) {
	    spliceinfo->ambig_qstarts[nspliceends] = (int) (genomicpos - left);
	    spliceinfo->ambig_probs_5[nspliceends] = acceptor_prob;
	    nspliceends += 1;
	  }
	  genomicpos++;
	}
      }
    }

    debug13(printf("Got %d spliceends\n",nspliceends));
    return nspliceends;
  }
}


static int
count_spliceends_trim3_sense (double *max_prob, Univcoord_T *end_genomicpos,
			      Univcoord_T start_genomicpos, Univcoord_T middle_genomicpos,
			      int *mismatch_positions, int nmismatches,
			      Univcoord_T left, Univcoord_T chroffset, double end_splicesite_prob_match,
			      int max_nconsecutive, bool plusp) {
  int nspliceends = 0;
  Univcoord_T genomicpos;
  double donor_prob, acceptor_prob;
  int querypos;
  int mismatchi;
  int nconsecutive;

  *max_prob = 0.0;

  if (plusp) {
    debug13(printf("Case 1\n"));
    /* assert(start_genomicpos > *end_genomicpos); */
    genomicpos = start_genomicpos;
    while (genomicpos > middle_genomicpos) {
      donor_prob = Maxent_hr_donor_prob(genomicpos,chroffset); /* Case 1 */
      debug13(printf("3', watson, sense anti %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos - left,donor_prob));
      if (donor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
      }
      genomicpos--;
    }
      
    querypos = genomicpos - left;
    mismatchi = 0;
    while (mismatchi < nmismatches && mismatch_positions[mismatchi] > querypos) {
      mismatchi++;
    }
    nconsecutive = 0;
    /* was (nconsecutive < max_nconsecutive || nspliceends == 0) */
    while (genomicpos > *end_genomicpos && nconsecutive < max_nconsecutive) {
      donor_prob = Maxent_hr_donor_prob(genomicpos,chroffset); /* Case 1 */
      debug13(printf("3', watson, sense anti %u %u %d %f.  nconsecutive %d\n",
		     genomicpos,genomicpos-chroffset,querypos,donor_prob,nconsecutive));
      if (donor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
      }
      if (mismatchi < nmismatches && querypos == mismatch_positions[mismatchi]) {
	nconsecutive = 0;
	mismatchi++;
      } else {
	nconsecutive++;
      }
      querypos--;
      genomicpos--;
    }
    *end_genomicpos = genomicpos;
    debug13(printf("\n"));

  } else {
    debug13(printf("Case 5\n"));
    /* assert(start_genomicpos > *end_genomicpos); */
    genomicpos = start_genomicpos;
    while (genomicpos > middle_genomicpos) {
      acceptor_prob = Maxent_hr_antiacceptor_prob(genomicpos,chroffset); /* Case 5 */
      debug13(printf("3', watson, sense forward %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos - left,acceptor_prob));
      if (acceptor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
      }
      genomicpos--;
    }
      
    querypos = genomicpos - left;
    mismatchi = 0;
    while (mismatchi < nmismatches && mismatch_positions[mismatchi] > querypos) {
      mismatchi++;
    }
    nconsecutive = 0;
    /* was (nconsecutive < max_nconsecutive || nspliceends == 0) */
    while (genomicpos > *end_genomicpos && nconsecutive < max_nconsecutive) {
      acceptor_prob = Maxent_hr_antiacceptor_prob(genomicpos,chroffset); /* Case 5 */
      debug13(printf("3', watson, sense forward %u %u %d %f.  nconsecutive %d\n",
		     genomicpos,genomicpos-chroffset,querypos,acceptor_prob,nconsecutive));
      if (acceptor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
      }
      if (mismatchi < nmismatches && querypos == mismatch_positions[mismatchi]) {
	nconsecutive = 0;
	mismatchi++;
      } else {
	nconsecutive++;
      }
      querypos--;
      genomicpos--;
    }
    *end_genomicpos = genomicpos;
    debug13(printf("\n"));
  }

  return nspliceends;
}


static int
count_spliceends_trim3_anti (double *max_prob, Univcoord_T *end_genomicpos,
			     Univcoord_T start_genomicpos, Univcoord_T middle_genomicpos,
			     int *mismatch_positions, int nmismatches,
			     Univcoord_T left, Univcoord_T chroffset, double end_splicesite_prob_match,
			     int max_nconsecutive, bool plusp) {
  int nspliceends = 0;
  Univcoord_T genomicpos;
  double donor_prob, acceptor_prob;
  int querypos;
  int mismatchi;
  int nconsecutive;

  *max_prob = 0.0;

  if (plusp) {
    debug13(printf("Case 5\n"));
    /* assert(start_genomicpos > *end_genomicpos); */
    genomicpos = start_genomicpos;
    while (genomicpos > middle_genomicpos) {
      acceptor_prob = Maxent_hr_antiacceptor_prob(genomicpos,chroffset); /* Case 5 */
      debug13(printf("3', watson, sense forward %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos - left,acceptor_prob));
      if (acceptor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
      }
      genomicpos--;
    }
      
    querypos = genomicpos - left;
    mismatchi = 0;
    while (mismatchi < nmismatches && mismatch_positions[mismatchi] > querypos) {
      mismatchi++;
    }
    nconsecutive = 0;
    /* was (nconsecutive < max_nconsecutive || nspliceends == 0) */
    while (genomicpos > *end_genomicpos && nconsecutive < max_nconsecutive) {
      acceptor_prob = Maxent_hr_antiacceptor_prob(genomicpos,chroffset); /* Case 5 */
      debug13(printf("3', watson, sense forward %u %u %d %f.  nconsecutive %d\n",
		     genomicpos,genomicpos-chroffset,querypos,acceptor_prob,nconsecutive));
      if (acceptor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
      }
      if (mismatchi < nmismatches && querypos == mismatch_positions[mismatchi]) {
	nconsecutive = 0;
	mismatchi++;
      } else {
	nconsecutive++;
      }
      querypos--;
      genomicpos--;
    }
    *end_genomicpos = genomicpos;
    debug13(printf("\n"));

  } else {
    debug13(printf("Case 1\n"));
    /* assert(start_genomicpos > *end_genomicpos); */
    genomicpos = start_genomicpos;
    while (genomicpos > middle_genomicpos) {
      donor_prob = Maxent_hr_donor_prob(genomicpos,chroffset); /* Case 1 */
      debug13(printf("3', watson, sense anti %u %u %d %f mm\n",genomicpos,genomicpos-chroffset,genomicpos - left,donor_prob));
      if (donor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
      }
      genomicpos--;
    }
      
    querypos = genomicpos - left;
    mismatchi = 0;
    while (mismatchi < nmismatches && mismatch_positions[mismatchi] > querypos) {
      mismatchi++;
    }
    nconsecutive = 0;
    /* was (nconsecutive < max_nconsecutive || nspliceends == 0) */
    while (genomicpos > *end_genomicpos && nconsecutive < max_nconsecutive) {
      donor_prob = Maxent_hr_donor_prob(genomicpos,chroffset); /* Case 1 */
      debug13(printf("3', watson, sense anti %u %u %d %f.  nconsecutive %d\n",
		     genomicpos,genomicpos-chroffset,querypos,donor_prob,nconsecutive));
      if (donor_prob >= end_splicesite_prob_match) {
	nspliceends += 1;
      }
      if (mismatchi < nmismatches && querypos == mismatch_positions[mismatchi]) {
	nconsecutive = 0;
	mismatchi++;
      } else {
	nconsecutive++;
      }
      querypos--;
      genomicpos--;
    }
    *end_genomicpos = genomicpos;
    debug13(printf("\n"));
  }

  return nspliceends;
}


int
Splice_trim_novel_spliceends_3 (Splicetype_T *splicetype, Spliceinfo_T spliceinfo,
				int *found_sensedir, int try_sensedir,
				Univcoord_T left, int qstart, int qend, int querylength,
				int *mismatch_positions, int nmismatches,
				Univcoord_T chroffset, double end_splicesite_prob_match,
				int max_nconsecutive, bool plusp) {
  int nspliceends = 0, n_sense_spliceends, n_anti_spliceends;
  Univcoord_T start_genomicpos, middle_genomicpos, end_genomicpos,
    end_genomicpos_sense, end_genomicpos_anti, genomicpos;
  double max_sense_prob = 0.0, max_antisense_prob = 0.0, donor_prob, acceptor_prob;


  debug13(printf("\nEntered Splice_trim_novel_spliceends_3 with try_sensedir %d\n",try_sensedir));
  /* assert(try_sensedir != SENSE_NULL); -- Not true from Substring_extend_anchor procedures */

  middle_genomicpos = left + qend;

  if ((start_genomicpos = middle_genomicpos + END_SPLICESITE_SEARCH_MM) > left + querylength) {
    start_genomicpos = left + querylength;
  }

  if (middle_genomicpos < left + qstart + END_SPLICESITE_SEARCH) {
    end_genomicpos = left + qstart;
  } else {
    end_genomicpos = middle_genomicpos - END_SPLICESITE_SEARCH;
  }

  if (end_genomicpos < left + qstart + MIN_EXON_LENGTH) {
    end_genomicpos = left + qstart + MIN_EXON_LENGTH;
  }

  debug13(printf("\n1 Set end points for 3' trim to be %u..%u..%u\n",
		 start_genomicpos - chroffset,middle_genomicpos - chroffset,end_genomicpos - chroffset));

  if (try_sensedir == SENSE_FORWARD) {
    nspliceends = count_spliceends_trim3_sense(&max_sense_prob,&end_genomicpos,
					       start_genomicpos,middle_genomicpos,
					       mismatch_positions,nmismatches,
					       left,chroffset,end_splicesite_prob_match,
					       max_nconsecutive,plusp);
  } else if (try_sensedir == SENSE_ANTI) {
    nspliceends = count_spliceends_trim3_anti(&max_antisense_prob,&end_genomicpos,
					      start_genomicpos,middle_genomicpos,
					      mismatch_positions,nmismatches,
					      left,chroffset,end_splicesite_prob_match,
					      max_nconsecutive,plusp);
  } else {
    end_genomicpos_sense = end_genomicpos_anti = end_genomicpos;
    n_sense_spliceends = count_spliceends_trim3_sense(&max_sense_prob,&end_genomicpos_sense,
						      start_genomicpos,middle_genomicpos,
						      mismatch_positions,nmismatches,
						      left,chroffset,end_splicesite_prob_match,
						      max_nconsecutive,plusp);
    n_anti_spliceends = count_spliceends_trim3_anti(&max_antisense_prob,&end_genomicpos_anti,
						    start_genomicpos,middle_genomicpos,
						    mismatch_positions,nmismatches,
						    left,chroffset,end_splicesite_prob_match,
						    max_nconsecutive,plusp);
    if (max_sense_prob > max_antisense_prob) {
      nspliceends = n_sense_spliceends;
      end_genomicpos = end_genomicpos_sense;
      *found_sensedir = SENSE_FORWARD;
    } else if (max_antisense_prob > max_sense_prob) {
      nspliceends = n_anti_spliceends;
      end_genomicpos = end_genomicpos_anti;
      *found_sensedir = SENSE_ANTI;
    } else {
      debug13(printf("Cannot distinguish between sense and antisense at end\n"));
      *found_sensedir = SENSE_NULL;
      /* ambig_qends and mbig_probs_3 not yet allocated */
      return 0;
    }
  }


  if (nspliceends == 0) {
    debug13(printf("Got no spliceends\n"));
    *found_sensedir = SENSE_NULL;
    return 0;

  } else if (nspliceends > MAX_NSPLICEENDS) {
    debug13(printf("Too many spliceends\n"));
    *found_sensedir = SENSE_NULL;
    return 0;

  } else {
    *found_sensedir = try_sensedir;

    debug13(printf("Going from genomicpos %u down to %u\n",start_genomicpos,end_genomicpos));
    nspliceends = 0;

    if (try_sensedir == SENSE_FORWARD) {
      if (plusp) {
	*splicetype = DONOR;
	
	genomicpos = start_genomicpos;
	while (genomicpos > end_genomicpos) {
	  donor_prob = Maxent_hr_donor_prob(genomicpos,chroffset); /* Case 1 */
	  if (donor_prob >= end_splicesite_prob_match) {
	    spliceinfo->ambig_qends[nspliceends] = (int) (genomicpos - left);
	    spliceinfo->ambig_probs_3[nspliceends] = donor_prob;
	    nspliceends += 1;
	  }
	  genomicpos--;
	}

      } else {
	*splicetype = ANTIACCEPTOR;
      
	genomicpos = start_genomicpos;
	while (genomicpos > end_genomicpos) {
	  acceptor_prob = Maxent_hr_antiacceptor_prob(genomicpos,chroffset); /* Case 5 */
	  if (acceptor_prob >= end_splicesite_prob_match) {
	    spliceinfo->ambig_qends[nspliceends] = (int) (genomicpos - left);
	    spliceinfo->ambig_probs_3[nspliceends] = acceptor_prob;
	    nspliceends += 1;
	  }
	  genomicpos--;
	}
      }

    } else {
      if (plusp) {
	*splicetype = ANTIACCEPTOR;
      
	genomicpos = start_genomicpos;
	while (genomicpos > end_genomicpos) {
	  acceptor_prob = Maxent_hr_antiacceptor_prob(genomicpos,chroffset); /* Case 5 */
	  if (acceptor_prob >= end_splicesite_prob_match) {
	    spliceinfo->ambig_qends[nspliceends] = (int) (genomicpos - left);
	    spliceinfo->ambig_probs_3[nspliceends] = acceptor_prob;
	    nspliceends += 1;
	  }
	  genomicpos--;
	}

      } else {
	*splicetype = DONOR;
	
	genomicpos = start_genomicpos;
	while (genomicpos > end_genomicpos) {
	  donor_prob = Maxent_hr_donor_prob(genomicpos,chroffset); /* Case 1 */
	  if (donor_prob >= end_splicesite_prob_match) {
	    spliceinfo->ambig_qends[nspliceends] = (int) (genomicpos - left);
	    spliceinfo->ambig_probs_3[nspliceends] = donor_prob;
	    nspliceends += 1;
	  }
	  genomicpos--;
	}
      }
    }

    debug13(printf("Got %d spliceends\n",nspliceends));
    return nspliceends;
  }
}

