/* hall_symbol.c */
/* Copyright (C) 2010 Atsushi Togo */

#include <stdio.h>
#include "lattice.h"
#include "hall_symbol.h"
#include "spg_database.h"
#include "spacegroup.h"
#include "symmetry.h"
#include "mathfunc.h"

#include "debug.h"

/* See: R. W. Grosse-Kunstleve, Acta Cryst. (1999). A55, 383-395 */
/* */
/* S is the Smith normal form, S = U*M*V */
/* M = R - I, where R is virtical stack of generator rotation matrices */
/* and I is virtical stack of identity matrices. */
/* */
/* Origin shift is given by (V*S+*U)*dw where dw is translations */
/* corresponding to those obtained by symmetry finder and from */
/* International table. */
/* S+ is given by the operations of inversion of the diagonal elements */
/* and then transpose. */

static double tricli_VSpU[][3][9] = {
  { /* 1 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
};

static int tricli_generators[][3][9] = {
  { /* 1 */
    {  1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    {  1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
};

static double monocli_A_VSpU[][3][9] = {
  { /* 1 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, -1, 0, 0, 1.0/2, 0,  0,  0,  0, },
  },
  { /* 5 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 1, 0, 0, -1.0/2, 0,  0,  0,  0, },
  },
  { /* 7 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 1, 0, 0, -1.0/2, 0,  0,  0,  0, },
  },
  { /* 11 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 1, 0, 0, -1.0/2, 0,  0,  0,  0, },
  },
};

static double monocli_B_VSpU[][3][9] = {
  { /* 1 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 5 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 7 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 11 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
};

static double monocli_C_VSpU[][3][9] = {
  { /* 1 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 5 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 7 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 9 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 11 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
};

static double monocli_I_VSpU[][3][9] = {
  { /* 1 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 1.0/2, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 1.0/2, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 3 */
    { -1, 1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 1, -1.0/2, 0, -1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 5 */
    { -1, 0, 1.0/2,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, -1.0/2, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 7 */
    { 0, 1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, -1.0/2, 0, 1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 9 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 11 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
};

static double monocli_VSpU[][3][9] = {
  { /* 1 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 5 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 7 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 11 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
};

static int monocli_generators[][3][9] = {
  { /* 1 */
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 3 */
    {  -1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    {  -1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 5 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 7 */
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 9 */
    {  1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    {  1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 11 */
    {  1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    {  1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
};

static double ortho_A_VSpU[][3][9] = {
  { /* 1 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 1, 0, 0, -1.0/2, 0,  0,  0,  0, },
  },
  { /* 3 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 1, 0, 0, -1.0/2, 0,  0,  0,  0, },
  },
  { /* 4 */
    { -1.0/2, 0, 0, 0, 0, 0, 0, 0, 0, },
    { 0, -1, 0, 0, 0, -1.0/2, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 5 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, -1.0/2, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 1, 0, 0, 0, 0, 0, -1.0/2, 0, },
  },
  { /* 7 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, -1, 0, 0, 1.0/2, 0,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, -1, 0, 0, 1.0/2, 0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, -1, 0, 0, 0, 1.0/2, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 11 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, -1.0/2, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, -1, 0, 0, 0, 0, 0, 1.0/2, 0, },
  },
};

static double ortho_B_VSpU[][3][9] = {
  { /* 1 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { -1.0/2, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 1.0/2, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 0, -1.0/2, 0, 0, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
  { /* 5 */
    { 0, 0, 0, -1, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, -1.0/2, 0, 0, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 7 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 0, -1.0/2, 0, 0, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
  { /* 11 */
    { -1.0/2, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
    { -1.0/2, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
};

static double ortho_C_VSpU[][3][9] = {
  { /* 1 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 3 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 1.0/2, 0, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 5 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { -1.0/2, 0, 0, 1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, 0, -1.0/2, },
  },
  { /* 7 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 11 */
    { 0, 0, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, -1.0/2, 0, 0, 0, 0, 0, 0, },
  },
};

static double ortho_F_VSpU[][3][9] = {
  { /* 1 */
    { -1.0/2, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
    { 1.0/2, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 3 */
    { 0, 1.0/2, 0, -1.0/4, -1.0/4, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, -1.0/4, -1.0/4, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, -1.0/4, 3.0/4, 0,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 1, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 0, 1, 0, 0, -1.0/2, 0, },
  },
  { /* 5 */
    { 0, 1, 0, 0, -1, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 1, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 0, -1, 0, 0, 1.0/2, 0, },
  },
  { /* 7 */
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 9 */
    { 0, -1, 0, 0, 1, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, -1, 0, 0, 0, 0, 0, 1.0/2, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 0, 1, 0, 0, -1.0/2, 0, },
  },
  { /* 11 */
    { 0, -1, 0, 0, 1, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, -1, 0, 0, 0, 0, 0, 1.0/2, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 0, -1, 0, 0, 1.0/2, 0, },
  },
};

static double ortho_I_VSpU[][3][9] = {
  { /* 1 */
    { -1, 0, 1.0/2,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, -1.0/2, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1.0/2, -1.0/2, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1.0/2, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, -1.0/2, 0, -1, 0, 1.0/2, 0, 0, },
    { 0, 0, -1.0/2, 0, 0, 0, 0, 0, 0, },
  },
  { /* 5 */
    { -1.0/2, 0, 0, 0, 1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
    { -1.0/2, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 1.0/2, 0, -1, 0, -1.0/2, 0, 0, },
    { 0, 0, -1.0/2, 0, 0, 0, 0, 0, 0, },
  },
  { /* 7 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 0, -1, 0, 1.0/2, 0, -1.0/2, },
    { 0, 0, 0, 0, 0, 0, 0, 0, -1.0/2, },
  },
  { /* 11 */
    { -1.0/2, 0, 0, 0, 1.0/2, 0,  0,  0,  0, },
    { -1.0/2, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, -1.0/2, 0, 0, -1.0/2, 0, 0, 0, 0, },
    { 0, 1.0/2, 0, 0, -1.0/2, 0, -1.0/2, 0, 0, },
  },
};

static double ortho_VSpU[][3][9] = {
  { /* 1 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 3 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 4 */
    { -1.0/2, 0, 0, 0, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, -1.0/2, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 5 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, -1.0/2, 0, 0, 0, 0, 0, },
    { 0, -1.0/2, 0, 0, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, 0, -1.0/2, },
  },
  { /* 7 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 0, -1.0/2, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 11 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, -1.0/2, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, -1.0/2, 0, 0, 0, 0, 0, 0, },
  },
};
  
static int ortho_generators[][3][9] = {
  { /* 1 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 3 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 5 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 7 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 9 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 11 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
};

static double trigo_VSpU[][3][9] = {
  { /* 1 */
    { -2.0/3, 1.0/3, 0,  0,  0,  0,  0,  0,  0, },
    { -1.0/3, -1.0/3, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { -6, 3, 0, 4, 0, 0,  0,  0,  0, },
    { -3, 1, 0, 2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 3 */
    { 0, 1.0/3, 0, -2.0/3, 0, 0,  0,  0,  0, },
    { 0, -1.0/3, 0, -1.0/3, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 1, 0, -2, 0, 0, 1, 0, 0, },
    { 0, -1, 0, 1, 0, 0, -1, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 5 */
    { 0, -1, 0, -2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, -1, 0, -2, 0, 0, 0, 0, 0, },
    { 0, -1, 0, -1, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, 0, -1.0/2, },
  },
  { /* 7 */
    { 0, -1, 0, -2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 8 */
    { 0, -1, 0, -2, 0, 0, 0, 0, 0, },
    { 0, -1, 0, -1, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 9 */
    { 0, -1, 0, -2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 10 */
    { 0, -1, 0, -2, 0, 0, 0, 0, 0, },
    { 0, -1, 0, -1, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 11 */
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 1, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
    { 1, -1, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 13 */
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 1, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 14 */
    { 0, -1, 0, 0, 0, 0, 0, 0, 0, },
    { 0, 1, 0, -1, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 15 */
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 1, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 16 */
    { 0, -1, 0, 0, 0, 0, 0, 0, 0, },
    { 0, -1, 0, 1, 0, 0, 0, 0, 0, },
    { 0, 0, -1.0/2, 0, 0, 0, 0, 0, 0, },
  },
  { /* 17 */
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 1, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 18 */
    { 0, -1, 0, 0, 0, 0, 0, 0, 0, },
    { 0, -1, 0, 1, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 19 */
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 1, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 20 */
    { 0, -1, 0, 0, 0, 0, 0, 0, 0, },
    { 0, -1, 0, 1, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
};

static int trigo_generators[][3][9] = {
  { /* 1 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 3 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 5 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 7 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 9 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    {  0, -1, 0, 1, -1, 0, 0, 0, 1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 11 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 13 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 14 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 15 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 16 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 17 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 18 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 19 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 20 */
    {  0, 1, 0, -1, 1, 0, 0, 0, -1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
};

static double rhombo_VSpU[][3][9] = {
  { /* 1 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 1,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 1, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { 0, 1.0/2, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 1.0/2, 0, -1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 4 */
    { -1, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
    { 1, 0, 0, -1, 0, 1.0/2, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 5 */
    { -1, 0, 0, 0, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, -1, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { 1, 0, 0, -1, 0, 0, 0, -1.0/2, 0, },
  },
  { /* 7 */
    { -1, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
    { -1, 0, 0, 1, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 8 */
    { -1, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
    { -1, 0, 0, 1, 0, -1.0/2, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 9 */
    { -1, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
    { -1, 0, 0, 1, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 10 */
    { -1, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
    { -1, 0, 0, 1, 0, -1.0/2, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 11 */
    { -1.0/2, -1.0/2, 1.0/2,  0,  0,  0,  0,  0,  0, },
    { 1.0/2, -1.0/2, -1.0/2,  0,  0,  0,  0,  0,  0, },
    { -1.0/2, 1.0/2, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    { -2, 0, 2, 1.0/2, -1, 0,  0,  0,  0, },
    { 1, 0, -1, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 13 */
    { -1, 0, 0, 0, 0, 1.0/2,  0,  0,  0, },
    { 1, 0, 0, -1, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 14 */
    { -1, 0, 0, 0, 0, 1.0/2, 0, 0, 0, },
    { 1, 0, 0, -1, 0, -1.0/2, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 15 */
    { 0, -1.0/2, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { -1, 1.0/2, 0, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 16 */
    { 0, 0, 0, -1, 0, 0, 0, -1.0/2, 0, },
    { 0, 0, 0, 0, 0, 0, 0, -1.0/2, 0, },
    { -1, 0, 0, 1, 0, 0, 0, 1.0/2, 0, },
  },
  { /* 17 */
    { 0, -1.0/2, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { -1, 1.0/2, 0, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 18 */
    { -1, 0, 0, 0, 0, 1.0/2, 0, 0, 0, },
    { -1, 0, 0, 1, 0, 1.0/2, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 19 */
    { 0, -1.0/2, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { -1, 1.0/2, 0, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 20 */
    { -1, 0, 0, 0, 0, 1.0/2, 0, 0, 0, },
    { -1, 0, 0, 1, 0, 1.0/2, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
};


static int rhombo_generators[][3][9] = {
  { /* 1 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 3 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 5 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 7 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 9 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 11 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 13 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 14 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 15 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 16 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 17 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 18 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 19 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 20 */
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  0, 1, 0, 1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
};

static double hexa_VSpU[][3][9] = {
  { /* 1 */
    { -1, 1, 0,  0,  0,  0,  0,  0,  0, },
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { -1, 1, 0, 0, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 3 */
    { 1, 0, 0, -1, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 4 */
    { 1, 0, 0, -1, 0, 0, 0, 0, 0, },
    { -1, 0, 0, 0, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 5 */
    { -1, 0, 0, -1, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { -1, 0, 0, -1, 0, 0, 0, 0, 0, },
    { -1, 0, 0, 0, 0, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, 0, -1.0/2, },
  },
  { /* 7 */
    { -1.0/3, -1.0/3, 0,  0,  0,  0,  0,  0,  0, },
    { 1.0/3, -2.0/3, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { -1, -1, 0, 1, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 9 */
    { -1.0/3, 0, 0, -1.0/3, 0, 0,  0,  0,  0, },
    { 1.0/3, 0, 0, -2.0/3, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { -1, 0, 0, -1, 0, 0, 1, 0, 0, },
    { 1, 0, 0, 0, 0, 0, -1, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 11 */
    { -1, 0, 0, 1, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 12 */
    { -1, 0, 0, 1, 0, 0, 0, 0, 0, },
    { -1, 0, 0, 2, 0, 0, 0, 0, 0, },
    { 0, 0, -1.0/2, 0, 0, 0, 0, 0, 0, },
  },
};

static int hexa_generators[][3][9] = {
  { /* 1 */
    {  1, -1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    {  1, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 3 */
    {  1, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    {  1, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 5 */
    {  1, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    {  1, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 7 */
    {  -1, 1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    {  -1, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 9 */
    {  -1, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    {  -1, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  0, -1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 11 */
    {  -1, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    {  -1, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  0, 1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
};


static double cubic_F_VSpU[][3][9] = {
  { /* 1 */
    { 0, 0, -1,  0,  0,  0,  0,  0,  0, },
    { 1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { -1, 0, -1, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 1, 0, 0, 0, 1.0/2, 0,  0,  0,  0, },
  },
  { /* 3 */
    { 0, 0, -1.0/2, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 1.0/2, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, -1, 0, -1, 0, 0, 1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 5 */
    { 0, 1.0/2, -1.0/2, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, -1.0/2, 1.0/2, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, -1.0/2, -1.0/2, 0, 1.0/2, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 2, 0, -2, 0, 0, 1.0/2, -1, 0, },
    { 0, -1, 0, 1, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
  { /* 7 */
    { -1.0/2, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
    { 1.0/2, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 1.0/4, -1.0/4, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -3.0/4, -1.0/4, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 1.0/4, -1.0/4, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, -1, -2, 0, 0, 3.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 11 */
    { 0, 1.0/2, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, -1, 1.0/2, 0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, -2, 4, 0, 0, -3.0/2, 1, 0, },
    { 0, 0, 1, -2, 0, 0, 1.0/2, -1, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
  { /* 13 */
    { -1.0/4, 1.0/4, 1.0/2,  0,  0,  0,  0,  0,  0, },
    { -1.0/4, -3.0/4, -1.0/2,  0,  0,  0,  0,  0,  0, },
    { -1.0/4, 1.0/4, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 14 */
    { -1, 0, 1, 1, 1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { -1, 0, 0, 1, 1.0/2, 0,  0,  0,  0, },
  },
  { /* 15 */
    { -1.0/4, 1.0/4, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1.0/4, -3.0/4, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { -1.0/4, 1.0/4, 0, 1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 16 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, -1, 0, 1, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 17 */
    { 0, 0, 1.0/2, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, -1.0/2, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, -1.0/2, 0, 0,  0,  0,  0, },
  },
  { /* 18 */
    { 0, 2, 0, 2, 0, 0, -3.0/2, -1, 0, },
    { 0, -1, 0, -1, 0, 0, 1.0/2, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
  { /* 19 */
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 20 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 21 */
    { 0, -1.0/2, 0, -1.0/2, 0, -1.0/2,  0,  0,  0, },
    { 0, -1.0/2, 0, 1.0/2, 0, 1.0/2,  0,  0,  0, },
    { 0, -1.0/2, 0, 1.0/2, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 22 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 1, 0, 0, 0, 1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 23 */
    { 0, 0, 1.0/2, -1.0/2, 0, 1.0/2,  0,  0,  0, },
    { 0, 0, 1.0/2, 1.0/2, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, -1.0/2, -1.0/2, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 24 */
    { 0, 0, 2, 0, 0, 0, 1.0/2, 1, 0, },
    { 0, 0, -1, 0, 0, 0, -1.0/2, -1, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
};
  
static double cubic_I_VSpU[][3][9] = {
  { /* 1 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 2, 0, -1.0/2, 0, -1,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { 0, -1, 0, -1, 0, 0,  0,  0,  0, },
    { 1, -1, 0, -1, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 4 */
    { -2, -3, 0, -1, 0, 0, 1, -1, 1, },
    { -1, -3, 0, -1, 0, 0, 1, -1, 1, },
    { 0, -1, 0, 0, 0, 0, 0, 0, 0, },
  },
  { /* 5 */
    { -1, 0, 1, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 1, -1, 0, 0,  0,  0,  0, },
    { 1, 0, -1, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { -1, 0, 1, -1, 0, 0, 0, 0, 0, },
    { 0, 0, 1, -1, 0, 0, 0, 0, 0, },
    { 1, 0, -3, 4, 0, 0, -1, -1, -1, },
  },
  { /* 7 */
    { -1, 0, 1.0/2,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, -1.0/2, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 9 */
    { 1, 0, -1, -2, 0, -1,  0,  0,  0, },
    { 1, 0, -1, -1, 0, 0,  0,  0,  0, },
    { 1, 0, -1, -1, 0, -1,  0,  0,  0, },
  },
  { /* 10 */
    { -1, 0, 0, 0, 0, -1, 0, -1, 1, },
    { -1, 0, 0, 1, 0, 0, 0, -1, 1, },
    { -1, 0, 0, 1, 0, -1, 0, -1, 1, },
  },
  { /* 11 */
    { -1, 0, 1, 0, 0, -1,  0,  0,  0, },
    { 3, 0, -1, -3, 0, 2,  0,  0,  0, },
    { -3, 0, 1, 3, 0, -3,  0,  0,  0, },
  },
  { /* 12 */
    { -1, 0, 0, 2, 0, -1, -1, 0, 0, },
    { 1, 0, 0, -1, 0, 2, 0, -1, -1, },
    { -1, 0, 0, 1, 0, -3, 0, 1, 1, },
  },
  { /* 13 */
    { -3.0/4, 1.0/4, 1.0/4,  0,  0,  0,  0,  0,  0, },
    { -1.0/4, -1.0/4, -1.0/4,  0,  0,  0,  0,  0,  0, },
    { -1.0/2, 1.0/2, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 14 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { -2, 1, 0, 1, 0, 0,  0,  0,  0, },
  },
  { /* 15 */
    { 1, 2, -5, -7, 0, 0,  0,  0,  0, },
    { 1, 1, -4, -5, 0, 0,  0,  0,  0, },
    { 0, 1, -2, -2, 0, 0,  0,  0,  0, },
  },
  { /* 16 */
    { -4, -3, 0, 13, 0, 0, -5, 5, 5, },
    { -3, -3, 0, 11, 0, 0, -4, 4, 4, },
    { -2, -1, 0, 6, 0, 0, -2, 2, 2, },
  },
  { /* 17 */
    { -2, 1, 0, 1, 0, 0,  0,  0,  0, },
    { 1, -1, 0, -1, 0, 0,  0,  0,  0, },
    { 2, -1, 0, -2, 0, 0,  0,  0,  0, },
  },
  { /* 18 */
    { -2, 3, 0, 7, 0, 0, -3, -2, -2, },
    { 1, -3, 0, -7, 0, 0, 3, 2, 2, },
    { -2, 5, 0, 12, 0, 0, -5, -4, -4, },
  },
  { /* 19 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 20 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, -1.0/2,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 21 */
    { -1, 0, 0, 0, 0, -1,  0,  0,  0, },
    { -1, 0, 0, 1, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1, 0, -1,  0,  0,  0, },
  },
  { /* 22 */
    { -1, 0, 0, 0, 0, -1, 0, 0, 0, },
    { -1, 0, 0, 1, 0, 0, 0, 0, 0, },
    { -1, 0, 0, 1, 0, -1, 0, 0, 0, },
  },
  { /* 23 */
    { 1, 0, 0, 0, -2, 1,  0,  0,  0, },
    { -1, 0, 0, 0, 1, -1,  0,  0,  0, },
    { 1, 0, 0, 0, -1, 0,  0,  0,  0, },
  },
  { /* 24 */
    { 1, 0, 0, -2, 0, 3, 0, -2, 0, },
    { -1, 0, 0, 1, 0, -2, 0, 1, 0, },
    { 1, 0, 0, -1, 0, 1, 0, -1, 0, },
  },
};

static double cubic_VSpU[][3][9] = {
  { /* 1 */
    { -1.0/2, 1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { -1.0/2, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 3 */
    { -1.0/2, 1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { -1.0/2, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { -1.0/2, 1.0/2, 0, 1, 0, 0,  0,  0,  0, },
  },
  { /* 4 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, -1, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 5 */
    { -1.0/2, 1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { -1.0/2, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 1.0/2, -1.0/2, 0, -1, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 2, 0, 0, 0, 0, 1.0/2, -1, 0, },
    { 0, -1, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
  { /* 7 */
    { -1.0/2, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { -1.0/2, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 9 */
    { 0, -1.0/2, 0, -1, 0, -1,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, -1,  0,  0,  0, },
  },
  { /* 10 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 1, -1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 11 */
    { 0, -1.0/2, 0, -1, 0, 1,  0,  0,  0, },
    { 0, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 1.0/2, 0, 0, 0, -1,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, -2, 0, 2, 1.0/2, -1, 0, },
    { 0, 0, 0, 1, 0, -1, -1.0/2, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
  { /* 13 */
    { -1.0/2, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 1.0/2, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 14 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 15 */
    { -1.0/2, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 1.0/2, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { -1.0/2, -1.0/2, 0, 1, 0, 0,  0,  0,  0, },
  },
  { /* 16 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, -1, 0, 0, 0, 0, 1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 17 */
    { -1.0/2, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 1.0/2, -1.0/2, 0, 0, 0, 0,  0,  0,  0, },
    { 1.0/2, 1.0/2, 0, -1, 0, 0,  0,  0,  0, },
  },
  { /* 18 */
    { 0, -2, 0, 0, 0, 0, 1.0/2, 1, 0, },
    { 0, 1, 0, 0, 0, 0, -1.0/2, -1, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
  { /* 19 */
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 20 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 21 */
    { 0, 0, -1.0/2, -1, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 1,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 22 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 1, -1.0/2, 0, 0, },
    { 0, 0, 0, 1, 0, 0, -1.0/2, 0, 0, },
  },
  { /* 23 */
    { 0, 0, 1.0/2, -1, 0, 0,  0,  0,  0, },
    { 0, 0, 1.0/2, 0, 0, -1,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 24 */
    { 0, 0, 0, -2, 0, 2, 1.0/2, -1, 0, },
    { 0, 0, 0, 1, 0, -1, -1.0/2, 0, 0, },
    { 0, 0, 0, -1, 0, 0, 1.0/2, 0, 0, },
  },
};

static int cubic_generators[][3][9] = {
  { /* 1 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 3 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 5 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 7 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 9 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 11 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    {  -1, 0, 0, 0, -1, 0, 0, 0, 1, },
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 13 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 14 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 15 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 16 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 17 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 18 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 19 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 20 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 21 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 22 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  0, 0, 1, 1, 0, 0, 0, 1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 23 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 24 */
    {  1, 0, 0, 0, 1, 0, 0, 0, -1, },
    {  0, 0, -1, -1, 0, 0, 0, -1, 0, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
};


static double tetra_I_VSpU[][3][9] = {
  { /* 1 */
    { -1, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
    { 0, -1, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 2, 0, -1.0/2, 0, -1,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, -1, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 3 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, -1, 0, 0, 0,  0,  0,  0, },
  },
  { /* 4 */
    { -1, 0, 0, 0, -1, 0, 1.0/2, 0, -1.0/2, },
    { 0, 0, 0, 0, -1, 0, 1.0/2, 0, -1.0/2, },
    { 1, 0, 0, 0, 1, 0, -1, 0, 0, },
  },
  { /* 5 */
    { -1, 0, 0, 0, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 0, -1, 0,  0,  0,  0, },
  },
  { /* 6 */
    { -1, 0, 0, 0, -1, 0, -1.0/2, 0, 1.0/2, },
    { 0, 0, 0, 0, -1, 0, -1.0/2, 0, 1.0/2, },
    { -1, 0, 0, 0, -1, 0, 0, 0, 0, },
  },
  { /* 7 */
    { -3.0/4, 1.0/4, 1.0/4,  0,  0,  0,  0,  0,  0, },
    { -1.0/4, -1.0/4, -1.0/4,  0,  0,  0,  0,  0,  0, },
    { -1.0/2, 1.0/2, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { -2, 1, 0, 1, 0, 0,  0,  0,  0, },
  },
  { /* 9 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, -1, -1, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { -1, 0, 0, 0, 1, 0, -1.0/2, 0, 1.0/2, },
    { 0, 0, 0, 0, -1, 0, 1.0/2, 0, -1.0/2, },
    { 1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 11 */
    { -3.0/4, 1.0/4, 0, 0, 1.0/4, 0,  0,  0,  0, },
    { -1.0/4, -1.0/4, 0, 0, -1.0/4, 0,  0,  0,  0, },
    { -1.0/2, 1.0/2, 0, 0, -1.0/2, 0,  0,  0,  0, },
  },
  { /* 12 */
    { -1, 0, 0, 0, 1, 0, 1.0/2, 0, -1.0/2, },
    { 0, 0, 0, 0, -1, 0, -1.0/2, 0, 1.0/2, },
    { -1, 0, 0, 0, 1, 0, 1, 0, -1, },
  },
};

static double tetra_VSpU[][3][9] = {
  { /* 1 */
    { -1.0/2, 1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { -1.0/2, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, 0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 3 */
    { -1, 0, 0, 0, 1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, -1.0/2,  0,  0,  0, },
  },
  { /* 4 */
    { -1, 0, 0, 0, 1.0/2, 0, 0, 0, 0, },
    { 0, 0, 0, 0, -1.0/2, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 5 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { -1, 0, 0, 1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, 0, 0, 0, 0,  0,  0,  0, },
  },
  { /* 6 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { -1, 0, 0, 0, 0, 0, 1.0/2, 0, 0, },
    { 0, 0, 0, 0, 0, 0, 0, 0, -1.0/2, },
  },
  { /* 7 */
    { -1.0/2, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 1.0/2, -1.0/2, 0,  0,  0,  0,  0,  0,  0, },
    { 0, 0, -1.0/2,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 9 */
    { -1, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, 0, 0, -1.0/2, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 10 */
    { -1, 0, 0, 0, -1.0/2, 0, 0, 0, 0, },
    { 0, 0, 0, 0, -1.0/2, 0, 0, 0, 0, },
    { 0, 0, 0, 0, 0, -1.0/2, 0, 0, 0, },
  },
  { /* 11 */
    { 0, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 1, 0, 0, -1.0/2, 0, 0,  0,  0,  0, },
    { 0, 0, -1.0/2, 0, 0, 0,  0,  0,  0, },
  },
  { /* 12 */
    { 0, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 1, 0, 0, 0, 0, 0, -1.0/2, 0, 0, },
    { 0, 0, -1.0/2, 0, 0, 0, 0, 0, 0, },
  },
};

static int tetra_generators[][3][9] = {
  { /* 1 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 2 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 3 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 4 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 5 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 6 */
    {  0, -1, 0, 1, 0, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 7 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 8 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 9 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 10 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  1, 0, 0, 0, -1, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
  { /* 11 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {   0,  0,  0,  0,  0,  0,  0,  0,  0, },
  },
  { /* 12 */
    {  0, 1, 0, -1, 0, 0, 0, 0, -1, },
    {  -1, 0, 0, 0, 1, 0, 0, 0, 1, },
    {  -1, 0, 0, 0, -1, 0, 0, 0, -1, },
  },
};



static int find_hall_symbol( double origin_shift[3],
			     const Centering centering,
			     SPGCONST Symmetry *symmetry );
static int is_hall_symbol( double shift[3],
			   const int hall_number,
			   SPGCONST Symmetry *symmetry,
			   Centering centering,
			   SPGCONST int generators[3][9],
			   SPGCONST double VSpU[3][9] );
static int is_hall_symbol_cubic( double shift[3],
				 SPGCONST Symmetry *symmetry,
				 Centering centering );
static int is_hall_symbol_hexa( double shift[3],
				SPGCONST Symmetry *symmetry );
static int is_hall_symbol_rhombo( double shift[3],
				  SPGCONST Symmetry *symmetry );
static int is_hall_symbol_trigonal( double shift[3],
				    SPGCONST Symmetry *symmetry );
static int is_hall_symbol_tetra( double shift[3],
				 SPGCONST Symmetry *symmetry,
				 const Centering centering );
static int is_hall_symbol_ortho( double shift[3],
				 SPGCONST Symmetry *symmetry,
				 const Centering centering );
static int is_hall_symbol_monocli( double shift[3],
				   SPGCONST Symmetry *symmetry,
				   const Centering centering );
static int is_hall_symbol_tricli( double shift[3],
				  SPGCONST Symmetry *symmetry );
static int get_translations( double trans[3][3],
			     SPGCONST Symmetry *symmetry,
			     Centering centering,
			     SPGCONST int rot[3][3][3] );
static void transform_translation( double trans_reduced[3],
				   const Centering centering,
				   const double trans[3] );
static void transform_translation_inverse( double trans[3],
					   const Centering centering,
					   const double trans_reduced[3] );
static int get_origin_shift( double shift[3],
			     const int hall_number,
			     SPGCONST int rot[3][3][3],
			     SPGCONST double trans[3][3],
			     const Centering centering,
			     SPGCONST double VSpU[3][9] );
static void unpack_generators( int rot[3][3][3], int generators[3][9] );
static int set_dw( double dw[3],
		   const int hall_number, 
		   const int operation_index[2],
		   SPGCONST int rot[3][3],
		   const double trans[3],
		   const Centering centering );
static int is_match_database( const int hall_number,
			      const double shift[3],
			      SPGCONST Symmetry *symmetry,
			      const Centering centering );


static double tolerance;
static double lattice[3][3];

int hal_get_hall_symbol( double origin_shift[3],
			 const Centering centering,
			 SPGCONST Symmetry *symmetry,
			 SPGCONST double bravais_lattice[3][3],
			 const double symprec )
{
  mat_copy_matrix_d3( lattice, bravais_lattice );
  tolerance = symprec;

  return find_hall_symbol( origin_shift, centering, symmetry );
}

static int find_hall_symbol( double origin_shift[3],
			     const Centering centering,
			     SPGCONST Symmetry *symmetry )
{
  int hall_number = 0;

  /* CUBIC IT: 195-230, Hall: 489-530 */
  hall_number = is_hall_symbol_cubic( origin_shift,
				      symmetry,
				      centering );
  if ( hall_number ) { goto end; }

  /* HEXA, IT: 168-194, Hall: 462-488 */
  hall_number = is_hall_symbol_hexa( origin_shift, symmetry );
  if ( hall_number ) { goto end; }

  /* TRIGO, IT: 143-167, Hall: 430-461 */
  hall_number = is_hall_symbol_trigonal( origin_shift, symmetry );
  if ( hall_number ) { goto end; }

  /* RHOMB, IT: 143-167, Hall: 430-461 */
  hall_number = is_hall_symbol_rhombo( origin_shift, symmetry );
  if ( hall_number ) { goto end; }

  /* TETRA, IT: 75-142, Hall: 349-429 */
  hall_number = is_hall_symbol_tetra( origin_shift, symmetry, centering );
  if ( hall_number ) { goto end; }
  
  /* ORTHO, IT: 16-74, Hall: 108-348 */
  hall_number = is_hall_symbol_ortho( origin_shift, symmetry, centering );
  if ( hall_number ) { goto end; }

  /* MONOCLI, IT: 3-15, Hall: 3-107 */
  hall_number = is_hall_symbol_monocli( origin_shift, symmetry, centering );
  if ( hall_number ) { goto end; }

  /* TRICLI, IT: 1-2, Hall: 1-2 */
  hall_number = is_hall_symbol_tricli( origin_shift, symmetry );
  if ( hall_number ) { goto end; }

 end:
  return hall_number;
}

static int is_hall_symbol_cubic( double shift[3],
				 SPGCONST Symmetry *symmetry,
				 Centering centering )
{
  int i, hall_number;
  Symmetry *conv_symmetry;
  double trans_mat[3][3] = { { 0, 0, 1 },
			     { 0,-1, 0 },
			     { 1, 0, 0 } };

  for ( i = 0; i < 24; i++) {
    for ( hall_number = 489; hall_number < 531; hall_number++ ) {
      /* Special case of Pa-3 (205) */
      if ( hall_number==501 ) {
	if ( is_hall_symbol( shift,
			     501,
			     symmetry,
			     centering,
			     cubic_generators[i],
			     cubic_VSpU[i] ) ) { goto found; }

	/* Try another basis */
	conv_symmetry = spa_get_conventional_symmetry( trans_mat,
						       centering,
						       symmetry );
	
	if ( is_hall_symbol( shift,
			     501,
			     conv_symmetry,
			     centering,
			     cubic_generators[i],
			     cubic_VSpU[i] ) ) { goto found; }

	sym_free_symmetry( conv_symmetry );
	continue;
      }
      
      if ( centering==NO_CENTER ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     cubic_generators[i],
			     cubic_VSpU[i] ) ) { goto found; }
      }

      if ( centering==BODY ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     cubic_generators[i],
			     cubic_I_VSpU[i] ) ) { goto found; }
      }

      if ( centering==FACE ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     cubic_generators[i],
			     cubic_F_VSpU[i] ) ) { goto found; }
      }
    }
  }

  return 0;

 found:
  return hall_number;

}

static int is_hall_symbol_hexa( double shift[3],
				SPGCONST Symmetry *symmetry )
{
  int i, hall_number;

  for ( i = 0; i < 12; i++) {
    for ( hall_number = 462; hall_number < 489; hall_number++ ) {
      if ( is_hall_symbol( shift,
			   hall_number,
			   symmetry,
			   NO_CENTER,
			   hexa_generators[i],
			   hexa_VSpU[i] ) ) { goto found; }
    }
  }

  return 0;

 found:
  return hall_number;
}

static int is_hall_symbol_trigonal( double shift[3],
				    SPGCONST Symmetry *symmetry )
{
  int i, hall_number;

  for ( i = 0; i < 20; i++) {
    for ( hall_number = 430; hall_number < 462; hall_number++ ) {
      if ( is_hall_symbol( shift,
			   hall_number,
			   symmetry,
			   NO_CENTER,
			   trigo_generators[i],
			   trigo_VSpU[i] ) ) { goto found; }
    }
  }

  return 0;

 found:
  return hall_number;

}

static int is_hall_symbol_rhombo( double shift[3],
				  SPGCONST Symmetry *symmetry )
{
  int i, hall_number;

  for ( i = 0; i < 20; i++) {
    for ( hall_number = 430; hall_number < 462; hall_number++ ) {
      if ( is_hall_symbol( shift,
			   hall_number,
			   symmetry,
			   NO_CENTER,
			   rhombo_generators[i],
			   rhombo_VSpU[i] ) ) { goto found; }
    }
  }

  return 0;

 found:
  return hall_number;

}

static int is_hall_symbol_tetra( double shift[3],
				 SPGCONST Symmetry *symmetry,
				 const Centering centering )
{
  int i,  hall_number;

  for ( i = 0; i < 12; i++) {
    for ( hall_number = 349; hall_number < 429; hall_number++ ) {
      if ( centering==NO_CENTER ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     tetra_generators[i],
			     tetra_VSpU[i] ) ) { goto found; }
      }

      if ( centering==BODY ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     tetra_generators[i],
			     tetra_I_VSpU[i] ) ) { goto found; }
      }
    }
  }

  return 0;

 found:
  return hall_number;
}

static int is_hall_symbol_ortho( double shift[3],
				 SPGCONST Symmetry *symmetry,
				 const Centering centering )
{
  int hall_number;
  int i;

  for ( i = 0; i < 12; i++) {
    for ( hall_number = 108; hall_number < 348; hall_number++ ) {
      if ( centering==NO_CENTER ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     ortho_generators[i],
			     ortho_VSpU[i] ) ) { goto found; }
      }

      if ( centering==BODY ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     ortho_generators[i],
			     ortho_I_VSpU[i] ) ) { goto found; }
      }

      if ( centering==FACE ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     ortho_generators[i],
			     ortho_F_VSpU[i] ) ) { goto found; }
      }

      if ( centering==A_FACE ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     ortho_generators[i],
			     ortho_A_VSpU[i] ) ) { goto found; }
      }

      if ( centering==B_FACE ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     ortho_generators[i],
			     ortho_B_VSpU[i] ) ) { goto found; }
      }

      if ( centering==C_FACE ) {
	if ( is_hall_symbol( shift,
			     hall_number,
			     symmetry,
			     centering,
			     ortho_generators[i],
			     ortho_C_VSpU[i] ) ) { goto found; }
      }
    }
  }

  return 0;

 found:
  return hall_number;

}

static int is_hall_symbol_monocli( double shift[3],
				   SPGCONST Symmetry *symmetry,
				   const Centering centering )
{
  int hall_number;
  int i;

  for ( i = 0; i < 12; i++) {
    for ( hall_number = 3; hall_number < 108; hall_number++ ) {
      if ( centering==NO_CENTER ) {
	if ( is_hall_symbol( shift,
			      hall_number,
			      symmetry,
			      centering,
			      monocli_generators[i],
			      monocli_VSpU[i] ) ) { goto found; }
      }

      if ( centering==A_FACE ) {
	if ( is_hall_symbol( shift,
			      hall_number,
			      symmetry,
			      centering,
			      monocli_generators[i],
			      monocli_A_VSpU[i] ) ) { goto found; }
      }

      if ( centering==B_FACE ) {
	if ( is_hall_symbol( shift,
			      hall_number,
			      symmetry,
			      centering,
			      monocli_generators[i],
			      monocli_B_VSpU[i] ) ) { goto found; }
      }

      if ( centering==C_FACE ) {
	if ( is_hall_symbol( shift,
			      hall_number,
			      symmetry,
			      centering,
			      monocli_generators[i],
			      monocli_C_VSpU[i] ) ) { goto found; }
      }

      if ( centering==BODY ) {
	if ( is_hall_symbol( shift,
			      hall_number,
			      symmetry,
			      centering,
			      monocli_generators[i],
			      monocli_I_VSpU[i] ) ) { goto found; }
      }

    
    }
  }
  return 0;

 found:
  return hall_number;

}

static int is_hall_symbol_tricli( double shift[3],
				  SPGCONST Symmetry *symmetry )
{
  int i, hall_number;

  for ( i = 0; i < 2; i++) {
    for ( hall_number = 1; hall_number < 3; hall_number++ ) {
      if ( is_hall_symbol( shift,
			    hall_number,
			    symmetry,
			    NO_CENTER,
			    tricli_generators[i],
			    tricli_VSpU[i] ) ) { goto found; }
    }
  }

  return 0;

 found:
  return hall_number;

}

static void unpack_generators( int rot[3][3][3], int generators[3][9] )
{
  int i, j, k;
  for ( i = 0; i < 3; i++ ) {
    for ( j = 0; j < 3; j++ ) {
      for ( k = 0; k < 3; k++ ) {
	rot[i][j][k] = generators[i][j*3+k];
      }
    }
  }
}

static int is_hall_symbol( double shift[3],
			   const int hall_number,
			   SPGCONST Symmetry *symmetry,
			   Centering centering,
			   SPGCONST int generators[3][9],
			   SPGCONST double VSpU[3][9] )
{
  int is_origin_shift;
  int operation_index[2];
  int rot[3][3][3];
  double trans[3][3];

  unpack_generators( rot, generators );

  debug_print("Start Hall symbol check\n");
  debug_print("Generators\n");
  debug_print_matrix_i3( rot[0] );
  debug_print_matrix_i3( rot[1] );
  debug_print_matrix_i3( rot[2] );
  debug_print("Test Hall symbol #%d.\n", hall_number);

  spgdb_get_operation_index( operation_index, hall_number );
  if ( ! ( operation_index[0] == symmetry->size ) ) {
    debug_print("Hall symbol #%d doesn't match.\n", hall_number );
    goto not_found;
  }

  if ( get_translations( trans, symmetry, centering, rot ) ) {
    is_origin_shift = get_origin_shift( shift,
					hall_number,
					rot,
					trans,
					centering,
					VSpU );

    debug_print("origin shift: %f %f %f\n",
		shift[0],
		shift[1],
		shift[2]);

    if ( is_origin_shift ) {
      if ( is_match_database( hall_number,
			      shift,
			      symmetry,
			      centering ) ) {
	debug_print("Match with database %d\n", hall_number);
	goto found;
      }
    }
    debug_print("Hall symbol #%d doesn't match.\n", hall_number);
  } else {
    goto not_found;
  }

 not_found:
  return 0;

 found:
  return 1;
}

static int get_translations( double trans[3][3],
			     SPGCONST Symmetry *symmetry,
			     Centering centering,
			     SPGCONST int rot[3][3][3] )
{
  int i, j;
  int is_found;
  static SPGCONST int zero[3][3] = { { 0, 0, 0 }, 
				     { 0, 0, 0 }, 
				     { 0, 0, 0 }, };

  for ( i = 0; i < 3; i++ ) {
    for ( j = 0; j < 3; j++ ) {
      trans[i][j] = 0;
    }
  }

  for ( i = 0; i < 3; i++ ) {
    if ( mat_check_identity_matrix_i3( rot[i], zero ) ) { continue; }
    is_found = 0;
    for ( j = 0; j < symmetry->size; j++ ) {
      if ( mat_check_identity_matrix_i3( symmetry->rot[j], rot[i] ) ) {
	mat_copy_vector_d3( trans[i], symmetry->trans[j] );
	is_found = 1;
	debug_print("trans%d found! %f %f %f\n", i, trans[i][0], trans[i][1], trans[i][2] );
	break;
      }
    }
    if ( ! is_found ) {
      goto not_found;
    }
  }

  debug_print("all translations found!\n");
  return 1;

 not_found:

  return 0;
}

static void transform_translation( double trans_reduced[3],
				   const Centering centering,
				   const double trans[3] )
{
  int i;
  static int M_bcc[3][3] = {
    { 0, 1, 1 },
    { 1, 0, 1 },
    { 1, 1, 0 },
  };
  static int M_fcc[3][3] = {
    {-1, 1, 1 },
    { 1,-1, 1 },
    { 1, 1,-1 },
  };
  static int M_ac[3][3] = {
    { 1, 0, 0 },
    { 0, 1, 1 },
    { 0,-1, 1 },
  };
  static int M_bc[3][3] = {
    { 1, 0, 1 },
    { 0, 1, 0 },
    {-1, 0, 1 },
  };
  static int M_cc[3][3] = {
    { 1, 1, 0 },
    {-1, 1, 0 },
    { 0, 0, 1 },
  };

  switch ( centering ) {
  case NO_CENTER:
    mat_copy_vector_d3( trans_reduced, trans );
    break;
  case BODY:
    mat_multiply_matrix_vector_id3( trans_reduced, M_bcc, trans );
    break;
  case FACE:
    mat_multiply_matrix_vector_id3( trans_reduced, M_fcc, trans );
    break;
  case A_FACE:
    mat_multiply_matrix_vector_id3( trans_reduced, M_ac, trans );
    break;
  case B_FACE:
    mat_multiply_matrix_vector_id3( trans_reduced, M_bc, trans );
    break;
  case C_FACE:
    mat_multiply_matrix_vector_id3( trans_reduced, M_cc, trans );
    break;
  default:
    break;
  }

  for ( i = 0; i < 3; i++ ) {
    trans_reduced[i] -= mat_Nint( trans_reduced[i] );
  }
}

static void transform_translation_inverse( double trans[3],
					   const Centering centering,
					   const double trans_reduced[3] )
{
  int i;
  static double M_bcc_inv[3][3] = {
    {-0.5, 0.5, 0.5},
    { 0.5,-0.5, 0.5},
    { 0.5, 0.5,-0.5},
  };
  static double M_fcc_inv[3][3] = {
    { 0.0, 0.5, 0.5},
    { 0.5, 0.0, 0.5},
    { 0.5, 0.5, 0.0},
  };
  static double M_ac_inv[3][3] = {
    { 1.0, 0.0, 0.0},
    { 0.0, 0.5,-0.5},
    { 0.0, 0.5, 0.5},
  };
  static double M_bc_inv[3][3] = {
    { 0.5, 0.0,-0.5},
    { 0.0, 1.0, 0.0},
    { 0.5, 0.0, 0.5},
  };
  static double M_cc_inv[3][3] = {
    { 0.5,-0.5, 0.0},
    { 0.5, 0.5, 0.0},
    { 0.0, 0.0, 1.0},
  };

  switch ( centering ) {
  case NO_CENTER:
    mat_copy_vector_d3( trans, trans_reduced );
    break;
  case BODY:
    mat_multiply_matrix_vector_d3( trans, M_bcc_inv, trans_reduced );
    break;
  case FACE:
    mat_multiply_matrix_vector_d3( trans, M_fcc_inv, trans_reduced );
    break;
  case A_FACE:
    mat_multiply_matrix_vector_d3( trans, M_ac_inv, trans_reduced );
    break;
  case B_FACE:
    mat_multiply_matrix_vector_d3( trans, M_bc_inv, trans_reduced );
    break;
  case C_FACE:
    mat_multiply_matrix_vector_d3( trans, M_cc_inv, trans_reduced );
    break;
  default:
    break;
  }

  for ( i = 0; i < 3; i++ ) {
    trans[i] -= mat_Nint( trans[i] );
  }
}

static int get_origin_shift( double shift[3],
			     const int hall_number,
			     SPGCONST int rot[3][3][3],
			     SPGCONST double trans[3][3],
			     const Centering centering,
			     SPGCONST double VSpU[3][9] )
{
  int i, j;
  int operation_index[2];
  double dw[9], tmp_dw[3], tmp_shift[3];

  debug_print("*** get_origin_shift ***\n");

  spgdb_get_operation_index( operation_index, hall_number );

  /* The obtained dw is reduced to that of primitve cell by centerings. */
  for ( i = 0; i < 3; i++ ) {
    /* Zero matrix is the sign to set dw 0 */
    if ( mat_get_determinant_i3( rot[i] ) == 0 ) {
      for ( j = 0; j < 3; j++ ) {
	dw[i*3+j] = 0;
      }
    } else {
      if ( set_dw( tmp_dw, hall_number, operation_index,
		   rot[i], trans[i], centering ) ) {
	for ( j = 0; j < 3; j++ ) {
	  dw[i*3+j] = tmp_dw[j];
	}
      } else {
	goto not_found;
      }
    }
  }


#ifdef DEBUG
  printf("VSpU\n");
  for ( i = 0; i < 3; i++ ) {
    for ( j = 0; j < 9; j++ ) {
      printf("%4.1f ", VSpU[i][j]);
    }
    printf("\n");
  }
#endif

  /* VSpU*dw is given for the primitive cell if there is centering. */
  for ( i = 0; i < 3; i++ ) {
    tmp_shift[i] = 0;
    for ( j = 0; j < 9; j++ ) {
      tmp_shift[i] += VSpU[i][j] * dw[j];
    }
  }
  /* Transform VSpU*dw to that of the conventional unit cell. */
  transform_translation_inverse( shift, centering, tmp_shift );

  return 1;

 not_found:
  return 0;
}

static int set_dw( double dw[3],
		   const int hall_number, 
		   const int operation_index[2],
		   SPGCONST int rot[3][3],
		   const double trans[3],
		   const Centering centering )
{
  int i, j, tmp_hall_num;
  int rot_db[3][3];
  double trans_db[3], tmp_dw[3];

  debug_print("Generator\n");
  debug_print_matrix_i3( rot );

  for ( i = 0; i < operation_index[0]; i++ ) {
    /* rotation matrix matching and set difference of translations */
    tmp_hall_num = spgdb_get_operation( rot_db,
					trans_db,
					operation_index[1] + i );
    if ( ! ( tmp_hall_num  == hall_number ) ) {
      warning_print("spglib BUG: line %d, %s\n", __LINE__, __FILE__);
      warning_print("spglib BUG: %d != %d\n", tmp_hall_num, hall_number);
      goto err;
    }

    if ( mat_check_identity_matrix_i3( rot_db, rot ) ) {
      for ( j = 0; j < 3; j++ ) {
	tmp_dw[j] = trans_db[j] - trans[j];
      }
      /* Transform dw to that of primitive cell if there is centering. */
      transform_translation( dw, centering, tmp_dw );

      debug_print("trans(db): %f %f %f\n",
		  trans_db[0], trans_db[1], trans_db[2] );
      debug_print("trans    : %f %f %f\n",
		  trans[0], trans[1], trans[2] );

      debug_print("dw       : %f %f %f\n",
		  dw[0], dw[1], dw[2] );

      goto found;
    }
  }

  /* Not found */
 err:
  return 0;
  
 found:
  return 1;
}

static int is_match_database( const int hall_number,
			      const double origin_shift[3],
			      SPGCONST Symmetry *symmetry,
			      const Centering centering )
{
  int i, j, k, is_found;
  int operation_index[2];
  int rot_db[3][3];
  int found_list[192];
  double trans_db[3], conv_trans[3], tmp_vec[3];

  spgdb_get_operation_index( operation_index, hall_number );

  for ( i = 0; i < symmetry->size; i++ ) { found_list[i] = 0; }

  for ( i = 0; i < symmetry->size; i++ ) {
    is_found = 0;
    for ( j = 0; j < operation_index[0]; j++ ) {
      /* rotation matrix matching and set difference of translations */
      if (! ( spgdb_get_operation( rot_db,
				   trans_db,
				   operation_index[1]+j )==hall_number )) {
	goto not_found;
      }

      if ( mat_check_identity_matrix_i3( symmetry->rot[i], rot_db ) ) {
	mat_multiply_matrix_vector_id3( tmp_vec, rot_db, origin_shift );
	for ( k = 0; k < 3; k++ ) {
	  conv_trans[k] = tmp_vec[k] + symmetry->trans[i][k] - origin_shift[k];
	}
	
	if ( cel_is_overlap( conv_trans,
			     trans_db,
			     lattice,
			     tolerance ) && ( ! found_list[j] ) ) {
	  found_list[j] = 1;
	  is_found = 1;
	  
	  debug_print("---match!(%d)---\n", i+1);
	  debug_print_matrix_i3( rot_db );
	  debug_print("trans(db)   : %f %f %f\n",
		      trans_db[0], trans_db[1], trans_db[2] );
	  debug_print("trans       : %f %f %f\n",
		      symmetry->trans[i][0],
		      symmetry->trans[i][1],
		      symmetry->trans[i][2] );
	  debug_print("trans (conv): %f %f %f\n",
		      conv_trans[0], conv_trans[1], conv_trans[2] );

	  break;
	} else {
	  ;
	  debug_print("---not match!(%d)---\n", i+1);
	  debug_print_matrix_i3( rot_db );
	  debug_print("trans(db): %f %f %f\n", trans_db[0], trans_db[1], trans_db[2] );
	  debug_print("trans    : %f %f %f\n",
		      symmetry->trans[i][0],
		      symmetry->trans[i][1],
		      symmetry->trans[i][2] );
	  debug_print("trans (conv): %f %f %f\n",
		      conv_trans[0], conv_trans[1], conv_trans[2] );
	}
      }
    }
    if ( ! is_found ) {
      goto not_found;
    }
  }

  debug_print("is_match OK!\n");
  return 1;

 not_found:
  return 0;
}

