var count = (function quickTest(BigNumber) {

    var start = +new Date(),
        log,
        error,
        undefined,
        passed = 0,
        total = 0,
        n = 'null',
        N = 'NaN',
        I = 'Infinity';

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    BigNumber.config({
        DECIMAL_PLACES : 20,
        ROUNDING_MODE : 4,
        ERRORS : true,
        RANGE : 1E9,
        EXPONENTIAL_AT : 1E9
    });

    log('\n Starting tests...');

    function Tplus(addendA, addendB, expected) {
        assert(String(expected), String(new BigNumber(addendA).plus(addendB)));
        assert(String(expected), String(new BigNumber(addendA).plus(new BigNumber(addendB))));
    }

    Tplus('0.0000002081', '951', '951.0000002081');
    Tplus('-0.00000000000101', '290.41', '290.40999999999899');
    Tplus('591', '-0.0000000000000023311', '590.9999999999999976689');
    Tplus('0.000000000006331', '-0.0000901', '-0.000090099993669');
    Tplus('-0.000000000000000101', '8.811', '8.810999999999999899');
    Tplus('9.71', '25.2521', '34.9621');
    Tplus('12.61', '22581', '22593.61');
    Tplus('11', '-38.91', '-27.91');
    Tplus('-0.00000000138751', '2.11', '2.10999999861249');
    Tplus('3762.2711', '-744.88784434996040047048938241', '3017.38325565003959952951061759');
    Tplus('-0.000000000000000025415273976364280222001', '-492878574085250915901', '-492878574085250915901.000000000000000025415273976364280222001');
    Tplus('426904249750.96038450081242691', '56433.631', '426904306184.59138450081242691');
    Tplus('-0.000000000000000134741', '32233.1553231', '32233.155323099999999865259');
    Tplus('-20527603846670651', '-19635621.807985261', '-20527603866306272.807985261');
    Tplus('-86085754207.708686061', '-1', '-86085754208.708686061');
    Tplus('-4.96914656033745696212200401', '-1.241', '-6.21014656033745696212200401');
    Tplus('329060927228852916217075.1962341', '4100482994459676741454741', '4429543921688529657671816.1962341');
    Tplus('615901601', '-6473391841168.33422241481', '-6472775939567.33422241481');
    Tplus('-5812703356423471', '-0.0000000000000000000000000000000000000001512809242910987744755268662361', '-5812703356423471.0000000000000000000000000000000000000001512809242910987744755268662361');
    Tplus('-964262.331151767124531', '-56784.754061', '-1021047.085212767124531');
    Tplus('-10804741298331762053003.61', '-18727784577281', '-10804741317059546630284.61');
    Tplus('2455658857532054049742171.776541', '-11', '2455658857532054049742160.776541');
    Tplus('-8.419821', '-1121', '-1129.419821');
    Tplus('-14118910.8630102182791', '2231454017.61', '2217335106.7469897817209');
    Tplus('-35192.01', '30.31011', '-35161.69989');
    Tplus('21', '1214.75047979608491', '1235.75047979608491');

    function Tminus(minuend, subtrahend, expected) {
        assert(String(expected), String(new BigNumber(minuend).minus(subtrahend)));
        assert(String(expected), String(new BigNumber(minuend).minus(new BigNumber(subtrahend))));
    }

    Tminus('4.851', '41', '-36.149');
    Tminus('21', '13081', '-13060');
    Tminus('0.00000000056061', '-0.1401', '0.14010000056061');
    Tminus('61', '11', '50');
    Tminus('-1137.71', '-74.211', '-1063.499');
    Tminus('-1', '-0.0000000000000131', '-0.9999999999999869');
    Tminus('-0.000513821', '0.261', '-0.261513821');
    Tminus('-0.000401', '-0.0002471', '-0.0001539');
    Tminus('1', '31', '-30');
    Tminus('-19851', '-1.51', '-19849.49');
    Tminus('8016778098131', '6331.62851', '8016778091799.37149');
    Tminus('-270.9996571', '23960950851', '-23960951121.9996571');
    Tminus('-7986807223743730.000000000000000000000000000000000001', '-7173955.760471', '-7986807216569774.239529000000000000000000000000000001');
    Tminus('1.48968081', '242980.00000000000000000000000000000000000000000000000001', '-242978.51031919000000000000000000000000000000000000000001');
    Tminus('-118422270.0000000000000000000000000001', '14530.0001', '-118436800.0001000000000000000000000001');
    Tminus('9330969181250.000000000000000000000000000000000000001', '50.71', '9330969181199.290000000000000000000000000000000000001');
    Tminus('152192135853041', '-12.46214721', '152192135853053.46214721');
    Tminus('29353.8632116991', '29.491', '29324.3722116991');
    Tminus('-24835555113.72271', '16121.144514291', '-24835571234.867224291');
    Tminus('-55940242410820.01', '384349225568820.00000001', '-440289467979640.01000001');
    Tminus('283535576260.000000000000000000000000000000001', '14.121', '283535576245.879000000000000000000000000000001');
    Tminus('480.0000000000000000000000000000001', '-164098203.91', '164098683.9100000000000000000000000000001');
    Tminus('29364155498.97581', '-2885650871', '32249806369.97581');
    Tminus('4.1021951', '-165723099860.0000000000000000000000000001', '165723099864.1021951000000000000000000001');
    Tminus('-530.81', '56840.01', '-57370.82');

    function Ttimes(multiplicand, multiplier, expected) {
        assert(String(expected), String(new BigNumber(multiplicand).times(multiplier)));
        assert(String(expected), String(new BigNumber(multiplicand).times(new BigNumber(multiplier))));
    }

    Ttimes('-21', '131', '-2751');
    Ttimes('111', '0.000000000841', '0.000000093351');
    Ttimes('40.71', '21', '854.91');
    Ttimes('-0.0000000000036161', '-1', '0.0000000000036161');
    Ttimes('7.8121', '36.121', '282.1808641');
    Ttimes('1', '46.221', '46.221');
    Ttimes('-128.21', '-1', '128.21');
    Ttimes('-180766.9302821', '291', '-52603176.7120911');
    Ttimes('3714507177452301036251', '6867200.8062651448391', '25508266683878109110598887924.4875346622141');
    Ttimes('0.0000000000000000000000654341', '0.00000000000000034029824715606780001', '0.000000000000000000000000000000000000022267109534234856032634341');
    Ttimes('2902811542.1751', '21.7592802616132547300301', '63163089892.83378534432594919064247051');
    Ttimes('-0.0000004236706737870443510907850088041', '-2870.7155483080953029218351', '0.00121623799060263520422140469527350360094358075600762840391');
    Ttimes('806512666721.29120081', '-8211', '-6622275506448522.04985091');
    Ttimes('24.0654772221', '270.37988121', '6506.820872573358786741');
    Ttimes('128868235795911', '-4897225358224393204401', '-631096792209375818286886121943004311');
    Ttimes('-47584.99084549466512725031', '0.000000000000000000000000000000000000000000154041881', '-0.00000000000000000000000000000000000000733008149720777859166674211023311');
    Ttimes('-98.9161', '-18710464022168031', '1850766130263175171.1991');
    Ttimes('-141', '1387696355.96881144331', '-195665186191.60241350671');
    Ttimes('-546863822.0898879518644444391', '-16471500507546.2972881', '9007667723112297421676.09483644538995959345560471');
    Ttimes('0.32596723638370271', '833.075641', '271.55536439535165708668711');
    Ttimes('-363496.57279544085492428981', '-39403998991', '14323218587663509.49683689305463158171');
    Ttimes('3.11', '36726825625425561919475.9510331', '114220427695073497569570.207712941');

    function Tdiv(dividend, divisor, expected) {
        assert(String(expected), String(new BigNumber(dividend).div(divisor)));
        assert(String(expected), String(new BigNumber(dividend).div(new BigNumber(divisor))));
    }

    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 0});
    Tdiv('999.5', 1, '1000');
    BigNumber.config({DECIMAL_PLACES : 0, ROUNDING_MODE : 6});
    Tdiv('-223891', '-0.000000000000000000005965201', '37532850946682266029258696');
    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 5});
    Tdiv('-2.51', '21', '-0.1195238095238095238');
    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 0});
    Tdiv('-9.11', '-5.61', '1.6239');
    BigNumber.config({DECIMAL_PLACES : 18, ROUNDING_MODE : 5});
    Tdiv('-789.23741', '2996.9101', '-0.263350378778462524');
    BigNumber.config({DECIMAL_PLACES : 19, ROUNDING_MODE : 1});
    Tdiv('86.41', '0.00000000000000901', '9590455049944506.1043285238623751387');
    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 2});
    Tdiv('-1.031', '-11', '0.093727272728');
    BigNumber.config({DECIMAL_PLACES : 2, ROUNDING_MODE : 1});
    Tdiv('1', '1.691', '0.59');
    BigNumber.config({DECIMAL_PLACES : 12, ROUNDING_MODE : 0});
    Tdiv('-0.0000000210121', '2.81', '-0.000000007478');
    BigNumber.config({DECIMAL_PLACES : 25, ROUNDING_MODE : 6});
    Tdiv('2844.11', '-1.681', '-1691.9155264723378941106484236');

    BigNumber.config({EXPONENTIAL_AT : 0} );

    BigNumber.config({DECIMAL_PLACES : 18, ROUNDING_MODE : 1});
    Tdiv('1.59425939501316841840968291765489155608078575071e+9', '-3.08038002682802076462608674211e+24', '-5.17e-16');
    BigNumber.config({DECIMAL_PLACES : 32, ROUNDING_MODE : 1});
    Tdiv('-2.7776797347790808834515841467871979111321511e+24', '3.3415875353512401063640184461e+3', '-8.3124553985000249637933047177077877273241264998630019e+20');
    BigNumber.config({DECIMAL_PLACES : 44, ROUNDING_MODE : 1});
    Tdiv('1.256012023778514778181132285841e+17', '-5.465631858655701463629691267993890472939145146541478599849959175484551e+15', '-2.298017971681387626003643396450012590362066053e+1');
    BigNumber.config({DECIMAL_PLACES : 14, ROUNDING_MODE : 0});
    Tdiv('1.378656411343789736916650471e+20', '-1.354076170203205324162874106485640621111e+19', '-1.018152775804995e+1');
    BigNumber.config({DECIMAL_PLACES : 37, ROUNDING_MODE : 4});
    Tdiv('1.577809310668934678754980869332886769208471e+23', '6.93677728955357384105850187632701172506828921e+11', '2.27455667784669044615322243776416258218503335378e+11');
    BigNumber.config({DECIMAL_PLACES : 14, ROUNDING_MODE : 3});
    Tdiv('-5.3967707960012001e+16', '9.289226824998276692885063155660376484197905615639719776994658491e+23', '-5.80971e-8');
    BigNumber.config({DECIMAL_PLACES : 16, ROUNDING_MODE : 0});
    Tdiv('-1.7511951356552580494958278051e+28', '3.1019236701050639775694537481e+28', '-5.645513306895602e-1');

    BigNumber.config({EXPONENTIAL_AT : 1E9});

    function Tsqrt(expected, actual) {
        assert(expected, new BigNumber(actual).sqrt().toString())
    }

    BigNumber.config({DECIMAL_PLACES : 96, ROUNDING_MODE : 2});
    Tsqrt('4949554350723.461435143921800511302655747834828959172999243354749297834846074272605873894779037895263058987767', '24498088270765545887271078.495701603537518591752268455746');

    BigNumber.config({DECIMAL_PLACES : 92, ROUNDING_MODE : 2});
    Tsqrt('34024250164506590698940450635557.52953427667455604566896038084827726129860253810440168217368794648838157166948213114635075589', '1157649599256926764239267638462457331268783688179593190844781949.0026');

    BigNumber.config({DECIMAL_PLACES : 17, ROUNDING_MODE : 4});
    Tsqrt('44.40502294864850084', '1971.80606307');

    BigNumber.config({DECIMAL_PLACES : 81, ROUNDING_MODE : 3});
    Tsqrt('53661161093118917.324727467936347226946388973481645944102836490011463645342312303037845424027327131', '2879520209861659438086912639838761.96836512');

    BigNumber.config({DECIMAL_PLACES : 97, ROUNDING_MODE : 1});
    Tsqrt('14192376.643123461322847366137186892667087869800484670508360531721790645551341594437545247745705539930735', '201423554780276.3686384009255372');

    BigNumber.config({DECIMAL_PLACES : 73, ROUNDING_MODE : 3});
    Tsqrt('4855847283088595702176944.8112532951528228009352578372167502504010846351042776625505257853252286361', '23579252836678896488410638492374748088831131237827.786414707871968195569');

    BigNumber.config({DECIMAL_PLACES : 4, ROUNDING_MODE : 4});
    Tsqrt('2553252336.0331', '6519097491458426345.3554782053764871036657698339691144086');

    BigNumber.config({DECIMAL_PLACES : 61, ROUNDING_MODE : 5});
    Tsqrt('313415140.8429479671745256541153927986977746351343070479080617690487091', '98229050509604911');

    BigNumber.config({DECIMAL_PLACES : 24, ROUNDING_MODE : 0});
    Tsqrt('1.537808174192351787259717', '2.3648539806128145775');

    BigNumber.config({DECIMAL_PLACES : 20, ROUNDING_MODE : 4});
    Tsqrt('62.30949119560079364913', '3882.4726930546528212066');

    BigNumber.config({DECIMAL_PLACES : 60, ROUNDING_MODE : 1});
    Tsqrt('20416269776260474006.927083982432268806569265768750039002235065141713481228428261', '416824071577046905366185925400975939799.8');

    BigNumber.config({DECIMAL_PLACES : 82, ROUNDING_MODE : 4});
    Tsqrt('174228234.709795821707519878032640346805381350977820681861118903584344693715322225768054125', '30355477770091701.53717021425249799086667828628302277998976752563');

    BigNumber.config({DECIMAL_PLACES : 86, ROUNDING_MODE : 0});
    Tsqrt('8346080245081.12842427016192329441165264650792513611706834892396914269233387534464159247031782538653', '69657055457333468703224935.763141948224298');

    BigNumber.config({DECIMAL_PLACES : 20, ROUNDING_MODE : 4});

    function Tpow(expected, n, exp) {
        assert(String(expected), String(new BigNumber(n).pow(exp)))
    }

    BigNumber.config({EXPONENTIAL_AT : 0});

    Tpow('6.1203577224967e+13', '61203577224967', 1);
    Tpow('-2.6742593337474927167986914400257603e+34', '-82827', 7);
    Tpow('3.490938536648870448335810684618787855050175354916141007545163425555957335829e+75', '247548469', 9);
    Tpow('6.634204312890625e+15', '-95', 8);
    Tpow('-9.171271e+1', '-91.71271', 1);
    Tpow('5.65625765123605544067809018296482219675982471958093705254464e+59', '-9093998202', 6);
    Tpow('1e+0', '-5420353171006060062', 0);
    Tpow('5.93548365959371477646876280183482286894691258676416e+26', '840398698.65610156', 3);
    Tpow('1.016984074247269470395836690098169093010136836967e+39', '21584.7', 9);
    Tpow('-8.983272e+1', '-89.83272', 1);
    Tpow('5.308416e+6', '-48', 4);
    Tpow('3.83432050166120236679168e+23', '52088', 5);

    BigNumber.config({ROUNDING_MODE : 0});

    Tpow('-5.196101e-14', '-453.8', -5);
    Tpow('1e-20', '8308633', -17);
    Tpow('1e-20', '834', -23);
    Tpow('1.6666666666666666667e-1', '6', -1);
    Tpow('-1e-20', '-71', -29);
    Tpow('9.765625e-4', '-2', -10);
    Tpow('1e-20', '62.8159321', -18);
    Tpow('-6.333754988633e-8', '-2.0557545', -23);
    Tpow('1e-20', '33', -24);
    Tpow('-1e-20', '-5235.7923', -27);
    Tpow('-1.29728108e-12', '-9169', -3);
    Tpow('2.323057312542e-8', '3', -16);

    BigNumber.config({DECIMAL_PLACES : 20, ROUNDING_MODE : 4, EXPONENTIAL_AT : 1E9});

    function Tmod(a, b, expected) {
        assert(String(expected), String(new BigNumber(a).mod(new BigNumber(b))));
    }

    Tmod('-6591', '8.81', '-1.12');
    Tmod('1.11', '11', '1.11');
    Tmod('0.00000000432701', '271', '0.00000000432701');
    Tmod('0.0000000000215581', '-0.000000000002301', '0.0000000000008491');
    Tmod('-0.0000000000000591', '-1.71', '-0.0000000000000591');
    Tmod('-0.0000841', '6.981', '-0.0000841');
    Tmod('-2.4471', '-14359441882754.391', '-2.4471');
    Tmod('2.31614421', '2097.74828131', '2.31614421');
    Tmod('203239281', '-1092.361', '55.145');
    Tmod('-46.3306582161', '133931.6342079549871', '-46.3306582161');
    Tmod('-6038925.2969825025061', '-331548954963.41', '-6038925.2969825025061');
    Tmod('1338199.302617179356379591035511', '568.441', '89.188617179356379591035511');
    Tmod('-71', '2935538995927946.15230380421', '-71');

    function TcomparedTo(a, b, expected) {
        assert(String(expected), String(new BigNumber(a).cmp(b)));
        assert(String(expected), String(new BigNumber(a).cmp(new BigNumber(b))));
    }

    TcomparedTo(1, 0, 1);
    TcomparedTo(1, -0, 1);
    TcomparedTo(-1, 0, -1);
    TcomparedTo(-1, -0, -1);
    TcomparedTo(1, Infinity, -1);
    TcomparedTo(1, -Infinity, 1);
    TcomparedTo(-1, Infinity, -1);
    TcomparedTo(-1, -Infinity, 1);
    TcomparedTo(0, 1, -1);
    TcomparedTo(0, -1, 1);
    TcomparedTo(-0, 1, -1);
    TcomparedTo(-0, -1, 1);
    TcomparedTo('51065.7214018651', '21', '1');
    TcomparedTo('1', '-16594.7051', '1');
    TcomparedTo('0.0000000000000000000010511609090481', '-11020751', '1');
    TcomparedTo('-5.8551401', '-0.000000000000000000000001123091759805151', '-1');
    TcomparedTo('11527.63339771', '-0.00000000000000000000004429516529932561', '1');
    TcomparedTo('45875.11', '405079444.71', '-1');
    TcomparedTo('1426.641', '20.9286431', '1');
    TcomparedTo('4.657171', '-50001.941', '1');
    TcomparedTo('-190771', '-686381.317791', '1');
    TcomparedTo('-0.0000000000000000000161', '12855824.31', '-1');
    TcomparedTo('21', '353565391626181', '-1');
    TcomparedTo('398788950.02241', '-46.7685077791', '1');
    TcomparedTo('-78.15411', '-105.31471806641', '1');
    TcomparedTo('41', '-71', '1');
    TcomparedTo('-51', '-0.00000474054791', '-1');
    TcomparedTo('-41', '17.241', '-1');
    TcomparedTo('11', '113010.8763991', '-1');
    TcomparedTo('-61', '3702421785.491', '-1');
    TcomparedTo('-0.00000000000000000000000000000003464171', '220.26991', '-1');
    TcomparedTo('0.000013781', '0.00000000501652742971', '1');
    TcomparedTo('-14762.331', '-1.11', '-1');
    TcomparedTo('31', '61', '-1');
    TcomparedTo('0.0000000000000000000000000000000000000000145668763947641', '398563006.4471', '-1');
    TcomparedTo('493.11824741', '736531', '-1');
    TcomparedTo('75077024.211', '1', '1');
    TcomparedTo('-90.74569446258221', '-21', '-1');
    TcomparedTo('15654315.61266191', '-11', '1');
    TcomparedTo('1.494421', '41821961', '-1');
    TcomparedTo('-26.51', '-0.00863321', '-1');
    TcomparedTo('-3.21', '30292255112081', '-1');
    TcomparedTo('-3.20191e+3', '5.11e-30', '-1');
    TcomparedTo('-1.08821e+5', '1.939813191e+9', '-1');
    TcomparedTo('-7.592791e-17', '-9.291e-27', '-1');
    TcomparedTo('-4.901e+0', '-1.39230141e-31', '-1');
    TcomparedTo('-2.8785721351e+5', '-1.084971e+3', '-1');
    TcomparedTo('6.8649111e+1', '2.1e+1', '1');
    TcomparedTo('1.4421e+4', '8.7711e+1', '1');
    TcomparedTo('1.497851e-9', '1.81661e+2', '-1');
    TcomparedTo('-8.631953001e+6', '6.310131231e+0', '-1');
    TcomparedTo('4.91e+0', '5.381e-15', '1');
    TcomparedTo('4.21e+0', '-1.63069631e+3', '1');
    TcomparedTo('-4.081e+1', '-7.801e+1', '1');
    TcomparedTo('-3.1e+1', '-8.41e+0', '-1');
    TcomparedTo('-1.1e+1', '1.037475521e+4', '-1');
    TcomparedTo('-1e+0', '1.6500661e+5', '-1');
    TcomparedTo('2.07751e-24', '2.88755811e+5', '-1');
    TcomparedTo('1e+0', '-1e+0', '1');
    TcomparedTo('3.821e-23', '4.25729041e-22', '-1');
    TcomparedTo('4.831e+1', '-1.01e+0', '1');
    TcomparedTo('-1.61071e+1', '8.22705601e+8', '-1');
    TcomparedTo('2.474493111e-20', '-9.0664201e+7', '1');
    TcomparedTo('4.27903551e+8', '1e+0', '1');
    TcomparedTo('1.1e+1', '-3.1e+1', '1');
    TcomparedTo('-6.3993191e+3', '2.9980701e+0', '-1');
    TcomparedTo('-3.7599935031e+1', '-1.14011e+5', '1');
    TcomparedTo('-1.631920621e+1', '-5.947675381e+5', '1');
    TcomparedTo('2.0690564941e-24', '-5.690831e+4', '1');
    TcomparedTo('-1.30864281e+0', '-6.1e+1', '1');

    function Tabs(expected, value){
        assert(String(expected), new BigNumber(String(value)).abs().toString());
    }

    Tabs(1, 1);
    Tabs(1, -1);
    Tabs(0.5, '0.5');
    Tabs(0.5, '-0.5');
    Tabs(0.1, 0.1);
    Tabs(0.1, -0.1);
    Tabs(1.1, 1.1);
    Tabs(1.1, -1.1);
    Tabs(1.5, '1.5');
    Tabs(1.5, '-1.5');
    Tabs('Infinity', Infinity);
    Tabs('Infinity', -Infinity);
    Tabs(NaN, NaN);
    Tabs(0, 0);
    Tabs(0, -0);

    function Tneg(expected, value){
        assert(String(expected), new BigNumber(value).neg().toString());
    }

    Tneg(-4, 4);
    Tneg(-2147483648, 2147483648);
    Tneg(-0.25, 0.25);
    Tneg(-0.0625, 0.0625);
    Tneg(-1, 1);
    Tneg(1, -1);
    Tneg(0, 0);
    Tneg(NaN, NaN);
    Tneg(-Infinity, Infinity);
    Tneg(Infinity, -Infinity);
    Tneg('-5.0600621890668482322956892808849303', '5.0600621890668482322956892808849303');
    Tneg('7', '-7');
    Tneg('-6.1095374220609', '6.1095374220609');
    Tneg('9.01', '-9.01');
    Tneg('-1.016984074247269470395836690098169093010136836967', '1.016984074247269470395836690098169093010136836967');
    Tneg('-1.497639134680472576', '1.497639134680472576');
    Tneg('-4.1717657571404248', '4.1717657571404248');

    function Tceil(expected, value) {
        assert(String(expected), new BigNumber(String(value)).ceil().toString());
    }

    Tceil(0, 0);
    Tceil(0, '0.000');
    Tceil(0, -0);
    Tceil(Infinity, Infinity);
    Tceil(-Infinity, -Infinity);
    Tceil(NaN, NaN);
    Tceil(1, 0.1);
    Tceil(1, 0.49999999999999994);
    Tceil(1, 0.5);
    Tceil(1, 0.7);
    Tceil(0, -0.1);
    Tceil(0, -0.49999999999999994);
    Tceil(0, -0.5);
    Tceil(0, -0.7);
    Tceil(1, 1);
    Tceil(2, 1.1);
    Tceil(2, 1.5);
    Tceil(2, 1.7);
    Tceil(-1, -1);
    Tceil(-1, -1.1);
    Tceil(-1, -1.5);
    Tceil(-1, -1.7);
    Tceil('-2075364', '-2075364.364286541923');
    Tceil('60593539780450631', '60593539780450631');
    Tceil('65937898671515', '65937898671515');
    Tceil('-39719494751819198566798', '-39719494751819198566798.578');
    Tceil('92627382695288166557', '92627382695288166556.8683774524284866028260448205069');
    Tceil('-881574', '-881574');
    Tceil('-3633239209', '-3633239209.654526163275621746013315304191073405508491056');
    Tceil('-23970335459820625362', '-23970335459820625362');
    Tceil('131869457416154038', '131869457416154038');
    Tceil('-2685', '-2685');
    Tceil('-4542227860', '-4542227860.9511298545226');
    Tceil('2416872282', '2416872281.963955669484225137349193306323379254936827');
    Tceil('-757684868752087594264588207655', '-757684868752087594264588207655.27838048392835556');
    Tceil('-438798503526', '-438798503526.2317623894721299587561697');
    Tceil('801625782231888715214665', '801625782231888715214665');
    Tceil('-91881984778675238', '-91881984778675238');
    Tceil('327765350218284325239839632047', '327765350218284325239839632046.91682741746683081459605386');
    Tceil('-7469045007691432294', '-7469045007691432294.362757245');
    Tceil('8365540212937142194319515218790', '8365540212937142194319515218789.4106658678537421977827');
    Tceil('-14108', '-14108.495051214515');
    Tceil('49104502', '49104501.10055989379655329194309526150310568683504206945625');
    Tceil('131370407', '131370406.330005158136313262837556068534122953');
    Tceil('3017', '3017');

    function Tfloor(expected, value) {
        assert(String(expected), new BigNumber(String(value)).floor().toString());
    }

    Tfloor(0, 0);
    Tfloor(0, '0.000');
    Tfloor(-0, -0);
    Tfloor(Infinity, Infinity);
    Tfloor(-Infinity, -Infinity);
    Tfloor(NaN, NaN);
    Tfloor(0, 0.1);
    Tfloor(0, 0.49999999999999994);
    Tfloor(0, 0.5);
    Tfloor(0, 0.7);
    Tfloor(-1, -0.1);
    Tfloor(-1, -0.49999999999999994);
    Tfloor(-1, -0.5);
    Tfloor(-1, -0.7);
    Tfloor(1, 1);
    Tfloor(1, 1.1);
    Tfloor(1, 1.5);
    Tfloor(1, 1.7);
    Tfloor(-1, -1);
    Tfloor(-2, -1.1);
    Tfloor(-2, -1.5);
    Tfloor(-2, -1.7);
    Tfloor('-2075365', '-2075364.364286541923');
    Tfloor('60593539780450631', '60593539780450631');
    Tfloor('65937898671515', '65937898671515');
    Tfloor('-39719494751819198566799', '-39719494751819198566798.578');
    Tfloor('92627382695288166556', '92627382695288166556.8683774524284866028260448205069');
    Tfloor('-881574', '-881574');
    Tfloor('-3633239210', '-3633239209.654526163275621746013315304191073405508491056');
    Tfloor('-23970335459820625362', '-23970335459820625362');
    Tfloor('131869457416154038', '131869457416154038');
    Tfloor('-4542227861', '-4542227860.9511298545226');
    Tfloor('2416872281', '2416872281.963955669484225137349193306323379254936827');
    Tfloor('-757684868752087594264588207656', '-757684868752087594264588207655.27838048392835556');
    Tfloor('-438798503527', '-438798503526.2317623894721299587561697');
    Tfloor('801625782231888715214665', '801625782231888715214665');
    Tfloor('-91881984778675238', '-91881984778675238');
    Tfloor('327765350218284325239839632046', '327765350218284325239839632046.91682741746683081459605386');
    Tfloor('-7469045007691432295', '-7469045007691432294.362757245');
    Tfloor('8365540212937142194319515218789', '8365540212937142194319515218789.4106658678537421977827');
    Tfloor('-14109', '-14108.495051214515');
    Tfloor('49104501', '49104501.10055989379655329194309526150310568683504206945625');
    Tfloor('131370406', '131370406.330005158136313262837556068534122953');
    Tfloor('3017', '3017');

    function Tround(expected, value, decimalPlaces, roundingMode) {
        assert(String(expected), new BigNumber(String(value)).round(decimalPlaces, roundingMode).toString());
    }

    BigNumber.config({ROUNDING_MODE : 7});

    Tround(0, 0);
    Tround(0, -0);
    Tround(Infinity, Infinity);
    Tround(-Infinity, -Infinity);
    Tround(NaN, NaN);
    Tround(1, 0.5);
    Tround(1, 0.7);
    Tround(1, 1);
    Tround(1, 1.1);
    Tround(1, 1.49999);
    Tround(0, -0.5);
    Tround(-1, -0.5000000000000001);
    Tround(-1, -0.7);
    Tround(-1, -1);
    Tround(-1, -1.1);
    Tround(-1, -1.49999);
    Tround(-1, -1.5);
    Tround(9007199254740990, 9007199254740990);
    Tround(9007199254740991, 9007199254740991);
    Tround(-9007199254740990, -9007199254740990);
    Tround(-9007199254740991, -9007199254740991);
    Tround('-4.1787893', '-4.178789328', 7, 1);
    Tround('372', '372', 3, 6);
    Tround('-9.1645', '-9.1645354', 4, 6);
    Tround('83.241462', '83.241462', 6, 6);
    Tround('-0.00001262', '-0.00001262', 8, 6);
    Tround('63.4147', '63.414735', 4, 3);
    Tround('-2', '-2', 3, 5);
    Tround('538601216', '538601216', 2, 3);
    Tround('2138522', '2138522', 6, 1);
    Tround('301527.309', '301527.309', 3, 5);
    Tround('-7.568', '-7.5683', 3, 5);
    Tround('-1969563', '-1969563', 8, 1);
    Tround('-249815496353142605063083339706423770452529255711064841247978.1649', '-249815496353142605063083339706423770452529255711064841247978.16482575714251625720191747855256', 4, 3);
    Tround('-8757939868931375459275247990933877309', '-8757939868931375459275247990933877309', 37, 5);
    Tround('556228688742576287254121716253667920283844645854057755303259369', '556228688742576287254121716253667920283844645854057755303259369', 25, 3);
    Tround('0', '0.00000000000000000000000000000000000000000000000029171651270270187897342247995644546232788426', 35, 3);
    Tround('-0.0000058979376152458865088785307705', '-0.000005897937615245886508878530770431196412050562641578155968', 34, 0);
    Tround('-0.00000000000000847145658', '-0.00000000000000847145657888758222743159237', 23, 3);
    Tround('-6049825281564367887763596943301191584240212075976455', '-6049825281564367887763596943301191584240212075976455', 53, 0);
    Tround('-64680661822322715719008107701612.741312367131318202976964422', '-64680661822322715719008107701612.74131236713131820297696442216284615573809', 27, 4);
    Tround('0', '-0.0000000000000000000000000000000000048810387700772580768508931440796984745771392', 29, 2);
    Tround('232730083859746141163.7653402888578932348762190816502', '232730083859746141163.76534028885789323487621908165018267559238917447610653431235463768839108', 31, 5);
    Tround('-499311204526771278437755030091270601.471802335', '-499311204526771278437755030091270601.47180233568698638607829334564', 9, 2);
    Tround('-4793558721695298336734196955569628362250.1', '-4793558721695298336734196955569628362250.107513868618556605590137359591047843592593', 1, 6);
    Tround('0.0000000000000000000000000000000000000622', '0.000000000000000000000000000000000000062176566496', 40, 5);
    Tround('-393323198834362857317921391989917230.1575124832953752419421908305674188204', '-393323198834362857317921391989917230.15751248329537524194219083056741882044261538788798', 37, 5);
    Tround('0', '-0.000000000000000169092170076859363', 3, 5);

    function TbaseIn(expected, value, base) {
        assert(expected, new BigNumber(value, base).toString())
    }

    BigNumber.config({DECIMAL_PLACES : 40, ROUNDING_MODE : 2});

    TbaseIn('-729.0001524157902758725803993293705227861606', '-1000000.00000001', 3);
    TbaseIn('-4096.0000152587890625', '-1000000.00000001', 4);
    TbaseIn('-15625.00000256', '-1000000.00000001', 5);
    TbaseIn('-46656.0000005953741807651272671848803536046334', '-1000000.00000001', 6);
    TbaseIn('-117649.0000001734665255574303432156634721649541', '-1000000.00000001', 7);
    TbaseIn('-262144.000000059604644775390625', '-1000000.00000001', 8);
    TbaseIn('-531441.0000000232305731254187746379102835730507', '-1000000.00000001', 9);
    TbaseIn('-1000000.00000001', '-1000000.00000001', 10);
    TbaseIn('-1771561.0000000046650738020973341431092840981941', '-1000000.00000001', 11);
    TbaseIn('-2985984.000000002325680393613778387440938881268', '-1000000.00000001', 12);
    TbaseIn('-4826809.0000000012258947398402566721524761600832', '-1000000.00000001', 13);
    TbaseIn('-7529536.0000000006776036154587122781861854381443', '-1000000.00000001', 14);
    TbaseIn('-11390625.0000000003901844231062338058222831885383', '-1000000.00000001', 15);
    TbaseIn('-16777216.00000000023283064365386962890625', '-1000000.00000001', 16);
    TbaseIn('-24137569.0000000001433536083296850401481727781882', '-1000000.00000001', 17);
    TbaseIn('-34012224.0000000000907444262711670884293370452072', '-1000000.00000001', 18);
    TbaseIn('-47045881.0000000000588804597472215429921222500439', '-1000000.00000001', 19);
    TbaseIn('-64000000.0000000000390625', '-1000000.00000001', 20);
    TbaseIn('-85766121.0000000000264390375792455941496210138949', '-1000000.00000001', 21);
    TbaseIn('-113379904.0000000000182229445394427114965206410085', '-1000000.00000001', 22);
    TbaseIn('-148035889.0000000000127696005408659110598172017909', '-1000000.00000001', 23);
    TbaseIn('-191102976.0000000000090846890375538218259411675049', '-1000000.00000001', 24);
    TbaseIn('-244140625.0000000000065536', '-1000000.00000001', 25);
    TbaseIn('-308915776.0000000000047886513275010026255956100003', '-1000000.00000001', 26);
    TbaseIn('-387420489.0000000000035407061614721497695336509027', '-1000000.00000001', 27);
    TbaseIn('-481890304.0000000000026468891228855948366647868677', '-1000000.00000001', 28);
    TbaseIn('-594823321.000000000001999014833671504164315094574', '-1000000.00000001', 29);
    TbaseIn('-729000000.0000000000015241579027587258039932937052', '-1000000.00000001', 30);
    TbaseIn('-887503681.0000000000011724827159637921277158030113', '-1000000.00000001', 31);
    TbaseIn('-1073741824.0000000000009094947017729282379150390625', '-1000000.00000001', 32);
    TbaseIn('-1291467969.0000000000007110309102419347878538765581', '-1000000.00000001', 33);
    TbaseIn('-1544804416.0000000000005599750325378321880787999147', '-1000000.00000001', 34);
    TbaseIn('-1838265625.0000000000004440743054270216786320984887', '-1000000.00000001', 35);
    TbaseIn('-2176782336.0000000000003544704151217464391770978328', '-1000000.00000001', 36);

    BigNumber.config({DECIMAL_PLACES : 51, ROUNDING_MODE : 4});
    TbaseIn('1072424547177.982891327541533302850175278158817253467459228776101', 'donxvwix.zdts', 36);

    BigNumber.config({DECIMAL_PLACES : 86});
    TbaseIn('824178538787196749922434027872451367594239056.93473392033316110609748881918323116875731371037529199959272159196515804304815690839727', '402kfhkd37bt5n8scr1ir9ndlrnipig.s17oe7rkhi91bh', 30);

    BigNumber.config({DECIMAL_PLACES : 84});
    TbaseIn('9560389177469634483515162.499335215204179931606951906984830542647805834768203380512715304247460734647288652625', '195qdkkqsa8shmhp9e.edr89', 29);

    BigNumber.config({DECIMAL_PLACES : 65});
    TbaseIn('5955289028666603391738616069969.70235175643053599414353772852590894151261634747374296179598348968', '8qp28dk3js2iqksmqaqnq.lntnif5qh', 31);

    BigNumber.config({DECIMAL_PLACES : 49});
    TbaseIn('27603501710202437282037.4945845176631161140013607910579053986520224457133', '42545302442500101544532043113.254455200225412543300022520330204033', 6);

    BigNumber.config({DECIMAL_PLACES : 39});
    TbaseIn('9464300204295306111422098057.77248824166891668678144149703717008528', '25473f3dbce5cf3hg8318d7.dg52d120b14ea966a7ag06a2gh03', 18);

    BigNumber.config({DECIMAL_PLACES : 15});
    TbaseIn('133262758349237628352120716402.993431117739119', '3bkobquqthhfbndsmv3i.vp8o0sc4ldtji02mmgqr7blpdjgk', 32);

    BigNumber.config({DECIMAL_PLACES : 65});
    TbaseIn('171510920999634527633.53051379043557196404057602235264411208736518600450525086556631034', '1fqecn4264r1is.ijur8yj41twl9', 35);

    BigNumber.config({DECIMAL_PLACES : 48});
    TbaseIn('325927753012307620476767402981591827744994693483231017778102969592507', 'c16de7aa5bf90c3755ef4dea45e982b351b6e00cd25a82dcfe0646abb', 16);

    BigNumber.config({DECIMAL_PLACES : 48});
    TbaseIn('72783.559378210242248003991012349918599484318629885897', '11c4f.8f33690f15e13146d99092446da', 16);

    BigNumber.config({DECIMAL_PLACES : 81});
    TbaseIn('8535432796511493691316991730196733707212461.36382685871580896850891461623808', '9l0ah4mf8a0kcgn44oji4kh7l6fbenb.929jlggo43612jfn', 25);

    function TbaseOut(expected, value, base) {
        assert(expected, new BigNumber(value).toString(base))
    }

    BigNumber.config({DECIMAL_PLACES : 40});

    TbaseOut('1111100111.11111111101111100111011011001000101101', '999.999', 2);
    TbaseOut('1101000.2222220210221200021112101120001001102122', '999.999', 3);
    TbaseOut('33213.3333233213123020231003211120010012021032', '999.999', 4);
    TbaseOut('12444.4444141414141414141414141414141414141414', '999.999', 5);
    TbaseOut('4343.5554412021453552313300510342441202145355', '999.999', 6);
    TbaseOut('2625.6664412312515410253566644123125154102536', '999.999', 7);
    TbaseOut('1747.7773716662132071260101422335136152375747', '999.999', 8);
    TbaseOut('1330.8882385024534603137817105667427164213206', '999.999', 9);
    TbaseOut('999.999', '999.999', 10);
    TbaseOut('829.aa973a4913443083925788751826a12209617036', '999.999', 11);
    TbaseOut('6b3.bba3320237938b1770119a7a5699606ab3b294a9', '999.999', 12);
    TbaseOut('5bb.ccaa5926381c699320b4278283cb62c494684889', '999.999', 13);
    TbaseOut('515.ddb38266d3049bdac14b5d165539a5b770ad942', '999.999', 14);
    TbaseOut('469.eeb9595959595959595959595959595959595959', '999.999', 15);
    TbaseOut('3e7.ffbe76c8b4395810624dd2f1a9fbe76c8b439581', '999.999', 16);
    TbaseOut('37d.ggc18275989641138dge6d54eg08g529aeb6faa1', '999.999', 17);
    TbaseOut('319.hhc307dh7b6deeha40a6b43307dh7b6deeha40a7', '999.999', 18);
    TbaseOut('2eb.iic2ch24i43efhgd61307b0e199g6fb74a55dcac', '999.999', 19);
    TbaseOut('29j.jjc', '999.999', 20);
    TbaseOut('25c.kkbfaii9d8gg7b6ee594cc372aa15087k2j65i0h', '999.999', 21);
    TbaseOut('219.llb7g822a4ceh5a0fag44k937cak18la89ii6f3', '999.999', 22);
    TbaseOut('1ka.mmaj3f2cgc81fe490kebi8f9m173blcfb08mj6j3', '999.999', 23);
    TbaseOut('1hf.nna4590djiena4590djiena4590djiena4590djj', '999.999', 24);
    TbaseOut('1eo.oo99999999999999999999999999999999999999', '999.999', 25);
    TbaseOut('1cb.pp8b0g5lb0g5lb0g5lb0g5lb0g5lb0g5lb0g5lb1', '999.999', 26);
    TbaseOut('1a0.qq78f2dle09cnjl9h6mblfcj9ij58geni8qkbn0n', '999.999', 27);
    TbaseOut('17j.rr619hjdigg3g9nr2pf067h6qc2j7ar95474jjq6', '999.999', 28);
    TbaseOut('15d.ss4hkojk13n4rnf1dpo9ham09782ifbgg63d28cl', '999.999', 29);
    TbaseOut('139.tt3', '999.999', 30);
    TbaseOut('117.uu16eq9rha657cl1g2ngcbdir7m8tmihjp2dsf4t', '999.999', 31);
    TbaseOut('v7.vuv7di5k75c10oidqboqjuv7di5k75c10oidqbor', '999.999', 32);
    TbaseOut('u9.wvu22k10p1i619fhuaq3d5gsfqkr584weg2bahcu', '999.999', 33);
    TbaseOut('td.xwsnmjjt3i7kw38x68nvs0igtcqmabee4ufqd1up', '999.999', 34);
    TbaseOut('sj.yxr4d4d4d4d4d4d4d4d4d4d4d4d4d4d4d4d4d4d4', '999.999', 35);
    TbaseOut('rr.zypcdtnwjl0v3qs82axzf9i56mgpcdtnwjl0v3qs', '999.999', 36);

    BigNumber.config({DECIMAL_PLACES : 20});
    TbaseOut('b8b057c619.881cda90778727315519', '240240367775.61294728461437827', 14);

    BigNumber.config({DECIMAL_PLACES : 57});
    TbaseOut('3uke1r.qog7jpuls3wumhw9r6ij66xbtr9fxmcxo7pjkdsets6obc03cdkpohtp4', '177198677.78587988442236198004', 34);

    BigNumber.config({DECIMAL_PLACES : 53});
    TbaseOut('rxp7iouofqsa.5xr0ytex5w3u84ss4bmqr4d2r58uk0w6vou7okvti3sqaf8xdt7m4', '2699846669670878465.17042631933406076262750289627556', 35);

    BigNumber.config({DECIMAL_PLACES : 10});
    TbaseOut('71.4dadah84a6', '127.26419', 18);

    BigNumber.config({DECIMAL_PLACES : 62});
    TbaseOut('pho3b9iei6hp0dcf72fih.ph9881jc7d7c0c0mgpgd1ji1k02ola0l976leng417ng4dma8k5jj95gommail', '511945071604551808598197283361.9872166378092262156654881169655510', 26);

    BigNumber.config({DECIMAL_PLACES : 16});
    TbaseOut('98.1902712204800298', '107.16547', 11);

    BigNumber.config({DECIMAL_PLACES : 20});

    function TtoString(expected, value) {
        assert(expected, new BigNumber(value).toString())
    }

    TtoString('NaN', NaN);
    TtoString('Infinity', 1/0);
    TtoString('-Infinity', -1/0);
    TtoString('0', 0);
    TtoString('9', 9);
    TtoString('90', 90);
    TtoString('90.12', 90.12);
    TtoString('0.1', 0.1);
    TtoString('0.01', 0.01);
    TtoString('0.0123', 0.0123);
    TtoString('111111111111111111111',   '111111111111111111111');
    TtoString('1111111111111111111111',  '1111111111111111111111');
    TtoString('11111111111111111111111', '11111111111111111111111');
    TtoString('0.00001', 0.00001);
    TtoString('0.000001', 0.000001);

    BigNumber.config({EXPONENTIAL_AT : 0});

    TtoString('5.73447902457635174479825134e+14', '573447902457635.174479825134');
    TtoString('1.07688e+1', '10.7688');
    TtoString('3.171194102379077141557759899307946350455841e+27', '3171194102379077141557759899.307946350455841');
    TtoString('4.924353466898191177698653319742594890634579e+37', '49243534668981911776986533197425948906.34579');
    TtoString('6.85558243926569397328633907445409866949445343654692955e+18', '6855582439265693973.28633907445409866949445343654692955');
    TtoString('1e+0', '1');
    TtoString('2.1320000803e+7', '21320000.803');
    TtoString('5.0878741e+4', '50878.741');
    TtoString('5.1932898288391e+8', '519328982.88391');
    TtoString('5.690616778176956027307884725933410349604387915634524e+49', '56906167781769560273078847259334103496043879156345.24');
    TtoString('3.25099780528575457273246693147645e+15', '3250997805285754.572732466931476450');
    TtoString('3e+0', '3');
    TtoString('2.5811494197573291905990947355226e+13', '25811494197573.291905990947355226');
    TtoString('5.60372259169833930471746454125e+13', '56037225916983.3930471746454125');
    TtoString('1.2615810663732236602461593613783e+7', '12615810.663732236602461593613783');
    TtoString('1.4654366449266911507705477499035407722184965108377032e+11', '146543664492.66911507705477499035407722184965108377032');
    TtoString('6.4986735507448912857131832908423940757e+38', '649867355074489128571318329084239407570.0');
    TtoString('3.6146989180120676857245474944e+3', '3614.6989180120676857245474944');
    TtoString('9.928654762302286149994896625074e+4', '99286.54762302286149994896625074');
    TtoString('3.46424170787806074650506079e+3', '3464.24170787806074650506079');
    TtoString('1.25934313355319666474752550204680303068865719647e+33', '1259343133553196664747525502046803.03068865719647');
    TtoString('1.23014105337660651106978059198916100450966081493207e+27', '1230141053376606511069780591.98916100450966081493207');
    TtoString('1.386164712267169624993434287237e+23', '138616471226716962499343.4287237');
    TtoString('2.66076369930322488334961932e+3', '2660.76369930322488334961932');

    BigNumber.config({EXPONENTIAL_AT : 1E9});

    function TtoExponential(expected, value, decimalPlaces){
        assert(String(expected), new BigNumber(value).toExponential(decimalPlaces));
    }

    TtoExponential('1e+0', 1, undefined);
    TtoExponential('1.1e+1', 11, undefined);
    TtoExponential('1.12e+2', 112, undefined);
    TtoExponential('1e+0', 1, 0);
    TtoExponential('1e+1', 11, 0);
    TtoExponential('1e+2', 112, 0);
    TtoExponential('1.0e+0', 1, 1);
    TtoExponential('1.1e+1', 11, 1);
    TtoExponential('1.1e+2', 112, 1);
    TtoExponential('1.00e+0', 1, 2);
    TtoExponential('1.10e+1', 11, 2);
    TtoExponential('1.12e+2', 112, 2);
    TtoExponential('1.000e+0', 1, 3);
    TtoExponential('1.100e+1', 11, 3);
    TtoExponential('1.120e+2', 112, 3);
    TtoExponential('1e-1', 0.1, undefined);
    TtoExponential('1.1e-1', 0.11, undefined);
    TtoExponential('1.12e-1', 0.112, undefined);
    TtoExponential('1e-1', 0.1, 0);
    TtoExponential('1e-1', 0.11, 0);
    TtoExponential('1e-1', 0.112, 0);
    TtoExponential('1.0e-1', 0.1, 1);
    TtoExponential('1.1e-1', 0.11, 1);
    TtoExponential('1.1e-1', 0.112, 1);
    TtoExponential('1.00e-1', 0.1, 2);
    TtoExponential('1.10e-1', 0.11, 2);
    TtoExponential('1.12e-1', 0.112, 2);
    TtoExponential('1.000e-1', 0.1, 3);
    TtoExponential('1.100e-1', 0.11, 3);
    TtoExponential('1.120e-1', 0.112, 3);

    BigNumber.config({ROUNDING_MODE : 0});

    TtoExponential('-5.3453435435e+8', '-53453.435435E4', undefined);
    TtoExponential('-8.8254658100092746334967191957167916942544e+17', '-882546581000927463.34967191957167916942543286', 40);
    TtoExponential('-4.794121828559674450610889008537305783490457e-9', '-0.00000000479412182855967445061088900853730578349045628396662493370334888944406719979291547717079', 42);
    TtoExponential('3.6149e+33', '3614844933096444884855774574994631.0106397808', 4);
    TtoExponential('5.582954000000000e-12', '0.000000000005582954', 15);
    TtoExponential('-3.88740271991885914774802363151163005925700000000000000000e-24', '-0.000000000000000000000003887402719918859147748023631511630059257', 56);

    BigNumber.config({ROUNDING_MODE : 1});

    TtoExponential('0e+0', '-0.0E-0', undefined);
    TtoExponential('-2.856376815219143184897347685012382222462687620998915470135915e+6', '-2856376.815219143184897347685012382222462687620998915470135915511363444', 60);
    TtoExponential('7.75700e-24', '0.000000000000000000000007757', 5);
    TtoExponential('7.0e-1', '0.7', 1);
    TtoExponential('5.2109749078977455423107465583658126e+37', '52109749078977455423107465583658126637', 34);
    TtoExponential('3.631093819552528994444977110063007461579154042777868294000e-29', '0.00000000000000000000000000003631093819552528994444977110063007461579154042777868294', 57);
    TtoExponential('-9.893937860425888e+8', '-989393786.042588804219191', 15);

    BigNumber.config({ROUNDING_MODE : 2});

    TtoExponential('-5.28860565e-16', '-0.00000000000000052886056528317233012115396784629214632', 8);
    TtoExponential('6.4114675970838738000e-18', '0.0000000000000000064114675970838738', 19);
    TtoExponential('8.00000000000000000000e-20', '0.00000000000000000008', 20);
    TtoExponential('2.74000064578288771723078597711103520450391668117802304078152085625023633681179e+24', '2740000645782887717230785.977111035204503916681178023040781520856250236336811781347278', 77);
    TtoExponential('8.1936742669491704846805837777816457628e-16', '0.00000000000000081936742669491704846805837777816457628', 37);
    TtoExponential('-7.2157448e+14', '-721574484716710.00141299844961546', 7);
    TtoExponential('-5.321807464703650000000e-15', '-0.00000000000000532180746470365', 21);
    TtoExponential('-4.449e+27', '-4449471658582621135143349142.228707647170080816912435271162', 3);

    BigNumber.config({ROUNDING_MODE : 3});

    TtoExponential('-9.99999999000000009e+8', '-999999999.000000009e-0', undefined);
    TtoExponential('-3.99764422903251220452704763278376060858663250289320247532595e+24', '-3997644229032512204527047.63278376060858663250289320247532594416986984981431156065660613', 59);
    TtoExponential('5.534083545686157907280686578717428772e+12', '5534083545686.157907280686578717428772', 36);
    TtoExponential('5.00000000e-9', '0.000000005', 8);
    TtoExponential('-4.08363116583051e+14', '-408363116583051', 14);
    TtoExponential('9.278230415634296945273818e+19', '92782304156342969452.738186255580532649103987374718221928871873054827841260470670536425', 24);

    BigNumber.config({ROUNDING_MODE : 4});

    TtoExponential('-5.002239116605888927178702930656e-39', '-0.00000000000000000000000000000000000000500223911660588892717870293065633642', 30);
    TtoExponential('-8.52292947230244775435e+29', '-852292947230244775434968241532.494643593912804433318745222587246680109833509655450267792446', 20);
    TtoExponential('-6.1169514510867e+10', '-61169514510.8673382', 13);
    TtoExponential('-8.05745763527307676170759722175169266017831695215e+48', '-8057457635273076761707597221751692660178316952146', 47);
    TtoExponential('-4.923572102098e+10', '-49235721020.9847017846898652687600227388412980598816', 12);
    TtoExponential('-7.981341661715027117746906076515945e+41', '-798134166171502711774690607651594491039629', 33);
    TtoExponential('-8.00e-3', '-0.008', 2);

    BigNumber.config({ROUNDING_MODE : 5});

    TtoExponential('4.95474614815842e+38', '495474614815842191683004449862568813538.573064401156', 14);
    TtoExponential('-8.9667567079038139e+16', '-89667567079038139', 16);
    TtoExponential('-7.0e+2', '-703', 1);
    TtoExponential('-2.6249e+33', '-2624861185343559570287214983819906', 4);
    TtoExponential('-6.510119186347371697501169416839709631422185436811698613000000000000000000000000000000e-31', '-0.0000000000000000000000000000006510119186347371697501169416839709631422185436811698613', 84);
    TtoExponential('7.73e+3', '7729', 2);
    TtoExponential('1.4393781011009257793117531801549e+4', '14393.781011009257793117531801548751', 31);

    BigNumber.config({ROUNDING_MODE : 6});

    TtoExponential('-4.3502707501164e+36', '-4350270750116411997402439304498892819', 13);
    TtoExponential('9.5e-21', '0.0000000000000000000094520280724178734152', 1);
    TtoExponential('1.39631186750554172785676012693418617250072200744214625994656047727270672248243741907e+34', '13963118675055417278567601269341861.725007220074421462599465604772727067224824374190703237660781', 83);
    TtoExponential('5.9446570e-26', '0.00000000000000000000000005944657036540768164877637239177740419063920648', 7);
    TtoExponential('7.00000e-12', '0.000000000007', 5);

    BigNumber.config({ROUNDING_MODE : 4});

    TtoExponential('-2.033619450856645241153977e+0', '-2.03361945085664524115397653636144859', 24);
    TtoExponential('1.130e+8', '112955590.0430616', 3);
    TtoExponential('-2.1366468193419876852426155614364269e+10', '-21366468193.419876852426155614364269', 34);
    TtoExponential('5.82086615659566151529e+7', '58208661.56595661515285734890860077163', 20);

    function TtoFixed(expected, value, decimalPlaces){
        assert(String(expected), new BigNumber(value).toFixed(decimalPlaces));
    }

    TtoFixed('NaN', NaN, 2);
    TtoFixed('Infinity', 1/0, 2);
    TtoFixed('-Infinity', -1/0, 2);

    TtoFixed('1111111111111111111111.00000000', '1111111111111111111111', 8);
    TtoFixed('0.1', 0.1, 1);
    TtoFixed('0.10', 0.1, 2);
    TtoFixed('0.100', 0.1, 3);
    TtoFixed('0.01', 0.01, 2);
    TtoFixed('0.010', 0.01, 3);
    TtoFixed('0.0100', 0.01, 4);
    TtoFixed('0.00', 0.001, 2);
    TtoFixed('0.001', 0.001, 3);
    TtoFixed('0.0010', 0.001, 4);
    TtoFixed('1.0000', 1, 4);
    TtoFixed('1.0', 1, 1);
    TtoFixed('1', 1, 0);
    TtoFixed('12', 12, 0);
    TtoFixed('1', 1.1, 0);
    TtoFixed('12', 12.1, 0);
    TtoFixed('1', 1.12, 0);
    TtoFixed('12', 12.12, 0);
    TtoFixed('0.0000006', 0.0000006, 7);
    TtoFixed('0.00000006', 0.00000006, 8);
    TtoFixed('0.000000060', 0.00000006, 9);
    TtoFixed('0.0000000600', 0.00000006, 10);
    TtoFixed('0', 0, 0);
    TtoFixed('0.0', 0, 1);
    TtoFixed('0.00', 0, 2);

    BigNumber.config({ROUNDING_MODE : 0});

    TtoFixed('6552606716539719300271040797809220.3', '6552606716539719300271040797809220.237838405', 1);
    TtoFixed('25605410.260045950231371', '25605410.260045950231370974066', 15);
    TtoFixed('-65593283.0000', '-65593283', 4);
    TtoFixed('-2238743875407788208860272230040067273281097722822075126.88000000000', '-2238743875407788208860272230040067273281097722822075126.88', 11);
    TtoFixed('-1714042659419404211.0000000', '-1714042659419404211', 7);
    TtoFixed('580861301.000', '580861301', 3);
    TtoFixed('-495746734.000', '-495746734', 3);
    TtoFixed('-909962202.00000', '-909962202', 5);
    TtoFixed('-6588.00', '-6588', 2);

    BigNumber.config({ROUNDING_MODE : 1});

    TtoFixed('0.3', '0.3', 1);
    TtoFixed('-200258348374.3', '-200258348374.3595802551014614089785610548492885372322083789', 1);
    TtoFixed('-8996550690041157178188143247448286757711.5580857413', '-8996550690041157178188143247448286757711.55808574133329', 10);
    TtoFixed('-3172413669280032477.00', '-3172413669280032477', 2);
    TtoFixed('6547946357820.750067107731812021675', '6547946357820.750067107731812021675351468709784004', 21);
    TtoFixed('24188393190716631282750407915133817062572333100239.0000000000000000000', '24188393190716631282750407915133817062572333100239', 19);
    TtoFixed('539982361589798324286450574560330534901309503.82000000', '539982361589798324286450574560330534901309503.82', 8);
    TtoFixed('829898800701640360552652.0000', '829898800701640360552652', 4);
    TtoFixed('-1585782773394.4', '-1585782773394.44501382110847', 1);
    TtoFixed('-7604844176594943774211951887242195107.399', '-7604844176594943774211951887242195107.399576743428669618164', 3);

    BigNumber.config({ROUNDING_MODE : 2});

    TtoFixed('7810971819938620325901899057970512790433.0000000000000000', '7810971819938620325901899057970512790433', 16);
    TtoFixed('-623395.0', '-623395', 1);
    TtoFixed('81334094079413997693749807.000000000000', '81334094079413997693749807', 12);
    TtoFixed('790143966.5', '790143966.406169480041929356421', 1);
    TtoFixed('-0.000485', '-0.0004851099615478', 6);
    TtoFixed('843581583868277359263.9721265', '843581583868277359263.97212648230399373761717', 7);
    TtoFixed('-921038771017147427324121032.000000', '-921038771017147427324121032', 6);
    TtoFixed('-823.0', '-823', 1);
    TtoFixed('505.0', '505', 1);

    BigNumber.config({ROUNDING_MODE : 3});

    TtoFixed('0.0000', '0.00000000000584', 4);
    TtoFixed('-79055907138737329232.165677999092552368', '-79055907138737329232.1656779990925523674', 18);
    TtoFixed('47206303530045536472421888342339.00000000', '47206303530045536472421888342339', 8);
    TtoFixed('3974.0', '3974', 1);
    TtoFixed('1296297430771.39', '1296297430771.391670729445980444292', 2);
    TtoFixed('0.000016', '0.00001686928031', 6);
    TtoFixed('-0.00001', '-0.0000000000008', 5);
    TtoFixed('4883938724383.000', '4883938724383', 3);
    TtoFixed('-5.4115622307', '-5.411562230628539329345282639155988044655000352687981880842', 10);
    TtoFixed('-0.01', '-0.0000000000155280115429218804426906540524', 2);
    TtoFixed('-2730930776035874709906.00000', '-2730930776035874709906', 5);
    TtoFixed('8419336352549424092369623.00000', '8419336352549424092369623', 5);

    BigNumber.config({ROUNDING_MODE : 4});

    TtoFixed('733744593401073823825766410831877679446.0000000000000000000', '733744593401073823825766410831877679446', 19);
    TtoFixed('-64.6849459', '-64.6849458687691227978', 7);
    TtoFixed('-0.000000', '-0.00000000009', 6);
    TtoFixed('-62537287527837589298857228059657673223234916.95923265430000000', '-62537287527837589298857228059657673223234916.9592326543', 17);
    TtoFixed('3393668096256773847245721315080265089731.000000', '3393668096256773847245721315080265089731', 6);
    TtoFixed('0.0', '0.0000000000000056674956638008432348702401392', 1);
    TtoFixed('72516372734.6', '72516372734.6447', 1);
    TtoFixed('-418.28', '-418.2800731793741351', 2);
    TtoFixed('0.00', '0.001', 2);

    BigNumber.config({ROUNDING_MODE : 5});

    TtoFixed('0.00', '0.00000000000000070362', 2);
    TtoFixed('682377946933004839.0000000', '682377946933004839', 7);
    TtoFixed('0.000000000', '0.00000000000000301722508588270616971784651380892069087869575', 9);
    TtoFixed('-356330174906.7', '-356330174906.737986270704', 1);
    TtoFixed('771875969530483104163034745359742192923504.000000000000000', '771875969530483104163034745359742192923504', 15);
    TtoFixed('-85.57612133364', '-85.576121333642541652128540737082', 11);
    TtoFixed('-796870619982006783.00000000', '-796870619982006783', 8);
    TtoFixed('985819.0', '985819', 1);

    BigNumber.config({ROUNDING_MODE : 6});

    TtoFixed('-6614662975368684488885953285955838893900074215956.00', '-6614662975368684488885953285955838893900074215956', 2);
    TtoFixed('8642393776898.000', '8642393776898', 3);
    TtoFixed('3612207878528.1106344698085142865', '3612207878528.1106344698085142865438924419', 19);
    TtoFixed('977910386.93', '977910386.9343537', 2);
    TtoFixed('-8937937492688941768401276851642629965923372.0', '-8937937492688941768401276851642629965923372', 1);
    TtoFixed('-8327876772303865517454752161.4376112752921904774', '-8327876772303865517454752161.43761127529219047742910886', 19);
    TtoFixed('-27707.00', '-27707', 2);
    TtoFixed('-7287595765894242586375.92700435484', '-7287595765894242586375.927004354837723619581861634319', 11);
    TtoFixed('-0.00000', '-0.000000000000023670858459165486137218740358207859', 5);
    TtoFixed('574676836802209077.64156', '574676836802209077.641563875647945277493356873895', 5);
    TtoFixed('-3041038455237222898218053653661.87952947815', '-3041038455237222898218053653661.87952947815', 11);
    TtoFixed('-628244132307.000', '-628244132307', 3);

    BigNumber.config({ROUNDING_MODE : 4});

    function TtoFraction(expected, value, maxDenominator){
        assert(expected, new BigNumber(value).toFraction(maxDenominator).toString());
    }

    TtoFraction('5,1', '5.1582612935891', '3');
    TtoFraction('14645,1797', '8.14969395596340', '4682');
    TtoFraction('350921,81990', '4.28004634702', '82418');
    TtoFraction('923754598113149102531799372705758450549,100000000', '9237545981131491025317993727057.58450549');
    TtoFraction('23665467152910525354658972818814752082,49447', '478602688796297558085606261629921.9787246948249979233505266225354557384', '57466');
    TtoFraction('12401124714135748113774134798397,2', '6200562357067874056887067399198.66971421701', '2');
    TtoFraction('5747456325233732815460960864983211610837351385715799969377,100000000000000000000000000000', '57474563252337328154609608649.83211610837351385715799969377');
    TtoFraction('4744,61', '77.7705294446793765233206385561288346447', '86');
    TtoFraction('230406626515463254737498487452583023685806305331639,100000000000000000000000000', '2304066265154632547374984.87452583023685806305331639');
    TtoFraction('8452816572418429518944119647,5000', '1690563314483685903788823.9294');
    TtoFraction('48239499471878294436129777581,10000000000000000', '4823949947187.82944361297775810');
    TtoFraction('45172634468362586403517152343110493066053,74300', '607976237797612199239800166125309462.530995962577562', '89241');
    TtoFraction('245629009658126699398365025147112883705717,1250000000000000000000000000000', '196503207726.5013595186920201176903069645736');
    TtoFraction('19575206081236584421562241755643406299,500000000000000000000000000000000000', '39.150412162473168843124483511286812598');
    TtoFraction('124335953351113119989680198250484096210027176457861877,20000000000000000000000', '6216797667555655999484009912524.20481050135882289309385');
    TtoFraction('62359286968916651021817694022163661098254437249041482587,100000000000000000000000000', '623592869689166510218176940221.63661098254437249041482587');
    TtoFraction('1406253392003122107350210973,50', '28125067840062442147004219.46', '90');
    TtoFraction('736337826,1', '736337826.0', '1');
    TtoFraction('2294354321,3019', '759971.62007285863174441704674479534091075', '5946');
    TtoFraction('342555042697121035185957633789368546657896318423873,50000000000000000000000000', '6851100853942420703719152.67578737093315792636847746');
    TtoFraction('3511582655801640591,52', '67530435688493088.2885316751682999639624370547426320', '73');
    TtoFraction('697806103903081,1', '697806103903081.3297023089715266281593302923590', '1');
    TtoFraction('504239849856718524776121635107896338763895,26083', '19332126283660565302155489595057943440.70448184108567544983900489517214731', '26482');
    TtoFraction('3513961,10', '351396.0988885676988110229927700324193', '11');
    TtoFraction('1514998954038665195379885,15263', '99259578984384799540.056673', '55079');
    TtoFraction('1190687310674838338273481559,50000', '23813746213496766765469.63118');
    TtoFraction('31747884008318370382077439489,40', '793697100207959259551935987.225181146039', '45');
    TtoFraction('39451289109544785156061546711681,59993', '657598204949657212609163514.271348320970259397527780637349273932', '61325');
    TtoFraction('3859539118592118171061,65', '59377524901417202631.7076617756627530296590420377861', '248');
    TtoFraction('2529860848590369124121129138,36761', '68819152052184900414056.449443703377131320899467509', '59208');
    TtoFraction('17331073,2', '8665536.3781112033855216846406980971689', '2');
    TtoFraction('11088761069693879,25000', '443550442787.75516');
    TtoFraction('37298162459148947522489,474', '78688106453900733169.80801275320205353359142143262532', '517');
    TtoFraction('12411782390212011049587005051032466579057,100000000000000000000000000000000000000', '124.11782390212011049587005051032466579057');
    TtoFraction('288377889236302421024476469650301879883847,50000000000000000000000', '5767557784726048420.48952939300603759767694');
    TtoFraction('64269805556832147,1', '64269805556832146.8921574533060466484388410550211', '2');
    TtoFraction('1019428223041680221,2', '509714111520840110.44623631764611004359625363775303233110', '2');
    TtoFraction('15446409508173177876456900755482517,250000', '61785638032692711505827603021.9300680');
    TtoFraction('794012664285496747924956753856029102507539,10000000', '79401266428549674792495675385602910.2507539');
    TtoFraction('1254937597799095507639361,5', '250987519559819101527872.2');
    TtoFraction('6200289913593672418541428454415821,29', '213803100468747324777290636359166.2447131068603257501', '29');
    TtoFraction('28363420172665408049802993720278,1', '28363420172665408049802993720278.0', '53164');
    TtoFraction('614972069955669535218947919309413188717428438,76097', '8081423314397013485668921499000133891183.99461213951158548435230604703', '85075');
    TtoFraction('456397843623966327008858405616124567381,62500000000000000000', '7302365497983461232.141734489857993078096');
    TtoFraction('18578778016377890885386246496020651,1000000000000000000000000', '18578778016.377890885386246496020651');

    function TtoPrecision(expected, value, precision){
        assert(String(expected), new BigNumber(value).toPrecision(precision));
    }

    BigNumber.config({EXPONENTIAL_AT : [-7, 40]});

    TtoPrecision('1e+27', '1.2345e+27', 1);
    TtoPrecision('1.2e+27', '1.2345e+27', 2);
    TtoPrecision('1.23e+27', '1.2345e+27', 3);
    TtoPrecision('1.235e+27', '1.2345e+27', 4);
    TtoPrecision('1.2345e+27', '1.2345e+27', 5);
    TtoPrecision('1.23450e+27', '1.2345e+27', 6);
    TtoPrecision('1.234500e+27', '1.2345e+27', 7);

    TtoPrecision('-1e+27', '-1.2345e+27', 1);
    TtoPrecision('-1.2e+27', '-1.2345e+27', 2);
    TtoPrecision('-1.23e+27', '-1.2345e+27', 3);
    TtoPrecision('-1.235e+27', '-1.2345e+27', 4);
    TtoPrecision('-1.2345e+27', '-1.2345e+27', 5);
    TtoPrecision('-1.23450e+27', '-1.2345e+27', 6);
    TtoPrecision('-1.234500e+27', '-1.2345e+27', 7);

    BigNumber.config({ROUNDING_MODE : 0});

    TtoPrecision('-0.000090000000', '-0.00009', 8);
    TtoPrecision('-7e-7', '-0.0000007', 1);
    TtoPrecision('68.9316834061848', '68.931683406184761912218250317', 15);
    TtoPrecision('7.8601018089704732e+27', '7860101808970473167417935916.60087069', 17);
    TtoPrecision('3.21445885399803244067719798337437062000000e-11', '0.0000000000321445885399803244067719798337437062', 42);
    TtoPrecision('-8171786349835057630612358814.162756978', '-8171786349835057630612358814.162756977984', 37);
    TtoPrecision('3340.9039701', '3340.903970019817086594869184429527413533291595472085', 11);
    TtoPrecision('-7269097658095414435895.9161181115739745427300313060', '-7269097658095414435895.916118111573974542730031306', 50);
    TtoPrecision('0.00000632207', '0.00000632206077863', 6);
    TtoPrecision('6e+2', '573', 1);
    TtoPrecision('7.4e-7', '0.000000738', 2);
    TtoPrecision('-5.031561e-7', '-0.0000005031560306227217140253964236911907612837', 7);
    TtoPrecision('-4.291e+11', '-429050053964', 4);
    TtoPrecision('8.514e+7', '85131637', 4);
    TtoPrecision('-3.4e-9', '-0.000000003326783057540398442677461', 2);
    TtoPrecision('6.9404295962722512e-20', '0.00000000000000000006940429596272251146200868514973032594273', 17);
    TtoPrecision('-828376248340605120247.15155295014', '-828376248340605120247.15155295013990774586360178257303370779', 32);


    BigNumber.config({ROUNDING_MODE : 1});

    TtoPrecision('-844789036.5239726', '-844789036.52397268892', 16);
    TtoPrecision('-5056.20629012767878749185273209679064306054', '-5056.206290127678787491852732096790643060542', 42);
    TtoPrecision('-0.3287519131314873763501859870298952500', '-0.32875191313148737635018598702989525', 37);
    TtoPrecision('-60729764', '-60729764', 8);
    TtoPrecision('-7.622e-14', '-0.00000000000007622481594531380999826456196664586', 4);
    TtoPrecision('-4686402261639729535.736324492474', '-4686402261639729535.7363244924747488', 31);
    TtoPrecision('-2.0', '-2', 2);
    TtoPrecision('-13801188035233586637950193108.13592574381473451125649500', '-13801188035233586637950193108.135925743814734511256495', 55);
    TtoPrecision('0.0000807327587149839799300000', '0.00008073275871498397993', 24);
    TtoPrecision('-6.000000e-8', '-0.00000006', 7);
    TtoPrecision('-3.83574993e+11', '-383574993535', 9);
    TtoPrecision('7.6987000000000000e-14', '0.000000000000076987', 17);
    TtoPrecision('80928866804.6112050947427973', '80928866804.6112050947427973864826014844575374353', 27);
    TtoPrecision('-0.00730140', '-0.0073014067221009206110062377503733', 6);

    BigNumber.config({ROUNDING_MODE : 2});

    TtoPrecision('-0.0300', '-0.0300921721159558', 3);
    TtoPrecision('65317841202.20949859371772273480125', '65317841202.2094985937177227348012464402154', 34);
    TtoPrecision('-8.9231575495202e+29', '-892315754952021994731329589682.1894180393920044085713', 14);
    TtoPrecision('-2.8075679202e-8', '-0.0000000280756792028583066', 11);
    TtoPrecision('9.71456e+9', '9714558552', 6);
    TtoPrecision('2.9514099281e-10', '0.00000000029514099281', 11);
    TtoPrecision('-1.24459e+14', '-124459985101107', 6);
    TtoPrecision('0.0000734657394154607815562372000000', '0.0000734657394154607815562372', 30);
    TtoPrecision('1.78719530353972e+15', '1787195303539715', 15);
    TtoPrecision('-2.8e+9', '-2861102528', 2);
    TtoPrecision('-8.74480375581000e-9', '-0.00000000874480375581', 15);
    TtoPrecision('-1792404726015427380.248150830448457643618022', '-1792404726015427380.248150830448457643618022', 43);
    TtoPrecision('-678437320202616518.2220157912209286', '-678437320202616518.22201579122092864', 34);

    BigNumber.config({ROUNDING_MODE : 3});

    TtoPrecision('-39449414270333.925852213835', '-39449414270333.925852213834759031494508489474', 26);
    TtoPrecision('-7.50437989976', '-7.50437989975503711836768', 12);
    TtoPrecision('-0.000004303975760000000', '-0.00000430397576', 16);
    TtoPrecision('-16040233916257241895.97650633973989', '-16040233916257241895.9765063397398857', 34);
    TtoPrecision('-7438.9287248601393819', '-7438.9287248601393818639176907606', 20);
    TtoPrecision('9.857465584298e-7', '0.000000985746558429876825600458537705318327799', 13);
    TtoPrecision('532637.9095983547284850466577958315920', '532637.90959835472848504665779583159203905641996', 37);
    TtoPrecision('-1.40416695292e+30', '-1404166952915258058306475434520.7856110230505157', 12);
    TtoPrecision('60346876.6670832429026869255506808488', '60346876.6670832429026869255506808488', 36);
    TtoPrecision('-2.52466133e+23', '-252466132238128405832984', 9);
    TtoPrecision('55', '55', 2);
    TtoPrecision('8', '8', 1);

    BigNumber.config({ROUNDING_MODE : 4});

    TtoPrecision('7.905300379788e+16', '79053003797878062.6454954', 13);
    TtoPrecision('-6.83490000000e-13', '-0.00000000000068349', 12);
    TtoPrecision('-62760641815.69084973661201201', '-62760641815.690849736612012010742308663', 28);
    TtoPrecision('0.000704', '0.000704496313', 3);
    TtoPrecision('82926865286287.8852357368342860830310721063079299643', '82926865286287.88523573683428608303107210630792996432', 51);
    TtoPrecision('-0.00032388272393900301214220090249', '-0.00032388272393900301214220090248744799603424908', 29);
    TtoPrecision('8.6e+12', '8621641486938.4837308885005093571508566552428700982454', 2);
    TtoPrecision('2', '2', 1);
    TtoPrecision('1.4641440117052559075e+20', '146414401170525590746.047955203899370771105088', 20);
    TtoPrecision('3511.925583', '3511.925583', 10);
    TtoPrecision('2861824.253079699095728', '2861824.253079699095727765750377038689', 22);
    TtoPrecision('-3.940097756e+10', '-39400977564.548924098664431671700066962', 10);
    TtoPrecision('-888', '-888', 3);
    TtoPrecision('-0.000302106125213724988141721256104', '-0.00030210612521372498814172125610432438685', 30);
    TtoPrecision('6943.4804552555315615809650428503', '6943.480455255531561580965042850266831249032130818358478956', 32);

    BigNumber.config({ROUNDING_MODE : 5});

    TtoPrecision('-1408003897645960.648499616456', '-1408003897645960.648499616456', 28);
    TtoPrecision('-7719307749101742537.6299396338672184', '-7719307749101742537.6299396338672184334306', 35);
    TtoPrecision('-1.0', '-1', 2);
    TtoPrecision('-8.28e+14', '-827860423543649', 3);
    TtoPrecision('0.00054398953021585321711560388890', '0.00054398953021585321711560388889590290139888', 29);
    TtoPrecision('-4.409e-9', '-0.000000004408792', 4);
    TtoPrecision('4.0000e-10', '0.0000000004', 5);
    TtoPrecision('3.40e+16', '34001779327925905', 3);
    TtoPrecision('-9.03e+34', '-90332622851356543193546536340366547', 3);
    TtoPrecision('-4.5320e+16', '-45320100856429143.39155209710530673318222777', 5);
    TtoPrecision('3.618e+30', '3618328715720583671291544414202', 4);
    TtoPrecision('-1003.61140', '-1003.61139687804673322250551', 9);
    TtoPrecision('-8139415035028632370.38737', '-8139415035028632370.38736602659835', 24);
    TtoPrecision('8e+7', '83198058', 1);

    BigNumber.config({ROUNDING_MODE : 6});

    TtoPrecision('0.08000', '0.08', 4);
    TtoPrecision('-4.5132e+21', '-4513243388120382069815.8508153058993058875', 5);
    TtoPrecision('-73549', '-73549.2594630551663822238', 5);
    TtoPrecision('1.275868004728922895890883e+29', '127586800472892289589088296800.6', 25);
    TtoPrecision('-0.0003715444034899460421534099962225699000', '-0.0003715444034899460421534099962225699', 37);
    TtoPrecision('-6.9625565265e+24', '-6962556526511822306135536', 11);
    TtoPrecision('1.67583703641e+13', '16758370364138.915293525076269061228714877', 12);
    TtoPrecision('-173594.95064085553515176707313947534918109631092170', '-173594.950640855535151767073139475349181096310921699', 50);
    TtoPrecision('-6.9503965525e+19', '-69503965525000308384.151383', 11);
    TtoPrecision('4.411225e+20', '441122486054080817112', 7);
    TtoPrecision('2.467044064783596937642371770e+31', '24670440647835969376423717700462.39', 28);
    TtoPrecision('3.9711897549481645654e+24', '3971189754948164565361634.8039734590476326224193520402091769', 20);
    TtoPrecision('-1.4757613208690e+21', '-1475761320868963235919.64499841336073105746686372924161', 14);
    TtoPrecision('91683083887068.6191146', '91683083887068.61911461351134520171343337804061135', 21);
    TtoPrecision('-7923074181102822.578', '-7923074181102822.5778', 19);

    n = new BigNumber(1);
    assert(true, n.isFinite());
    assert(false, n.isNaN());
    assert(false, n.isNegative());
    assert(false, n.isZero());
    assert(true, n.equals(n));
    assert(true, n.equals(n, 2));
    assert(true, n.equals('1.0', 20));
    assert(true, n.equals(new BigNumber(1, 10), 26));
    assert(true, n.equals(new BigNumber(1), 27));
    assert(true, n.equals(1, 36));
    assert(true, n.greaterThan(0.99999));
    assert(false, n.greaterThanOrEqualTo(1.1));
    assert(true, n.lessThan(1.001));
    assert(true, n.lessThanOrEqualTo(2));
    assert(n.toString(), n.valueOf());

    n = new BigNumber('-0.1');
    assert(true, n.isF());
    assert(false, n.isNaN());
    assert(true, n.isNeg());
    assert(false, n.isZ());
    assert(false, n.equals(0.1));
    assert(false, n.greaterThan(-0.1));
    assert(true, n.greaterThanOrEqualTo(-1));
    assert(true, n.lessThan(-0.01));
    assert(false, n.lessThanOrEqualTo(-1));
    assert(n.toString(), n.valueOf());

    n = new BigNumber(Infinity);
    assert(false, n.isFinite());
    assert(false, n.isNaN());
    assert(false, n.isNegative());
    assert(false, n.isZero());
    assert(true, n.eq('Infinity'));
    assert(true, n.eq(1/0));
    assert(true, n.gt('9e999'));
    assert(true, n.gte(Infinity));
    assert(false, n.lt(Infinity));
    assert(true, n.lte(Infinity));
    assert(n.toString(), n.valueOf());

    n = new BigNumber('-Infinity');
    assert(false, n.isF());
    assert(false, n.isNaN());
    assert(true, n.isNeg());
    assert(false, n.isZ());
    assert(false, n.equals(Infinity));
    assert(true, n.equals(-1/0));
    assert(false, n.greaterThan(-Infinity));
    assert(true, n.greaterThanOrEqualTo('-Infinity', 8));
    assert(true, n.lessThan(0));
    assert(true, n.lessThanOrEqualTo(Infinity));
    assert(n.toString(), n.valueOf());

    n = new BigNumber('0.0000000');
    assert(true, n.isFinite());
    assert(false, n.isNaN());
    assert(false, n.isNegative());
    assert(true, n.isZero());
    assert(true, n.eq(-0));
    assert(true, n.gt(-0.000001));
    assert(false, n.gte(0.1));
    assert(true, n.lt(0.0001));
    assert(true, n.lte(-0));
    assert(n.toString(), n.valueOf());

    n = new BigNumber(-0);
    assert(true, n.isF());
    assert(false, n.isNaN());
    assert(true, n.isNeg());
    assert(true, n.isZ());
    assert(true, n.equals('0.000'));
    assert(true, n.greaterThan(-1));
    assert(false, n.greaterThanOrEqualTo(0.1));
    assert(false, n.lessThan(0));
    assert(false, n.lessThan(0, 36));
    assert(true, n.lessThan(0.1));
    assert(true, n.lessThanOrEqualTo(0));
    assert(n.toString(), n.valueOf());

    n = new BigNumber('NaN');
    assert(false, n.isFinite());
    assert(true, n.isNaN());
    assert(false, n.isNegative());
    assert(false, n.isZero());
    assert(false, n.eq(NaN));
    assert(false, n.eq(Infinity));
    assert(false, n.gt(0));
    assert(false, n.gte(0));
    assert(false, n.lt(1));
    assert(false, n.lte(-0));
    assert(false, n.lte(-1));
    assert(n.toString(), n.valueOf());

    BigNumber.config({ ERRORS : false });

    n = new BigNumber('hiya');
    assert(false, n.isFinite());
    assert(true, n.isNaN());
    assert(false, n.isNegative());
    assert(false, n.isZero());
    assert(false, n.equals(0));
    assert(false, n.greaterThan(0));
    assert(false, n.greaterThanOrEqualTo(-Infinity));
    assert(false, n.lessThan(Infinity));
    assert(false, n.lessThanOrEqualTo(0));
    assert(n.toString(), n.valueOf());

    BigNumber.config({ ERRORS : true });

    n = new BigNumber('-1.234e+2');
    assert(true, n.isF());
    assert(false, n.isNaN());
    assert(true, n.isNeg());
    assert(false, n.isZ());
    assert(true, n.eq(-123.4, 10));
    assert(true, n.gt('-ff', 16));
    assert(true, n.gte('-1.234e+3'));
    assert(true, n.lt(-123.39999));
    assert(true, n.lte('-123.4e+0'));
    assert(n.toString(), n.valueOf());

    n = new BigNumber('5e-200');
    assert(true, n.isFinite());
    assert(false, n.isNaN());
    assert(false, n.isNegative());
    assert(false, n.isZero());
    assert(true, n.equals(5e-200));
    assert(true, n.greaterThan(5e-201));
    assert(false, n.greaterThanOrEqualTo(1));
    assert(true, n.lessThan(6e-200));
    assert(true, n.lessThanOrEqualTo(5.1e-200));
    assert(n.toString(), n.valueOf());

    n = new BigNumber('1');
    assert(true, n.equals(n));
    assert(true, n.equals(n.toS()));
    assert(true, n.equals(n.toString()));
    assert(true, n.equals(n.valueOf()));
    assert(true, n.equals(n.toFixed()));
    assert(true, n.equals(1));
    assert(true, n.equals('1e+0'));
    assert(false, n.equals(-1));
    assert(false, n.equals(0.1));

    BigNumber.config({ ERRORS : false });

    assert(false, new BigNumber(NaN).equals(0));
    assert(false, new BigNumber(null).equals(0));
    assert(false, new BigNumber(undefined).equals(0));
    assert(false, new BigNumber(Infinity).equals(0));
    assert(false, new BigNumber([]).equals(0));
    assert(false, new BigNumber([]).equals(0));
    assert(false, new BigNumber({}).equals(0));
    assert(false, new BigNumber('').equals(0));
    assert(false, new BigNumber('   ').equals(0));
    assert(false, new BigNumber('\t').equals(0));
    assert(false, new BigNumber('gerg').equals(0));
    assert(false, new BigNumber(new Date).equals(0));
    assert(false, new BigNumber(new RegExp).equals(0));
    assert(false, new BigNumber(0.1).equals(0));
    assert(false, new BigNumber(1e9 + 1).equals(1e9));
    assert(false, new BigNumber(1e9 - 1).equals(1e9));
    assert(true, new BigNumber(1e9 + 1).equals(1e9 + 1));
    assert(true, new BigNumber(1).equals(1));
    assert(false, new BigNumber(1).equals(-1));
    assert(false, new BigNumber(NaN).equals('efffe'));

    assert(false, new BigNumber('b').greaterThan('a'));
    assert(false, new BigNumber('a').lessThan('b', 10));
    assert(true, new BigNumber('a', 16).lessThanOrEqualTo('ff', 16));
    assert(true, new BigNumber('b', 16).greaterThanOrEqualTo(9, 16));

    BigNumber.config({ ERRORS : true });

    assert(true, new BigNumber(10).greaterThan(10, 2));
    assert(true, new BigNumber(10).lessThan(10, 36));
    assert(false, new BigNumber(NaN).lessThan(NaN));
    assert(false, new BigNumber(Infinity).lessThan(-Infinity));
    assert(false, new BigNumber(Infinity).lessThan(Infinity));
    assert(true, new BigNumber(Infinity, 10).lessThanOrEqualTo(Infinity, 2));
    assert(false, new BigNumber(NaN).greaterThanOrEqualTo(NaN));
    assert(true, new BigNumber(Infinity).greaterThanOrEqualTo(Infinity));
    assert(true, new BigNumber(Infinity).greaterThanOrEqualTo(-Infinity));
    assert(false, new BigNumber(NaN).greaterThanOrEqualTo(-Infinity));
    assert(true, new BigNumber(-Infinity).greaterThanOrEqualTo(-Infinity));

    assert(false, new BigNumber(2, 10).greaterThan(10, 2));
    assert(false, new BigNumber(10, 2).lessThan(2, 10));
    assert(true, new BigNumber(255).lessThanOrEqualTo('ff', 16));
    assert(true, new BigNumber('a', 16).greaterThanOrEqualTo(9, 16));
    assert(false, new BigNumber(0).lessThanOrEqualTo('NaN'));
    assert(false, new BigNumber(0).greaterThanOrEqualTo(NaN));
    assert(false, new BigNumber(NaN, 2).lessThanOrEqualTo('NaN', 36));
    assert(false, new BigNumber(NaN, 36).greaterThanOrEqualTo(NaN, 2));
    assert(false, new BigNumber(0).lessThanOrEqualTo(-Infinity));
    assert(true, new BigNumber(0).greaterThanOrEqualTo(-Infinity));
    assert(true, new BigNumber(0).lessThanOrEqualTo('Infinity', 36));
    assert(false, new BigNumber(0).greaterThanOrEqualTo('Infinity', 36));
    assert(false, new BigNumber(10).lessThanOrEqualTo(20, 4));
    assert(true, new BigNumber(10).lessThanOrEqualTo(20, 5));
    assert(false, new BigNumber(10).greaterThanOrEqualTo(20, 6));

    assert(false, new BigNumber(1.23001e-2).lessThan(1.23e-2));
    assert(true, new BigNumber(1.23e-2).lt(1.23001e-2));
    assert(false, new BigNumber(1e-2).lessThan(9.999999e-3));
    assert(true, new BigNumber(9.999999e-3).lt(1e-2));

    assert(false, new BigNumber(1.23001e+2).lessThan(1.23e+2));
    assert(true, new BigNumber(1.23e+2).lt(1.23001e+2));
    assert(true, new BigNumber(9.999999e+2).lessThan(1e+3));
    assert(false, new BigNumber(1e+3).lt(9.9999999e+2));

    assert(false, new BigNumber(1.23001e-2).lessThanOrEqualTo(1.23e-2));
    assert(true, new BigNumber(1.23e-2).lte(1.23001e-2));
    assert(false, new BigNumber(1e-2).lessThanOrEqualTo(9.999999e-3));
    assert(true, new BigNumber(9.999999e-3).lte(1e-2));

    assert(false, new BigNumber(1.23001e+2).lessThanOrEqualTo(1.23e+2));
    assert(true, new BigNumber(1.23e+2).lte(1.23001e+2));
    assert(true, new BigNumber(9.999999e+2).lessThanOrEqualTo(1e+3));
    assert(false, new BigNumber(1e+3).lte(9.9999999e+2));

    assert(true, new BigNumber(1.23001e-2).greaterThan(1.23e-2));
    assert(false, new BigNumber(1.23e-2).gt(1.23001e-2));
    assert(true, new BigNumber(1e-2).greaterThan(9.999999e-3));
    assert(false, new BigNumber(9.999999e-3).gt(1e-2));

    assert(true, new BigNumber(1.23001e+2).greaterThan(1.23e+2));
    assert(false, new BigNumber(1.23e+2).gt(1.23001e+2));
    assert(false, new BigNumber(9.999999e+2).greaterThan(1e+3));
    assert(true, new BigNumber(1e+3).gt(9.9999999e+2));

    assert(true, new BigNumber(1.23001e-2).greaterThanOrEqualTo(1.23e-2));
    assert(false, new BigNumber(1.23e-2).gte(1.23001e-2));
    assert(true, new BigNumber(1e-2).greaterThanOrEqualTo(9.999999e-3));
    assert(false, new BigNumber(9.999999e-3).gte(1e-2));

    assert(true, new BigNumber(1.23001e+2).greaterThanOrEqualTo(1.23e+2));
    assert(false, new BigNumber(1.23e+2).gte(1.23001e+2));
    assert(false, new BigNumber(9.999999e+2).greaterThanOrEqualTo(1e+3));
    assert(true, new BigNumber(1e+3).gte(9.9999999e+2));

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];;
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;