//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Support/Util/ItemFileNameUtil.cpp
//! @brief     Defines auxiliary functions in ItemFileNameUtils namespace.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Support/Util/ItemFileNameUtil.h"
#include "GUI/Support/Util/Path.h"

namespace {

const QString jobdata_file_prefix = "jobdata";
const QString refdata_file_prefix = "refdata";
const QString realdata_file_prefix = "realdata";
const QString instrument_file_prefix = "instrdata";
const QString nativedata_file_prefix = "nativedata";

//! Constructs the name of the file for intensity data.
QString intensityDataFileName(const QString& itemName, const QString& prefix)
{
    QString bodyName = GUI::Base::Path::getValidFileName(itemName);
    return QString("%1_%2_0.int.gz").arg(prefix).arg(bodyName);
}

} // namespace


//! Constructs the name of the file with simulated intensities.

QString GUI::Model::FilenameUtil::jobResultsFileName(const QString& itemName)
{
    return intensityDataFileName(itemName, jobdata_file_prefix);
}

//! Constructs the name of the file with reference data.

QString GUI::Model::FilenameUtil::jobReferenceFileName(const QString& itemName)
{
    return intensityDataFileName(itemName, refdata_file_prefix);
}

QString GUI::Model::FilenameUtil::jobNativeDataFileName(const QString& id)
{
    return intensityDataFileName(id, nativedata_file_prefix);
}

//! Constructs the name of the intensity file belonging to real data item.

QString GUI::Model::FilenameUtil::realDataFileName(const QString& dataName)
{
    return intensityDataFileName(dataName, realdata_file_prefix);
}

QString GUI::Model::FilenameUtil::nativeDataFileName(const QString& dataName)
{
    return intensityDataFileName(dataName, nativedata_file_prefix);
}


//! Returns list of fileName filters related to nonXML data stored by JobModel and RealModel.

QStringList GUI::Model::FilenameUtil::nonXMLFileNameFilters()
{
    QStringList result = QStringList() << QString(jobdata_file_prefix + "_*.int.gz")
                                       << QString(refdata_file_prefix + "_*.int.gz")
                                       << QString(realdata_file_prefix + "_*.int.gz")
                                       << QString(nativedata_file_prefix + "_*.int.gz")
                                       << QString(instrument_file_prefix + "_*.int.gz");

    return result;
}
