/*
 *                          libcollatinus.h
 *
 *  This file is part of COLLATINVS.
 *                                                                            
 *  COLLATINVS is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *                                                                            
 *  COLLATINVS is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *                                                                            
 *  You should have received a copy of the GNU General Public License
 *  along with COLLATINVS; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

// AGENDA : toutes les defs de libcollatinus.cc  placer dans le header !!

/*
 *
 *  . flexions, synthse de formes.
 *
 *  . renvois "cf." trop vagues.
 *      - mettre en place un renvoi exclusif : 
 *        utiliser la balise "pl. " pour les formes
 *        sans singulier.
 *        Le mme rsultat peut tre obtenu en
 *          - dtectant -ae, -i, -es (ambigu) ;
 *          - liminant les lemmatisations au singulier.
 *
 *       D'autres renvois exclusifs sont sans doute  mettre en place.
 *
 */

#ifndef LIBCOLLATINUS_H
#define LIBCOLLATINUS_H

#include <map>
#include <list>
#include <string>

/*   Fonctions de conversion  */
std::wstring ini (std::wstring c, int n); // ini  donne la chaine c tronque de n caractres  droite
std::wstring jviu (std::wstring ch);      // jviu transforme la graphie ramiste en graphie ancienne
std::wstring basDeCasse (std::wstring s);
std::wstring indente (std::wstring s);
std::wstring sinesuffixo (std::wstring e, std::wstring s); // utile pour les formes en -que ou -ue.
int aRomano (std::wstring R); // Fonction de conversion des chiffres romains en chiffres arabes.

/* classes */
class Tdes
{
    private:
        std::wstring gr; // graphie
        int c;     // cas
        int g;     // genre
        int n;     // nombre
        int d;     // degr (gradus)
        int p;     // personne
        int t;     // temps
        int m;     // mode
        int v;     // voix
        int mo;    // modle (paradigma)
        int rn;    // numro de radical (radix_num)
    public:
        /*
         * Format d'une ligne de lemmata.fr :
         * graph|casus|genus|numerus|gradus|persona|tempus|modus|uox|paradigma|radix_num
         */
        Tdes (std::wstring linea);
        Tdes ( std::wstring l, int k, int ge, int num, int deg, 
                int per, int tem, int mod, int vo, 
                int  mdl, int radn);
        // affectations
        void da_c (int x);
        void da_g (int x);
        void da_n (int x);
        void da_d  (int x);
        void da_p  (int x);
        void da_t  (int x);
        void da_m  (int x);
        void da_v  (int x);
        void da_mo (int x);
        void da_rn (int x);
        // consultation string
        std::wstring graphie ();
        int mdl ();
        std::wstring modele ();
        int casus ();
        std::wstring cas ();
        int genus ();
        std::wstring genre ();
        int persona ();
        std::wstring personne ();
        int numerus ();
        std::wstring nombre ();
        int gradus ();
        std::wstring degre ();
        int tempus ();
        std::wstring temps ();
        // consultation integer
        int modus ();
        std::wstring mode ();
        int uox ();
        std::wstring voix ();
        int nr ();
        std::wstring morpho ();
        std::wstring doc ();
        void affiche ();
};

class AnalyseMorpho : public Tdes
{
    private:
        std::wstring lemme;
    public:
        std::wstring lemma ();
        AnalyseMorpho ( 
                std::wstring l, int k, int ge, int num, int deg, 
                int per, int tem, int mod, int vo, 
                int  mdl, int radn);
};

typedef std::multimap<std::wstring, AnalyseMorpho> ListeAnalyses;

/* classe des irrguliers */
class Tirr
{
    private:
        std::wstring gr;
        std::wstring k;
        int c;
        int g;
        int n;
        int d;
        int p;
        int t;
        int m;
        int v;
    public:
        int casus ();
        int genus ();
        int gradus ();
        int persona ();
        int numerus ();
        int tempus ();
        int modus ();
        int uox ();
        Tirr (std::wstring linea);
        std::wstring canon ();
        std::wstring graphie ();
        std::wstring morpho ();
        void affiche ();
};

/* classe des radicaux */

class Tradix
{
    private:
        std::wstring can;
        int mo;
        int num;
    public:
        Tradix (std::wstring c, int m, int n);
        int mdl ();
        int nr ();
        std::wstring canon ();
        std::wstring doc ();
        void affiche ();
};   

/* classe des entres */
class Tentree
{
    private:
        std::wstring k;
        int mdl;
        std::wstring radp;
        std::wstring rads;
        std::wstring def;
        bool vide;
        std::wstring ote2 (std::wstring k);
    public:
        Tentree (std::wstring linea);
        Tentree ();
        std::wstring canon ();
        int modele ();
        char pars ();
        std::wstring radix ();
        std::wstring radical (int n);
        std::wstring  grmodele ();
        std::wstring rperfectum ();
        std::wstring rsupinum () ;
        std::wstring definition () ;
        bool est_neutre (); // vrai si la dfinition inclut la chane ", .n ".
        bool est_pluriel ();
        bool est_deponent ();
        std::wstring morphoK ();
        AnalyseMorpho amMorphoK ();
        AnalyseMorpho amMorphoKN ();
        std::wstring doc ();
        bool est_vide ();
        void affiche ();
};


/* classe lexique */
class TLexicum
{
    private:
        bool uacuum;
        // liste d'entres
        typedef std::map<std::wstring, Tentree*> c2e;
        // liste des dsinences
        typedef std::multimap<std::wstring, Tdes> c2d; 
        // liste des irrguliers
        typedef std::multimap<std::wstring, Tirr> c2i;
        // liste des radicaux
        typedef std::multimap<std::wstring, Tradix> c2r;
        // std::multimap des dsinences
        c2d desinentiae;
        // std::multimap des irrguliers
        c2i irregulares;
        // std::multimap des entres
        c2e entrees;
        // std::multimap des radicaux
        c2r radices;
        // type liste de radicaux
        typedef std::list<Tradix> lr;
        void radices_adde (std::wstring k, std::wstring r, int m, int n);
        std::wstring ote2 (std::wstring k);
    public:
        typedef std::map<std::wstring, std::wstring> ee;
        typedef ee::iterator e2m;
        ee liste_e;
        TLexicum (std::string url);
        TLexicum ();
        Tentree *renvoi (Tentree *e);  // renvoi d'entre si la df. commence par cf.
        Tentree * entree (std::wstring k);
        void uniLanalyses (ListeAnalyses& l1, ListeAnalyses& l2);
        std::wstring irregulier (std::wstring f);
        std::wstring radical (std::wstring r);
        std::wstring desinence (std::wstring d);
        void ajoute (std::wstring e, std::wstring ch);
        std::wstring analyses (std::wstring f);
        void lajoute (std::wstring l, ListeAnalyses& la, AnalyseMorpho am);
        void lajoute (std::wstring l, ListeAnalyses& la, Tdes d);
        ListeAnalyses lanalyses (std::wstring f);
        std::wstring lemmatiseM (std::wstring f); // analyse morphologique
        std::wstring lemmatise (std::wstring f); // simple lemmatisation
        std::wstring desN (Tentree* e, int c, int n);
        std::wstring desA (Tentree* e, int c, int n, int g, int d);
        std::wstring synthA (Tentree* e, int c, int n, int g, int d);
        std::wstring synthPart (Tentree* e, int c, int g, int n, int t, int v);
        std::wstring synthN (Tentree *e, int c, int n);
        std::wstring synthP (Tentree* e, int c, int n, int g);
        std::wstring synthV (Tentree* e, int p, int n, int t, int m, int v, int rn=1);
        std::wstring flechis (Tentree* e);
};

// fonctions globales qui vitent une 
// double dclaration de lexicum

void lexicumLege (std::string lang, std::string c);
void lexicumDic (std::string lang);
void activeCalepin (bool a);
bool calepino_actif ();
std::wstring ambrogio (std::wstring m);
std::wstring lemmatise (std::wstring f);
std::wstring lemmatiseM (std::wstring f);
Tentree * entree (std::wstring linea);
std::wstring flechis (Tentree* e);
ListeAnalyses lanalyses (std::wstring f);
void uniLanalyses (ListeAnalyses& l1, ListeAnalyses& l2);

#endif
