#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""launcher.py """

# Copyright (C) 2008, 2009, 2010, 2011 Federico Brega, Pierluigi Villani

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

import sys
import locale, gettext
import os
import glal_selected

def setup_gettext():
    """Application translation system setup"""

    #Setup translation.
    langs = []
    mylocale = locale.getdefaultlocale()[0]
    if mylocale:
        langs = [mylocale]
    #Other languages in the system.
    language = os.environ.get('LANGUAGE', None)
    if language:
        langs += language.split(':')
    #If there isn't any better use this.
    langs += ["en_CA", "en_US"]
    po_path = os.path.join(os.path.dirname(os.path.abspath(__file__)),
                                           '..','po')

    if not os.path.isdir(po_path):
        po_path = "/usr/share/locale"
    gettext.bindtextdomain("cyclograph", po_path)
    gettext.textdomain("cyclograph")
    lang = gettext.translation("cyclograph", po_path,
                                     languages=langs, fallback=True)
    lang.install(unicode=True)

def main():
    setup_gettext()
    if len(sys.argv) < 2:
        #If there has been an error with the commandline specified frontend
        #try others in the order (PyQt gets first because is the supported
        #frontend on Mac Os and Windows where users might want a graphical
        #interface.
        if glal_selected.use('qt'):
            import cyclograph
            cyclograph.main_qt()
        elif glal_selected.use('gtk3'):
            import cyclograph
            cyclograph.main_gtk3()
        elif glal_selected.use('gtk'):
            import cyclograph
            cyclograph.main_gtk()
        elif glal_selected.use('cli'):
            import cyclograph
            cyclograph.main_cli()
        else:
            sys.exit(2)
    else:
        #also file=* is an argument even if it is not space separed.
        arguments = set([token.partition('=')[0] for token in sys.argv])
        if arguments & set(('-q', '--qt')):
            if glal_selected.use('qt') != 'qt':
                sys.stderr.write('Qt backend is not installed correctely\n')
                sys.exit(3)
            import cyclograph
            cyclograph.main_qt()
        elif arguments & set(('-g', '--gtk')):
            if glal_selected.use('gtk') != 'gtk':
                sys.stderr.write('GTK+ 2 backend is not installed correctely\n')
                sys.exit(3)
            import cyclograph
            cyclograph.main_gtk()
        elif arguments & set(('-G', '--gtk3')):
            if glal_selected.use('gtk3') != 'gtk3':
                sys.stderr.write('GTK+ 3 backend is not installed correctely\n')
                sys.exit(3)
            import cyclograph
            cyclograph.main_gtk3()
        elif arguments & set(('-f', '--file', '-e', '--extension')):
            if glal_selected.use('cli') != 'cli':
                sys.stderr.write('Fatal error, cli backend cannot be started\n')
                sys.exit(3)
            import cyclograph
            cyclograph.main_cli()
        else:
            glal_selected.use('cli')
            import cyclograph
            cyclograph.main_cli()
            sys.exit(1)

if __name__ == '__main__':
    main()

# vim:sw=4:softtabstop=4:expandtab
