/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gnuradio/io_signature.h>
#include "select_vectors_impl.h"

namespace gr {
  namespace dab {

select_vectors::sptr
select_vectors::make(size_t itemsize, unsigned int length, unsigned int num_select, unsigned int num_skip)
{
  return gnuradio::get_initial_sptr
    (new select_vectors_impl(itemsize, length, num_select, num_skip));
}

select_vectors_impl::select_vectors_impl(size_t itemsize, unsigned int length, unsigned int num_select, unsigned int num_skip)
  : gr::block("select_vectors",
             gr::io_signature::make (1, 1, itemsize*length),
             gr::io_signature::make (1, 1, itemsize*length)),
  d_itemsize(itemsize), d_length(length), d_num_select(num_select), d_num_skip(num_skip), d_index(num_select+num_skip) /* <- dont output anything before 1st trigger */
{
  assert(d_num_select!=0);
  assert(d_length!=0);
  set_tag_propagation_policy(TPP_DONT);
}

void 
select_vectors_impl::forecast (int noutput_items, gr_vector_int &ninput_items_required)
{
  int in_req  = noutput_items * (d_num_skip+d_num_select)/d_num_select; /* very coarse .. */

  unsigned ninputs = ninput_items_required.size ();
  for (unsigned i = 0; i < ninputs; i++)
    ninput_items_required[i] = in_req;
}

int 
select_vectors_impl::general_work (int noutput_items,
                        gr_vector_int &ninput_items,
                        gr_vector_const_void_star &input_items,
                        gr_vector_void_star &output_items)
{
  const char *iptr = (const char *) input_items[0];
  
  char *optr = (char *) output_items[0];

  int n_consumed = 0;
  int n_produced = 0;

  std::vector<int> tag_positions;
  int next_tag_position = -1;
  int next_tag_position_index = -1;

  // Get all stream tags with key "dab_sync", and make a vector of the positions.
  // "next_tag_position" contains the position within "iptr where the next "dab_sync" stream tag is found
  std::vector<tag_t> tags;
  get_tags_in_range(tags, 0, nitems_read(0), nitems_read(0) + ninput_items[0], pmt::mp("first"));
  for(int i=0;i<tags.size();i++) {
      int current;
      current = tags[i].offset - nitems_read(0);
      tag_positions.push_back(current);
      next_tag_position_index = 0;
  }
  if(next_tag_position_index >= 0) {
      next_tag_position = tag_positions[next_tag_position_index];
  }
  //


  while (n_consumed < ninput_items[0] && n_produced < noutput_items) {

    if (next_tag_position == n_consumed) { /* new frame starts */
      // Action when stream tags is found:
      d_index=0;
      //

      next_tag_position_index++;
      if (next_tag_position_index == tag_positions.size()) {
        next_tag_position_index = -1;
        next_tag_position = -1;
      }
      else {
        next_tag_position = tag_positions[next_tag_position_index];
      }
    }
    
    /* select this vector? */
    if (d_index >= d_num_skip && d_index < d_num_select + d_num_skip) {
      /* trigger signal */
      if (d_index == d_num_skip) // Add stream tag to the output for the first symbol after 'd_num_skip'
        add_item_tag(0, nitems_written(0) + n_produced, pmt::intern("first"), pmt::intern(""), pmt::intern("select_vectors"));
      /* data */
      memcpy(optr, iptr, d_length*d_itemsize);
      iptr += d_length*d_itemsize;
      optr += d_length*d_itemsize;
      n_produced++;
    } else {
      iptr += d_length*d_itemsize;
    }
  
    d_index++;
    n_consumed++;
  }

  consume_each(n_consumed);
  return n_produced;
}

}
}

