/*
 *  $Id: psdf2d.c 22381 2019-08-13 07:05:39Z yeti-dn $
 *  Copyright (C) 2009-2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/inttrans.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/stats.h>
#include <libprocess/linestats.h>
#include <libgwydgets/gwygraph.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "preview.h"

#define PSDF2D_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

/* Keep it simple and use a predefined set of zooms, these seem suitable. */
typedef enum {
    ZOOM_1 = 1,
    ZOOM_4 = 4,
    ZOOM_16 = 16
} ZoomType;

enum {
    GRAPH_WIDTH = 320,
    NLINES = 12,
    MIN_RESOLUTION = 4,
    MAX_RESOLUTION = 16384,
    MAX_THICKNESS = 128,
};

typedef struct {
    GwyMaskingType masking;
    ZoomType zoom;
    gboolean create_image;
    gboolean zoomed_image;
    gboolean separate;
    gboolean fixres;
    gint resolution;
    gint thickness;
    GwyInterpolationType interpolation;
    GwyWindowingType windowing;
    GwyAppDataId target_graph;
} PSDFArgs;

typedef struct {
    PSDFArgs *args;
    GtkWidget *create_image;
    GtkWidget *zoomed_image;
    GtkWidget *dialogue;
    GtkObject *resolution;
    GtkWidget *fixres;
    GtkObject *thickness;
    GtkWidget *interpolation;
    GtkWidget *windowing;
    GtkWidget *view;
    GtkWidget *masking;
    GSList *zoom;
    gdouble hx;
    gdouble hy;
    GwyDataField *dfield;
    GwyDataField *mask;
    GwyDataField *psdf;
    GwyDataField *modulus;
    GwySelection *selection;
    GtkWidget *graph;
    GwyDataLine *line;
    GwyGraphModel *gmodel;
    GtkWidget *separate;
    GtkWidget *target_graph;
    GwyContainer *mydata;
} PSDFControls;

static gboolean      module_register        (void);
static void          psdf2d                 (GwyContainer *data,
                                             GwyRunType run);
static void          psdf2d_dialogue        (PSDFArgs *args,
                                             GwyContainer *data,
                                             GwyDataField *dfield,
                                              GwyDataField *mask,
                                             gint id);
static void          masking_changed        (GtkComboBox *combo,
                                             PSDFControls *controls);
static void          zoom_changed           (GtkRadioButton *button,
                                             PSDFControls *controls);
static void          create_image_changed   (PSDFControls *controls);
static void          zoomed_image_changed   (PSDFControls *controls);
static void          fixres_changed         (PSDFControls *controls,
                                             GtkToggleButton *check);
static void          resolution_changed     (PSDFControls *controls,
                                             GtkAdjustment *adj);
static void          thickness_changed      (PSDFControls *controls,
                                             GtkAdjustment *adj);
static void          interpolation_changed  (GtkComboBox *combo,
                                             PSDFControls *controls);
static void          windowing_changed      (PSDFControls *controls,
                                             GtkComboBox *combo);
static void          separate_changed       (PSDFControls *controls);
static void          selection_changed      (PSDFControls *controls,
                                             gint hint);
static void          recalculate_psdf       (PSDFControls *controls);
static void          update_sensitivity     (PSDFControls *controls);
static void          target_graph_changed   (PSDFControls *controls);
static void          update_curve           (PSDFControls *controls,
                                             gint i);
static void          init_graph_model_units (GwyGraphModel *gmodel,
                                             GwyDataField *psdf);
static void          calculate_zoomed_fields(PSDFControls *controls);
static GwyDataField* cut_field_to_zoom      (GwyDataField *dfield,
                                             gint zoom);
static void          create_output_graphs   (PSDFControls *controls,
                                             GwyContainer *data);
static gint          create_results         (GwyContainer *data,
                                             GwyDataField *psdf,
                                             GwySelection *selection,
                                             gint oldid,
                                             gint zoom);
static void          calculate_psdf         (GwyDataField *dfield,
                                             GwyDataField *mask,
                                             const PSDFArgs *args,
                                             GwyDataField *psdf,
                                             GwyDataField *modulus);
static void          psdf2d_load_args       (GwyContainer *container,
                                             PSDFArgs *args);
static void          psdf2d_save_args       (GwyContainer *container,
                                             PSDFArgs *args);
static void          psdf2d_sanitize_args   (PSDFArgs *args);

static const PSDFArgs psdf2d_defaults = {
    GWY_MASK_IGNORE, ZOOM_1, TRUE, TRUE,
    FALSE, FALSE, 120, 1,
    GWY_INTERPOLATION_LINEAR,
    GWY_WINDOWING_HANN,
    GWY_APP_DATA_ID_NONE,
};

static GwyAppDataId target_id = GWY_APP_DATA_ID_NONE;

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Calculates two-dimensional power spectrum density function "
       "and extracts its linear profiles."),
    "Yeti <yeti@gwyddion.net>",
    "1.3",
    "David Nečas (Yeti)",
    "2009",
};

GWY_MODULE_QUERY2(module_info, psdf2d)

static gboolean
module_register(void)
{
    gwy_process_func_register("psdf2d",
                              (GwyProcessFunc)&psdf2d,
                              N_("/_Statistics/2D _PSDF..."),
                              /*GWY_STOCK_PSDF_SECTION*/ NULL,
                              PSDF2D_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Calculate 2D power spectrum density"));

    return TRUE;
}

static void
psdf2d(GwyContainer *data, GwyRunType run)
{
    PSDFArgs args;
    GwyDataField *dfield, *mask, *psdf, *modulus;
    gint id;

    g_return_if_fail(run & PSDF2D_RUN_MODES);
    g_return_if_fail(g_type_from_name("GwyLayerPoint"));
    psdf2d_load_args(gwy_app_settings_get(), &args);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_MASK_FIELD, &mask,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield);

    if (run == GWY_RUN_IMMEDIATE) {
        /* Is it reasonable to simply do nothing when settings say to not
         * create the PSDF image? */
        if (args.create_image) {
            psdf = gwy_data_field_new(1, 1, 1.0, 1.0, FALSE);
            modulus = gwy_data_field_new(1, 1, 1.0, 1.0, FALSE);
            calculate_psdf(dfield, mask, &args, psdf, modulus);
            create_results(data, psdf, NULL, id,
                           args.zoomed_image ? args.zoom : ZOOM_1);
            g_object_unref(psdf);
            g_object_unref(modulus);
        }
    }
    else {
        psdf2d_dialogue(&args, data, dfield, mask, id);
        psdf2d_save_args(gwy_app_settings_get(), &args);
    }
}

static void
psdf2d_dialogue(PSDFArgs *args,
                GwyContainer *data,
                GwyDataField *dfield,
                GwyDataField *mask,
                gint id)
{
    GtkWidget *hbox, *vbox, *notebook, *label, *hbox2, *align;
    GtkTable *table;
    GtkDialog *dialogue;
    GwyGraph *graph;
    GwyGraphArea *area;
    PSDFControls controls;
    gint response, row;
    GSList *l;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.dfield = dfield;
    controls.mask = mask;
    controls.psdf = gwy_data_field_new(1, 1, 1.0, 1.0, FALSE);
    controls.modulus = gwy_data_field_new(1, 1, 1.0, 1.0, FALSE);
    calculate_psdf(dfield, mask, args, controls.psdf, controls.modulus);
    controls.hx = gwy_data_field_get_dx(dfield)/(2*G_PI);
    controls.hy = gwy_data_field_get_dx(dfield)/(2*G_PI);

    controls.dialogue = gtk_dialog_new_with_buttons(_("Power Spectrum Density"),
                                                    NULL, 0,
                                                    GTK_STOCK_CLEAR,
                                                    RESPONSE_CLEAR,
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_CANCEL,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    NULL);
    dialogue = GTK_DIALOG(controls.dialogue);
    gtk_dialog_set_default_response(dialogue, GTK_RESPONSE_OK);
    gtk_dialog_set_response_sensitive(dialogue, GTK_RESPONSE_OK, FALSE);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);

    hbox = gtk_hbox_new(FALSE, 2);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       TRUE, TRUE, 4);

    /***** PSDF preview *****/
    controls.mydata = gwy_container_new();
    calculate_zoomed_fields(&controls);
    gwy_container_set_string_by_name(controls.mydata, "/0/base/palette",
                                     g_strdup("DFit"));
    gwy_container_set_enum_by_name(controls.mydata, "/0/base/range-type",
                                   GWY_LAYER_BASIC_RANGE_AUTO);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, FALSE);
    controls.selection = create_vector_layer(GWY_DATA_VIEW(controls.view), 0,
                                             "Point", TRUE);
    gwy_selection_set_max_objects(controls.selection, NLINES);
    g_object_set(gwy_data_view_get_top_layer(GWY_DATA_VIEW(controls.view)),
                 "draw-as-vector", TRUE,
                 NULL);
    g_signal_connect_swapped(controls.selection, "changed",
                             G_CALLBACK(selection_changed), &controls);

    align = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
    gtk_container_add(GTK_CONTAINER(align), controls.view);
    gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 4);

    /***** Graph *****/
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 4);

    controls.gmodel = gwy_graph_model_new();
    g_object_set(controls.gmodel,
                 "title", _("PSDF Section"),
                 "axis-label-bottom", "k",
                 "axis-label-left", "W",
                 NULL);
    init_graph_model_units(controls.gmodel, controls.psdf);
    controls.line = gwy_data_line_new(1, 1.0, FALSE);

    controls.graph = gwy_graph_new(controls.gmodel);
    graph = GWY_GRAPH(controls.graph);
    gtk_widget_set_size_request(controls.graph, GRAPH_WIDTH, 120);
    gwy_graph_set_axis_visible(graph, GTK_POS_LEFT, FALSE);
    gwy_graph_set_axis_visible(graph, GTK_POS_RIGHT, FALSE);
    gwy_graph_set_axis_visible(graph, GTK_POS_TOP, FALSE);
    gwy_graph_set_axis_visible(graph, GTK_POS_BOTTOM, FALSE);
    gwy_graph_enable_user_input(graph, FALSE);
    area = GWY_GRAPH_AREA(gwy_graph_get_area(graph));
    gwy_graph_area_enable_user_input(area, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), controls.graph, TRUE, TRUE, 0);

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(vbox), notebook, FALSE, FALSE, 0);

    /***** PSDF controls *****/
    table = GTK_TABLE(gtk_table_new(6 + 4*(!!mask), 3, FALSE));
    gtk_table_set_row_spacings(table, 2);
    gtk_table_set_col_spacings(table, 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(table),
                             gtk_label_new("PSDF"));
    row = 0;

    hbox2 = gtk_hbox_new(FALSE, 8);
    gtk_table_attach(table, hbox2, 0, 4, row, row+1, GTK_FILL, 0, 0, 0);
    label = gtk_label_new(_("Zoom:"));
    gtk_box_pack_start(GTK_BOX(hbox2), label, FALSE, FALSE, 0);

    controls.zoom = gwy_radio_buttons_createl(G_CALLBACK(zoom_changed),
                                              &controls,
                                              args->zoom,
                                              "1×", ZOOM_1,
                                              "4×", ZOOM_4,
                                              "16×", ZOOM_16,
                                              NULL);
    for (l = controls.zoom; l; l = g_slist_next(l)) {
        GtkWidget *widget = GTK_WIDGET(l->data);
        gtk_box_pack_start(GTK_BOX(hbox2), widget, FALSE, FALSE, 0);
    }
    row++;

    controls.windowing
        = gwy_enum_combo_box_new(gwy_windowing_type_get_enum(), -1,
                                 G_CALLBACK(gwy_enum_combo_box_update_int),
                                 &args->windowing, args->windowing, TRUE);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row, _("_Windowing type:"), NULL,
                            GTK_OBJECT(controls.windowing),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    g_signal_connect_swapped(controls.windowing, "changed",
                             G_CALLBACK(windowing_changed), &controls);
    row++;

    gtk_table_set_row_spacing(table, row-1, 8);
    controls.create_image
        = gtk_check_button_new_with_mnemonic(_("Create PSDF image"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.create_image),
                                 args->create_image);
    gtk_table_attach(table, controls.create_image,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls.create_image, "toggled",
                             G_CALLBACK(create_image_changed), &controls);
    row++;

    controls.zoomed_image
        = gtk_check_button_new_with_mnemonic(_("Only zoomed part"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.zoomed_image),
                                 args->zoomed_image);
    gtk_table_attach(table, controls.zoomed_image,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls.zoomed_image, "toggled",
                             G_CALLBACK(zoomed_image_changed), &controls);
    row++;

    if (mask) {
        controls.masking
            = gwy_enum_combo_box_new(gwy_masking_type_get_enum(), -1,
                                     G_CALLBACK(masking_changed), &controls,
                                     args->masking, TRUE);
        gwy_table_attach_adjbar(GTK_WIDGET(table), row++, _("_Masking:"), NULL,
                                GTK_OBJECT(controls.masking),
                                GWY_HSCALE_WIDGET_NO_EXPAND);
    }

    /***** Graph controls *****/
    table = GTK_TABLE(gtk_table_new(4, 3, FALSE));
    gtk_table_set_row_spacings(table, 2);
    gtk_table_set_col_spacings(table, 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(table),
                             gtk_label_new("Graph"));
    row = 0;

    controls.resolution = gtk_adjustment_new(controls.args->resolution,
                                             MIN_RESOLUTION, MAX_RESOLUTION,
                                             1, 10, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row,
                            _("_Fixed resolution:"), NULL,
                            controls.resolution,
                            GWY_HSCALE_CHECK | GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls.resolution, "value-changed",
                             G_CALLBACK(resolution_changed), &controls);
    controls.fixres = gwy_table_hscale_get_check(controls.resolution);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.fixres),
                                 controls.args->fixres);
    g_signal_connect_swapped(controls.fixres, "toggled",
                             G_CALLBACK(fixres_changed), &controls);
    row++;

    controls.thickness = gtk_adjustment_new(controls.args->thickness,
                                            1, MAX_THICKNESS, 1, 10, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row, _("_Thickness:"), _("px"),
                            controls.thickness, GWY_HSCALE_SNAP);
    g_signal_connect_swapped(controls.thickness, "value-changed",
                             G_CALLBACK(thickness_changed), &controls);
    row++;

    controls.separate = gtk_check_button_new_with_mnemonic(_("_Separate curves"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.separate),
                                 args->separate);
    gtk_table_attach(table, controls.separate,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls.separate, "toggled",
                             G_CALLBACK(separate_changed), &controls);
    row++;

    controls.interpolation
        = gwy_enum_combo_box_new(gwy_interpolation_type_get_enum(), -1,
                                 G_CALLBACK(interpolation_changed),
                                 &controls,
                                 controls.args->interpolation, TRUE);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row,
                            _("_Interpolation type:"), NULL,
                            GTK_OBJECT(controls.interpolation),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    controls.target_graph = create_target_graph(&args->target_graph,
                                                GTK_WIDGET(table), row++,
                                                controls.gmodel);
    g_signal_connect_swapped(controls.target_graph, "changed",
                             G_CALLBACK(target_graph_changed), &controls);

    update_sensitivity(&controls);

    gtk_widget_show_all(controls.dialogue);
    do {
        response = gtk_dialog_run(dialogue);
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(controls.dialogue);
            case GTK_RESPONSE_NONE:
            goto finalize;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_CLEAR:
            gwy_selection_clear(controls.selection);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    if (args->create_image) {
        gwy_data_field_invalidate(controls.psdf);
        create_results(data, controls.psdf, controls.selection, id,
                       args->zoomed_image ? args->zoom : ZOOM_1);
    }
    create_output_graphs(&controls, data);
    gtk_widget_destroy(controls.dialogue);
finalize:
    g_object_unref(controls.mydata);
    g_object_unref(controls.psdf);
    g_object_unref(controls.modulus);
    g_object_unref(controls.gmodel);
    g_object_unref(controls.line);
}

static void
masking_changed(GtkComboBox *combo, PSDFControls *controls)
{
    controls->args->masking = gwy_enum_combo_box_get_active(combo);
    recalculate_psdf(controls);
}

static void
create_image_changed(PSDFControls *controls)
{
    PSDFArgs *args = controls->args;
    args->create_image
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(controls->create_image));
    update_sensitivity(controls);
}

static void
zoomed_image_changed(PSDFControls *controls)
{
    PSDFArgs *args = controls->args;
    args->zoomed_image
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(controls->zoomed_image));
}

static void
zoom_changed(GtkRadioButton *button, PSDFControls *controls)
{
    PSDFArgs *args = controls->args;
    ZoomType zoom = gwy_radio_buttons_get_current(controls->zoom);
    GwyDataField *psdf;
    gdouble xoff, yoff;

    if (button && zoom == args->zoom)
        return;

    psdf = gwy_container_get_object(controls->mydata,
                                    gwy_app_get_data_key_for_id(0));
    xoff = gwy_data_field_get_xoffset(psdf);
    yoff = gwy_data_field_get_yoffset(psdf);
    args->zoom = zoom;
    calculate_zoomed_fields(controls);
    gwy_set_data_preview_size(GWY_DATA_VIEW(controls->view), PREVIEW_SIZE);
    psdf = gwy_container_get_object(controls->mydata,
                                    gwy_app_get_data_key_for_id(0));
    xoff -= gwy_data_field_get_xoffset(psdf);
    yoff -= gwy_data_field_get_yoffset(psdf);
    gwy_selection_move(controls->selection, xoff, yoff);
}

static void
resolution_changed(PSDFControls *controls, GtkAdjustment *adj)
{
    controls->args->resolution = gwy_adjustment_get_int(adj);
    /* Resolution can be changed only when fixres == TRUE */
    selection_changed(controls, -1);
}

static void
thickness_changed(PSDFControls *controls, GtkAdjustment *adj)
{
    controls->args->thickness = gwy_adjustment_get_int(adj);
    selection_changed(controls, -1);
}

static void
fixres_changed(PSDFControls *controls, GtkToggleButton *check)
{
    controls->args->fixres = gtk_toggle_button_get_active(check);
    selection_changed(controls, -1);
}

static void
interpolation_changed(GtkComboBox *combo, PSDFControls *controls)
{
    controls->args->interpolation = gwy_enum_combo_box_get_active(combo);
    selection_changed(controls, -1);
}

static void
separate_changed(PSDFControls *controls)
{
    PSDFArgs *args = controls->args;
    args->separate
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(controls->separate));
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->target_graph),
                                   !args->separate);
}

static void
windowing_changed(PSDFControls *controls, GtkComboBox *combo)
{
    controls->args->windowing = gwy_enum_combo_box_get_active(combo);
    recalculate_psdf(controls);
}

static void
recalculate_psdf(PSDFControls *controls)
{
    calculate_psdf(controls->dfield, controls->mask, controls->args,
                   controls->psdf, controls->modulus);
    calculate_zoomed_fields(controls);
    selection_changed(controls, -1);
}

static void
selection_changed(PSDFControls *controls, gint hint)
{
    gint i, n;

    n = gwy_selection_get_data(controls->selection, NULL);
    if (hint < 0) {
        gwy_graph_model_remove_all_curves(controls->gmodel);
        for (i = 0; i < n; i++)
            update_curve(controls, i);
    }
    else
        update_curve(controls, hint);

    update_sensitivity(controls);
}

static void
update_sensitivity(PSDFControls *controls)
{
    PSDFArgs *args = controls->args;
    gint n;

    n = gwy_graph_model_get_n_curves(controls->gmodel);
    gtk_widget_set_sensitive(controls->zoomed_image, args->create_image);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialogue),
                                      GTK_RESPONSE_OK,
                                      args->create_image || n);
}

static void
target_graph_changed(PSDFControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);

    gwy_data_chooser_get_active_id(chooser, &controls->args->target_graph);
}

static void
calculate_zoomed_fields(PSDFControls *controls)
{
    ZoomType zoom = controls->args->zoom;
    GwyDataField *zoomed;

    zoomed = cut_field_to_zoom(controls->modulus, zoom);
    gwy_container_set_object(controls->mydata,
                             gwy_app_get_data_key_for_id(0), zoomed);
    gwy_data_field_data_changed(zoomed);
    g_object_unref(zoomed);
}

static GwyDataField*
cut_field_to_zoom(GwyDataField *dfield, gint zoom)
{
    GwyDataField *zoomed;
    guint xres, yres, width, height;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    width = (xres/zoom) | 1;
    height = (yres/zoom) | 1;
    if (width < 17)
        width = MAX(width, MIN(17, xres));
    if (height < 17)
        height = MAX(height, MIN(17, yres));

    if (width >= xres && height >= yres)
        return g_object_ref(dfield);

    zoomed = gwy_data_field_area_extract(dfield,
                                         xres - width - (xres - width)/2,
                                         yres - height - (yres - height)/2,
                                         width, height);
    gwy_data_field_set_xoffset(zoomed, -0.5*gwy_data_field_get_xreal(zoomed));
    gwy_data_field_set_yoffset(zoomed, -0.5*gwy_data_field_get_yreal(zoomed));

    return zoomed;
}

static void
update_curve(PSDFControls *controls, gint i)
{
    GwyGraphCurveModel *gcmodel;
    GwyDataField *modulus;
    gdouble xy[4], h, xofffull, yofffull;
    gint xl0, yl0, xl1, yl1;
    gint n, lineres;
    gchar *desc;

    if (!gwy_selection_get_object(controls->selection, i, xy)) {
        g_return_if_reached();
    }

    modulus = gwy_container_get_object(controls->mydata,
                                       gwy_app_get_data_key_for_id(0));
    xy[0] += gwy_data_field_get_xoffset(modulus);
    xy[1] += gwy_data_field_get_yoffset(modulus);

    /* The ω=0 pixel is always at res/2, for even dimensions it means it is
     * shifted half-a-pixel to the right from the precise centre. */
    xl0 = gwy_data_field_get_xres(controls->psdf)/2;
    yl0 = gwy_data_field_get_yres(controls->psdf)/2;

    xofffull = gwy_data_field_get_xoffset(controls->psdf);
    yofffull = gwy_data_field_get_yoffset(controls->psdf);
    xl1 = floor(gwy_data_field_rtoj(controls->psdf, xy[0] - xofffull));
    yl1 = floor(gwy_data_field_rtoi(controls->psdf, xy[1] - yofffull));

    /* FIXME FIXME FIXME: Is this correct? */
    h = hypot(controls->hx*xy[0], controls->hy*xy[1])/hypot(xy[0], xy[1]);

    if (!controls->args->fixres) {
        lineres = GWY_ROUND(hypot(abs(xl0 - xl1) + 1, abs(yl0 - yl1) + 1));
        lineres = MAX(lineres, MIN_RESOLUTION);
    }
    else
        lineres = controls->args->resolution;

    gwy_data_field_get_profile(controls->psdf, controls->line,
                               xl0, yl0, xl1, yl1,
                               lineres,
                               controls->args->thickness,
                               controls->args->interpolation);
    gwy_data_line_multiply(controls->line, h);

    n = gwy_graph_model_get_n_curves(controls->gmodel);
    if (i < n) {
        gcmodel = gwy_graph_model_get_curve(controls->gmodel, i);
    }
    else {
        gcmodel = gwy_graph_curve_model_new();
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "color", gwy_graph_get_preset_color(i),
                     NULL);
        gwy_graph_model_add_curve(controls->gmodel, gcmodel);
        g_object_unref(gcmodel);
    }

    gwy_graph_curve_model_set_data_from_dataline(gcmodel, controls->line, 0, 0);
    desc = g_strdup_printf(_("PSDF %.0f°"), 180.0/G_PI*atan2(-xy[1], xy[0]));
    g_object_set(gcmodel, "description", desc, NULL);
    g_free(desc);
}

static void
init_graph_model_units(GwyGraphModel *gmodel,
                       GwyDataField *dfield)
{
    GwySIUnit *unit;

    unit = gwy_data_field_get_si_unit_xy(dfield);
    unit = gwy_si_unit_duplicate(unit);
    g_object_set(gmodel, "si-unit-x", unit, NULL);
    g_object_unref(unit);

    unit = gwy_data_field_get_si_unit_z(dfield);
    unit = gwy_si_unit_duplicate(unit);
    g_object_set(gmodel, "si-unit-y", unit, NULL);
    g_object_unref(unit);
}

static void
create_output_graphs(PSDFControls *controls, GwyContainer *data)
{
    GwyGraphCurveModel *gcmodel;
    GwyGraphModel *gmodel;
    PSDFArgs *args = controls->args;
    gchar *s;
    gint i, n;

    n = gwy_selection_get_data(controls->selection, NULL);
    if (!n)
        return;

    if (!args->separate) {
        gwy_app_add_graph_or_curves(controls->gmodel,
                                    data, &args->target_graph, 1);
        return;
    }

    for (i = 0; i < n; i++) {
        gmodel = gwy_graph_model_new_alike(controls->gmodel);
        gcmodel = gwy_graph_model_get_curve(controls->gmodel, i);
        gcmodel = gwy_graph_curve_model_duplicate(gcmodel);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
        g_object_get(gcmodel, "description", &s, NULL);
        g_object_set(gmodel, "title", s, NULL);
        g_free(s);
        gwy_app_data_browser_add_graph_model(gmodel, data, TRUE);
        g_object_unref(gmodel);
    }
}

/* Convert points to lines from the origin, which is assumed to be in the
 * centre. */
static void
add_line_selection_from_points(GwyContainer *data,
                               GwyDataField *dfield,
                               gint id,
                               GwySelection *pointsel)
{
    GwySelection *linesel;
    gint nsel, i;
    gdouble *seldata;
    gdouble xreal, yreal;
    gchar *key;

    if (!pointsel || !(nsel = gwy_selection_get_data(pointsel, NULL)))
        return;

    linesel = g_object_new(g_type_from_name("GwySelectionLine"), NULL);
    g_return_if_fail(linesel);
    gwy_selection_set_max_objects(linesel, 1024);
    seldata = g_new(gdouble, 4*nsel);
    xreal = gwy_data_field_get_xreal(dfield);
    yreal = gwy_data_field_get_yreal(dfield);

    for (i = 0; i < nsel; i++) {
        seldata[4*i + 0] = 0.5*xreal;
        seldata[4*i + 1] = 0.5*yreal;
        gwy_selection_get_object(pointsel, i, seldata + 4*i + 2);
    }

    gwy_selection_set_data(linesel, nsel, seldata);
    g_free(seldata);

    key = g_strdup_printf("/%d/select/line", id);
    gwy_container_set_object_by_name(data, key, linesel);
    g_object_unref(linesel);
}

static gint
create_results(GwyContainer *data, GwyDataField *psdf,
               GwySelection *selection,
               gint oldid, gint zoom)
{
    GwyDataField *zoomed;
    GQuark quark;
    gint newid;

    zoomed = cut_field_to_zoom(psdf, zoom);
    newid = gwy_app_data_browser_add_data_field(zoomed, data, TRUE);

    add_line_selection_from_points(data, zoomed, newid, selection);
    gwy_app_set_data_field_title(data, newid, _("2D PSDF"));
    quark = gwy_app_get_data_palette_key_for_id(newid);
    gwy_container_set_const_string(data, quark, "DFit");
    quark = gwy_app_get_data_range_type_key_for_id(newid);
    gwy_container_set_enum(data, quark, GWY_LAYER_BASIC_RANGE_AUTO);
    gwy_app_channel_log_add_proc(data, oldid, newid);

    g_object_unref(zoomed);

    return newid;
}

static void
calculate_psdf(GwyDataField *dfield, GwyDataField *mask,
               const PSDFArgs *args,
               GwyDataField *psdf, GwyDataField *modulus)
{
    guint xres, yres, n, i;
    gdouble *p;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    n = xres*yres;
    gwy_data_field_area_2dpsdf_mask(dfield, psdf, mask, args->masking,
                                    0, 0, xres, yres, args->windowing, 1);

    /* We do not really care about modulus units nor its absolute scale.
     * We just have it to display the square root... */
    gwy_data_field_assign(modulus, psdf);
    p = gwy_data_field_get_data(modulus);
#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(p,n) private(i)
#endif
    for (i = 0; i < n; i++)
        p[i] = p[i] >= 0.0 ? sqrt(p[i]) : -sqrt(-p[i]);
}

static const gchar create_image_key[]  = "/module/psdf2d/create_image";
static const gchar fixres_key[]        = "/module/psdf2d/fixres";
static const gchar interpolation_key[] = "/module/psdf2d/interpolation";
static const gchar masking_key[]       = "/module/psdf2d/masking";
static const gchar resolution_key[]    = "/module/psdf2d/resolution";
static const gchar separate_key[]      = "/module/psdf2d/separate";
static const gchar thickness_key[]     = "/module/psdf2d/thickness";
static const gchar windowing_key[]     = "/module/psdf2d/windowing";
static const gchar zoom_key[]          = "/module/psdf2d/zoom";
static const gchar zoomed_image_key[]  = "/module/psdf2d/zoomed_image";

static void
psdf2d_sanitize_args(PSDFArgs *args)
{
    args->masking = gwy_enum_sanitize_value(args->masking,
                                            GWY_TYPE_MASKING_TYPE);
    args->create_image = !!args->create_image;
    args->zoomed_image = !!args->zoomed_image;
    args->separate = !!args->separate;
    args->fixres = !!args->fixres;
    if (args->zoom != ZOOM_1 && args->zoom != ZOOM_4 && args->zoom != ZOOM_16)
        args->zoom = psdf2d_defaults.zoom;
    args->resolution = CLAMP(args->resolution, MIN_RESOLUTION, MAX_RESOLUTION);
    args->thickness = CLAMP(args->thickness, 1, MAX_THICKNESS);
    args->interpolation = gwy_enum_sanitize_value(args->interpolation,
                                                  GWY_TYPE_INTERPOLATION_TYPE);
    args->windowing = gwy_enum_sanitize_value(args->windowing,
                                              GWY_TYPE_WINDOWING_TYPE);
    gwy_app_data_id_verify_graph(&args->target_graph);
}

static void
psdf2d_load_args(GwyContainer *container,
                 PSDFArgs *args)
{
    *args = psdf2d_defaults;

    gwy_container_gis_enum_by_name(container, masking_key, &args->masking);
    gwy_container_gis_boolean_by_name(container, create_image_key,
                                      &args->create_image);
    gwy_container_gis_boolean_by_name(container, zoomed_image_key,
                                      &args->zoomed_image);
    gwy_container_gis_boolean_by_name(container, separate_key, &args->separate);
    gwy_container_gis_boolean_by_name(container, fixres_key, &args->fixres);
    gwy_container_gis_int32_by_name(container, resolution_key,
                                    &args->resolution);
    gwy_container_gis_int32_by_name(container, thickness_key, &args->thickness);
    gwy_container_gis_enum_by_name(container, interpolation_key,
                                   &args->interpolation);
    gwy_container_gis_enum_by_name(container, windowing_key, &args->windowing);
    gwy_container_gis_enum_by_name(container, zoom_key, &args->zoom);
    args->target_graph = target_id;
    psdf2d_sanitize_args(args);
}

static void
psdf2d_save_args(GwyContainer *container,
                 PSDFArgs *args)
{
    gwy_container_set_enum_by_name(container, masking_key, args->masking);
    gwy_container_set_boolean_by_name(container, create_image_key,
                                      args->create_image);
    gwy_container_set_boolean_by_name(container, zoomed_image_key,
                                      args->zoomed_image);
    gwy_container_set_boolean_by_name(container, separate_key, args->separate);
    gwy_container_set_boolean_by_name(container, fixres_key, args->fixres);
    gwy_container_set_int32_by_name(container, resolution_key,
                                    args->resolution);
    gwy_container_set_int32_by_name(container, thickness_key, args->thickness);
    gwy_container_set_enum_by_name(container, interpolation_key,
                                   args->interpolation);
    gwy_container_set_enum_by_name(container, windowing_key, args->windowing);
    gwy_container_set_enum_by_name(container, zoom_key, args->zoom);
    target_id = args->target_graph;
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
