/*  This file is part of the libdepixelize project
    Copyright (C) 2013 Vinícius dos Santos Oliveira <vini.ipsmaker@gmail.com>

    GNU Lesser General Public License Usage
    This library is free software; you can redistribute it and/or modify it
    under the terms of the GNU Lesser General Public License as published by the
    Free Software Foundation; either version 2.1 of the License, or (at your
    option) any later version.
    You should have received a copy of the GNU Lesser General Public License
    along with this library.  If not, see <http://www.gnu.org/licenses/>.

    GNU General Public License Usage
    Alternatively, this library may be used under the terms of the GNU General
    Public License as published by the Free Software Foundation, either version
    2 of the License, or (at your option) any later version.
    You should have received a copy of the GNU General Public License along with
    this library.  If not, see <http://www.gnu.org/licenses/>.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.
*/

#ifndef LIBDEPIXELIZE_TRACER_CURVATURE_H
#define LIBDEPIXELIZE_TRACER_CURVATURE_H

#include "point.h"
#include <cmath>

namespace Tracer {

/**
 * Curvature function for a quadratic Bézier curve where the points are know.
 * Its main use is to numerically integrate the curvature function and then
 * smooth the B-Splines generated by the Kopf-Lischinski algorithm.
 */
template<class T>
struct Curvature
{
    Curvature(Point<T> p0, Point<T> c1, Point<T> p2) :
        p0(p0), c1(c1), p2(p2)
    {}

    T operator()(T t) const;

    /**
     * The derivative of x
     */
    T xPrime(T t) const;

    /**
     * The derivative of y
     */
    T yPrime(T t) const;

    /**
     * The second derivative of x
     */
    T xPrimePrime() const;

    /**
     * The second derivative of y
     */
    T yPrimePrime() const;

    Point<T> p0, c1, p2;
};

template<class T>
T Curvature<T>::operator()(T t) const
{
    T num = xPrime(t) * yPrimePrime() - yPrime(t) * xPrimePrime();
    T den = std::pow(xPrime(t) * xPrime(t) + yPrime(t) * yPrime(t), T(3) / 2);
    return num / den;
}

template<class T>
T Curvature<T>::xPrime(T t) const
{
    return (1-t)*2*(c1.x-p0.x) + t*2*(p2.x-c1.x);
}

template<class T>
T Curvature<T>::yPrime(T t) const
{
    return (1-t)*2*(c1.y-p0.y) + t*2*(p2.y-c1.y);
}

template<class T>
T Curvature<T>::xPrimePrime() const
{
    return 2 * (p2.x - 2*c1.x + p0.x);
}

template<class T>
T Curvature<T>::yPrimePrime() const
{
    return 2 * (p2.y - 2*c1.y + p0.y);
}

} // namespace Tracer

#endif // LIBDEPIXELIZE_TRACER_CURVATURE_H

/*
  Local Variables:
  mode:c++
  c-file-style:"stroustrup"
  c-file-offsets:((innamespace . 0)(inline-open . 0)(case-label . +))
  indent-tabs-mode:nil
  fill-column:99
  End:
*/
// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=8:softtabstop=4:encoding=utf-8:textwidth=99 :
